C---------------------------------------------------------------------
      PROGRAM main
C---------------------------------------------------------------------
C     12/01/13/ Aitbala Sargent and James Fastook
C        Solves 2-D shelf equations using
C          (a) direct linear algorithm and
C          (b) traditional iterative non-linear algorithm.
C     Ice parameters: rho, rhow, n, g, and
C     model parameters: wx,wy,U0,tana,k1,k2,Lx,LY and
C     scale parameter: xs, zs, us, taus
C         - are assigned in "parameter2d.h".
C
C     Linear and non-linear algorithm solutions are dumped to files
C          2du.data,2dv.data,2dtaux.data,2dtauy.data,etc.
C     Time of calculations and errors of the algorithms are dumpled to files
C          2dtime.data and 2derr.data.
C---------------------------------------------------------------------
      IMPLICIT none
      INTEGER nx,ny
      print *,'enter the size of the problem, nx'
        read *,nx
      print *,'enter the size of the problem, ny'
        read *,ny
      call algorithm(nx,ny)
      STOP
      END
C---------------------------------------------------------------------
      SUBROUTINE algorithm(nx,ny)
C---------------------------------------------------------------------
C      - creates/opens files to record the time and errors (2dtime.data, 2derr.data)
C      - calls EXACT to calculate the manufactured analytical solution
C      - calls direct to calculate solution using new linear algorithm
C      - calls ITERAT to calculate solution using traditional iterative algorithm
C---------------------------------------------------------------------
      IMPLICIT none
      INCLUDE "parameter2d.h"
      DOUBLE PRECISION x(nx),y(nx),u(nx,ny),v(nx,ny)
      DOUBLE PRECISION h(nx,ny),b(nx,ny),s(nx,ny)
      DOUBLE PRECISION xmu(nx,ny),fx(nx,ny),fy(nx,ny)
      DOUBLE PRECISION uex(nx,ny),vex(nx,ny),tauxe(nx,ny),tauye(nx,ny)
      DOUBLE PRECISION taux(nx,ny),tauy(nx,ny)
      DOUBLE PRECISION dsdx,dsdy,erru,errv,errx,erry
      DOUBLE PRECISION errui,errvi,errxi,erryi
      INTEGER nx,ny,iter
      REAL dtime,ta(2),t1,t2
      LOGICAL existtime,existerr
C ---
C   create or open files to write the errors and time of calculation of algorithms.
C ---
      INQUIRE(FILE='2dtime.data',EXIST=existtime)
      if(existtime) then
        OPEN(20,file='2dtime.data',status='old',access='append')
      else
        OPEN(20,file='2dtime.data',status='new',access='append')
      endif
      INQUIRE(FILE='2derr.data',EXIST=existerr)
      if(existerr) then
        OPEN(21,file='2derr.data',status='old',access='append')
      else
        OPEN(21,file='2derr.data',status='new',access='append')
      endif
C ---
C     CALL mesh(nx,ny)   ! write grid nodes for graphing
C ---
C --- calculate manufactured analytical solution of 2d shelf equations
C ---
      CALL EXACT(nx,ny,x,y,uex,vex,tauxe,tauye,
     &           h,b,s,dsdx,dsdy,xmu,fx,fy)
      print *,'grid size = (',nx,'x',ny,')'
C ---
C ---- solve 2d shelf model using new linear algorithm -------------
C ---
      t1 = dtime(ta)
      CALL direct(nx,ny,x,y,u,v,taux,tauy,h,s,dsdx,dsdy,fx,fy,xmu,
     & uex,vex,tauxe,tauye,erru,errv,errx,erry)
      t1 = dtime(ta)
      print *,'Direct method took:',t1,' seconds.'
      print *,'erru=',erru,' errv=',errv,
     & ' errtaux=',errx,' errtauy=',erry
C ---
C ---- solve 2d shelf model using traditional non-linear algorithm ---
C ---
      t2 = dtime(ta)
      CALL ITERAT(nx,ny,x,y,u,v,taux,tauy,h,s,fx,fy,
     &    uex,vex,tauxe,tauye,errui,errvi,errxi,erryi,iter)
      t2 = dtime(ta)
      print *,'Iterative method took:',t2,' seconds.'
      print *,'errui=',errui,' errvi=',errvi,
     & ' errtauxi=',errxi,' errtauyi=',erryi,' iter=',iter
      write(20,10) nx,t1,t2
      close(20)
      write(21,10) nx,erru,errv,errx,erry,
     & errui,errvi,errxi,erryi,iter
      close(21)
10    format(i6,2x,8(e11.5,2x),i6)
C ---
C --- dump the calculated solutions
C ---
      CALL dump(nx,ny,x,y,u,v,taux,tauy,h,b,s,dsdx,dsdy,xmu,fx,fy,
     &          uex,vex,tauxe,tauye)
      RETURN
      END
C---------------------------------------------------------------------
      SUBROUTINE dump(nx,ny,x,y,u,v,taux,tauy,h,b,s,dsdx,dsdy,
     &          xmu,fx,fy,uex,vex,tauxe,tauye)
C---------------------------------------------------------------------
C     Dumps values of exact and calculated solutions 
C---------------------------------------------------------------------
      IMPLICIT none
      INCLUDE "parameter2d.h"
      DOUBLE PRECISION x(nx),y(nx),h(nx,ny),b(nx,ny),s(nx,ny)
      DOUBLE PRECISION xmu(nx,ny),fx(nx,ny),fy(nx,ny)
      DOUBLE PRECISION uex(nx,ny),vex(nx,ny),tauxe(nx,ny),tauye(nx,ny)
      DOUBLE PRECISION u(nx,ny),v(nx,ny),taux(nx,ny),tauy(nx,ny)
      DOUBLE PRECISION dsdx,dsdy,dx,dy,xx,yy
      INTEGER nx,ny,i,j
          OPEN(11,file='2db.data')
          OPEN(12,file='2ds.data')
          OPEN(13,file='2du.data')
          OPEN(14,file='2dv.data')
          OPEN(15,file='2dmux.data')
          OPEN(16,file='2dfx.data')
          OPEN(17,file='2dfy.data')
          OPEN(20,file='2dtaux.data')
          OPEN(21,file='2dtauy.data')
          OPEN(22,file='2due.data')
          OPEN(23,file='2dve.data')
          OPEN(24,file='2dtauxe.data')
          OPEN(25,file='2dtauye.data')
          OPEN(26,file='2dhe.data')
          DO i=1,nx
          DO j=1,ny
            xx=x(i)*xs
            yy=y(j)*xs
            WRITE(11,*) xx,yy,b(i,j)*zs
            WRITE(12,*) xx,yy,s(i,j)*zs
            WRITE(26,*) xx,yy,h(i,j)*zs
            WRITE(13,*) xx,yy,u(i,j)*us
            WRITE(14,*) xx,yy,v(i,j)*us
            WRITE(15,*) xx,yy,xmu(i,j)
            WRITE(16,*) xx,yy,fx(i,j)
            WRITE(17,*) xx,yy,fy(i,j)
            WRITE(22,*) xx,yy,uex(i,j)*us
            WRITE(23,*) xx,yy,vex(i,j)*us
          ENDDO
          ENDDO
          dx = x(2)-x(1)
          dy = y(2)-y(1)
          DO i=1,nx-1
          DO j=1,ny-1
            xx=(x(i)+dx)*xs
            yy=(y(j)+dy)*xs
            WRITE(20,*) xx,yy,taux(i,j)*taus
            WRITE(21,*) xx,yy,tauy(i,j)*taus
            WRITE(24,*) xx,yy,tauxe(i,j)*taus
            WRITE(25,*) xx,yy,tauye(i,j)*taus
          ENDDO
          ENDDO
        DO i=11,25
          CLOSE(i)
        ENDDO
      END
C---------------------------------------------------------------------
      SUBROUTINE EXACT(nx,ny,x,y,u,v,taux,tauy,
     &                 h,b,s,dsdx,dsdy,xmu,fx,fy)
C---------------------------------------------------------------------
C     Calculates manufactured analytical solutions of 2D shelf equations
C---------------------------------------------------------------------
      IMPLICIT none
      INCLUDE "parameter2d.h"
      DOUBLE PRECISION x(nx),y(nx),u(nx,ny),v(nx,ny)
      DOUBLE PRECISION h(nx,ny),b(nx,ny),s(nx,ny)
      DOUBLE PRECISION xmu(nx,ny),fx(nx,ny),fy(nx,ny)
      DOUBLE PRECISION taux(nx,ny),tauy(nx,ny)
      DOUBLE PRECISION sinx,cosx,sin2x,cos2x,cosx2
      DOUBLE PRECISION siny,cosy,sin2y,cos2y,cosy2
      DOUBLE PRECISION dudx,dudy,d2udx2,d2udy2,d2udxdy
      DOUBLE PRECISION dvdx,dvdy,d2vdx2,d2vdy2,d2vdxdy
      DOUBLE PRECISION dhdx,dhdy,d2hdx2,d2hdy2,d2hdxdy
      DOUBLE PRECISION dsdx,dsdy,dx,dy,xx,yy,hh,V0
      DOUBLE PRECISION xnu,ee,xmuu,dmudx,dmudy
      INTEGER nx,ny,i,j,ip,jp
C ---
        dx=Lx/(nx-1)
        dy=Ly/(ny-1)
        V0 = U0*wx/wy
C ---
C---  calculate velocity, bed, surface, ice thickness,
C---      and RHS functions,fx and fy, at the nodes of the grid.
C ---
      do i=1,nx
        xx = (i-1)*dx
        x(i) = xx
        sinx   = sin(wx*xx)
        cosx   = cos(wx*xx)
        cosx2  = cosx*cosx
        sin2x  = 2*sinx*cosx
        cos2x  = cosx*cosx-sinx*sinx
      do j=1,ny
        yy = (j-1)*dy
        y(j) = yy
        siny   = sin(wy*yy)
        cosy   = cos(wy*yy)
        cosy2  = cosy*cosy
        sin2y  = 2*siny*cosy
        cos2y  = cosy*cosy-siny*siny
        u(i,j) = U0*cosx*siny
        v(i,j) = -V0*sinx*cosy
        s(i,j) = -delta*tana*xx
        hh = k1*cosx2*cosy2+k2
        h(i,j) = hh
        b(i,j) = s(i,j)-hh
c......... derivatives ...........
        dudx    = -U0*wx*sinx*siny
        dudy    = U0*wy*cosx*cosy
        d2udx2  = -wx**2*u(i,j)
        d2udy2  = -wy*wy*u(i,j)
        d2udxdy = -U0*wy*wx*sinx*cosy
          dvdx    = -V0*wx*cosx*cosy
          dvdy    = V0*wy*sinx*siny
          d2vdx2  = -wx*wx*v(i,j)
          d2vdy2  = -wy*wy*v(i,j)
          d2vdxdy = V0*wx*wy*cosx*siny
       dsdx = -delta*tana
       dsdy = 0.d0
          dhdx = -k1*wx*sin2x*cosy2
          dhdy = -k1*wy*cosx2*sin2y
          d2hdx2 = -2*k1*wx*wx*cos2x*cosy2
          d2hdy2 = -2*k1*wy*wy*cosx2*cos2y
          d2hdxdy = k1*wx*wy*sin2x*sin2y
c......... xmu ....................
       xnu = dudx*dudx+dvdy*dvdy+dudx*dvdy
     &     + 0.25d0*(dudy+dvdx)*(dudy+dvdx)
       ee = (1.-n)/(2.*n)
       xmuu = xnu**ee
       xmu(i,j) = xmuu
       dmudx = ee*xmuu/xnu
     &       * (d2udx2*(2*dudx+dvdy)+d2vdxdy*(2*dvdy+dudx)
     &         +0.5*(d2udxdy+d2vdx2)*(dudy+dvdx))
       dmudy = ee*xmuu/xnu
     &       * (d2vdy2*(2*dvdy+dudx)+d2udxdy*(2*dudx+dvdy)
     &         +0.5*(d2vdxdy+d2udy2)*(dudy+dvdx))
c.......... rhs ...................
       fx(i,j) = 2*(hh*dmudx+xmuu*dhdx)*(2*dudx+dvdy)
     &         + 2*xmuu*hh*(2*d2udx2+d2vdxdy)
     &         + (hh*dmudy+xmuu*dhdy)*(dudy+dvdx)
     &         + xmuu*hh*(d2udy2+d2vdxdy)-hh*dsdx
       fy(i,j) = 2*(hh*dmudy+xmuu*dhdy)*(2*dvdy+dudx)
     &         + 2*xmuu*hh*(2*d2vdy2+d2udxdy)
     &         + (hh*dmudx+xmuu*dhdx)*(dudy+dvdx)
     &         + xmuu*hh*(d2vdx2+d2udxdy)-hh*dsdy
      enddo
      enddo
C ---
C---  calculate stress components, taux and tauy, at the centroids of the grid.
C ---
      do i=1,nx-1
        ip = i+1
      do j=1,ny-1
        jp = j+1
        xmuu = 0.25*(xmu(i,j)+xmu(ip,j)+xmu(i,jp)+xmu(ip,jp))
        hh = 0.25*(h(i,j)+h(ip,j)+h(i,jp)+h(ip,jp))
        dudx = 0.5*(u(ip,jp)+u(ip,j)-u(i,jp)-u(i,j))/dx
        dvdx = 0.5*(v(ip,jp)+v(ip,j)-v(i,jp)-v(i,j))/dx
        dudy = 0.5*(u(ip,jp)+u(i,jp)-u(ip,j)-u(i,j))/dy
        taux(i,j) = 2*xmuu*hh*dudx
        tauy(i,j) = xmuu*hh*(dudy+dvdx)
      enddo
      enddo
      RETURN
      END
C---------------------------------------------------------------------
      SUBROUTINE mesh(nx,ny)
C---------------------------------------------------------------------
C     builds grid to construct graphs
C---------------------------------------------------------------------
      IMPLICIT none
      INTEGER nx,ny,i,j,node1,node2,node3,node4
        OPEN(95,file='mesh.ijk')
        DO i=1,nx-1
        DO j=1,ny-1
          node1=(j-1)*nx+i
          node4=node1+1
          node2=node1+ny
          node3=node2+1
          write(95,*) node1-1,node2-1,node4-1
          write(95,*) node2-1,node3-1,node4-1
        ENDDO
        ENDDO
        CLOSE(95)
       RETURN
       END
C---------------------------------------------------------------------
      SUBROUTINE direct(nx,ny,x,y,u,v,taux,tauy,h,s,dsdx,dsdy,fx,fy,xmu,
     & uex,vex,tauxe,tauye,erru,errv,errx,erry) ! calc. u,v and taux,tauy
C 
C       solves 2d shelf model using new linear algorithm
C---------------------------------------------------------------------
      IMPLICIT none
      INCLUDE "parameter2d.h"
      DOUBLE PRECISION Lh,dx,dy,dx2,dy2,dx2inv,avk,avh,alh,alk
      DOUBLE PRECISION fxt,fxb,fxr,fxl,fyt,fyb,fyr,fyl,sinx,sina,yy
      DOUBLE PRECISION phii,phjj,denom,errx,erry,erru,errv,ttx,tty
      DOUBLE PRECISION fxij,fxijm,aij,hij,hij2,hijm2,tauxh,tauyh
      DOUBLE PRECISION dsdx,dsdy
      DOUBLE PRECISION x(nx),y(nx),h(nx,ny),s(nx,ny),xmu(nx,ny)
      DOUBLE PRECISION fx(nx,ny),fy(nx,ny),ffx(nx,ny),ffy(nx,ny)
      DOUBLE PRECISION uex(nx,ny),vex(nx,ny),tauxe(nx,ny),tauye(nx,ny)
      DOUBLE PRECISION u(nx,ny),v(nx,ny),taux(nx,ny),tauy(nx,ny)
      DOUBLE PRECISION phix(nx,ny),phiy(nx,ny),al(nx),a(nx),b(nx),c(nx)
     &  ,alf(nx),betx(nx),bety(nx),ckx(nx,ny),cky(nx,ny),vk(nx,ny)
      INTEGER nx,ny,i,ip,im,ip2,im2,j,jp,jm,jp2,jm2,k,ll,llm
C
        dx=Lx/(nx-1)
        dy=Ly/(ny-1)
        dx2 = dx*dx
        dy2 = dy*dy
        dx2inv = 1./dx2
C---
C---  define Right-Hand Side function: 
C---    ff = - d[h ds/dx +fx]/dx + d[h ds/dy + fy]/dy 
C---
      do i=1,nx-1
        ip = i+1
        im = i-1
        ip2= i+2
        im2= i-2
      do j=1,ny-1
        jp = j+1
        jm = j-1
        jp2= j+2
        jm2= j-2
       fxt = 0.5*(h(ip,jp)+h(i,jp))*( s(ip,jp)-s(i,jp))/dx
     &       +0.5*(fx(ip,jp)+fx(i,jp)) 
       fxb = 0.5*(h(ip,j)+h(i,j))*( s(ip,j)-s(i,j))/dx
     &       +0.5*(fx(ip,j)+fx(i,j)) 
        fyr = 0.5*(h(ip,jp)+h(ip,j))*( s(ip,jp)-s(ip,j) )/dy
     &       +0.5*(fy(ip,jp)+fy(ip,j)) 
        fyl = 0.5*(h(i,jp)+h(i,j))*( s(i,jp)-s(i,j) )/dy
     &       +0.5*(fy(i,jp)+fy(i,j)) 
       if(i.eq.nx-1) then
        fxr = 0.5*(h(i,j)+h(i,jp))*
     &       0.5*(3.*s(i,jp)-4.*s(im,jp)+s(im2,jp)
     &           +3.*s(i,j)-4.*s(im,j)+s(im2,j) )/(2.*dx)
       else
        fxr = 0.5*(h(ip,j)+h(ip,jp))*
     &       0.5*( s(ip2,jp)-s(i,jp)+s(ip2,j)-s(i,j) )/(2.*dx)
     &       +0.5*(fx(ip,j)+fx(ip,jp)) 
       endif
       if(i.eq.1) then
        fxl = 0.5*(h(i,j)+h(i,jp))*
     &       0.5*(-3.*s(i,jp)+4.*s(ip,jp)-s(ip2,jp)
     &           -3.*s(i,j)+4.*s(ip,j)-s(ip2,j) )/(2.*dx)
       else
        fxl = 0.5*(h(i,j)+h(i,jp))*
     &       0.5*( s(ip,jp)-s(im,jp)+s(ip,j)-s(im,j) )/(2.*dx)
     &       +0.5*(fx(i,j)+fx(i,jp)) 
       endif
       if(j.eq.ny-1) then
        fyt = 0.5*(h(ip,jp)+h(i,jp))*
     &       0.5*( 3.*s(ip,j)-4.*s(ip,jm)+s(ip,jm2)
     &            +3.*s(i,j)-4.*s(i,jm)+s(i,jm2) )/(2.*dy)
     &       +0.5*(fy(ip,jp)+fy(i,jp)) 
       else
        fyt = 0.5*(h(ip,jp)+h(i,jp))*
     &       0.5*( s(ip,jp2)-s(ip,j)+s(i,jp2)-s(i,j) )/(2.*dy)
     &       +0.5*(fy(ip,jp)+fy(i,jp)) 
       endif
       if(j.eq.1) then
        fyb = 0.5*(h(ip,j)+h(i,j))*
     &       0.5*(-3.*s(ip,j)+4.*s(ip,jp)-s(ip,jp2)
     &            -3.*s(i,j)+4.*s(i,jp)-s(i,jp2) )/(2.*dy)
     &       +0.5*(fy(ip,j)+fy(i,j)) 
       else
        fyb = 0.5*(h(ip,j)+h(i,j))*
     &       0.5*( s(ip,jp)-s(ip,jm)+s(i,jp)-s(i,jm) )/(2.*dy)
     &       +0.5*(fy(ip,j)+fy(i,j)) 
       endif
        ffx(i,j) = (fxr-fxl)/dx - (fyt-fyb)/dy
        ffy(i,j) = (fxt-fxb)/dy + (fyr-fyl)/dx
      enddo
      enddo
C---
C---  adjust the bordering boundary conditions
C---
      do j=1,ny-1
        i = 2
        ffx(i,j) = ffx(i,j)-tauxe(i-1,j)/dx2
        ffy(i,j) = ffy(i,j)-tauye(i-1,j)/dx2
        i = nx-2
        ffx(i,j) = ffx(i,j)-tauxe(i+1,j)/dx2
        ffy(i,j) = ffy(i,j)-tauye(i+1,j)/dx2
      enddo
      do i=1,nx-1
        j = 2
        ffx(i,j) = ffx(i,j)-tauxe(i,j-1)/dy2
        ffy(i,j) = ffy(i,j)-tauye(i,j-1)/dy2
        j = ny-2
        ffx(i,j) = ffx(i,j)-tauxe(i,j+1)/dy2
        ffy(i,j) = ffy(i,j)-tauye(i,j+1)/dy2
      enddo
C---
C---  Solve tridiagonal equations for j=2, ..., ny-2
C---
      Lh = Ly-dy
      avk = sqrt(2.d0/Lh)
      avh = pi/Lh
      alh = pi*dy/(2.d0*Lh)
      alk = 4.d0/dy2
C---
C---  1. calculate al(k) and vk(k,y)
C---
      do k=2,ny-2
        sinx=sin(alh*(k-1))
        al(k)=alk*sinx*sinx
      do j=2,ny-2
         yy = (j-1)*dy
         sina = sin(avh*(k-1)*yy)
         vk(k,j) = avk*sin(avh*(k-1)*yy)
      enddo
      enddo
C---
C---  2. calculate phix(k,x) and phiy(k,x) - the RHS functions
C---
      do i=2,nx-2
        do k=2,ny-2
          phii = 0.d0
          phjj = 0.d0
          do j=2,ny-2
            phii = phii+ffx(i,j)*vk(k,j)
            phjj = phjj+ffy(i,j)*vk(k,j)
          enddo
          phix(k,i) = dy*phii 
          phiy(k,i) = dy*phjj 
        enddo
      enddo
C---
C---   3. calculate elements of three-diagonal matrix: a, b, and c
C---
      do k=2,ny-2
         b(1) = 1.d0
         c(1) = 0.d0
         phix(k,1) = 0.d0
         phiy(k,1) = 0.d0
         do i=2,nx-2
           a(i-1) = dx2inv
           b(i) = -2.*dx2inv-al(k)
           c(i) = dx2inv
         enddo
         a(nx-2) = 0.d0
         b(nx-1) = 1.d0
         phix(k,nx-1) = 0.d0
         phiy(k,nx-1) = 0.d0
C---
C---   4. solving a system of three-diagonal equations: ckx(i,.) and cky(i,.)
C---
           alf(1) = -c(1)/b(1)
           betx(1) = phix(k,1)/b(1)
           bety(1) = phiy(k,1)/b(1)
           do ll = 2,nx-1
             llm = ll-1
             denom = b(ll)+a(llm)*alf(llm)
             alf(ll) = -c(ll)/denom
             betx(ll) = (phix(k,ll)-a(llm)*betx(llm))/denom
             bety(ll) = (phiy(k,ll)-a(llm)*bety(llm))/denom
           enddo
           ckx(k,nx-1) = betx(ny-1)
           cky(k,nx-1) = bety(ny-1)
           do ll = ny-2,1,-1
             ckx(k,ll) = alf(ll)*ckx(k,ll+1)+betx(ll)
             cky(k,ll) = alf(ll)*cky(k,ll+1)+bety(ll)
           enddo
       enddo
C---
C---   5. calculating taux(i,...) = sum ckx(k,x)*vk(k,y)
C---              and tauy(i,...) = sum cky(k,x)*vk(k,y)
C---
       errx = 0.d0
       erry = 0.d0
       do i=2,nx-2
       do j=2,ny-2
         ttx = 0.d0
         tty = 0.d0
         do k=2,ny-2
           ttx = ttx+ckx(k,i)*vk(k,j)
           tty = tty+cky(k,i)*vk(k,j)
         enddo
         taux(i,j) = ttx
         tauy(i,j) = tty
         errx = errx+(taux(i,j)-tauxe(i,j))*(taux(i,j)-tauxe(i,j))
         erry = erry+(tauy(i,j)-tauye(i,j))*(tauy(i,j)-tauye(i,j))
       enddo       !  end of j-cycle
           taux(i,ny-1) = tauxe(i,ny-1)
           taux(i,1) = tauxe(i,1)
           tauy(i,ny-1) = tauye(i,ny-1)
           tauy(i,1) = tauye(i,1)
       enddo       !  end of i-cycle
       do j=1,nx-1
         taux(1,j) = tauxe(1,j)
         taux(nx-1,j) = tauxe(nx-1,j)
         tauy(1,j) = tauye(1,j)
         tauy(nx-1,j) = tauye(nx-1,j)
       enddo
       errx = sqrt(errx/(nx*ny))
       erry = sqrt(erry/(nx*ny))
C---
C---   Now calculate velocity
C---
C---     assign boundary conditions: u(1,j) and v(1,j)
      do j=1,ny                   ! left border
        u(1,j) = uex(1,j)
        v(1,j) = vex(1,j)
      enddo
C---     assign boundary conditions: u(i,1) and v(i,1)
      do i=1,nx                   ! bottom border
        u(i,1) = uex(i,1)
        v(i,1) = vex(i,1)
      enddo
C---
C---     calculate u(i,j) inside of the domain
C---
      do i=1,nx-1
        ip = i+1
      do j=2,ny-1
        jp = j+1
        jm = j-1
        hij2 = 0.5*(h(i,j)+h(ip,j)+h(i,jp)+h(ip,jp))
        tauyh = tauy(i,j)/hij2
        tauxh = taux(i,j)/hij2
        aij = tauxh*tauxh+tauyh*tauyh
c       if(aij .le. 0.) then
c         aij = -(abs(aij))**((n-1.)/2.)
c       else
c         aij = aij**((n-1.)/2.)
c       endif
        fxij = tauxh*aij
c
        hijm2 = 0.5*(h(i,jm)+h(ip,jm)+h(i,j)+h(ip,j))
        tauyh = tauy(i,jm)/hijm2
        tauxh = taux(i,jm)/hijm2
        aij = tauxh*tauxh+tauyh*tauyh
c       if(aij .le. 0.) then
c         aij = -(abs(aij))**((n-1.)/2.)
c       else
c         aij = aij**((n-1.)/2.)
c       endif
        fxijm = tauxh*aij
c
        u(ip,j)=u(i,j)+dx*0.5*(fxij+fxijm)
      enddo
      enddo
C---
C---     assign boundary conditions: u(i,ny) - top border
C---
      do i=2,nx                 
        u(i,ny) = uex(i,1)
      enddo
C---
C---    calculate v(i,j) for i=2,nx,  j=2,ny-1
C---
      erru = 0.
      errv = 0.
      do i=1,nx-1
        ip = i+1
      do j=2,ny-1
        jp = j+1
        jm = j-1
        hij = 0.25*(h(i,j)+h(ip,j)+h(i,jp)+h(ip,jp))
        hij2 = 2.*hij
        tauyh = tauy(i,j)/hij2
        tauxh = taux(i,j)/hij2
        aij = tauxh*tauxh+tauyh*tauyh
c       if(aij .le. 0.) then
c         aij = -(abs(aij))**((n-1.)/2.)
c       else
c         aij = aij**((n-1.)/2.)
c       endif
        fxij = 2.*tauyh*aij
c
        hij = 0.25*(h(i,jm)+h(ip,jm)+h(i,j)+h(ip,j))
        hij2 = 2.*hij
        tauyh = tauy(i,jm)/hij2
        tauxh = taux(i,jm)/hij2
        aij = tauxh*tauxh+tauyh*tauyh
c       if(aij .le. 0.) then
c         aij = -(abs(aij))**((n-1.)/2.)
c       else
c         aij = aij**((n-1.)/2.)
c       endif
        fxijm = 2.*tauyh*aij
        v(ip,j)=v(i,j)+dx*0.5*(fxij+fxijm)
     &          -0.25*dx/dy*( u(i,jp)-u(i,jm)+u(ip,jp)-u(ip,jm))
        errv = errv+(v(ip,j)-vex(ip,j))*(v(ip,j)-vex(ip,j))
        erru = erru+(u(ip,j)-uex(ip,j))*(u(ip,j)-uex(ip,j))
      enddo
      enddo
C---
C    boundary conditions
C---
      do i=1,nx
        v(i,ny) = vex(i,ny)
      enddo
      do i=1,nx
        taux(i,ny)=taux(i,ny-1)
        tauy(i,ny)=tauy(i,ny-1)
      enddo
      do j=1,ny
        taux(nx,j)=taux(nx-1,j)
        tauy(nx,j)=tauy(nx-1,j)
      enddo
      erru = sqrt(erru/(nx*ny))
      errv = sqrt(errv/(nx*ny))
      RETURN
      END
C---------------------------------------------------------------------
      SUBROUTINE UPDATENU(nx,ny,x,y,u,v,xmu)
C       calculates ice viscocity (xmu)
C---------------------------------------------------------------------
      IMPLICIT none
      INCLUDE "parameter2d.h"
      DOUBLE PRECISION x(nx),y(nx),u(nx,ny),v(nx,ny),xmu(nx,ny)
      DOUBLE PRECISION dx,dy,dx2,dy2,dudx,dvdx,dudy,dvdy,xmuu,eps,exp
      INTEGER nx,ny,i,ip,j,jp
      PARAMETER(eps=1.d-15)
C ---
      exp = (1.d0-n)/(2.d0*n)
      dx = x(2)-x(1)
      dy = y(2)-y(1)
      dx2 = 2.*dx
      dy2 = 2.*dy
      DO i=1,nx-1
        ip = i+1
      DO j=1,ny-1
        jp = j+1
        dudx = (u(ip,jp)+u(ip,j)-u(i,jp)-u(i,j))/dx2
        dvdx = (v(ip,jp)+v(ip,j)-v(i,jp)-v(i,j))/dx2
        dudy = (u(ip,jp)+u(i,jp)-u(ip,j)-u(i,j))/dy2
        dvdy = (v(ip,jp)+v(i,jp)-v(ip,j)-v(i,j))/dy2
        xmuu =dudx*dudx+dvdy*dvdy+0.25*(dudy+dvdx)*(dudy+dvdx)+dudx*dvdy
        xmu(i,j) = (abs(xmuu)+eps)**exp
      ENDDO
      ENDDO
      DO j=1,ny-1
       xmu(nx,j)=xmu(nx-1,j)
      ENDDO
      DO i=1,nx
       xmu(i,ny)=xmu(i,ny-1)
      ENDDO
C ---
      RETURN
      END
C---------------------------------------------------------------------
      SUBROUTINE ITERAT(nx,ny,x,y,u,v,taux,tauy,h,s,fx,fy,
     &    uex,vex,tauxe,tauye,errui,errvi,errxi,erryi,iter) 
C        solves 2d shelf model using traditional non-linear algorithm 
C---------------------------------------------------------------------
      IMPLICIT none
      INCLUDE "parameter2d.h"
C ---
      DOUBLE PRECISION x(nx),y(nx),u(nx,ny),v(nx,ny)
      DOUBLE PRECISION h(nx,ny),s(nx,ny),xmu(nx,ny)
      DOUBLE PRECISION fx(nx,ny),fy(nx,ny)
      DOUBLE PRECISION uex(nx,ny),vex(nx,ny),tauxe(nx,ny),tauye(nx,ny)
      DOUBLE PRECISION taux(nx,ny),tauy(nx,ny)
      DOUBLE PRECISION taux1(nx,ny),tauy1(nx,ny)
      DOUBLE PRECISION fxx(nx,ny),fyy(nx,ny),ai(nx,ny),aj(nx,ny)
C ---
      DOUBLE PRECISION dx,dy,dx2,dy2,dxdyi,dxi,dyi,xx,dudx,dvdx,dudy
      DOUBLE PRECISION erroru,errorv,errxi,erryi,errui,errvi
      DOUBLE PRECISION v1,vn,u1,un,xijm,ximj,ximjm,ff,e1,e2,xmuu
      DOUBLE PRECISION aijm,aijp,aimj,aipj,aij,bij,fipj,fimj,fijp,fijm
      DOUBLE PRECISION dsdx,dsdy,tx2tyij,tx2tyijm,fxmax,fymax,hh
      DOUBLE PRECISION eps,errold
      INTEGER nx,ny,maxit,iter,i,ip,im,j,jp,jm
C ---
      maxit = 600000
      errold = 1.d-18
        eps = 1.d-5
        dx=x(2)-x(1)
        dy=y(2)-y(1)
        dxi = 1./(dx*dx)
        dyi = 1./(dy*dy)
        dxdyi = 1./(dx*dy)
        dx2 = 1./(2*dx)
        dy2 = 1./(2*dy)
C ---
C     calculate the norm of the RHSs
C ---
      fxmax = 0.
      fymax = 0.
      do i=2,nx-1
         im = i-1
         ip = i+1
       do j=2,ny-1
         jm = j-1
         jp = j+1
         fyy(i,j) = h(i,j)*(s(i,jp)-s(i,jm))*dy2+fy(i,j)
         fxx(i,j) = h(i,j)*(s(ip,j)-s(im,j))*dx2+fx(i,j)
         if(abs(fxx(i,j)) .gt. fxmax) fxmax = abs(fxx(i,j))
         if(abs(fyy(i,j)) .gt. fymax) fymax = abs(fyy(i,j))
       enddo 
       enddo 
C ---
C     assign the initial approximation
C ---
        xx = rand(1)
        do i = 1,nx
        do j = 1,ny
          v(i,j) = rand(0)
          u(i,j) = rand(0)
        enddo
        enddo
C ---
C     assign boundary conditions at x=0 and x=1
C ---
      do j = 1,ny
        u(1,j) = uex(1,j)
        v(1,j) = vex(1,j)
        u(nx,j) = uex(nx,j)
        v(nx,j) = vex(nx,j)
      enddo
C ---
C     assign boundary conditions at y=0 and y=1 
C ---
      do i = 1,nx
        u(i,1) = uex(i,1)
        v(i,1) = vex(i,1)
        u(i,ny) = uex(i,ny)
        v(i,ny) = vex(i,ny)
      enddo
      iter = 0
C ---
C     start iterations 
C ---
1     iter = iter + 1
      call UPDATENU(nx,ny,x,y,u,v,xmu)
C ---
C     calculate xnu*h at the middle of the cells' sides 
C ---
      do i=2,nx-1
         ip = i+1
         im = i-1
      do j=2,ny-1
         jm = j-1
         jp = j+1
         xijm = xmu(i,jm)*0.25*(h(i,jm)+h(ip,jm)+h(i,j)+h(ip,j))
         ximj = xmu(im,j)*0.25*(h(im,j)+h(i,j)+h(im,jp)+h(i,jp))
         ximjm = xmu(im,jm)*0.25*(h(im,jm)+h(i,jm)+h(im,j)+h(i,j))
        ai(im,j) = 0.5*(ximj+ximjm)
        aj(i,jm) = 0.5*(ximjm+xijm)
      enddo
      enddo
      do i=2,nx-1
        j = ny
        jm = j-1
        im = i-1
        ip = i+1
        xijm = xmu(i,jm)*0.25*(h(i,jm)+h(ip,jm)+h(i,j)+h(ip,j))
        ximjm = xmu(im,jm)*0.25*(h(im,jm)+h(i,jm)+h(im,j)+h(i,j))
        aj(i,jm) = 0.5*(ximjm+xijm)
      enddo
      do j=2,ny-1
        i = nx
        jm = j-1
        jp = j+1
        im = i-1
        ximj = xmu(im,j)*0.25*(h(im,j)+h(i,j)+h(im,jp)+h(i,jp))
        ximjm = xmu(im,jm)*0.25*(h(im,jm)+h(i,jm)+h(im,j)+h(i,j))
        ai(im,j) = 0.5*(ximj+ximjm)
      enddo
C---
      do i=2,nx-1
         im = i-1
         ip = i+1
       do j=2,ny-1
         jm = j-1
         jp = j+1
         aijm = aj(i,jm)*dyi
         aijp = aj(i,j)*dyi
         aimj = ai(im,j)*dxi
         aipj = ai(i,j)*dxi
         aij = -aijm-aijp-4*aimj-4*aipj
         bij = -4*aijm-4*aijp-aimj-aipj
c ............
        fipj = ai(i,j)*dxdyi*0.25*(u(ip,jp)-u(ip,jm)+u(i,jp)-u(i,jm))
        fimj = ai(im,j)*dxdyi*0.25*(u(i,jp)-u(i,jm)+u(im,jp)-u(im,jm))
        fijp = aj(i,j)*dxdyi*0.5*(u(ip,jp)-u(im,jp)+u(ip,j)-u(im,j))
        fijm = aj(i,jm)*dxdyi*0.5*(u(ip,j)-u(im,j)+u(ip,jm)-u(im,jm))
         ff = fyy(i,j) - fipj+fimj-fijp+fijm
        v(i,j) = (ff-aimj*v(im,j)-4*aijm*v(i,jm)
     &              -aipj*v(ip,j)-4*aijp*v(i,jp))/bij
c .............
        fipj = ai(i,j)*dxdyi*0.5*(v(ip,jp)-v(ip,jm)+v(i,jp)-v(i,jm))
        fimj = ai(im,j)*dxdyi*0.5*(v(i,jp)-v(i,jm)+v(im,jp)-v(im,jm))
        fijp = aj(i,j)*dxdyi*0.25*(v(ip,jp)-v(im,jp)+v(ip,j)-v(im,j))
        fijm = aj(i,jm)*dxdyi*0.25*(v(ip,j)-v(im,j)+v(ip,jm)-v(im,jm))
         ff = fxx(i,j) - fipj+fimj-fijp+fijm
        u(i,j) = (ff-4*aimj*u(im,j)-aijm*u(i,jm)
     &              -4*aipj*u(ip,j)-aijp*u(i,jp))/aij
c .............
       enddo      ! end of j-cycle 
      enddo    !  end of i-cycle
C ---
C     calculate errors
C ---
        erroru = 0.
        errorv = 0.
      do i=2,nx-1
         im = i-1
         ip = i+1
       do j=2,ny-1
         jm = j-1
         jp = j+1
         aijm = aj(i,jm)*dyi
         aijp = aj(i,j)*dyi
         aimj = ai(im,j)*dxi
         aipj = ai(i,j)*dxi
         aij = -aijm-aijp-4*aimj-4*aipj
         bij = -4*aijm-4*aijp-aimj-aipj
          fipj = ai(i,j)*dxdyi*0.5*(v(ip,jp)-v(ip,jm)+v(i,jp)-v(i,jm))
          fimj = ai(im,j)*dxdyi*0.5*(v(i,jp)-v(i,jm)+v(im,jp)-v(im,jm))
          fijp = aj(i,j)*dxdyi*0.25*(v(ip,jp)-v(im,jp)+v(ip,j)-v(im,j))
          fijm = aj(i,jm)*dxdyi*0.25*(v(ip,j)-v(im,j)+v(ip,jm)-v(im,jm))
          ff = fxx(i,j) - fipj+fimj-fijp+fijm
          e1 = abs(u(i,j)*aij - (ff-4*aimj*u(im,j)-aijm*u(i,jm)
     &              -4*aipj*u(ip,j)-aijp*u(i,jp)))
          fipj = ai(i,j)*dxdyi*0.25*(u(ip,jp)-u(ip,jm)+u(i,jp)-u(i,jm))
          fimj = ai(im,j)*dxdyi*0.25*(u(i,jp)-u(i,jm)+u(im,jp)-u(im,jm))
          fijp = aj(i,j)*dxdyi*0.5*(u(ip,jp)-u(im,jp)+u(ip,j)-u(im,j))
          fijm = aj(i,jm)*dxdyi*0.5*(u(ip,j)-u(im,j)+u(ip,jm)-u(im,jm))
         ff = fyy(i,j) - fipj+fimj-fijp+fijm
          e2 = abs(v(i,j)*bij - (ff-aimj*v(im,j)-4*aijm*v(i,jm)
     &              -aipj*v(ip,j)-4*aijp*v(i,jp)))
          if(e1 .gt. erroru) erroru = e1
          if(e2 .gt. errorv) errorv = e2
        enddo
        enddo
C ---
C     check the stopping criteria
C ---
      if(erroru .le. eps*fxmax .and. errorv .le. eps*fymax) goto 11
        if(iter .gt. maxit) then
          print *,' iter exceeded maxit=',maxit
          goto 11
        endif
        iter=iter+1
        goto 1
C ---
C      Iterations are done, velocities are determined,
C         calculate stresses: taux and tauy
C ---
11    continue
      errxi = 0.
      erryi = 0.
      do i = 1,nx-1
      do j = 1,ny-1
        ip = i+1
        jp = j+1
        hh = 0.25*(h(i,j)+h(ip,j)+h(i,jp)+h(ip,jp))
        xmuu = xmu(i,j)*hh
        dudx = (u(ip,j)+u(ip,jp)-u(i,j)-u(i,jp))*dx2
        dvdx = (v(ip,j)+v(ip,jp)-v(i,j)-v(i,jp))*dx2
        dudy = (u(i,jp)+u(ip,jp)-u(i,j)-u(ip,j))*dy2
        taux(i,j) = 2*xmuu*dudx
        tauy(i,j) = xmuu*(dudy+dvdx)
        e1 = taux(i,j)-tauxe(i,j)
        e2 = tauy(i,j)-tauye(i,j)
        errxi = errxi+e1*e1
        erryi = erryi+e2*e2
         e1 = u(i,j)-uex(i,j)
         e2 = v(i,j)-vex(i,j)
        errui = errxi+e1*e1
        errvi = erryi+e2*e2
      enddo
      enddo
       errxi = sqrt(errxi/(nx*ny))
       erryi = sqrt(erryi/(nx*ny))
       errui = sqrt(errui/(nx*ny))
       errvi = sqrt(errvi/(nx*ny))
      do i=1,nx-1
        taux(i,ny) = taux(i,ny-1)
        tauy(i,ny) = tauy(i,ny-1)
      enddo
      do j=1,ny
        taux(nx,j) = taux(nx-1,j)
        tauy(nx,j) = tauy(nx-1,j)
      enddo
      return
      end
C---------------------------------------------------------------------
