/*---------------------------------------*/
/* schemes.c (last modified: 12/06/2013) */
/* advection and dispersion schemes      */
/*---------------------------------------*/

#include "define.h"
#include "variables.h"

// SET BOUNDARY CONDITIONS
void Openbound(double* co, int s)
{
    if(U[2]>=0.0)
        co[1]=co[1]- (co[1]-v[s].clb)*U[2]*((double)DELTI)/((double)DELXI);
    else
        co[1]=co[1]- (co[3]-co[1])*U[2]*((double)DELTI)/((double)DELXI);

    if(U[M1]>=0.0)
        co[M]=co[M]- (co[M]-co[M2])*U[M1]*((double)DELTI)/((double)DELXI);
    else
        co[M]=co[M]- (v[s].cub-co[M])*U[M1]*((double)DELTI)/((double)DELXI);

    co[M1]=co[M];
}

// ADVECTION SCHEME
void TVD (double* co, int s)
{
    int j;
    double vx, cfl, rg, philen,f, phi, rat, rat1;
    double cold[M+1];

    for (j=1; j<=M; j++) cold[j]=co[j];

    for(j=1;j<=M2;j+=2)
    {
        vx=U[j+1];
        cfl =fabs(vx)*((double)DELTI)/(2.*(double)DELXI);

        //For positive velocity, vx(j) > 0
        if(vx>0.0)
        {
            f = cold[j+2]-cold[j];
            if (fabs(f)>1.e-35 && j!=1)
            {
                rg = (cold[j]-cold[j-2])/f;
                phi= (2.-cfl+rg*(1.+cfl))/3.;
                philen = Max(0.,Min(2.,Min(2.*rg,phi)));
            }
            else
            {
                philen = 0.0;
            }

            co[j+1] = cold[j] + 0.5*(1.0-cfl)*philen*f;
            fl[j+1] = vx*D[j+1]*co[j+1];
        }

        //For negative velocity, vx(j) < 0
        if(vx<=0.0)
        {
            f = cold[j]-cold[j+2];
            if (fabs(f)>1.e-35 && j!=M2)
            {
                rg = (cold[j+2]-cold[j+4])/f;
                phi= (2.-cfl+rg*(1.+cfl))/3.;
                philen = Max(0.,Min(2.,Min(2.*rg,phi)));
            }
            else
            {
                philen = 0.0;
            }

            co[j+1] = (cold[j+2] + 0.5*(1.0-cfl)*philen*f);
            fl[j+1] = vx*D[j+1]*co[j+1];
        }
    }

    for (j=3; j<=M2; j+=2)
    {
        rat=Dold2[j]/D[j];
        rat1=((double)DELTI/(2.*(double)DELXI))/D[j];
        cold[j]=rat*cold[j]-rat1*(fl[j+1]-fl[j-1]);
        co[j]=cold[j];
    }
}

// DISPERSION SCHEME
void Disp(double* co)
{
    int i,j;
    double a[M+1], b[M+1], c[M+1], r[M+1], di[M+1], gam[M+1];
    double bet, c1, c2;

    a[1] =0.0;
    b[1] =1.;
    c[1] =0.;
    di[1]=co[1];
    a[M] =0.0;
    b[M] =1.;
    c[M] =0.;
    di[M]=co[M];

    for(i=2; i<=M1; i+=1)
    {
        c1=disp[i-1]*D[i-1]/D[i];
        c2=disp[i+1]*D[i+1]/D[i];
        a[i]=-c1*(double)DELTI/(8*(double)DELXI*(double)DELXI);
        c[i]=-c2*(double)DELTI/(8*(double)DELXI*(double)DELXI);
        b[i]=1.+c2*(double)DELTI/(8*(double)DELXI*(double)DELXI)+c1*(double)DELTI/(8*(double)DELXI*(double)DELXI);
        r[i]=1.-c2*(double)DELTI/(8*(double)DELXI*(double)DELXI)-c1*(double)DELTI/(8*(double)DELXI*(double)DELXI);
    }

    for(i=3; i<=M1; i+=2)
    {
        di[i]=-c[i]*co[i+2]+r[i]*co[i]-a[i]*co[i-2];
    }

    if(b[1]==0.0)
    {
        printf("Error Disp.c: b[1]==0");
        exit(EXIT_SUCCESS);
    }

    bet=b[1];
    co[1]=di[1]/bet;

    for(j=3; j<=M1; j+=2)
    {
        gam[j]=c[j-2]/bet;
        bet=b[j]-a[j]*gam[j];
        co[j]=(di[j]-a[j]*co[j-2])/bet;
    }

    for(j=M3; j>=1; j-=2)
    {
        co[j]=co[j]-gam[j+2]*co[j+2];
    }
}


