!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
!  BOSVM (Balanced Optimality Structure Vegetation model) Version:1.0
!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!

! This model is develped to simulate land surface processes by including
! water, carbon, energy balances.

! Commands example:
! ./flux 0.5 1 30 3 30 A O

! alpha = 0.5; D = 1; Initial value of biomass = 30kgC; vegetation
! type = 3 (defensive woody); interation time is 30 (30*6yrs); name of
! forcing data = A.dat; name of output file = O.dat;

! Here alpha is the shoot root ratio; D is the canopy structure
! defined by ratio of relative LAI over relative crown area.

Program BOSVM
  implicit none

  ! forcing data
  real , dimension(:), allocatable :: t_a      !air temperature at 2 meters [K]
  real , dimension(:), allocatable :: p_suf    !surface pressure [Pa]
  real , dimension(:), allocatable :: SH       !specific humidity at 2 meters [kg.kg-1]
  real , dimension(:), allocatable :: u_w      !u wind speed [m.s-1]
  real , dimension(:), allocatable :: v_w      !v wind speed [m.s-1]
  real , dimension(:), allocatable :: r_lwd    !downawrd longwave radiation [w.m-2]
  real , dimension(:), allocatable :: r_swd    !downward shortwave radiation [w.m-2]
  real , dimension(:), allocatable :: P        !precipitation rate [kg.m-2.s-1]
  ! forcing data used in each time step
  real :: dt_a
  real :: dp_suf
  real :: dSH
  real :: du_w
  real :: dv_w
  real :: dr_lwd
  real :: dr_swd
  real :: dP

  ! physical variables used in the model
  real :: c_veg                 !biomass of vegetation [kgC]
  real :: c_leaf                !biomass of leaf [kgC]
  real :: c_root                !biomass of other[kgC]
  real :: c_stem
  real :: c_p                   !specific head capacity of moist
                                   !air [J.kg-1.K-1]
  real :: Aveg                  !area of fully veg covered [m2]
  real :: GPP                   !gross primary production
                                   ![kgC.m-2.s-1]
  real :: fc                    !leaf coverage [m2.m-2]
  real :: fs                    !proposion of soil part outside
                                   !crown area [m2.m-2]
  real :: D                     !function between the relation of D

  real , dimension(2) :: r_n    !net radiation [w.m-2] r_n(1):
                                   !vegetated area; r_n(2): bare soil
                                   !area
  real , dimension(2) :: a      !vegetation part albedo [1]
  real , dimension(2) :: t_s    !surface temperature [K]
  real , dimension(2) :: G      !soil heat flux of bare soil [w/m2]
  real , dimension(2) :: lET    !latent heat flux from bare soil [w.m-2]
  real , dimension(2) :: H      !sensible heat flux from vegetation [w.m-2]
  real , dimension(2) :: Cm     !transfer coefficient to momentum [1]
  real , dimension(2) :: Ch     !transfer coefficient to heat [1]
  real , dimension(2) :: Cq     !transfer coefficient to humidity [1]
  real , dimension(2) :: Jm     !momentum flux [J.m-2]
  real , dimension(2) :: Jq     !latent heat flux [w.m-2]
  real , dimension(2) :: Leak   !water leakage[kg.m-2.s-1]
  real :: NPP                   !Net primiary production [kgC.s-1]
  real :: vpD                   !vapor pressure deficit [Pa]
  real :: e_s                   !saturation vapor pressure [Pa]
  real :: e_a                   !actual vapor pressure [Pa]
  real , dimension(2) :: g_a    !aerodynamic conductance [m/s]
  real :: g_s                   !surface conductance [m/s]
  real :: rho_a                 !mean air density at constant pressure [kg.m-3]
  real :: t_kv                  !virtual temperature [K]
  real :: LAI                   !leaf area index [1]
  real :: alpha                 !parameter of biomass and LAI ［kgC.m-2］
  real :: g_soil                !bare soil evaporation conductance [m/s]
  real , dimension(2) :: t_soil !soil temperature [K]
  real , dimension(2) :: sm     !soil moisture [m3.m-3]
  real :: CA                    !crown area [m2]
  real :: pi                    !3.1415
  integer :: tSoil                 !soil texture symbols [-]
  integer :: tVeg                  !vegetation type [-] 1: Defensive
                                   !grass; 2: Offensive grass; 3:
                                   !Defensive woody; 4: Offensive
                                   !woody
  integer :: Veg                   !clare if it's bare soil or
                                   !vegetation, 1: vegetated area, 2:
                                   !bare soil
  integer :: Lay                   !clear if it's layer one or layer
                                   !two, 1: layer 1, 2: layer 2
  real :: dts = 3600*3             !time step, hours in second
  character(len = 100) :: file     !file name
  character(len = 100) :: afilei   !name of  forcing data file from command
  character(len = 100) :: afileo   !name of  outputfile from command
  character(10) :: aD, aalpha      !used to read data from command
  character(10) :: ac_veg, atVeg   !
  character(10) :: aniter          !how many time to run the 6-year data
  integer :: num, i, j             !interaction factors
  integer :: length  = 17534       !length of the forcing data (6 year, 3-hour time step)
  integer :: niter                 !number of iterations
  integer :: dnum                  !how many times divided the time step
  integer :: day = 0               !number of day
  integer :: ddts                  !new time scale [s]
  ! parameters from CHTESSEL model
  real :: f0                       !factor []


  include 'constants.f90'

  ! initial data settings
  ! ====================
  allocate(t_a(length))
  allocate(p_suf(length))
  allocate(SH(length))
  allocate(u_w(length))
  allocate(v_w(length))
  allocate(r_lwd(length))
  allocate(r_swd(length))
  allocate(P(length))

  call getarg(1,aalpha)
  call getarg(2,aD)
  call getarg(3,ac_veg)
  call getarg(4,atVeg)
  call getarg(5,aniter)
  call getarg(6,afilei)
  call getarg(7,afileo)
  close(1)
  close(2)
  close(3)
  close(4)
  close(5)
  close(6)
  close(7)
  read(aalpha,*)alpha
  read(aD,*)D
  read(ac_veg,*)c_veg
  read(atVeg,*)tVeg
  read(aniter,*)niter

  ! read data from file, file must be at the same directory as the excutable file
  ! forcing data must be formed as: air temperature; air pressure; specific humidity; wind speed u
  ! wind speed v; long wave incoming radiation; short wave incoming radiation; precipitation rate
  afilei = trim(afilei)//'.dat'
  open(1, file = afilei, status = 'OLD')
  do num = 1, length
     read(1,*)t_a(num), p_suf(num), SH(num), u_w(num), &
          & v_w(num), r_lwd(num), r_swd(num), P(num)
  end do
  close(1)

  ! settings for soil texture parameters
  ! 1.Coarse,   2.Medium, 3.Medium fine, 4.Fine,  5.Very fine, 6.Organic
  tSoil = 2
  pi = acos(0.0)
  a(2) = a_b
  ! initial vegetation structure vairables
  call InitialVeg(tVeg, c_veg, Aveg, alpha, D, c_leaf,&
       &c_root, LAI, fs, fc, CA, c_stem)

  sm(1) = sm_sat(tSoil)         ! soil moisture [m3/m3]
  sm(2) = sm(1)

  ! initialize soil temperature as air temperature at beginning
  t_soil(1) = t_a(1)
  t_soil(2) = t_a(1)
  t_s(1)    = t_a(1)
  t_s(2)    = t_a(1)
  lET(1)    = 20
  lET(2)    = 20
  H(1)      = 20
  H(2)      = 20
  G(1)      = 20
  G(2)      = 20


  ! initialize surface temperature at previous time step
  Cm(1) = 0.1
  Cm(2) = 0.1
  Ch(1) = 0.1
  Ch(2) = 0.1
  Cq(1) = 0.1
  Cq(2) = 0.1
  Jm(1) = 0.1
  Jm(2) = 0.1

  afileo = trim(afileo)//'.dat'   !name of output file
  open(1, file = afileo)          !prepare to write result in

  ddts = 5400                     !1.5 hour time scale [s]
  dnum = dts/ddts                 !how many times interval in 3-hour step, here is 3

  do num = 1, niter               ! 6-year data used for 10 times
     do i = 1, length-1
        do j = 1, dnum
           ! interpolation forcing data
           du_w = Interpolation(u_w(i),u_w(i+1),j, dnum)
           dv_w = Interpolation(v_w(i),v_w(i+1),j, dnum)
           dr_swd  = Interpolation(r_swd(i), r_swd(i+1), j, dnum)
           dr_lwd  = Interpolation(r_lwd(i), r_lwd(i+1), j, dnum)
           dSH     = Interpolation(SH(i), SH(i+1), j, dnum)
           dp_suf  = Interpolation(p_suf(i), p_suf(i+1),j, dnum)
           dt_a  = Interpolation(t_a(i), t_a(i+1), j, dnum)
           dP      = Interpolation(P(i) , P(i+1), j, dnum)

           !related variables calculation
           call UpdateFluxPara(LAI, dSH, dp_suf, dt_a, rho_a,&
                &vpD, e_a, a(1), c_p)

           ! bisection method to get surface temperature
           Veg = 1              !vegetated area
           call EnergyBalance(Veg, tVeg, dr_swd, dr_lwd,a(Veg),dv_w,&
                &du_w, t_soil(1), dt_a, r_n(Veg), H(Veg), lET(Veg),&
                &G(Veg), t_s(Veg), g_s, g_a(Veg), Cm(Veg), Ch(Veg),&
                &Cq(Veg), Jm(Veg), e_a, GPP, NPP, vpD, LAI, dp_suf,&
                &rho_a, sm(2), fc, c_root, CA, c_p)
           Veg = 2              !bare soil area
           call EnergyBalance(Veg, tVeg, dr_swd, dr_lwd,a(Veg),dv_w,&
                &du_w, t_soil(1), dt_a, r_n(Veg), H(Veg), lET(Veg),&
                &G(Veg), t_s(Veg), g_soil, g_a(Veg), Cm(Veg), Ch(Veg),&
                &Cq(Veg), Jm(Veg), e_a, GPP, NPP, vpD, LAI, dp_suf,&
                &rho_a, sm(1), fc, c_root, CA, c_p)

           ! update status
           call UpdateVegStatus(c_veg, c_leaf, c_root, NPP, fc,&
                &c_stem, tVeg)
           call VegUpdate(tVeg, c_veg, Aveg, alpha, D, c_leaf,&
                &c_root, LAI, fs, fc, CA, c_stem)
           call UpdateWaterStatus(sm(1), sm(2), fc, Leak(1),&
                &Leak(2), lET(1), lET(2), dP)
           call UpdateAlbedoStatus(a(1), LAI)

           call UpdateSoilStatus(sm(1), sm(2), t_s(1), t_s(2),&
                &t_soil(1), t_soil(2), fc)
        end do
        
        ! print output at the last 6-year circle
        if (num == niter) then
           write(1,*)t_s(1), r_n(1), lET(1), H(1), G(1), fc,&
                &c_veg, GPP, LAI, sm(2), Leak(2), P(i),&
                &g_s, CA, t_s(2), r_n(2), lET(2), H(2), G(2),&
                &c_leaf, c_root, g_soil,sm(1),Leak(1),t_a(i)&
                &,NPP, c_stem
        end if
     end do

  end do
  close(1)

  !-----------------------------------------------

contains

  !-----------------------------------------------
  subroutine EnergyBalance(pVeg, ptVeg, pr_swd, pr_lwd, pa, pUv,&
       &pUu, pTsoil, pTair, pRn, pH, plET, pG, pTs, pg_s, pg_a, pCm,&
       &pCh, pCq, pJm, pe_a, pGPP, pNPP, pvpD, pLAI, pp_suf, prho_a,&
       &psm, pfc, pc_root, pCA, pc_p)
    ! input
    real :: pr_swd           !
    real :: pr_lwd           !
    real :: pa               !surface albedo
    real :: pUv              !v_wind
    real :: pUu              !u_wind
    real :: pTsoil           !soil temperature
    real :: pTair            !air temperature
    real :: pe_a
    real :: pp_suf           !
    real :: pGPP             !
    real :: pNPP
    real :: pvpD
    real :: pLAI
    real :: prho_a           !air density
    real :: psm              !soil moisture at specified layer
    real :: pfc              !
    real :: pCA              !
    real :: pc_root          !
    real :: pc_p             !
    integer :: ptVeg            !
    integer :: pVeg
    ! media
    real :: zPreTs           !surface temperature last time step
    real :: zSair            !
    real :: zSs
    real :: zqair
    real :: zQov
    real :: zUn
    real :: zJq
    real :: zJs
    real :: zux
    real :: zqx
    real :: zsx
    real :: zwx
    real :: zdq_dt
    real :: zC_h1
    real :: zC_h3
    real :: zC_e2
    real :: zC_e3
    real :: zC_e4
    real :: zD_s1
    real :: zD_s2
    real :: zD_s4
    real :: zD_h1
    real :: zD_h2
    real :: zD_h4
    real :: zD_e1
    real :: zD_e2
    real :: zD_e4
    real :: zZ
    real :: zzi              !scale height of boundary layer[m]
    real :: zr_swd
    real :: zr_lwd
    integer :: zi
    real :: zG_t             !compensation concentration [ppm]
    real :: zL               !Obukhov length [m]
    ! output, here close the energy balance of current time step
    real :: pRn
    real :: pH
    real :: plET
    real :: pG
    real :: pTs
    real :: pg_s
    real :: pg_a
    real :: pCm
    real :: pCh
    real :: pCq
    real :: pJm

    zPreTs = pTs
    zSair = pc_p * pTair
    zzi   = 1000.0
    zJs   = pH
    zJq   = plET/l
    zqair = e * pe_a / pp_suf
    zr_swd = (1 - pa) * pr_swd
    zr_lwd = epsilon * pr_lwd - epsilon*sigma*zPreTs**4

    ! get surface conductance by previous surface temperature
    if (pVeg == 0) then         !if bare soil case
       pg_s = g_soilmax * f_w(psm)
    else                        !if vegetated area
       call GetGs(zPreTs, ptVeg, pLAI, prho_a, psm,&
            &pGPP, pNPP, pg_s, pvpD, pp_suf, pr_swd,&
            &pfc, pc_root, pCA)
    end if
    
    ! run three times to improve estimation of zL and g_a
    do zi = 1,3
       call GetL(zL, prho_a, zJq, zJs, pJm, pTair, zQov, pc_p)

       call UpdateFlux(zL, pCm, pCh, pCq, pJm, zJs,&
            &zJq, pTair, zPreTs, pe_a, pp_suf, pg_s,&
            &zUn, prho_a, pc_p)

       zdq_dt = e*0.6108e3*17.27*237.3*e0(zPreTs - 273)&
            &/((pp_suf*(zPreTs - 273 + 237.3)**2)*pp_suf)

       zC_h1 =  prho_a*pCh*zUn
       zC_h3 = -prho_a*pCh*zUn
       zC_e2 =  prho_a/(1/(pCq*zUn) +1/pg_s)
       zC_e3 = -prho_a*zdq_dt/(pc_p*(1/(pCq*zUn) +1/pg_s))
       zC_e4 = -prho_a*(e*e0(zPreTs - 273)&
            &*1e3/pp_suf - zdq_dt*zPreTs)/&
            &(1/(pCq*zUn) +1/pg_s)

       zZ    = (-4*epsilon*sigma*zPreTs**3 - 2*con_soil/z(1))&
            &/ pc_p + zC_h3 + l * zC_e3

       zD_s1 = -zC_h1 / zZ
       zD_s2 = -l * zC_e2 / zZ
       zD_s4 = (-zr_swd - zr_lwd - 4 * epsilon *&
            &sigma * zPreTs**4 - l * zC_e4 - 2 *&
            &con_soil * pTsoil / z(1)) / zZ

       zSs = zD_s1 * zSair + zD_s2 * zqair + zD_s4
       pTs = zSs/pc_p

       zD_h1 = zC_h1 + zC_h3*zD_s1
       zD_h2 =     0 + zC_h3*zD_s2
       zD_h4 =     0 + zC_h3*zD_s4
       zD_e1 =     0 + zC_e3*zD_s1
       zD_e2 = zC_e2 + zC_e3*zD_s2
       zD_e4 = zC_e4 + zC_e3*zD_s4

       pg_a = pCh * zUn

       pRn = zr_swd + zr_lwd - 4*epsilon*sigma*zPreTs&
            &**3*(pTs - zPreTs)
       pH  = -zD_h1*zSair - zD_h2*zqair - zD_h4
       plET= -l*(zD_e1*zSair + zD_e2*zqair + zD_e4)
       pG  = SoilHF(pTs, pTsoil)

       ! to get GPP NPP at this time step
       if (Veg /= 0) then
          call GetGs(pTs, ptVeg, pLAI, prho_a, psm,&
               &pGPP, pNPP, pg_s, pvpD, pp_suf,&
               &pr_swd, pfc, pc_root, pCA)
       end if

       zJs   = pH
       zJq   = plET/l

       zQov = (zJs - pTair * zJq *(c_v - c_d)) / (prho_a&
            &* pc_p) + 0.6 * pTair * zJq / prho_a
       if (zQov <= 0) then
          zwx = -(-zzi * gv * zQov / pTair)**(0.3334)
       else
          zwx = (zzi * gv * zQov / pTair)**(0.3334)
       end if
       zUn = sqrt(pUv**2 + pUu**2 + zwx**2)
       pJm = prho_a * pCm * zUn**2
    end do
  end subroutine EnergyBalance


  !-----------------------------------------------
  function f_w(psm)
    implicit none
    real :: psm              ! soil moisture [m3/m3]
    real :: f_w              ! soil water for surface resistance [-]

    if (psm < sm_r(tSoil)) then
       f_w = 1e-30
    else if (psm < sm_sat(tSoil)) then
       f_w = (psm - sm_r(tSoil))/(sm_sat(tSoil)&
            &- sm_r(tSoil))
    else
       f_w = 1.0
    end if

  end function f_w

  !-----------------------------------------------
  function e0(pt)
    implicit none
    real :: e0
    real :: pt                  !temperature [C]
    e0 = 0.6108*exp((17.27*pt)/(237.3+pt))
  end function e0

  !-----------------------------------------------
  function SoilHF(pts, pt_soil)
    implicit none
    real :: pts                 !surface temperature [k]
    real :: pt_soil             !soil temperature [K]
    real :: SoilHF              !soil heat flux [w/m2]
    SoilHF = - con_soil * 2* (pt_soil - pts)/z(1)
  end function SoilHF

  !-----------------------------------------------
  subroutine VegUpdate(ptVeg, pc_veg, pAveg, palpha, pD, pc_leaf,&
       &pc_root, pLAI, pfs, pfc, pCA, pc_stem)
    implicit none
    ! input
    real :: pc_veg           ! tatol biomass [kgC]
    real :: pD               ! parameter in soil proporsion [kgC]
    integer :: ptVeg
    ! output
    real :: pAveg            ! fully covered veg area [m2]
    real :: pc_leaf          ! leaf biomass [kgC]
    real :: pc_root          ! other biomass [kgC]
    real :: pLAI             ! Leaf Area Index [1]
    real :: pc_stem
    real :: pfs              ! soil proposion outside crown area [m2/m2]
    real :: zfa              !relative LAI [LAI/LAIref]
    real :: pfc              ! leave coverage [m2/m2]
    real :: pCA              ! crown area [m2]
    real :: zA               ! Total leaf area [m2]
    real :: palpha           ! ratio of leaf biomass to total [1]
    real :: zc_othr
    real :: zsum
    real :: zPreLAI
    pc_root = pc_veg*(1-palpha)
    zc_othr = pc_veg*palpha

    if (ptVeg == 1 .OR. ptVeg == 2) then !grassxxx
       pc_stem = 0.0
       pc_leaf = pc_veg*palpha
       zA = pc_leaf * SLA
       pfs     = sqrt(zA/(LAIref*CAref*pD))
       pLAI    = sqrt(zA*pD*LAIref/CAref)
       if (pfs > 1.0) then
          pfs = 1.0
          pLAI = zA/CAref
       end if
    else                                 !woody
       ! here I use the linearized method to update LAI
       zsum = pD*LAIref*CAref/SLA +&
            &al(tVeg)*(LAIref*pD)**1.667*CAref!total above ground biomass at the dual point
       ! theme 1: pfs<1.0 theme 2: pfs>1.0
       if (pfs<1.0 .AND. zsum>zc_othr) then !use theme 1
          call GetLAI1(pLAI, pfs, pD, zc_othr,&
               &pc_leaf, pc_stem)
       else if (pfs<1.0 .AND. zsum<zc_othr) then !use theme2
          pLAI = pD*LAIref
          call GetLAI2(pLAI, pfs, pD, zc_othr,&
               &pc_leaf, pc_stem)
       else if (pfs>=1.0 .AND. zsum>zc_othr) then !use theme1
          pLAI = pD*LAIref
          call GetLAI1(pLAI, pfs, pD, zc_othr,&
               &pc_leaf, pc_stem)
       else if (pfs>=1.0 .AND. zsum<zc_othr) then !use theme2
          call GetLAI2(pLAI, pfs, pD, zc_othr,&
               &pc_leaf, pc_stem)
       end if
    end if

    pfc     = (1.0 - exp(-k * pLAI)) * pfs
    pCA     = pfs * CAref

  end subroutine VegUpdate
  !-----------------------------------------------
  subroutine GetLAI1(pLAI, pfs, pD, pc_othr, pc_leaf, pc_stem)
    implicit none
    real :: pLAI
    real :: pfs
    real :: pD
    real :: pc_othr
    real :: pc_leaf
    real :: pc_stem
    real :: zPrefa
    real :: zfa
    real :: zE1, zE2
    real :: zD1, zD2, zD3, zD4
    ! when fs<1.0
    zPrefa = pLAI/LAIref
    zE1 = CAref*LAIref*zPrefa**2/(pD*SLA)
    zE2 = al(tVeg)*CAref*(LAIref**1.667)*zPrefa&
         &**2.667/pD
    zD1 = 2*CAref*LAIref*zPrefa/(pD*SLA)
    zD2 = -2*CAref*LAIref*zPrefa**2/(pD*SLA)
    zD3 = al(tVeg)*8.0*CAref*(LAIref**1.667)*&
         &zPrefa**1.667/(3.0*pD)
    zD4 = -al(tVeg)*8.0*CAref*(LAIref**1.667)*&
         &zPrefa**2.667/(3.0*pD)
    zfa = (pc_othr-zE1-zE2-zD2-zD4)/(zD1+zD3)
    pLAI = zfa*LAIref
    pc_leaf = CAref*LAIref*zfa**2/(pD*SLA)
    pc_stem = al(tVeg)*CAref*LAIref**1.667*&
         &zfa**2.667/pD
    pfs = zfa/pD
  end subroutine GetLAI1
  !-----------------------------------------------
  subroutine GetLAI2(pLAI, pfs, pD, pc_othr, pc_leaf, pc_stem)
    implicit none
    real :: pLAI
    real :: pfs
    real :: pD
    real :: pc_othr
    real :: pc_leaf
    real :: pc_stem
    real :: zPrefa
    real :: zfa
    real :: zE1, zE2
    real :: zD1, zD2, zD3, zD4
    ! when fs>1.0
    zPrefa = pLAI/LAIref
    zE1 = CAref*LAIref*zPrefa/SLA
    zE2 = al(tVeg)*(LAIref**1.667)*CAref*&
         &zPrefa**1.667
    zD1 = CAref*LAIref/SLA
    zD2 = -CAref*LAIref*zPrefa/SLA
    zD3 = al(tVeg)*5.0*(LAIref**1.667)*&
         &CAref*zPrefa**0.667/3.0
    zD4 = -al(tVeg)*5.0*(LAIref**1.667)*&
         &CAref*zPrefa**1.667/3.0
    zfa = (pc_othr-zE1-zE2-zD2-zD4)/(zD1+zD3)
    pLAI = zfa*LAIref
    pc_leaf = CAref*LAIref*zfa/SLA
    pc_stem = al(tVeg)*CAref*LAIref**1.667*&
         &zfa**1.667
    pfs = 1.0
  end subroutine GetLAI2
  !-----------------------------------------------
  subroutine InitialVeg(ptVeg, pc_veg, pAveg, palpha, pD, pc_leaf,&
       &pc_root, pLAI, pfs, pfc, pCA, pc_stem)
    implicit none
    ! input
    real :: pc_veg           ! tatol biomass [kgC]
    real :: pD               ! parameter in soil proporsion [kgC]
    integer :: ptVeg
    ! output
    real :: pAveg            ! fully covered veg area [m2]
    real :: pc_leaf          ! leaf biomass [kgC]
    real :: pc_root          ! other biomass [kgC]
    real :: pLAI             ! Leaf Area Index [1]
    real :: pc_stem
    real :: pfs              ! soil proposion outside crown area [m2/m2]
    real :: pfc              ! leave coverage [m2/m2]
    real :: pCA              ! crown area [m2]
    real :: palpha           ! ratio of leaf biomass to total [1]

    ! internal variables
    real :: zfa              !relative LAI [LAI/LAIref]
    real :: zA               ! Total leaf area [m2]
    real :: zc_above         ! above ground biomass [kgC]
    real :: zfa1, zfa2, zfam ! used in bisection method to calculate correct zfa
    real :: zfs1, zfs2, zfsm
    real :: zsum1, zsum2, zsumm

    ! calculate root biomass and above ground biomass
    pc_root  = pc_veg*(1-palpha)
    zc_above = pc_veg*palpha


    if (ptVeg == 1 .OR. ptVeg == 2) then !grassxxx
       pc_stem = 0.0
       pc_leaf = pc_veg*palpha
       zA      = pc_leaf * SLA  !
       pfs     = sqrt(zA/(LAIref*CAref*pD))
       pLAI    = sqrt(zA*pD*LAIref/CAref)
       if (pfs > 1.0) then      !control CA cannot larger than CAref,
                                !if CA=CAref, biomass will go to LAI
                                !only
          pfs = 1.0
          pLAI = zA/CAref
       end if
    else                                 !woody
       ! here we use bisection method to calculate the first LAI
       zfa1 = 0.0               !first guess of minimum relative LAI
       zfa2 = 3.0               !first guess of maximum relative LAI
       zfam = (zfa1+zfa2)/2.0   ! average
       do while (abs(zfa2-zfa1)>1e-3)
          zfs1 = zfa1/pD
          zfsm = zfam/pD
          if (zfs1 < 1.0) then
             zsum1 = CAref*LAIref*zfa1**2/(pD*SLA)&
                  &+ al(tVeg)*CAref*LAIref**1.667*&
                  &zfa1**2.667/pD - zc_above
          else
             zsum1 = CAref*LAIref*zfa1/SLA +&
                  &al(tVeg)*LAIref**1.667*CAref*&
                  &zfa1**1.667 - zc_above
          end if

          if (zfsm < 1.0) then
             zsumm = CAref*LAIref*zfam**2/(pD*SLA)&
                  &+ al(tVeg)*CAref*LAIref**1.667*&
                  &zfam**2.667/pD - zc_above
          else
             zsumm = CAref*LAIref*zfam/SLA +&
                  &al(tVeg)*LAIref**1.667*CAref*&
                  &zfam**1.667 - zc_above
          end if
          if (zsum1*zsumm > 0) then
             zfa1 = zfam
          else
             zfa2 = zfam
          end if
          zfam = (zfa1+zfa2)/2.0
       end do

       ! take average value of zfa1 and zfa2 as final relative LAI
       zfa = (zfa1+zfa2)/2.0
       pfs = min(1.0, zfa/pD)
       pLAI = zfa * LAIref
       pc_leaf = CAref*pfs*pLAI/SLA
       pc_stem = al(tVeg) * CAref * pfs*&
            &pLAI**(1.667)
    end if

    pfc     = (1.0 - exp(-k * pLAI)) * pfs
    pCA     = pfs * CAref

  end subroutine InitialVeg

  !-----------------------------------------------
  subroutine UpdateFluxPara(pLAI, pSH, pp_suf,&
       &pt_a, prho_a, pvpD, pe_a, pa_v, pc_p)
    implicit none
    ! input
    real :: pLAI                ! LAI [m2/m2]
    real :: pSH                 ! specific humidity [kg.kg-1]
    real :: pp_suf              ! surface pressure [Pa]
    real :: pt_a                ! air temperature [K]
    ! medium
    real :: zt_kv               ! virtual temperature [K]
    ! output
    real :: pe_a                ! actual vapor pressure [Pa]
    real :: prho_a              ! mean air density at constant pressure [kg.m-3]
    real :: pvpD                ! vapor pressure deficit [Pa]
    real :: pa_v                ! vegetation albedo [1]
    real :: pc_p                ! heat capacity of moist air

    include 'constants.f90'

    zt_kv = 1.01 * pt_a
    pe_a = pSH * pp_suf / (pSH + e - e*pSH)
    prho_a = pp_suf/(zt_kv * R)
    pvpD = GetvpD(pt_a, pSH, pp_suf)
    pa_v = a_v_min + (a_v_max - a_v_min)&
         &* exp(-k * pLAI)
    pc_p = c_d + (c_v - c_d) * pSH

  end subroutine UpdateFluxPara

  !-----------------------------------------------
  subroutine UpdateVegStatus(pc_veg, pc_leaf, pc_root,&
       &pnpp, pfc, pc_stem, ptVeg)
    implicit none
    ! output
    real :: pc_veg             ! leaf biomass [kgC]    
    real :: pc_leaf             ! leaf biomass [kgC]
    real :: pc_root             ! other biomass [kgC]
    real :: pnpp                ! net primary production [kgC.m-2.s-1]
    real :: pfc                 ! leaf coverage [m2/m2]
    real :: pc_stem
    real :: zAveg               ! veg area [m2]
    real :: zlit_root           ! litter production [kgC.s-1]
    real :: zlit_stem           ! litter production [kgC.s-1]
    real :: zlit_leaf           ! litter of leaf
    real :: zlitter             ! litter of leaf
    real :: zts                 ! time interval
    real :: znpp_leaf
    real :: znpp_root
    integer :: ptVeg
    real :: zD

    zts = real(ddts)
    ! biomass updated
    zAveg = pfc * CAref

    if (ptVeg == 1 .OR. ptVeg == 2) then !grass
       zlitter = pc_veg / tau_l          ! 1 year litter time
    else                                 !woody
       zlit_leaf = pc_leaf / tau_l
       zlit_stem = pc_stem / tau_s
       zlit_root = pc_root / tau_r
       zlitter = zlit_root + zlit_stem + zlit_leaf
    end if

    pc_veg = pc_veg + zts * (pnpp * zAveg&
         &- zlitter)

    pc_veg = max(0.1,pc_veg)    !keep total biomass not smaller than
                                !0.1kgC

  end subroutine UpdateVegStatus

  !-----------------------------------------------
  subroutine UpdateWaterStatus(psm_1, psm_2, pfc,&
       &pleak_1, pleak_2, plETv, plETb, prain)
    ! input
    real :: pfc              ! leaf coverage [kgC/m2]
    real :: plETv            ! lET at veg part [w/m2]
    real :: plETb            ! lET at bare part [w/m2]
    real :: prain            ! rainfall [kgH2O/m2/s]
    ! in/out
    real :: psm_1            ! soil moisture at layer 1[m3/m3]
    real :: psm_2            ! soil moisture at layer 2[m3/m3]
    real :: pleak_1          ! leakage between layer 1 and 2[kgH2O/m2/s]
    real :: pleak_2          ! leakage between layer 2 and 3[kgH2O/m2/s]
    real :: zW1              ! total amount of soil water[m3]
    real :: zW2              ! total amount of soil water[m3]
    real :: zleak1           ! extra leak [m3]
    real :: zleak2           ! extra leak [m3]
    real :: zDif             ! diffusion between layer 1 and layer2
    real :: zts
    zts = real(ddts)
    zleak1 = 0
    zleak2 = 0
    zW1 = psm_1 * CAref * z(1)
    zW2 = psm_2 * CAref * z(2)
    pleak_1 = Leakage(psm_1)
    pleak_2 = Leakage(psm_2)

    zW1 = zW1 + ddts * (prain - pleak_1 - plETb&
         &* (1 - pfc)/l) * CAref / rho_w
    if (zW1/(CAref * z(1)) > sm_sat(tSoil)) then
       psm_1  = sm_sat(tSoil)
       zleak1 = zW1 - sm_sat(tSoil) * CAref * z(1) ![m3]
       pleak_1 = pleak_1 + zleak1 * rho_w / (ddts * CAref)
    else
       psm_1 = zW1 / (CAref * z(1))
    end if

    zW2 = zW2 + ddts * (pleak_1 - pleak_2 - plETv&
         &* pfc / l) * CAref / rho_w
    if (zW2 / (CAref * z(2)) > sm_sat(tSoil)) then
       psm_2  = sm_sat(tSoil)
       zleak2 = zW2 - sm_sat(tSoil) * CAref * z(2) ![m3]
       pleak_2 = pleak_2 + zleak2 * rho_w / (ddts * CAref)
    else
       psm_2 = zW2 / (CAref * z(2))
    end if

  end subroutine UpdateWaterStatus

  !-----------------------------------------------
  subroutine UpdateAlbedoStatus(palbedo, pLAI)
    implicit none
    ! input
    real :: pLAI                ! leaf area index [1]
    ! output
    real :: palbedo             ! surface albedo [1]
    ! updata surface albedo
    palbedo = a_v_min + (a_v_max - a_v_min)&
         &* exp(-k * pLAI)
  end subroutine UpdateAlbedoStatus

  !-----------------------------------------------
  subroutine UpdateSoilStatus(psm_1, psm_2, pt_sv, pt_sb,&
       &pt_1, pt_2, pfc)
    implicit none
    ! input
    real :: psm_1       ! soil moisture at layer 1[m3/m3]
    real :: psm_2       ! soil moisture at layer 2[m3/m3]
    real :: pt_sv       ! veg surface temperature [K]
    real :: pt_sb       ! bare surface temperature [K]
    real :: pfc         ! leaf coverage [m3.m-3]
    ! output
    real :: pt_1        ! soil temperature at layer 1 [K]
    real :: pt_2        ! soil temperature at layer 2 [K]
    real :: zt_s        ! average surface temperature [K]
    real :: zcap_soil1  ! soil capacity at layer 1 [J.m-3.K-1]
    real :: zcap_soil2  ! soil capacity at layer 2 [J.m-3.K-1]
    real :: zG1         ! soil heat flux through layer 1 [w/m2]
    real :: zG2         ! soil heat flux through layer 2 [w/m2]
    real :: zts
    zts = real(ddts)
    ! soil temperature updated
    zt_s = pfc * pt_sv + (1-pfc) * pt_sb
    zcap_soil1 = rsCs * (1 - sm_sat(tSoil)) + psm_1 * rwCw
    zcap_soil2 = rsCs * (1 - sm_sat(tSoil)) + psm_2 * rwCw

    ! calculate soil heat flux
    zG1 = - con_soil * (pt_1 - zt_s) * 2 / z(1)
    zG2 = - con_soil * (pt_2 - pt_1) * 2 / (z(1) + z(2))

    pt_1 = pt_1 - zts * (1/zcap_soil1) * (zG2 - zG1) / z(1)
    pt_2 = pt_2 - zts * (1/zcap_soil2) * (  0 - zG2) / z(2)


  end subroutine UpdateSoilStatus

  !-----------------------------------------------
  function GetvpD(pts, pSH, pp_suf)
    implicit none
    real :: pts
    real :: pSH
    real :: pp_suf
    real :: GetvpD
    real :: ze_a
    real :: ze_s
    real :: test
    ze_a = pSH * pp_suf / (pSH + e - e*pSH)
    ze_s = e0(pts-273.16)*1e3
    test = e0(dt_a - 273.16)*1e3

    GetvpD = max(0.0, ze_s - ze_a)
  end function GetvpD

  !-----------------------------------------------
  function Diff(psm1, psm2)
    implicit none
    real :: psm1
    real :: psm2
    real :: zbc
    real :: zGsat            ![m/s]
    real :: zFsat            ![m]
    real :: zDiff1           !diffusivity
    real :: zDiff2           !diffusivity
    real :: Diff
    zbc   = 6.04
    zGsat = 0.57e-6
    zFsat = -0.338
    zDiff1 = -zbc*zGsat*zFsat*(psm1/sm_sat(tSoil))&
         &**(zbc+2.0)/sm_sat(tSoil)
    zDiff2 = -zbc*zGsat*zFsat*(psm2/sm_sat(tSoil))&
         &**(zbc+2.0)/sm_sat(tSoil)
    Diff   = 2.0*rho_w*(zDiff1 - zDiff2)*(psm2 - psm1)&
         &/(z(1) + z(2))
  end function Diff

  !-----------------------------------------------
  function Leakage(psm)
    implicit none
    real :: psm              ! soil moisture[m3.m-3]
    real :: Leakage          ! water leakage rate to deeper soil [kgH2O.m-2.s-1]
    real :: zh_soil          ! pressure head [m]
    real :: zh_soil_up       ! pressure head at uplayer[m]   
    real :: zh_soil_pwp      ! pressure head at wilting point[m]
    real :: zg_h             ! hydraulic conductivity [m.s-1]
    real :: zitem1           ! items for calculation
    real :: zitem2           ! items for calculation
    real :: zitem3           ! items for calculation
    real :: zm
    include 'constants.f90'

    zitem1  = a_s(tSoil)**(-1)
    if (psm > sm_r(tSoil)) then
       zitem2  = ((sm_sat(tSoil) - sm_r(tSoil)) / (psm&
            &- sm_r(tSoil)))**(n_s(tSoil)/(n_s(&
            &tSoil) - 1)) - 1
    else
       zitem2  = 1e30&
            &**(n_s(tSoil)/(n_s(tSoil) - 1)) - 1
    end if
    zitem3  = 1/n_s(tSoil)
    zh_soil = zitem1 * zitem2 ** zitem3

    zm = 1-1/n_s(tSoil)
    zitem1  = (1 + (a_s(tSoil) * zh_soil)&
         &**n_s(tSoil))**(-zm)
    zg_h    = G_sat(tSoil) * zitem1 ** 0.5&
         &* (1 - (1 - (zitem1)**(1/zm))**zm)**2

    Leakage = rho_w * zg_h

  end function Leakage

  !-----------------------------------------------
  function Interpolation(pa, pb, pj, pnum)
    implicit none
    ! input
    real :: pa               ! start value
    real :: pb               ! end value
    integer :: pj               ! number in the interpolation position
    integer :: pnum             ! how many parts separated in one time step
    real :: Interpolation

    Interpolation = pa + (pb - pa) * (pj - 1) / pnum
  end function Interpolation

  !-----------------------------------------------
  ! subroutine OutputAverage(pfile,pnum)
  !   character(len = 100) :: pfile !directory of file
  !   integer :: pnum               !number of columns in the data file
  !   real , dimension(pnum) :: zdata !data for average
  !   open(1, file = pfile)
  !   read(1,*)zdata(:)
  !   close(1)
  ! end subroutine OutputAverage

  !-----------------------------------------------
  subroutine GetGs(pts, ptVeg, pLAI, prho_a, psm,&
       &pGPP, pNPP, pg_s, pvpD, pP, pr_swd,&
       &pfc, pc_root, pca)
    ! input
    real :: pLAI             ![m2/m2]
    real :: prho_a           ![kgAir/m3]
    real :: psm              ![m3/m3]
    real :: pts              ![K]
    real :: pvpD             ![Pa]
    real :: pP               !surface pressure[Pa]
    real :: pr_swd           ![w/m2]
    real :: pfc              !crown area [m2]
    real :: pc_root
    real :: pca
    integer :: ptVeg
    ! media
    real :: zG_t
    real :: zf0              ![]
    real :: zgm_x            ![m/s]
    real :: zgm              ![m/s]
    real :: zDmax_s
    real :: zDmax_n
    real :: zDmax_x
    real :: zDmax
    real :: zf2              !relative soil moisture content
                                !(with effect of vegetation)
    real :: zf2c             !threshold point of soil moisture [1]
    real :: ztsc             !surface temperature with unit oC
    real :: zgm_s            ![m/s]
    real :: zgm_n            ![m/s]
    real :: zf0_n
    real :: zf0_s
    real :: zAm_max
    real :: zD_Dmax
    real :: zCmin
    real :: zAmin
    real :: zf
    real :: zAm
    real :: zEps
    real :: zRd
    real :: zKdf
    real :: zKdr
    real :: zKz
    real :: zIa
    real :: zAn
    real :: zgsc
    real :: zPAR
    real :: zgs
    real :: zvpD
    real :: zc_i
    integer :: zint
    ! output
    real :: pGPP
    real :: pNPP
    real :: pg_s
    real :: zfLAI

    ! transfer vpd unit from Pa to kg.kg-1
    zvpD = 0.622*pvpD/pP         ![kg.kg-1]
    zPAR = 0.48*pr_swd           !equal to 48% of incoming shortwave
                                 !radiation
    ztsc = pts - 273             !transfer unit from K to oC
    pGPP = 0                     !initial GPP, NPP and surface
                                 !conductance
    pNPP = 0
    pg_s = 0

    if (pc_root/pca < c_rm(ptVeg)) then !if root density smaller than
                                        !the maximum root density
       zf2 = pc_root * (psm - sm_pwp(tSoil))/&
            &((sm_cap(tSoil) - sm_pwp(tSoil))&
            &*pca*c_rm(ptVeg))
    else                        !no effect from root density
       zf2 = (psm - sm_pwp(tSoil))/&
            &(sm_cap(tSoil) - sm_pwp(tSoil))
    end if
    zf2 = max(0.0, zf2)         !keep 0<zf2<1
    zf2 = min(1.0, zf2)
    zDmax = 0.1                 ![kg/kg]
    zDmax_n = Dmax_n(ptVeg)
    zDmax_x = Dmax_x(ptVeg)
    zf0_s   = f0_s(ptVeg)
    zf0     = zf0_s
    zf2c    = f2c(ptVeg)
    zgm_s   = gm_s(ptVeg)

    if (ptVeg == 1 .OR. ptVeg == 2) then !grass
       zDmax_s = 1e-3*exp((ac - log(zgm_s*1e3))/bc)
       if(ptVeg == 1) then               !Def
          if (zf2 > zf2c) then
             zDmax = zDmax_n + (zDmax_s - zDmax_n)&
                  &*(zf2 - zf2c)/(1.0 - zf2c)
             zgm  = 1e-3*exp(ac - bc*log(zDmax*1e3))
          else
             zDmax = zDmax_n
             zgm   = 1e-3*exp(ac - bc*log(zDmax_n*1e3))&
                  &* zf2/zf2c
          end if
       else                              !Off
          if (zf2 > zf2c) then
             zDmax = zDmax_x + (zDmax_s - zDmax_x)&
                  &*(zf2 - zf2c)/(1.0 - zf2c)
             zgm   = 1e-3*exp(ac - bc*log(zDmax*1e3))
          else
             zDmax = zDmax_x*zf2/zf2c
             zgm   = 1e-3*exp(ac - bc*log(zDmax_x*&
                  &1e3))!*zf2/zf2c
          end if
       end if
    else                              !woody
       if(ptVeg == 3) then            !Def
          zgm = 1e-3*exp(dc - ec*zf0_s)
          zf0_n = (cc-log(zgm*1e3))/ec
          if (zf2 > zf2c) then
             zf0 = zf0_n + (zf0_s - zf0_n)*&
                  &(zf2 - zf2c)/(1.0 - zf2c)
          else
             zgm = zgm*zf2/zf2c
             zf0 = min(1.0, (cc - log(zgm*1e3))*zf2/&
                  &(ec*zf2c))
          end if
       else                           !Off
          zgm_s = 1e-3*exp(dc - ec*zf0_s)
          zgm_n = 1e-3*exp(cc - ec*zf0_s)
          if (zf2 > zf2c) then
             zf0 = zf0_s
             zgm = zgm_n + (zgm_s - zgm_n)*(zf2 -&
                  &zf2c)/(1.0 - zf2c)
          else
             zgm = zgm_n *zf2/zf2c
             zf0 = min(1.0, (cc - log(zgm*1e3))*zf2/&
                  &(ec*zf2c))
          end if
       end if
    end if

    ! calculate gm, Am_max and G_t at current surface temperature
    zgm = zgm * Q10**((ztsc-25)/10)/((1+exp(0.3*(T1_g&
         &- ztsc))) * (1+exp(0.3*(ztsc - T2_g))))
    zAm_max = Am_max(ptVeg) * Q10**((ztsc-25)/10)/((1+exp(0.3&
         &*(T1_A - ztsc)))*(1+exp(0.3*(ztsc - T2_A))))
    zG_t = G_x(ptVeg) * Q10**((ztsc-25)/10)

    zD_Dmax = min(1.0, zvpD/zDmax) !keep 0<zD_Dmax<1
    zD_Dmax = max(0.0, zD_Dmax)
    zf = zf0*(1 - zD_Dmax)
    zc_i = c_a * zf + (1 - zf)*zG_t
    zAm = zAm_max * (1 - exp(-12e-6*zgm*(zc_i-zG_t)&
         &/(zAm_max*22.4)))

    zEps = Eps0(ptVeg) * (zc_i - zG_t)/(zc_i + 2*zG_t)
    zRd = zAm/9.0

    zCmin = zG_t

    zAmin = zgm*(zCmin - zG_t)*12e-6*R*pts/pP

    ! start the integration
    do zint = 1, 3
       zKdf = 1-exp(-0.8*b2*pLAI*(1-x(zint))/2.0)
       zKdr = 1-exp(-Gr*b2*pLAI*(1-x(zint))/2.0)
       zKz  = fu*zKdf + (1 - fu)*zKdr
       zIa  = zPAR*(1-zKz)
       zIa  = max(0.0, zIa)
       if (zAm <= 0) then       !to avoid NaN
          zAn = 0
       else
          zAn  = (zAm + zRd) * (1-exp(-zEps * zIa/&
               &(zAm+zRd))) - zRd
       end if
       pNPP = pNPP + pLAI*w(zint)*zAn/2
       pGPP = pGPP + pLAI*w(zint)*(zAn+zRd)/2
       if (zIa == 0) then       !for the night case
          zgsc = 0
       else
          zgsc = 22.4e6 * (zAn + zRd * (1-(zAn + zRd)/(zAm&
               &+ zRd)))/(12*(c_a - zc_i))
       end if
       zgsc = max(0.0, zgsc)
       zgs  = 1.6 * zgsc
       pg_s = pg_s + pLAI*w(zint)*zgs/2
    end do

  end subroutine GetGs

  !-----------------------------------------------
  subroutine GetL(pL, prho_a, pJq, pJs, pJm, pTair, pQov, pc_p)
    ! input
    real :: prho_a           !air density [kgAir/m3]
    real :: pJq              !
    real :: pJs              !
    real :: pJm              !
    real :: pTair
    real :: pc_p
    ! media
    real :: zUx
    real :: zQx
    real :: zSx
    real :: zEps             !gas constants ratio
    ! output
    real :: pL
    real :: pQov
    zEps = 0.6

    zUx = sqrt(pJm/prho_a)

    pQov = (pJs - pTair * pJq *(c_v - c_d))&
         &/ (pc_p*prho_a) + zEps * pTair * pJq/prho_a

    pL = -zUx**3 * pTair / (k_1 * gv * pQov)

  end subroutine GetL

  !-----------------------------------------------
  subroutine UpdateFlux(pL, pCm, pCh, pCq, pJm,&
       &pJs, pJq, pTair, pTs, pe_a, pp_suf, pg_s,&
       &pUn, prho_a, pc_p)
    real :: pL
    real :: pCm
    real :: pCh
    real :: pCq
    real :: pJm
    real :: pJs
    real :: pJq
    real :: pTair
    real :: prho_a
    real :: pc_p
    real :: pTs
    real :: pe_a
    real :: pp_suf
    real :: pg_s
    real :: pUn
    real :: ze_s
    real :: zq_a
    real :: zq_s
    ze_s = e0(pTs - 273.16) * 1e3
    zq_s = e * ze_s / pp_suf
    zq_a = e * pe_a / pp_suf

    pCm = k_1**2 / (log((zn+z0m)/z0m) -&
         &PsiM((zn+z0m)/pL) + PsiM(z0m/pL))**2
    pCh = k_1**2 / ((log((zn+z0m)/z0m) - PsiM((zn+z0m)/pL)&
         &+ PsiM(z0m/pL)) * (log((zn+z0m)/z0h) - PsiH(&
         &(zn+z0m)/pL) + PsiH(z0h/pL)))
    pCq = k_1**2 / ((log((zn+z0m)/z0m) - PsiM((zn+z0m)/pL)&
         &+ PsiM(z0m/pL)) * (log((zn+z0m)/z0q) - PsiH(&
         &(zn+z0m)/pL) + PsiH(z0q/pL)))
    ! get new Jm, Js, Jq
    pUn = sqrt(pJm/(prho_a * pCm))
    pJs = prho_a * pCh * pUn * pc_p * (pTair - pTs)
    pJq = prho_a * pCq * pUn * pg_s * (zq_a - zq_s)

  end subroutine UpdateFlux

  !-----------------------------------------------
  function PsiM(pzeta)
    ! input
    real :: pzeta
    ! media
    real :: zx
    real, parameter :: zb = 2.0/3.0
    real, parameter :: za = 1.0
    real, parameter :: zc = 5.0
    real, parameter :: zd = 0.35
    ! output
    real :: PsiM

    zx = (1-16*pzeta)**0.25
    if (pzeta < 0.0) then
       PsiM = pi/2.0 - 2.0*atan(zx) + log(((1+zx)**2)&
            &*(1+zx**2)/8.0)
    else
       PsiM = -zb * (pzeta - zc/zd) * exp(-zd*pzeta)&
            &- za * pzeta - zb*zc/zd
    end if

  end function PsiM

  !-----------------------------------------------
  function PsiH(pzeta)
    real :: pzeta
    ! media
    real :: zx
    real, parameter :: zb = 2.0/3.0
    real, parameter :: za = 1.0
    real, parameter :: zc = 5.0
    real, parameter :: zd = 0.35
    ! output
    real :: PsiH

    zx = (1-16*pzeta)**0.25
    if (pzeta < 0.0) then
       PsiH = 2 * log((1.0 + zx**2)/2.0)
    else
       PsiH = -zb * (pzeta - zc/zd) * exp(-zd*pzeta)-&
            &(1+zb*za*pzeta)**1.5 - zb*zc/zd + 1
    end if
  end function PsiH
  !-----------------------------------------------
end program BOSVM
