% CryoGRID3 Xice
% main script

% Developed by: S. Westermann and M. Langer
% 21.11.2015
%
% -------------------------------------------------------------------------

clear all

%import CryoGrid modules
addpath('modules/cryoGridTechnical/')
addpath('modules/cryoGridSEB/')
addpath('modules/cryoGridSoil/')
addpath('modules/cryoGridSnow/')
addpath('modules/cryoGridExcessIce/')

%indentifies a run, dall data are stored in subirectory namend "run_number"
run_number='test';
mkdir(run_number)

%---- soil stratigraphy ---------------------------------------------------
% column 1: depth [m]; use a negative depth well above the higest possible
% snow cover as the first depth, e.g. 5m. This is needed for initialization.
% column 2: volumetric water content
% column 3: volumetric mineral content
% column 4: volumetric organic content
% column 5: not used
% column 6: code for soil type: 1: sand, 2: silt
% column 7: natural porosity - should be the same as 1-mineral-organic if no ground subsidence/thermokarst occurs


%Samoilov
PARA.soil.stratigraphy=[-5 0.25 0.2 0.05 0 1 0.75;  ...
    0.09 0.25 0.2 0.05 0 1 0.75; ...
    0.10 0.3 0.6 0.0 0 1 0.4; ...
    1.49 0.3 0.6 0 0 1 0.4; ...
    1.51 0.4 0.6 0 0 1 0.4; ...
    19.9 0.4 0.6 0 0 1 0.4; ...
    20.1 0.3 0.7 0 0 1 0.3; ...
    5000 0.3 0.7 0 0 1 0.3];

%------ model parameters --------------------------------------------------
PARA.soil.albedo=0.2;       % albedo snow-free surface
PARA.soil.albedoPond=0.07;  % albedo of water, used when the uppermost grod cell is 100% water due to modeled thermokarst development
PARA.soil.epsilon=0.97;     % emissvity snow-free surface
PARA.soil.z0=1e-3;          % roughness length [m] snow-free surface
PARA.soil.rs=50;            % surface resistance against evapotransiration [m^-1] snow-free surface
PARA.soil.Qgeo=0.05;        % geothermal heat flux [W/m2]
PARA.soil.kh_bedrock=3;     % thermal conductivity of the mineral soil fraction [W/mK]

PARA.soil.mobileWaterDomain=[0 15];  % soil domain where water from excess ice melt is mobile -> start and end [m] - if empty water is not mobile
PARA.soil.waterTable=0;              % depth at which a water table will form [m] - above excess water is removed, below it pools up
PARA.soil.perchedWaterTable=0;       % position of the perched water table relative to the surface
PARA.soil.saturationAbovePerchedWaterTable=1; % saturation of the layer above the perched water table

PARA.snow.max_albedo =0.85;   % albedo of fresh snow
PARA.snow.min_albedo=0.5;     % albedo of old snow
PARA.snow.epsilon=0.99;       % surface emissivity snow
PARA.snow.z0=5e-4; %1e-3;            % roughness length surface [m]
PARA.snow.rs=0;               % surface resistance -> should be 0 for snow
PARA.snow.rho_snow=225;       % density in [kg/m3]
PARA.snow.tau_1=86400;        % time constants of snow albedo change (according to ECMWF reanalysis) [sec]
PARA.snow.tau_a=0.008;
PARA.snow.tau_f=0.24;
PARA.snow.maxSnow=0.45;     % maximum snow depth that can be reached [m] - excess snow is removed in the model - if empty, no snow threshold
PARA.snow.albedo=PARA.snow.max_albedo;  % sets the initial albedo, do not modify
PARA.snow.extinction=25;    % light extinction coefficient of snow

PARA.technical.z=2;                         % height of input air temperature above ground in [m] - assumed constant even when snow depth increases
PARA.technical.SWEperCell=0.005;            % SWE per grid cell in [m] - determines size of snow grid cells
PARA.technical.maxSWE=0.3;                  % in [m] SWE
PARA.technical.arraySizeT=5002;             % number of values in the look-up tables for conductivity and capacity
PARA.technical.starttime=datenum(1979,5,1); % starttime of the simulation - if empty start from first value of time series
PARA.technical.endtime=[];                  % endtime of the simulation if empty end at last value of time series
PARA.technical.maxTimestep=0.1./3600./24;   % smallest possible time step in [days] - here 0.1 seconds
PARA.technical.minTimestep=300./3600./24;   % largest possible time step in [days] - here 300 seconds
PARA.technical.targetDeltaE=0.5e5;          % maximum energy change of a grid cell between time steps in [J/m3]
PARA.technical.outputTimestep=6/24;         % output time step in [days] - here one hour
PARA.technical.saveDate='01.01.';           % date of year when output file is written - no effect if "saveInterval" is empty
PARA.technical.saveInterval=1;              % interval [years] in which output files are written - if empty the entire time series is written - minimum is 1 year
PARA.technical.subsurfaceGrid = [[0:0.02:10] [10.05:0.05:18] [18.2:0.2:20] [21:1:30] [35:5:50] [60:10:100] [200:100:1000]]'; %the subsurface K-grid in [m]

% initial temperature profile -> first column depth [m] -> second column temperature [degree C]
%example profile - modify depending on your model application
PARA.Tinitial=[-5 5; 0 -10 ; 10 -10; 100 -8; 5000 10];


PARA.surf.albedo=PARA.soil.albedo;
PARA.surf.epsilon=PARA.soil.epsilon;
PARA.surf.z0=PARA.soil.z0;
PARA.surf.rs=PARA.soil.rs;

%---- loading of FORCING mat-file------------------------------------------
load('samoylov_forcing/samoylov_ERA_paper_1979_2014_spinup.mat')
FORCING.data.rainfall=FORCING.data.rainfall.*0;




%----------do not modify from here onwards---------------------------------
FORCING.i.snowfall=0;
FORCING.i.rainfall=0;
FORCING.i.Lin=0;
FORCING.i.Sin=0;
FORCING.i.Tair=0;
FORCING.i.wind=0;
FORCING.i.RH=0;
FORCING.i.q=0;

if isempty(PARA.technical.starttime)
    PARA.technical.starttime=FORCING.data.t_span(1,1);
end

if isempty(PARA.technical.endtime)
    PARA.technical.endtime=FORCING.data.t_span(end,1);
end

%----------------create and initialize the grids --------------------------
GRID.snow.snowCellSize=PARA.technical.SWEperCell/(PARA.snow.rho_snow/1000);
GRID.snow.snowGrid=[-1.*(PARA.technical.maxSWE./(PARA.technical.SWEperCell)+2).*GRID.snow.snowCellSize:GRID.snow.snowCellSize:-GRID.snow.snowCellSize]';

GRID.soil.soilGrid=PARA.technical.subsurfaceGrid;

K_grid =[GRID.snow.snowGrid; GRID.soil.soilGrid]; %grid on which the conductivty information is attached to (edges of grid cells)
cT_grid=(K_grid(1:end-1)+K_grid(2:end))/2; %grid on which heat capacity and temperature information lives (midpoints of grid cells)
cT_delta=(-cT_grid(1:end-1,1)+cT_grid(2:end,1));
K_delta=(-K_grid(1:end-1,1)+K_grid(2:end,1));

GRID.general.cT_grid=cT_grid;
GRID.general.K_grid=K_grid;
GRID.general.cT_delta=cT_delta;
GRID.general.K_delta=K_delta;

%set air grid
GRID.air.cT_domain= false(size(GRID.general.cT_grid));
GRID.air.cT_domain(1:length(GRID.snow.snowGrid))=1;
GRID.air.K_domain= false(size(GRID.general.K_grid));
GRID.air.K_domain(1:length(GRID.snow.snowGrid))=1;
[GRID.air.cT_domain_lb GRID.air.cT_domain_ub] = LayerIndex(GRID.air.cT_domain);
[GRID.air.K_domain_lb GRID.air.K_domain_ub]   = LayerIndex(GRID.air.K_domain);

%set soil grid
GRID.soil.cT_domain= false(size(GRID.general.cT_grid));
GRID.soil.cT_domain(end-length(GRID.soil.soilGrid)+2:end)=1;
GRID.soil.K_domain= false(size(GRID.general.K_grid));
GRID.soil.K_domain(end-length(GRID.soil.soilGrid)+1:end)=1;
[GRID.soil.cT_domain_lb GRID.soil.cT_domain_ub] = LayerIndex(GRID.soil.cT_domain);
[GRID.soil.K_domain_lb GRID.soil.K_domain_ub]   = LayerIndex(GRID.soil.K_domain);

%set snow grid
GRID.snow.cT_domain=logical(GRID.air.cT_domain.*0);
GRID.snow.K_domain=logical(GRID.air.K_domain.*0);
[GRID.snow.cT_domain_lb GRID.snow.cT_domain_ub] = LayerIndex(GRID.snow.cT_domain);
[GRID.snow.K_domain_lb GRID.snow.K_domain_ub]   = LayerIndex(GRID.snow.K_domain);


%interpolate input stratigraphy to the soil grid
[GRID.soil.cT_water,...
    GRID.soil.cT_mineral,...
    GRID.soil.cT_organic,...
    GRID.soil.cT_soilType,...
    GRID.soil.cT_natPor,...
    GRID.soil.K_water,...
    GRID.soil.K_mineral,...
    GRID.soil.K_organic,...
    GRID.soil.K_soilType] = createStratigraphy(PARA.soil.stratigraphy,...
    GRID.general.cT_grid(GRID.soil.cT_domain),...
    GRID.general.K_grid(GRID.soil.K_domain));

%----- initializie excess ground ice --------------------------------------
if isempty(PARA.soil.mobileWaterDomain)  % set the natural porosity that only water in "mobilewaterDomain" is mobile
    GRID.soil.cT_natPor=GRID.soil.cT_water;
else
    mobileWaterDomain = GRID.general.cT_grid(GRID.soil.cT_domain)>=PARA.soil.mobileWaterDomain(1) & GRID.general.cT_grid(GRID.soil.cT_domain)<=PARA.soil.mobileWaterDomain(2);
    GRID.soil.cT_natPor(~mobileWaterDomain)=GRID.soil.cT_water(~mobileWaterDomain);
end
GRID.soil.excessGroundIce = GRID.soil.cT_water>GRID.soil.cT_natPor; % lower the water table if air is present above the excess ground ice
firstCellExcessIce=find(GRID.soil.excessGroundIce(:,1)==1, 1, 'first');

if ~isempty(firstCellExcessIce) && firstCellExcessIce>1
    PARA.soil.waterTable=max(PARA.soil.waterTable,...
        sum((1 - GRID.soil.cT_water(1:firstCellExcessIce-1) - GRID.soil.cT_mineral(1:firstCellExcessIce-1) - GRID.soil.cT_organic(1:firstCellExcessIce-1))...
        .*GRID.general.K_delta(GRID.soil.cT_domain_ub:GRID.soil.cT_domain_ub+firstCellExcessIce-2)));
end

%----- initializie soil thermal properties --------------------------------
[GRID.soil.cT_frozen,...
    GRID.soil.cT_thawed,...
    GRID.soil.K_frozen,...
    GRID.soil.K_thawed,...
    GRID.soil.conductivity,...
    GRID.soil.capacity] = initialize(GRID.soil.cT_water,...
    GRID.soil.cT_mineral,...
    GRID.soil.cT_organic,...
    GRID.soil.cT_soilType,...
    GRID.soil.K_water,...
    GRID.soil.K_mineral,...
    GRID.soil.K_organic,...
    GRID.soil.K_soilType,...
    PARA.technical.arraySizeT,...
    GRID.general.cT_grid(GRID.soil.cT_domain),...
    PARA.soil.kh_bedrock);

%------ initializie snow properties----------------------------------------
GRID.snow.Snow_i=zeros(size(GRID.air.cT_domain));
GRID.snow.Snow_w=zeros(size(GRID.air.cT_domain));
GRID.snow.Snow_a=zeros(size(GRID.air.cT_domain));
GRID.snow.SWEinitial=0;

%---- initialize the surface energy balance struct ------------------------
SEB.Qnet=0;
SEB.Qh=0;
SEB.Qe=0;
SEB.Qg=0;
SEB.Sout=0;
SEB.Lout=0;
SEB.newSnow=0;
SEB.sublim=0;
SEB.meltwater=0;
SEB.L_star=-1000;
SEB.u_star=-1000;

%---- initialize temperature profile --------------------------------------
T = zeros(size(GRID.general.cT_grid));
T(GRID.air.cT_domain)   = FORCING.data.Tair(1);
T(GRID.soil.cT_domain)  = interp1(PARA.Tinitial(:,1), PARA.Tinitial(:,2), GRID.general.cT_grid(GRID.soil.cT_domain));

%---- preallocate temporary arrays ----------------------------------------
c_temp = zeros(size(GRID.general.cT_grid));
k_temp = zeros(size(GRID.general.cT_grid));
k_eff  = zeros(size(GRID.general.K_grid));

%------- unused grid cells --------------------------------
c_temp(GRID.air.cT_domain) = 4e5;   %set some value e.g. air
k_temp(GRID.air.cT_domain) = 0.025; %set some value e.g. air

%------- soil domain --------------------------------------
[c_temp(GRID.soil.cT_domain),...
    k_temp(GRID.soil.cT_domain)] = readThermalParameters(T(GRID.soil.cT_domain),...
    GRID.soil.cT_frozen,...
    GRID.soil.cT_thawed,...
    GRID.soil.capacity,...
    GRID.soil.K_frozen,...
    GRID.soil.K_thawed,...
    GRID.soil.conductivity,...
    PARA.technical.arraySizeT);

%------- snow domain --------------------------------------
c_temp(GRID.snow.cT_domain) = cap_snow(GRID.snow.Snow_i(GRID.snow.cT_domain),...
    GRID.snow.Snow_w(GRID.snow.cT_domain),...
    GRID.snow.Snow_a(GRID.snow.cT_domain));

k_temp(GRID.snow.cT_domain) = cond_snow(GRID.snow.Snow_i(GRID.snow.cT_domain),...
    GRID.snow.Snow_w(GRID.snow.cT_domain),...
    GRID.snow.Snow_a(GRID.snow.cT_domain));

%__________________________________________________________________________
%-------- provide arrays for data storage ---------------------------------
t=PARA.technical.starttime;
outputTime=t+PARA.technical.outputTimestep;
if ~isempty(PARA.technical.saveInterval)
    saveTime=datenum(str2num(datestr(t,'yyyy'))+1,  str2num(PARA.technical.saveDate(4:5)), str2num(PARA.technical.saveDate(1:2)))-PARA.technical.outputTimestep;
else
    saveTime=PARA.technical.endtime+PARA.technical.outputTimestep;
end

OUT = generateOUT();

Qh_sum=0;
Qe_sum=0;
Qnet_sum=0;
Qg_sum=0;
timestep_sum=0;

T_sum=0.*T;

disp('initialization successful');
save([run_number '/' run_number '_settings.mat'], 'FORCING', 'PARA', 'GRID')

%% ________________________________________________________________________
%  Time Integration
%_________________________________________________________________________
t_last=t;
counter=0;

while t<PARA.technical.endtime
    
    %------ interpolate forcing data to time t ----------------------------
    [FORCING]= interpolateForcingData(t, FORCING);
    
    % determine the thermal properties of the model domains ________
    %------- unused grid cells --------------------------------------------
    c_temp(GRID.air.cT_domain) = 4e5;   %set some value e.g. air
    k_temp(GRID.air.cT_domain) = 0.025; %set some value e.g. air
    
    %------- soil domain --------------------------------------------------
    [c_temp(GRID.soil.cT_domain),...
        k_temp(GRID.soil.cT_domain)] = readThermalParameters(T(GRID.soil.cT_domain),...
        GRID.soil.cT_frozen,...
        GRID.soil.cT_thawed,...
        GRID.soil.capacity,...
        GRID.soil.K_frozen,...
        GRID.soil.K_thawed,...
        GRID.soil.conductivity,...
        PARA.technical.arraySizeT);
    
    %------- snow domain --------------------------------------------------
    c_temp(GRID.snow.cT_domain) = cap_snow(GRID.snow.Snow_i(GRID.snow.cT_domain),...
        GRID.snow.Snow_w(GRID.snow.cT_domain),...
        GRID.snow.Snow_a(GRID.snow.cT_domain));
    
    k_temp(GRID.snow.cT_domain) = cond_snow(GRID.snow.Snow_i(GRID.snow.cT_domain),...
        GRID.snow.Snow_w(GRID.snow.cT_domain),...
        GRID.snow.Snow_a(GRID.snow.cT_domain));
    
    %------- interpolate conductivity to K-grid ---------------------------
    k_eff(2:end-1) = GRID.general.K_delta(1:end-1)./(2.*GRID.general.cT_delta) .* (1./k_temp(1:end-1)).^2 ...
        + GRID.general.K_delta(2:end)  ./(2.*GRID.general.cT_delta) .* (1./k_temp(2:end)).^2;
    k_eff(2:end-1) = k_eff(2:end-1).^(-0.5);
    k_eff(1)     = k_temp(1);
    k_eff(end)   = k_temp(end);
    
    %------ correct upper most value below air-domain ---------------------
    k_eff(GRID.air.K_domain_lb+1) = k_temp(GRID.air.K_domain_lb+1);
    
    %______________________________________________________________________
    %------ surface energy balance module ---------------------------------
    %set surface conditions
    PARA = surfaceCondition(GRID, PARA);
    
    %calculate the surface energy balance
    [dE_dt_SEB, SEB.Qnet, SEB.Qh, SEB.Qe, SEB.Qg, SEB.Sout, SEB.Lout, SEB.Sin_water] ...
        = surfaceEnergyBalance(T, FORCING, GRID, PARA, SEB.L_star);
    
    %------ soil module  --------------------------------------------------
    %calculate heat conduction
    dE_dt_cond=heatConduction(T,...
        PARA.soil.Qgeo,...
        k_eff,...
        GRID.general.cT_delta,...
        GRID.air.cT_domain_lb);
    
    %------ sum up fluxes --------------------------------------------
    dE_dt = dE_dt_SEB + dE_dt_cond;
    
    %------ determine optimal timestep ------------------------------------
    timestep=min([max([min([PARA.technical.minTimestep,...
        PARA.technical.targetDeltaE./(max(abs(dE_dt./GRID.general.K_delta).*(24.*3600)))]),...
        PARA.technical.maxTimestep]), outputTime-t]);
    
    %------ update T array ------------------------------------------------
    T = T + dE_dt./ c_temp./GRID.general.K_delta.*timestep.*24.*3600;
    T(GRID.air.cT_domain)=FORCING.i.Tair;
    
    %------ Xice water body scheme ----------------------------------------
    %simple water body scheme, mixes all water grid cells if the surface cell is unfrozen
    T(GRID.soil.cT_domain)=mixWaterLayer2(T(GRID.soil.cT_domain), GRID);
    
    %------- snow cover module --------------------------------------------
    [T GRID PARA SEB] = CryoGridSnow(T, GRID, FORCING, SEB, PARA, c_temp, timestep);
    [GRID T] = updateGRID_snow(T, GRID, PARA);
    
    %------- excess ice module --------------------------------------------
    if ~isempty(PARA.soil.mobileWaterDomain) && (sum(double(T(GRID.soil.cT_domain)>0 & GRID.soil.excessGroundIce==1))~=0) %&& isempty(GRID.snow.cT_domain_ub)
        GRID.soil.excessGroundIce = GRID.soil.excessGroundIce==1 & T(GRID.soil.cT_domain)<=0;  %remove the thawed cell from the list
        [GRID meltwaterGroundIce PARA] = excessGroundIceThaw4(T, GRID, PARA);
        PARA=setNewSurfaceParameters(GRID, PARA);    %modifies the surface parameters albedo and rs
    end
    
    %------- update Lstar for next time step ------------------------------
    [SEB.L_star SEB.u_star] = L_star(FORCING.i.wind,...
        PARA.technical.z,...
        PARA.surf.z0,...
        FORCING.i.Tair,...
        SEB.Qh,...
        SEB.Qe,...
        SEB.L_star);
    
    %__________ OUTPUT part _______________________________________________
    %---------- sum up ----------------------------------------------------
    timestep_sum=timestep_sum+(timestep*24*3600)*timestep;
    T_sum=T_sum+T.*timestep;
    Qe_sum=Qe_sum+SEB.Qe.*timestep;
    Qh_sum=Qh_sum+SEB.Qh.*timestep;
    Qnet_sum=Qnet_sum+SEB.Qnet.*timestep;
    Qg_sum=Qg_sum+SEB.Qg.*timestep;
    
    %----store in output table --------------------------------------------
    if  t==outputTime
        
        counter = counter+1;
        
        %average over timespan:
        dt_out=t-t_last;
        
        Qh=Qh_sum./dt_out;
        Qe=Qe_sum./dt_out;
        Qnet=Qnet_sum./dt_out;
        Qg=Qg_sum./dt_out;
        timestep_out=timestep_sum./dt_out;
        
        T_out=T_sum./dt_out;
        t_last=t;
        T_sum(:)=0;
        Qh_sum=0;
        Qe_sum=0;
        Qnet_sum=0;
        Qg_sum=0;
        timestep_sum=0;
        
        OUT.cryoGrid3=[OUT.cryoGrid3 [NaN(GRID.air.cT_domain_lb,1); T_out(GRID.air.cT_domain_lb+1:end,1)]];
        OUT.TIMESTEP=[OUT.TIMESTEP; timestep_out];
        OUT.timestamp=[OUT.timestamp; t];
        
        OUT.snow.outSnow_i=[OUT.snow.outSnow_i GRID.snow.Snow_i];
        OUT.snow.outSnow_a=[OUT.snow.outSnow_a GRID.snow.Snow_a];
        OUT.snow.outSnow_w=[OUT.snow.outSnow_w GRID.snow.Snow_w];
        
        OUT.SEB.Lsta=[OUT.SEB.Lsta SEB.L_star];
        OUT.SEB.QE=[OUT.SEB.QE; Qe];
        OUT.SEB.QH=[OUT.SEB.QH; Qh];
        OUT.SEB.QG=[OUT.SEB.QG; Qg];
        OUT.SEB.QNET=[OUT.SEB.QNET; Qnet];
        OUT.SEB.Tsurf=[OUT.SEB.Tsurf; T_out(GRID.air.cT_domain_lb+1,1)];
        OUT.SEB.runoff=[OUT.SEB.runoff; SEB.meltwater];
        OUT.SEB.sublimation=[OUT.SEB.sublimation; SEB.sublim];
        OUT.SEB.albedo_stored=[OUT.SEB.albedo_stored; PARA.surf.albedo];
        
        SEB.sublim=0;
        SEB.meltwater=0;
        
        OUT.soil.soil{1, size(OUT.soil.soil,2)+1}=[GRID.soil.cT_water GRID.soil.cT_mineral GRID.soil.cT_organic];
        OUT.soil.topPosition=[OUT.soil.topPosition; -GRID.general.K_grid(GRID.soil.cT_domain_ub)];
        OUT.soil.lakeDepth=[OUT.soil.lakeDepth; -GRID.general.K_grid(GRID.soil.cT_domain_ub+sum(GRID.soil.cT_water==1))];
        
        OUT.snow.topPosition=[OUT.snow.topPosition; -GRID.general.K_grid(GRID.snow.cT_domain_ub)];
        OUT.snow.botPosition=[OUT.snow.botPosition; -GRID.general.K_grid(GRID.snow.cT_domain_lb+1)];
        
        disp([datestr(now,'yyyy-mm-dd HH:MM:SS'),':  at ',datestr(t), ',  Average timestep: ',  num2str(timestep_out), ' seconds'])
        outputTime=round((outputTime+PARA.technical.outputTimestep)./PARA.technical.outputTimestep).*PARA.technical.outputTimestep;
        
        if  round((t-saveTime).*48)==0   % write yearly files
            save([run_number '/' run_number '_output' datestr(t,'yyyy')  '.mat'], 'OUT')
            OUT = generateOUT();
            saveTime=datenum(str2num(datestr(t,'yyyy'))+1, str2num(datestr(t,'mm')), str2num(datestr(t,'dd')), str2num(datestr(t,'HH')), str2num(datestr(t,'MM')), 0);
        end
    end
    %______________________________________________________________________
    %------- next time step -----------------------------------------------
    t=t+timestep;
    
end

save([run_number '/' run_number '_output' datestr(t,'yyyy')  '.mat'], 'OUT')






