% Tijn Berends, 11-02-2016
% c.j.berends@uu.nl
%
% This script demonstrates the optimized flood-fill algorithm for determing
% the land/ocean-mask using a 40km - 5km - 1km resolution sequence.
% Two demo data files, "Data_PD.nc" and "Data_30Ky.nc", are required for
% running the script. Please contact the author for acquiring them.
% 
% The program requires the following fields:
% - Hb_1km, Hb_5km, Hb_40km:  bedrock topography at different resolutions
% - Hbmax_5km, Hmax_40km:     maximum bedrock topography at low resolutions
% - Hi_1km, Hi_5km, Hi_40km:  ice thickness at different resolutions
% - SL_1km, SL_5km, SL_40km:  geoid at different resolutions

clc
clear all
close all

Hb_40km     = ncread('Data_PD.nc','Hb_40km');
Hb_5km      = ncread('Data_PD.nc','Hb_5km');
Hb_1km      = ncread('Data_PD.nc','Hb_1km');
Hi_40km     = ncread('Data_PD.nc','Hi_40km');
Hi_5km      = ncread('Data_PD.nc','Hi_5km');
Hi_1km      = ncread('Data_PD.nc','Hi_1km');
SL_40km     = ncread('Data_PD.nc','SL_40km');
SL_5km      = ncread('Data_PD.nc','SL_5km');
SL_1km      = ncread('Data_PD.nc','SL_1km');
Hbmax_40km  = ncread('Data_PD.nc','Hbmax_40km');
Hbmax_5km   = ncread('Data_PD.nc','Hbmax_5km');

% Sizes of the different arrays
C.NX40 = 181;         % Number of elements in the X (east-west)   direction on 40km resolution.
C.NY40 = 121;         % Number of elements in the Y (north-south) direction on 40km resolution.
C.NX5  = C.NX40 * 8;  % Same on 8km resolution.
C.NY5  = C.NY40 * 8;
C.NX1  = C.NX40 * 40; % Same on 1km resolution.
C.NY1  = C.NY40 * 40;

% Declaration of the Map and Stack arrays, used in the different Fill
% routines. The sizes of the Stack arrays have been experimentally set -
% the biggest possible sea (present-day) will use about 90% of the Stacks.

Map_40km       = zeros(C.NX40, C.NY40);
Map_5km        = zeros(C.NX5,  C.NY5 );
Map_1km        = zeros(C.NX1,  C.NY1 );
nMap_40km      = zeros(C.NX40, C.NY40);
nMap_5km       = zeros(C.NX5,  C.NY5 );
nMap_1km       = zeros(C.NX1,  C.NY1 );

Stack_40km     = zeros(1500,3);
Stack_5km      = zeros(25000,3);
Stack_1km      = zeros(220000,3);
nStack_40km    = zeros(1500,3);
nStack_5km     = zeros(25000,3);
nStack_1km     = zeros(220000,3);
StackN_40km    = 1;
StackN_5km     = 1;
StackN_1km     = 1;
nStackN_40km   = 1;
nStackN_5km    = 1;
nStackN_1km    = 1;

% The final sea depth field.
S_1km       = zeros(C.NX1,C.NY1);

%% Fill in the sea
% First do a 40km Hbmax fill.

% Start with seeds at middle western, middle eastern and middle northern
% edge of map. When looking at the 30 Ky data these three "basins" are not
% connected in this region so they need separate seeds.

nMap_40km(1,60)       = 1;
nStack_40km(1,:)      = [1, 60, 1];
nMap_40km(C.NX40, 60) = 1;
nStack_40km(2,:)      = [C.NX40, 60, 1];
nMap_40km(70, C.NY40) = 1;
nStack_40km(3,:)      = [70, C.NY40, 1];
nStackN_40km          = 4;

% Location and water depth of the "local minimum". These are needed when
% filling a lake, not so much when filling the sea. The Fill routine is
% designed to handle both.

lmx_40km = 1;
lmy_40km = 1;
Depth    = SL_40km(lmx_40km,lmy_40km) - Hbmax_40km(lmx_40km,lmy_40km);
IsSea    = true;

disp('Performing 40km Hbmax fill...')
tic
[Map_40km, Stack_40km, StackN_40km, nMap_40km, nStack_40km, nStackN_40km, Overflows] = Fill_40km(...
  C, nMap_40km, nStack_40km, nStackN_40km, Hbmax_40km, SL_40km, Hi_40km, lmx_40km, lmy_40km, Depth, S_1km, IsSea);
disp('Finished performing 40km Hbmax fill.')
toc

disp('Converting Map and Stack arrays from 40km to 8km...')
tic
[nMap_5km, nStack_5km, nStackN_5km] = Map_Stack_40km_to_5km(C, nMap_40km, nStack_40km, nStackN_40km, nMap_5km, nStack_5km, nStackN_5km);
disp('Finished converting M arrays from 40km to 8km.')
toc

% Reset local minimum location and depth for the 5km Fill routine.
lmx_5km = 1;
lmy_5km = 1;
Depth    = SL_5km(lmx_5km,lmy_5km) - Hbmax_5km(lmx_5km,lmy_5km);
IsSea    = true;

disp('Performing 8km Hbmax fill...')
tic
[Map_5km, Stack_5km, StackN_5km, nMap_5km, nStack_5km, nStackN_5km, Overflows] = Fill_5km(...
  C, nMap_5km, nStack_5km, nStackN_5km, Hbmax_5km, SL_5km, Hi_5km, lmx_5km, lmy_5km, Depth, S_1km, IsSea);
disp('Finished performing 8km Hbmax fill.')
toc

disp('Converting Map and Stack arrays from 8km to 1km...')
tic
[nMap_1km, nStack_1km, nStackN_1km] = Map_Stack_5km_to_1km(C, nMap_5km, nStack_5km, nStackN_5km, nMap_1km, nStack_1km, nStackN_1km);
disp('Finished converting M arrays from 8km to 1km.')
toc

% Reset local minimum location and depth for the 1km Fill routine.
lmx_1km = 1;
lmy_1km = 1;
Depth    = SL_1km(1,1) - Hb_1km(1,1);
IsSea    = true;

disp('Performing 1km Hb fill...')
tic
[Map_1km, Stack_1km, StackN_1km, nMap_1km, nStack_1km, nStackN_1km, Overflows] = Fill_1km(...
  C, nMap_1km, nStack_1km, nStackN_1km, Hb_1km, SL_1km, Hi_1km, lmx_1km, lmy_1km, Depth, S_1km, IsSea);
disp('Finished performing 1km Hb fill.')
toc

H = ShowMap(nMap_1km,Hi_1km);