function [x, value, data] = SolveProblem(M, y, options)
    %SolveProblem minimizes objective norm(M*x-y)^2 + alpha_1*norm(x)^2 + alpha_2*sum(x) subject to x >= 0
    % Depending on options, alphas are chosen accordingly to
    % options.regularization = 'Tikhonov' .... alpha_1 > 0, alpha_2 = 0
    % options.regularization = 'LASSO'    .... alpha_1 = 0, alpha_2 > 0
    % options.regularization = 'OLS'      .... alpha_1 = 0, alpha_2 = 0
    % For the last method, the matrix is purged by removing columns with small norm. To prevent this, set options.columnNormKeep = 0.
    % This package has been written by Lukas Adam, for further information, contact me at adam@utia.cas.cz
    
    %% Renorm the data (it often happens that M has very small norm)
    normConstant1 = max(M(:));
    MMod          = M / normConstant1;
    normConstant2 = max(y);
    yMod          = y / normConstant2;
    normConstant  = normConstant2 / normConstant1;
    %% Set parameters (options and alphas)
    if nargin < 3 || isempty(options)
        options = struct();
    end
    options = SetDefaultValue(options, 'regularization', 'Tikhonov');
    options = SetDefaultValue(options, 'alpha', 1e-3);
    options = SetDefaultValue(options, 'columnNormKeep', 1e-3);
    switch options.regularization
        case 'Tikhonov'
            options.columnNormKeep = 0;
            alpha1                 = options.alpha;
            alpha2                 = 0;
        case 'LASSO'
            options.columnNormKeep = 0;
            alpha1                 = 0;
            alpha2                 = options.alpha;
        case 'OLS'
            alpha1                 = 0;
            alpha2                 = 0;
        otherwise
            error('Regularization not defined. Please change options.regularization.');
    end
    % Remove the columns with small norm (for Tikhonov and LASSO only columns with zero norm are removed)
    columnNorm = sqrt(sum(MMod.^2));
    columnKeep = columnNorm > options.columnNormKeep*max(columnNorm);
    MMod       = MMod(:,columnKeep);
    x          = zeros(size(M,2),1);
    if cond(MMod'*MMod) >= 1e10
        fprintf('Matrix A''A is badly scaled. Check below for convergence.\n');
    end
    %% Run the optimization and do some postprocessing
    [x(columnKeep), lambda, iterNumber, residual]  = SolveSemismooth(MMod'*MMod+alpha1*eye(size(MMod,2)), -MMod'*yMod+alpha2*ones(size(MMod,2),1), zeros(size(MMod,2),1));
    if residual <= 1e-10
        converged = 1;
        fprintf('The optimization converged in %d iterations with residual = %d\n', iterNumber, residual);
    else
        converged = 0;
        fprintf('Hey, something went wrong. The optimization were not able to find the solution.');
    end    
    x                   = x*normConstant;
    value               = 0.5*norm(M*x-y)^2;
    % Save the variables
    data.x              = x;
    data.value          = value;    
    data.lambda         = lambda;
    data.iterNumber     = iterNumber;
    data.residual       = residual;
    data.alpha          = [alpha1 alpha2];
    data.columnNormKeep = options.columnNormKeep;
    data.converged      = converged;
end

function options = SetDefaultValue(options, fieldName, value)
    if ~isfield(options, fieldName) || isempty(options.(fieldName))
        options.(fieldName) = value;
    end
end