import sys

from numpy import *
from scipy import constants as co
import h5py

from IPython import embed

HEADER = "LUQUE {n} ELECTRONS:  ID (-1 for electrons, 0 for photons, 1 for positrons)   \t 	E (keV)	\t	x (cm)	\t	y (cm)	\t	z (cm)	\t	t (s)"


def get_parser():
    import argparse

    parser = argparse.ArgumentParser()
    parser.add_argument("ifiles", nargs='+', help="Input file(s)")
    parser.add_argument("--outfile", "-o", required=True, help="Output file")
    parser.add_argument("--forward", "-f", default=False, action='store_true',
                        help="Only count forward-directed particles")
    parser.add_argument("--noback", "-b", default=False, action='store_true',
                        help="Remove back-scatter (very slow)")

    parser.add_argument("--nparticles", default='XX',
                        help="Output file")

    return parser


def main():
    fname = sys.argv[1]
    parser = get_parser()
    args = parser.parse_args()

    pt, rt, tt = [], [], []
    for fname in args.ifiles:
        print(fname)
        f = h5py.File(fname, 'r')
        crossings = f['crossings']
        pid, p, r, t, wall = [array(crossings[k])
                              for k in ('id', 'p', 'r', 't', 'wall')]
        if args.forward:
            flt = p[:, 2] >=0
            p = p[flt, :]
            r = r[flt, :]
            t = t[flt]
            wall = wall[flt]
            pid = pid[flt]
            
        if args.noback:
            flt = filter_backscatter(pid, wall, t)
            p = p[flt, :]
            r = r[flt, :]
            t = t[flt]
            wall = wall[flt]
            pid = pid[flt]
            print("Skipping %d crossings" % (len(nonzero(logical_not(flt))[0])))
            
        pt.append(p)
        rt.append(r)
        tt.append(t)

    p, r, t = [concatenate(v) for v in (pt, rt, tt)]
    
    m = co.electron_mass
    p2 = sum(p**2, axis=1)
    eng = sqrt(p2 * co.c**2 + m**2 * co.c**4)
    k   = eng - m * co.c**2

    ids = full_like(eng, -1)

    savetxt(args.outfile,
            c_[ids, k / co.eV / co.kilo,
               r[:, 0] / co.centi, r[:, 1] / co.centi, r[:, 2] / co.centi, t],
            header=HEADER.format(n=args.nparticles))
    

def filter_backscatter(pid, wall, t):
    crossings = {}
    pid = pid.astype('i')
    flt = zeros_like(pid, dtype=bool)

    for i in range(len(pid)):
        # Assume a well ordering of t
        # (i.e. for each wall, crossings are in order)
        try:
            iprev, tprev = crossings[(pid[i], wall[i])]
            if t[i] < tprev:
                flt[i] = True
                flt[iprev] = False
                crossings[(pid[i], wall[i])] = (i, t[i])
                
        except KeyError:
            flt[i] = True
            crossings[(pid[i], wall[i])] = (i, t[i])

    return flt
    
        
if __name__ == '__main__':
    main()
    
    
