function UncertaintyAnalysis(fileName,windowsize,Porder,Tnominal)
% SedimentationRateEstimationPlusUncertainty - Estimates the sedimentation rate from 
% the estimated waveforms of the orbital components, together with
% corresponding uncertainties.
%
% About:
%   This script is part of the complementary material for the manuscript
%   entitled "Astronomical component estimation (ACE v.1) by time-variant
%   sinusoidal modeling", submitted to Geoscientific Model Development.
%
% Input:
%   fileName          *.mat file containing the estimated orbital
%                     component waveforms
%   windowsize        length of the analysis frame in meters
%   Porder            number of terms for the polynomial approximation.
%   Tnominal          nominal orbital component period in kyr
%
% Description:
%   From spatial-domain waveforms it estimates the instantaneous frequency
%   and sedimentation rate for orbital components. By means of Monte Carlo
%   simulations it provides the uncertainties on the sought instantaneous
%   frequency and sedimentation rate (one standard deviation).
% Example:
%   SedimentationRateEstimationPlusUncertainty('EstimatedComponents',25,3,[100 41])


ss = load(fileName);
signal = ss.orbitalComp;
fs = ss.fs;
depth = ss.depth;

[~,Nsources] = size(signal);

N = length(signal);
windowsize = round(windowsize*fs); % in samples

inc = round(windowsize/2);
Position = 1:inc:round(length(signal)-windowsize+1);

freqInstVar = zeros(N,Nsources);
SRVar = zeros(N,Nsources);

numItera = 100;
for p = 1:numItera % Monte Carlo iterations

    f0est = zeros(length(Position),Nsources);
    ooff = 0;
    normo = zeros(N,1);
    win = hanning(windowsize);
    winHarm = repmat(win,1,Nsources);
    
    freqInst = zeros(N,Nsources);
    f0inst = zeros(windowsize,Nsources);
    
    freqInstnew = zeros(N,Nsources);
    f0instnew = zeros(windowsize,Nsources);

    for i = 1:length(Position)
        seg = signal(Position(i):Position(i)+windowsize-1,:);
                
        for k = 1:Nsources
            x = seg(:,k);
            F = 2048; % FFT size
            X = abs(fft(x,F)).^2/N; % periodogram
            [~,maxInd] = max(X(1:F/2));
            f0est(i,k) = (maxInd-1)*fs/F; % estimated mean frequency
            
            [segEst,a,b,acov,bcov] = PolyEstimationWithUncertainty(seg(:,k),f0est(i,k),fs,Porder); % component waveform estimate
            A = sqrt(a.^2 + b.^2); % estimated envelope of an orbital component
            segFM = segEst./A; % remove envelope to obtain pure phase modulated signal
            Acov = sqrt(acov.^2 + bcov.^2); % estimated perturbed envelope
            segFMcov = segEst./Acov; % estimated perturbed phase modulated signal
            
            [~,c,d] = PolyEstimation(segFM,f0est(i,k),fs,Porder); % waveform estimate on pure phase modulated signal
            argEst = unwrap(-atan2(c,d)); % unwrap the instantaneous phase
            F0 = f0est(i,k) + (1/(2*pi))*diff(argEst)/(1/fs); % estimate instantaneous frequency
            f0inst(:,k) = [F0; F0(end)]';
            
            [~,~,~,ccov,dcov] = PolyEstimationWithUncertainty(segFMcov,f0est(i,k),fs,Porder); % waveform estimate on perturbed phase modulated signal
            argEstnew = unwrap(-atan2(ccov,dcov)); % unwrap the instantaneous phase of perturbed phase modulated signal
            F0 = f0est(i,k) + (1/(2*pi))*diff(argEstnew)/(1/fs); % estimate perturbed instantaneous frequency
            f0instnew(:,k) = [F0; F0(end)]';
        end
        
        freqInst(round(ooff)+windowsize,:) = 0; % combine the estimated segments
        freqInst((1:windowsize)+round(ooff),:) = freqInst((1:windowsize)+round(ooff),:) + f0inst.*winHarm;
        
        freqInstnew(round(ooff)+windowsize,:) = 0; % combine the estimated segments
        freqInstnew((1:windowsize)+round(ooff),:) = freqInstnew((1:windowsize)+round(ooff),:) + f0instnew.*winHarm;
        
        normo(round(ooff)+windowsize) = 0; % overlap-add procedure to account on the record edges
        normo((1:windowsize)+round(ooff)) = normo((1:windowsize)+round(ooff)) + win;
        ooff = ooff + inc;
    end
    freqInst = freqInst./repmat(normo,1,Nsources); % "True" instantaneous frequency
    freqInstnew = freqInstnew./repmat(normo,1,Nsources); % "Perturbed" instantaneous frequency    
    freqInstVar = freqInstVar + (freqInst - freqInstnew).^2; % cummulative depth-dependent instantaneous frequency variance
    
    SR = zeros(N,Nsources); % sedimentation rate in cm/kyr
    for r = 1:Nsources % SR for Nsources orbital components
        T = 100./freqInst(:,r); % x100 is to get centimeters
        SR(:,r) = T/Tnominal(r);
    end
    
    SRnew = zeros(N,Nsources); % perturbed sedimentation rate in cm/kyr
    for r = 1:Nsources % SR for Nsources orbital components
        T = 100./freqInstnew(:,r); % x100 is to get centimeters
        SRnew(:,r) = T/Tnominal(r);
    end

    SRVar = SRVar + (SR - SRnew).^2; % cummulative depth-dependent sedimentation rate variance
end

freqInstStd = sqrt(freqInstVar/(numItera-1)); % Instantaneous frequency uncertainty
SRStd = sqrt(SRVar/(numItera-1)); % sedimentation rate uncertainty


save('freqInstStd','freqInstStd'); % save output
save('SRStd.mat','SRStd'); % save output

return





function [signalEst,a,b,THETA] = PolyEstimation(signal,f0,fs,Porder)

N = length(signal);
t = (0:N-1)'/fs;

T = zeros(N,Porder); % time-basis for the polynomials
for i = 1:Porder
    T(:,i) = t.^(i-1);
end

X1 = repmat(sin(2*pi*f0*t),1,Porder); % model matrix - sine
X2 = repmat(cos(2*pi*f0*t),1,Porder); % model matrix - cosine

X1 = X1 .* T;
X2 = X2 .* T;
FI = [X1 X2];

THETA = FI\signal; % linear least-squares
signalEst = FI*THETA;

a = T*THETA(1:Porder); % sine polynomial
b = T*THETA(Porder+1:end); % cosine polynomial

return




function [signalEst,a,b,aCov,bCov] = PolyEstimationWithUncertainty(signal,f0,fs,Porder)

N = length(signal);
t = (0:N-1)'/fs;

T = zeros(N,Porder);
for i = 1:Porder
    T(:,i) = t.^(i-1);
end

X1 = repmat(sin(2*pi*f0*t),1,Porder);
X2 = repmat(cos(2*pi*f0*t),1,Porder);

X1 = X1 .* T;
X2 = X2 .* T;
FI = [X1 X2];

[THETA,~,~,S] = lscov(FI,signal);
signalEst = FI*THETA;

a = T*THETA(1:Porder); % sine polynomial
b = T*THETA(Porder+1:end); % cosine polynomial

THETAnewCov = mvnrnd(THETA,S,1); % multivariate distribution sampling
aCov = T*THETAnewCov(1:Porder)'; % sine polynomial perturbed
bCov = T*THETAnewCov(Porder+1:end)'; % cosine polynomial perturbed

return

