function OrbitalComponentEstimation(data,Nsources,bands,windowsize,Porder)
% OrbitalComponentEstimation - Extracts principal orbital components from
%   cyclostratigraphic data.
%
% About:
%   This script is part of the complementary material for the manuscript
%   entitled "Astronomical component estimation (ACE v.1) by time-variant
%   sinusoidal modeling", submitted to Geoscientific Model Development.
%
% Input:
%   data              Stratigraphic series: first column should be depth 
%                     (in meters), second column should be data value.
%                     !!!Uniform sampling is assumed.!!! and the depth
%                     scale can not contain negative and positive levels in
%                     the same file.
%   Nsources          Number of orbital components to extract
%   bands             (Nsources x 2)-matrix where each row encompasses
%                     the lower and upper frequency bound (in Hz) for a 
%                     given component.
%                     These frequency ranges can be determined using for
%                     example spectral anlysis (Periodogram, MTM etc.)
%   windowsize        length of the analysis frame in meters
%                     See Chapter 2.3 in Sinnesael et al., 2016, 
%                     Geoscientific Model Development for instructions
%   Porder            number of terms for the polynomial approximation.
%                     See Chapter 2.3 in Sinnesael et al., 2016, 
%                     Geoscientific Model Development for instructions
% Description:
%   Estimates the spatial-domain waveforms of orbital compnents by means of
%   non-stationary polyinomial-based sinusoidal modeling. Key reference: 
%   Zivanovic, M., and Schoukens, J. "On The Polynomial Approximation for 
%   Time-Variant Harmonic Signal Modeling", IEEE Transactions on Audio, 
%   Speech, and Language Processing, 19, 458?467, 2011.
%
% Example:
%   OrbitalComponentEstimation(data,2,[0.01 0.07; 0.18 0.25],80,2)

%depth
%------
depth_org = data(:,1);
%zero first level
depth = depth_org-data(1,1); 
%convert negative numbers to positve numbers
if depth(2,1)<0
depth = depth*-1;
end


signal = data(:,2);

N = length(depth);
Ts = depth(2); % sampling period in meters
fs = 1/Ts; % sampling frequency (uniform sampling assumed)
windowsize = round(windowsize*fs); % in samples


ooff = 0;
normo = zeros(N,1);
win = hanning(windowsize);
winHarm = repmat(win,1,Nsources);
sigo = zeros(N,Nsources);

inc = round(windowsize/4); % hop size between contiguous frames
Position = 1:inc:round(length(signal)-windowsize+1);
f0est = zeros(length(Position),Nsources);

for i = 1:length(Position)
    seg = signal(Position(i):Position(i)+windowsize-1);
    
    fprintf('processing frame %d of %d \r',i,length(Position));
    
    for k = 1:Nsources
        f0est(i,k) = MeanFrequency(hilbert(seg),bands(k,:),8*4096,fs); % estimate mean component frequencies
    end
        
    segEst = ExtractComponents(seg,f0est(i,:),fs,Porder); % extract component waveforms
                
    sigo(round(ooff)+windowsize,:) = 0; % combine the estimated segments
    sigo((1:windowsize)+round(ooff),:) = sigo((1:windowsize)+round(ooff),:) + segEst.*winHarm;
    
    normo(round(ooff)+windowsize) = 0; % overlap-add procedure to account on the record edges
    normo((1:windowsize)+round(ooff)) = normo((1:windowsize)+round(ooff)) + win;
    ooff = ooff + inc;
end
orbitalComp = sigo./repmat(max(0.1,normo),1,Nsources); % contains orbital component waveforms

%rescale to original depth scale
%depth = depth+data(1,1);

save('EstimatedComponents','orbitalComp','fs','depth','depth_org'); % save output

return





function [signalEst] = ExtractComponents(signal,f0,fs,Porder)
% Estimates orbital component waveforms by means of non-stationary
% polynomial-based sinusoidal modeling.

N = length(signal);
depth = (0:N-1)'/fs;

T = zeros(N,Porder);
for i = 1:Porder
    T(:,i) = depth.^(i-1); % time-basis for the polynomials
end

FI = [];
for p = 1:length(f0)
    X1 = repmat(sin(2*pi*f0(p)*depth),1,Porder); % model matrix - sines
    X2 = repmat(cos(2*pi*f0(p)*depth),1,Porder); % model matrix - cosines
    
    X1 = X1 .* T;
    X2 = X2 .* T;
    FI = [FI X1 X2];
end

THETA = FI\signal; % linear least squares for the polynomial coefficients

signalEst = zeros(N,length(f0));
for i = 1:length(f0) % get the waveforms from the model matrix and coefficients
    signalEst(:,i) = FI(:,(2*i-2)*Porder+1:2*i*Porder)*THETA((2*i-2)*Porder+1:2*i*Porder);
end

return





function [f0] = MeanFrequency(x,f0_lim,F,fs)
% Estimates mean frequency of the orbital component

N = length(x);

freqndx = round(f0_lim(1)/fs*F)+1:1:round(f0_lim(2)/fs*F)+1; % frequency band for a component
f0_set = fs*(freqndx-1)/F;

X = abs(fft(x,F)).^2/N; % periodogram

k0 = MaxPeak(X(freqndx)); % largest peak position

f0 = f0_set(k0); % estimated mean frequency in Hz

return




function [l] = MaxPeak(x)
% Estimates position of the largest peak in the periodogram

df = diff(x);
ndx = find(diff(sign(df))<0) + 1;
p = x(ndx);
[~,sort_ndx] = sort(p);

sort_ndx = flipud(sort_ndx);

l = ndx(sort_ndx(1));

return
