%
% About:
%   This script is part of the complementary material for the manuscript
%   entitled "Astronomical component estimation (ACE v.1) by time-variant
%   sinusoidal modeling", submitted to Geoscientific Model Development
%   (Sinnesael et al., 2016).(Last update: 26/08/2016).
%
%   This mainscript contains four main parts:
%
%   1) DATA INPUT: load your data, for example as a text file
%
%   2) PARAMETER SPECIFICATION: specify the model parameters
%
%   3) CALCULATIONS: run three functions "OrbitalComponentEstimation",
%   "SedimentationRateEstimation" and "UncertaintyAnalysis" with the
%   specified model parameters (2) and return numerical output.
%
%   4) BASIC GRAPHICAL OUTPUT: basic plots plotting per estimated component 
%   in function of depth: the estimated waveform, instantaneous frequency 
%   changes, estimated sedimentation rate and the corresponding 
%   uncertainties. The user is free to make customize the plotting 
%   statements or to export the variables and plot them elsewhere.  

% HINT: code can be disabled by putting a "%" in front of the line of code 
% (will turn into green color). The black lines in this script is the
% active executing code. 

% start timer
tic 

% clear previous variables and close all open figures
clear all
close all
clc

%______________________________________________________________
%
% 1) --- DATA INPUT ---
%______________________________________________________________
%
% - !!! DEPTH SCALE HAS TO BE UNIFORMLY SAMPLED!!! 
% IF YOUR ORIGINAL DATA IS NOT, WE SUGGEST INTERPOLATION BEFORE USING
% OF THIS ROUTINE

% - In the current code, the data is loaded as a text-file, but the user is
% free to use other formats (and adapt the MATLAB code accordingly).

%   data              Stratigraphic series: first column should be depth 
%                     (meters), second column should be data value.
%                     !!!Uniform sampling is assumed.!!! and the depth
%                     scale can not contain negative and positive levels in
%                     the same file.

data = load('test.txt');

%______________________________________________________________
%
% 2) --- PARAMETER SPECIFICATION ---
%______________________________________________________________
%
% Specification of the model parameters which will be used in the
% functions.
%
% Input:
%   Nsources          Number of orbital components to extract
%   bands             (Nsources x 2)-matrix where each row encompasses
%                     the lower and upper frequency bound (in Hz) for a 
%                     given component.
%                     These frequency ranges can be determined using for
%                     example spectral anlysis (Periodogram, MTM etc.)
%   windowsize        length of the analysis frame in meters
%                     See Chapter 2.3 in Sinnesael et al., 2016, 
%                     Geoscientific Model Development for instructions
%   Porder            number of terms for the polynomial approximation.
%                     See Chapter 2.3 in Sinnesael et al., 2016, 
%                     Geoscientific Model Development for instructions
%   Tnominal          nominal orbital component period in kyr
%                     Predicted periods for precession and obliquity can be
%                     found in Berger et al., (1992) or Laskar et al.,
%                     (2004)

Nsources = 2; % e.g. = 2
windowsize = 3; % e.g. = 3
Porder = 2; % e.g. = 2
bands = [1.78 2.57; 5.13 6.58]; % e.g. = [1.78 2.57; 5.13 6.58] 
Tnominal = [100 39]; % e.g. [30.5 17.5]

%______________________________________________________________
%
% 3) --- CALCULATIONS ---
%______________________________________________________________


% I) ORBITAL COMPONENT ESTIMATION (Function)
%______________________________________________________________

%CALCULATION
% function OrbitalComponentEstimation(data,Nsources,bands,windowsize,Porder)
OrbitalComponentEstimation(data,Nsources,bands,windowsize,Porder)

%RETURN OUTPUT
load('EstimatedComponents.mat')

% II) SEDIMENTATION RATE ESTIMATION (Function)
%______________________________________________________________

%CALCULATION
%function SedimentationRateEstimation(fileName,windowsize,Porder,Tnominal)
SedimentationRateEstimation('EstimatedComponents',windowsize,Porder,Tnominal)

%RETURN OUTPUT
load('InstanteousFrequecyEstimates.mat')
load('SedimentationRateEstimates.mat')

% III) UNCERTAINTY ANALYSIS (Function)
%_____________________________________________________________
%function UncertaintyAnalysis(fileName,windowsize,Porder,Tnominal)
UncertaintyAnalysis('EstimatedComponents',windowsize,Porder,Tnominal)

%RETURN OUTPUT
load('freqInstStd.mat') % one standard deviation
load('SRStd.mat') % one standard deviation


%______________________________________________________________
%
% 4) --- BASIC GRAPHICAL OUTPUT ---
%______________________________________________________________
%   
%   - plot the input data
%   - basic plots plotting per estimated component in function of depth: 
%   the estimated waveform, instantaneous frequency, changes, estimated 
%   sedimentation rate and the corresponding uncertainties. The user is 
%   free to make customize the plotting statements or to export the 
%   variables and plot them elsewhere. 
%
%   to label the axes correctly, check the units of the input files

% Plot the input data:

plot(data(:,1),data(:,2))
title('Input data') % change title ''
xlabel('depth, check your units')   % change x-axis labelling ''
ylabel('Signal')   % change y-axis labelling ''

% Plot for every estimated component:

for i = 1 : Nsources

figure()    
    
% Waveform
plot(depth_org,orbitalComp(:,i))    % plotting statement
title(['estimated component ', num2str(i)]) % change title ''
xlabel('depth, check your units')   % change x-axis labelling ''
ylabel('Amplitude')   % change y-axis labelling ''

figure()

% Instanteous frequency estimate
plot(depth_org,freqInst(:,i))
title(['estimated instantaneous frequency ', num2str(i)])
xlabel('depth, check your units')
ylabel('frequency, check your units')

figure()

% Sedimentation rate estimation
plot(depth_org,SR(:,i))
title(['estimated sedimentation rate ', num2str(i)])
xlabel('depth, check your units')
ylabel('sedimentation rate, check your units')

figure()

% Uncertainty on Instanteous frequency estimate, one standard deviation
plot(depth_org,freqInstStd(:,i))
title(['instantaneous frequency uncertainty ', num2str(i)])
xlabel('depth, check your units')
ylabel('frequency, check your units')

figure()

% Uncertainty on Sedimentation rate estimation, one standard deviation
plot(depth_org,SRStd(:,i))
title(['sedimentation rate uncertainty ', num2str(i)])
xlabel('depth, check your units')
ylabel('sedimentation rate , check your units')

%figure()

end

%end timer
toc 

%END ACEv1_MAINSCRIPT