/*---------------------------------------------------------------------------*\
  =========                 |
  \\      /  F ield         | OpenFOAM: The Open Source CFD Toolbox
   \\    /   O peration     |
    \\  /    A nd           | Copyright (C) 2011 OpenFOAM Foundation
     \\/     M anipulation  |
-------------------------------------------------------------------------------
License
    This file is part of OpenFOAM.

    OpenFOAM is free software: you can redistribute it and/or modify it
    under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    OpenFOAM is distributed in the hope that it will be useful, but WITHOUT
    ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
    FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
    for more details.

    You should have received a copy of the GNU General Public License
    along with OpenFOAM.  If not, see <http://www.gnu.org/licenses/>.

\*---------------------------------------------------------------------------*/

#include "HerschelBulkleyDebrisFlow.H"
#include "addToRunTimeSelectionTable.H"
#include "surfaceFields.H"

// * * * * * * * * * * * * * * Static Data Members * * * * * * * * * * * * * //

namespace Foam
{
namespace viscosityModels
{
    defineTypeNameAndDebug(HerschelBulkleyDebrisFlow, 0);

    addToRunTimeSelectionTable
    (
        viscosityModel,
        HerschelBulkleyDebrisFlow,
        dictionary
    );
}
}


// * * * * * * * * * * * * Private Member Functions  * * * * * * * * * * * * //

Foam::tmp<Foam::volScalarField>
Foam::viscosityModels::HerschelBulkleyDebrisFlow::calcNu() const
{
    dimensionedScalar tone("tone", dimTime, 1.0);
    dimensionedScalar rtone("rtone", dimless/dimTime, 1.0);
    
    tmp<volScalarField> sr(strainRate());


    return
    (
        min
        (
            nu0_,
            (tau0_ + k_*rtone*pow(tone*sr(), n_))
           /(max(sr(), dimensionedScalar ("VSMALL", dimless/dimTime, VSMALL)))
        )
    );
}


bool Foam::viscosityModels::HerschelBulkleyDebrisFlow::setHBParameters()
{
    Info<< "Line 0 ";
    if(Cv_ > dimensionedScalar ("CvLimit", dimless, 0.47))
    {
	tau00_ = tau00_ * exp(5*(Cv_ - dimensionedScalar ("CvLimit", dimless, 0.47)));
    }
    if(P0v_ > dimensionedScalar ("CvLimit", dimless, 0.27))
    {
	P0v_ = P0v_ * 0.7;
    }
    Info<< "Line 1 ";
    tau0_ = (tau00_/rhoF_)*Cv_*Cv_ * exp(22.0*Cv_*P0v_);
        Info<< "Line 3 ";
    k_ = 0.3 * tau0_ * dimensionedScalar("timeUnit", dimTime, 1.0);
    return true;
    
}

// * * * * * * * * * * * * * * * * Constructors  * * * * * * * * * * * * * * //

Foam::viscosityModels::HerschelBulkleyDebrisFlow::HerschelBulkleyDebrisFlow
(
    const word& name,
    const dictionary& viscosityProperties,
    const volVectorField& U,
    const surfaceScalarField& phi
)
:
    viscosityModel(name, viscosityProperties, U, phi),
    HerschelBulkleyDebrisFlowCoeffs_(viscosityProperties.subDict(typeName + "Coeffs")),
    waterContent_(HerschelBulkleyDebrisFlowCoeffs_.lookup("waterContent")),
    Cv_(HerschelBulkleyDebrisFlowCoeffs_.lookup("Cv")),
    P0v_(HerschelBulkleyDebrisFlowCoeffs_.lookup("P0v")),
    tau00_(HerschelBulkleyDebrisFlowCoeffs_.lookup("tau00")),
    rhoF_(HerschelBulkleyDebrisFlowCoeffs_.lookup("rhoF")),
    nu0_(HerschelBulkleyDebrisFlowCoeffs_.lookup("nu0")),
    tau0_(HerschelBulkleyDebrisFlowCoeffs_.lookup("tau0")),
    k_(HerschelBulkleyDebrisFlowCoeffs_.lookup("k")),
    n_(HerschelBulkleyDebrisFlowCoeffs_.lookup("n")),
    
    setPar_(setHBParameters()),
    nu_
    (
        IOobject
        (
            name,
            U_.time().timeName(),
            U_.db(),
            IOobject::NO_READ,
            IOobject::AUTO_WRITE
        ),
        calcNu()
    )
{}


// * * * * * * * * * * * * * * Member Functions  * * * * * * * * * * * * * * //

bool Foam::viscosityModels::HerschelBulkleyDebrisFlow::read
(
    const dictionary& viscosityProperties
)
{
    viscosityModel::read(viscosityProperties);

    HerschelBulkleyDebrisFlowCoeffs_ = viscosityProperties.subDict(typeName + "Coeffs");

    HerschelBulkleyDebrisFlowCoeffs_.lookup("waterContent") >> waterContent_;
    HerschelBulkleyDebrisFlowCoeffs_.lookup("Cv") >> Cv_;
    HerschelBulkleyDebrisFlowCoeffs_.lookup("P0v") >> P0v_;
    HerschelBulkleyDebrisFlowCoeffs_.lookup("tau00") >> tau00_;
    HerschelBulkleyDebrisFlowCoeffs_.lookup("rhoF") >> rhoF_;
    HerschelBulkleyDebrisFlowCoeffs_.lookup("nu0") >> nu0_;
	HerschelBulkleyDebrisFlowCoeffs_.lookup("n") >> n_;

    return true;
}


// ************************************************************************* //
