from pylab import *
import os

# Define input parameters

# case 0: Clear atmosphere with land surface (Lambertian)
# case 1: Clear atmosphere with ocean surface 
# case 2: Aerosol (desert)
# case 3: Aerosol (continental average)
# case 4: Water cloud (optical thickness 2)
# case 5: Ice cloud (solid column)
# case 6: Ice cloud (plate)
# case 7: Ice cloud (solid column rough)

ncases=8
cases=[0,1,2,3,4,5,6,7]

# Viewing angle
va=arange(-50., 1., 2., dtype='d')
umu=cos(va*pi/180.)

# Solar zenith angles
sza=[30.]

# Relative azimuth angles
phi=[0., 180.]
dphi=0

# wavelengths
lam=[443,670,865]

# Number of Stokes components
nstokes=4

# Number of photons
N = 1e6

# Initialize variables for radiance and standard deviation
radiance = zeros((len(va), len(sza), len(phi), len(lam), nstokes, ncases))
std = zeros((len(va), len(sza), len(phi), len(lam), nstokes, ncases))

# Path to uvspec executable
path=''

# Run RT calculation 
run_rt=1


if(run_rt):
    # Loop over all input parameters
    for case in cases:
        for ilam in range(len(lam)):
            for iphi in range(len(phi)):
                for isza in range(len(sza)):
                    for iumu in range(len(umu)):
                        
                        disp('run mystic: lam %g phi0 %g sza %g va %.2f case %d'%(lam[ilam], phi[iphi], sza[isza], 
                                                                                  (arccos(-umu[iumu])*180./pi), case))
                        # mystic.inp: template file which includes settings that are not changed
                        tmp = open('mystic.inp').read()  
                      
                        # mystic_run.inp: input file for uvspec, modified in the following 
                        inp = open('mystic_run.inp'%va[iumu], 'w')
                        inp.write(tmp)
                        inp.write('wavelength %g \n'%lam[ilam])
                        inp.write('umu %g \n'%umu[iumu])
                        inp.write('sza %g \n'%sza[isza])
                        inp.write('phi %g \n'%phi[iphi])
                        inp.write('mc_photons %d \n'%N)
                        if(case==0):
                            inp.write('albedo 0.0 \n') # Lambertian surface albedo of 0.2
                        elif(case==1):
                            inp.write('bpdf_tsang_u10 2 \n') # BPDF for ocean, wind speed 2
                        elif(case==2):
                             inp.write('aerosol_default \n')
                             inp.write('aerosol_species_file desert \n') # OPAC desert aerosol
                             inp.write('albedo 0.3 \n')
                             inp.write('mc_vroom on \n') # switch on variance reduction for spiky phase functions
                        elif(case==3):
                             inp.write('aerosol_default \n')
                             inp.write('aerosol_species_file maritime_clean \n') # OPAC 
                             inp.write('mc_vroom on \n') # switch on variance reduction for spiky phase functions
                        elif(case==4):  
                            inp.write('wc_file 1D wc.dat \n')
                            inp.write('wc_properties mie interpolate \n')  
                            inp.write('wc_modify tau set 10 \n') # cloud optical thickness 10
                            inp.write('mc_vroom on \n') # switch on variance reduction for spiky phase functions
                            inp.write('phi %g \n'%(phi[iphi]+dphi)) 
                        elif case==5:
                            inp.write('ic_file 1D ic.dat \n')
                            inp.write('ic_properties yang2013 interpolate\n')
                            inp.write('ic_habit_yang2013 solid_column smooth\n')
                            inp.write('ic_modify tau set 2 \n') # cloud optical thickness 10
                            inp.write('mc_vroom on \n') # switch on variance reduction for spiky phase functions
                            inp.write('phi %g \n'%(phi[iphi]+dphi))
                        elif case==6:
                            inp.write('ic_file 1D ic.dat \n')
                            inp.write('ic_properties yang2013 interpolate\n')
                            inp.write('ic_habit_yang2013 plate smooth\n')
                            inp.write('ic_modify tau set 2 \n') # cloud optical thickness 10
                            inp.write('mc_vroom on \n') # switch on variance reduction for spiky phase functions
                            inp.write('phi %g \n'%(phi[iphi]+dphi))
                        elif case==7:
                            inp.write('ic_file 1D ic.dat \n')
                            inp.write('ic_properties yang2013 interpolate\n')
                            inp.write('ic_habit_yang2013 solid_column severe\n')
                            inp.write('ic_modify tau set 2 \n') # cloud optical thickness 10
                            inp.write('mc_vroom on \n') # switch on variance reduction for spiky phase functions
                            inp.write('phi %g \n'%(phi[iphi]+dphi)) 

                        inp.close()
                        
                        fin, fout=os.popen2(path+'uvspec < mystic_run.inp')
                        os.wait()

                        # Write Stokes vector and standard deviation into variables
                        # I
                        radiance[iumu,isza,iphi,ilam,0,case]=loadtxt('mc.rad.spc')[0,4]
                        std[iumu,isza,iphi,ilam,0,case]=loadtxt('mc.rad.std.spc')[0,4]
                        # Q
                        radiance[iumu,isza,iphi,ilam,1,case]=loadtxt('mc.rad.spc')[1,4]
                        std[iumu,isza,iphi,ilam,1,case]=loadtxt('mc.rad.std.spc')[1,4]
                        # U
                        radiance[iumu,isza,iphi,ilam,2,case]=loadtxt('mc.rad.spc')[2,4]
                        std[iumu,isza,iphi,ilam,2,case]=loadtxt('mc.rad.std.spc')[2,4]


    np.save('radiance.npy', radiance)
    np.save('std.npy', std)

else:
    radiance=np.load('radiance.npy')
    std=np.load('std.npy')

# plot results 
figure(1, figsize=(8,20))
isza=0
ilam=0

colors=['b','g','r']
titles=['clear + land', 'clear + ocean', 'desert', 'maritime', 'liquid water cloud', 'ice solid-column smooth', 'ice plate smooth', 'ice solid-column rough']
for ilam in range(3):
    for case in range(ncases):
        subplot(8,2,2*case+1)
        errorbar(va, radiance[:,isza,0,ilam,0,case], std[:,isza,0,ilam,0,case], color=colors[ilam])
        errorbar(-va, radiance[:,isza,1,ilam,0,case], std[:,isza,1,ilam,0,case], color=colors[ilam])  
        ylabel(titles[case])
        if(case==7):
            xlabel('viewing angle [deg]')
        if(case==0):
            title('I')

        subplot(8,2,2*case+2)
        errorbar(va, radiance[:,isza,0,ilam,1,case], std[:,isza,0,ilam,1,case], color=colors[ilam])
        errorbar(-va, radiance[:,isza,1,ilam,1,case], std[:,isza,1,ilam,1,case], color=colors[ilam])
        if(case==7):
            xlabel('viewing angle [deg]')
        if(case==0):
            title('Q')

figure(1)
savefig('polarization_example.pdf')


