! 1D model of ice thermodynamics to look at diurnal cycle
! Code taken from CICE 
! Force ice with a sinusoidal atmos T so that it can be compared with
! analytical solution
!
! Alison McLaren
! May 2011
!
!==========================================================================
!
      program ice_model

      use ice_kinds_mod
      use ice_domain_size
      use ice_coupling
      use ice_therm_vertical

      implicit none

      integer (kind=int_kind),parameter :: &
         nx_block = 1    , &
         ny_block = 1    ! block dimensions

      integer (kind=int_kind) :: &
         icells          , &  ! number of cells with ice present
         my_task              ! processor number

      integer (kind=int_kind), dimension (nx_block*ny_block):: &
         indxi, indxj     ! compressed indices for cells with ice


      ! ice state variables
      real (kind=dbl_kind), dimension (nx_block,ny_block) :: &
         aicen , & ! concentration of ice
         vicen , & ! volume per unit area of ice          (m)
         vsnon     ! volume per unit area of snow         (m)

      real (kind=dbl_kind), dimension (nx_block,ny_block,max_ntrcr) :: &
         trcrn

      real (kind=dbl_kind), allocatable, dimension(:,:,:) :: &
         eicen     ! energy of melting for each ice layer (J/m^2)

      real (kind=dbl_kind), allocatable, dimension(:,:,:) :: &
         esnon     ! energy of melting for each snow layer (J/m^2)

      ! input from atmosphere
      real (kind=dbl_kind), dimension (nx_block,ny_block) :: &
         flw     , & ! incoming longwave radiation (W/m^2)
         potT    , & ! air potential temperature  (K) 
         Qa      , & ! specific humidity (kg/kg) 
         rhoa    , & ! air density (kg/m^3) 
         fsnow   , & ! snowfall rate (kg m-2 s-1)
         shcoef  , & ! transfer coefficient for sensible heat
         lhcoef      ! transfer coefficient for latent heat

      real (kind=dbl_kind), dimension (nx_block,ny_block) :: &
         fswsfc  , & ! SW absorbed at ice/snow surface (W m-2)
         fswint  , & ! SW absorbed in ice interior, below surface (W m-2)
         fswthrun    ! SW through ice to ocean         (W/m^2)

      real (kind=dbl_kind), allocatable, dimension (:,:,:) :: &
         Sswabs      ! SW radiation absorbed in snow layers (W m-2)

      real (kind=dbl_kind), allocatable, dimension (:,:,:) :: &
         Iswabs      ! SW radiation absorbed in ice layers (W m-2)

      ! input from ocean
      real (kind=dbl_kind), dimension (nx_block,ny_block) :: &
         fbot    , & ! ice-ocean heat flux at bottom surface (W/m^2)
         Tbot        ! ice bottom surface temperature (deg C)

      ! coupler fluxes to atmosphere
      real (kind=dbl_kind), dimension (nx_block,ny_block):: &
         fsensn  , & ! sensible heat flux (W/m^2) 
         fswabsn , & ! shortwave flux absorbed in ice and ocean (W/m^2) 
         flwoutn , & ! outgoing longwave radiation (W/m^2) 
         evapn       ! evaporative water flux (kg/m^2/s) 

      ! Note: these are intent out if calc_Tsfc = T, otherwise intent in
      real (kind=dbl_kind), dimension (nx_block,ny_block):: &
         flatn    , & ! latent heat flux   (W/m^2) 
         fsurfn   , & ! net flux to top surface, excluding fcondtopn
         fcondtopn    ! downward cond flux at top surface (W m-2)

      ! coupler fluxes to ocean
      real (kind=dbl_kind), dimension (nx_block,ny_block):: &
         freshn  , & ! fresh water flux to ocean (kg/m^2/s)
         fsaltn  , & ! salt flux to ocean (kg/m^2/s)
         fhocnn      ! net heat flux to ocean (W/m^2) 

      ! diagnostic fields
      real (kind=dbl_kind), dimension(nx_block,ny_block) :: &
         meltt    , & ! top ice melt             (m/step-->cm/day) 
         melts    , & ! snow melt                (m/step-->cm/day) 
         meltb    , & ! basal ice melt           (m/step-->cm/day) 
         congel   , & ! basal ice growth         (m/step-->cm/day) 
         snoice   , & ! snow-ice formation       (m/step-->cm/day) 
         mlt_onset, & ! day of year that sfc melting begins 
         frz_onset    ! day of year that freezing begins (congel or frazil) 

      real (kind=dbl_kind):: &
         yday      ! day of year

      logical (kind=log_kind) :: &
         l_stop   , &    ! if true, print diagnostics and abort on return
         print_diags_in_ice_therm


      integer (kind=int_kind) :: &
         istop, jstop    ! indices of grid cell where code aborts

      integer (kind=int_kind) :: &
         k, it, icycle, atmos_it, timestep_no, min_print         ! indices

      integer (kind=int_kind) :: &
         ncycles              ! no. of day cycles

      integer (kind=int_kind), dimension (4) :: &
         lyr_out           ! vertical indices on which to output

      integer (kind=int_kind) :: lyr_out_snow   ! as above but for snow, just one layer

      real (kind=dbl_kind) :: &
         dt   , & ! ice model timestep
         force_dt   , & ! forcing time step (i.e. coupling)
         atmos_dt     ! 'atmos model' timestep (frequency of calculating Tatmos)
	 
      real (kind=dbl_kind) :: & 
         hilyr        , &  ! thickness of each ice layer
         hslyr        , &  ! thickness of each snow layer
         depth_out1   , &  ! depths on which to output
         depth_out2   , &  ! depths on which to output
         depth_out3   , &  ! depths on which to output
         depth_out4   , &  ! depths on which to output
         depth_out_snow    ! snow depth on which to output

      real (kind=dbl_kind) :: & 
         Tatmos_a      , &   ! atmos T cycle amplitude (deg C)
         Tatmos_omega  , &   ! angular freq of atmos T cycle (1/s)
         Tatmos_offset , &   ! offset for atmos T 
                             ! so Tatmos = Tatmos_a*sin(Tatmos_omega*t)+Tatmos_offset
         Tatmos        , &   ! atmos T  (deg C)
         atmos_start_time ,& ! atmos model start time
         atmos_time_in_day

      real (kind=dbl_kind) :: & 
         Ti            , &   ! initial Ti
         slope               ! vertical gradient of initial T profile


      real (kind=dbl_kind), allocatable, dimension (:):: &
         depth             ! mid depth of each ice layer

      real (kind=dbl_kind), allocatable, dimension (:):: &
         depth_snow        ! mid depth of each snow layer

      real (kind=dbl_kind), dimension (nx_block,ny_block) :: &
         atm_force_avg     ! atmos forcing (either Tatmos or fcondtop depending on calc_Tsfc) 
                           ! averaged over forcing timestep 

      real (kind=dbl_kind), dimension (nx_block,ny_block):: &
         atm_force_avg_tmp    ! working array for accumulating atm_force_avg over atmos timesteps

      real (kind=dbl_kind), dimension (nx_block,ny_block) :: &
         Tn_top    , & ! T of top layer
         keffn_top    ! effective conductivity of top layer

      character (len=140) :: &
         filename_d   , &    ! output filename (data)
         filename_d2   , &   ! output filename (data on atmos timestep, 
                             !    calc_Tswfc=F only)
         filename_t          ! output filename (txt info)

      real (kind=dbl_kind) :: &
         time        ,& ! model time
         output_dt = 1.      ! output timestep

      real (kind=dbl_kind) :: ice_force_save(2), ice_force_use(2)

      real (kind=dbl_kind) :: ice_time, atmos_time

      real (kind=dbl_kind) :: time_in_day

      integer (kind=dbl_kind) :: ratio_it

      integer (kind=int_kind) :: coupling_period, ice_timestep_no, atmos_timestep_no, &
         n_cpl_in_day, n_ice_in_cpl, n_atm_in_cpl, abs_atmos_timestep_no, &
         abs_ice_timestep_no, atmos_nt_in_force, nt_in_day

      integer (kind=int_kind) :: nu_nml

      real (kind=dbl_kind) :: wind_speed

      character (len=10) :: atmos_timestep, ice_timestep, coupling_timestep, &
         snow_layer_thickness, ice_layer_thickness, coupling_method

      character (len=80) :: output_directory

      namelist / settings / atmos_timestep, ice_timestep, coupling_timestep, &
         snow_layer_thickness, ice_layer_thickness, coupling_method, output_directory, wind_speed

!=======================================================================
! Read namelist
      nu_nml = 120
      open(nu_nml, file = 'run_ice.namelist')
      read(nu_nml,nml=settings)
      close(nu_nml)

!=======================================================================

! Set output filenames based on namelist

      filename_d = trim(output_directory) // '/icedata_'//'snow_'//trim(snow_layer_thickness)//'_ice_'//&
         trim(ice_layer_thickness)//'_atmdt_'//trim(atmos_timestep)//'_icedt_'//trim(ice_timestep)//&
         '_cpldt_'//trim(coupling_timestep)//'_cplm_'//trim(coupling_method)//'.dat'
      filename_d2 = trim(output_directory) // '/atmdata_'//'snow_'//trim(snow_layer_thickness)//'_ice_'//&
         trim(ice_layer_thickness)//'_atmdt_'//trim(atmos_timestep)//'_icedt_'//trim(ice_timestep)//&
         '_cpldt_'//trim(coupling_timestep)//'_cplm_'//trim(coupling_method)//'.dat'
      filename_t = trim(output_directory) // '/text_'//'snow_'//trim(snow_layer_thickness)//'_ice_'//&
         trim(ice_layer_thickness)//'_atmdt_'//trim(atmos_timestep)//'_icedt_'//trim(ice_timestep)//&
         '_cpldt_'//trim(coupling_timestep)//'_cplm_'//trim(coupling_method)//'.txt'

      print*, 'filename_d = ', filename_d


      aicen(1,1)=1.0_dbl_kind
      vicen(1,1)=2.0_dbl_kind
      vsnon(1,1)=0.21_dbl_kind


! Set vertical resolution
! The following lines become invalid if the above values of
! aicen, vicen or vsnon are changed

      if (snow_layer_thickness == '1cm') then
         nslyr = int(vsnon(1,1) * 100. + .1)
         if (nslyr == 0) then
            nslyr = 1
         endif
      else if (snow_layer_thickness == '11cm') then
         nslyr = 1
      else
         print*, 'Invalid snow layer thickness setting'
         stop
      endif

      if (ice_layer_thickness == '1cm') then
         nilyr = int(vicen(1,1) * 100. + .1)
         print*, 'nilyr = ', nilyr
      else if (ice_layer_thickness == '25cm') then
         nilyr = 4
      else
         print*, 'Invalid ice layer thickness setting'
         stop
      endif
      ntilyr = ncat*nilyr
      ntslyr = ncat*nslyr


! Set the various timestep lengths.  min_print controls the
! frequency of data output, and is set to 60 for second-length
! timesteps to keep the files to a manageable size.

      if (atmos_timestep == '1s') then
         atmos_dt = c1
      else if (atmos_timestep == '20min') then
         atmos_dt = c20*c60
      else
         print*, 'Invalid atmos timestep setting'
      endif

      if (ice_timestep == '1s') then
         dt = c1
         min_print = 60_int_kind
      else if (ice_timestep == '1h') then
         dt = c60*c60
         min_print = 1_int_kind
      else
         print*, 'Invalid ice timestep setting'
      endif

      if (coupling_timestep == '1s') then
         force_dt = c1
      else if (coupling_timestep == '1h') then
         force_dt = c60*c60
      else if (coupling_timestep == '3h') then
         force_dt = c60*c60*c3
      else
         print*, 'Invalid coupling timestep setting'
      endif


! Set the coupling method
      if (coupling_method == 'jules') then
         calc_Tsfc = .false.
      else if (coupling_method == 'cice') then
         calc_Tsfc = .true.
      else
         print*, 'Invalid coupling method setting'
      endif

!===================================================================================
! Other settings not in namelist

      Tatmos_a=c3
      Tatmos_omega=2.0_dbl_kind*pi/(3600.0_dbl_kind*24.0_dbl_kind)
      Tatmos_offset=-c5
      
      ncycles=50          ! no. of cycles

      atmos_start_time = c0      ! can introduce a lag to atmos forcing

      depth_out_snow = 0.105_dbl_kind  ! Depths within snow on which to output T

      depth_out1=0.25_dbl_kind   ! Depths on which to output T
      depth_out2=0.75_dbl_kind
      depth_out3=1.25_dbl_kind
      depth_out4=1.75_dbl_kind
      
! ====================================================================
! Allocate variables dependent on nilyr and nslyr

   ! In run_ice
      print*, 'Before allocation, nilyr = ', nilyr
      allocate(eicen(nx_block,ny_block,nilyr))
      allocate(esnon(nx_block,ny_block,nslyr))
      allocate(Iswabs(nx_block,ny_block,nilyr))
      allocate(Sswabs(nx_block,ny_block,nslyr))
      allocate(depth(nilyr))
      allocate(depth_snow(nslyr))

   ! In ice_therm_vertical
      allocate(salin(nilyr+1))
      allocate(Tmlt(nilyr+1))

! ====================================================================
! Write out run info to txt file

      open(unit=11,file=filename_t)
      write(11,*) 'Data filename =',filename_d
      write(11,*) 'atmos_dt=',atmos_dt
      write(11,*) 'dt=',dt
      write(11,*) 'force_dt=',force_dt
      write(11,*) 'atmos_dt=',atmos_dt
      write(11,*) 'nilyr=',nilyr
      write(11,*) 'nslyr=',nslyr
      write(11,*) 'calc_Tsfc=',calc_Tsfc
      write(11,*) 'ncycles=',ncycles
      write(11,*) 'hice=',vicen(1,1)/aicen(1,1)
      write(11,*) 'Tatmos_a=',Tatmos_a
      write(11,*) 'Tatmos_omega=',Tatmos_omega
      write(11,*) 'Tatmos_offset=',Tatmos_offset
      write(11,*) 'atmos_start_time=',atmos_start_time

      open(unit=100,file='ice_diag.d')

!=====================================================================

! Initialise fields


! Initialise Tmlt and salin
      call init_thermo_vertical
      write(11,*) 'l_brine=',l_brine

! Initalise ice state

! Surface T
      Tatmos = Tatmos_a*sin(atmos_start_time*Tatmos_omega) + Tatmos_offset
      trcrn(1,1,1) = min(Tsmelt, Tatmos)

! Bottom T
      Tbot(1,1)=-1.8_dbl_kind

! T profile
      do k=1,nilyr 
          ! assume linear temp profile and compute enthalpy
          slope = Tbot(1,1) - trcrn(1,1,1)
             Ti = trcrn(1,1,1) &
                 + slope*(real(k,kind=dbl_kind)-p5) &
                          /real(nilyr,kind=dbl_kind)

           eicen(1,1,k) = &
                      -(rhoi * (cp_ice*(Tmlt(k)-Ti) &
                      + Lfresh*(c1-Tmlt(k)/Ti) - cp_ocn*Tmlt(k))) &
                      * vicen(1,1)/real(nilyr,kind=dbl_kind)
		      
	   !print*, 'At start, k, Ti = ', k, Ti
	   
	   if (k==1) then
	      Tn_top(1,1) = c0
	   endif
       enddo
       
       !print*, 'At start, Tatmos = ', Tatmos
      
      do k = 1,nslyr
         esnon(1,1,k) = (c0 - vsnon(1,1) * Lfresh * rhos) / real(nslyr)
      enddo

! Only 1 point
      icells=1
      indxi=1
      indxj=1
      my_task=0

      flw=c0
      potT=c0
      Qa=c0
      rhoa=c0
      fsnow=c0
      fbot=c0
      lhcoef=c0
!      shcoef=(1.20e-3_dbl_kind)*cp_air*rhoa(i,j)*wind(i,j)
      shcoef=(1.20e-3_dbl_kind)*cp_air*1.3_dbl_kind*wind_speed
      fswsfc=c0
      fswint=c0
      fswthrun=c0
      Sswabs=c0
      Iswabs=c0
      fsurfn=c0
      fcondtopn=c0
      fsensn=c0
      flatn=c0
      fswabsn=c0
      flwoutn=c0
      evapn=c0
      freshn=c0
      fsaltn=c0
      fhocnn=c0
      meltt=c0
      melts=c0
      meltb=c0
      congel=c0
      snoice=c0
      mlt_onset=c0   
      frz_onset=c0
      yday=c0
      l_stop=.false.
      istop=1
      jstop=1
      
      keffn_top(1,1) = c2 * kice * aicen(1,1) * c4 / vicen(1,1)
      
! Time quantities
      time_in_day = int(3600.0_dbl_kind*24.0_dbl_kind,kind=int_kind)
      n_cpl_in_day = int(time_in_day / force_dt)
      n_atm_in_cpl = int(force_dt / atmos_dt)
      n_ice_in_cpl = int(force_dt / dt)

! Time quantities
      nt_in_day = int(3600.0_dbl_kind*24.0_dbl_kind/dt,kind=int_kind)
      atmos_nt_in_force = force_dt / atmos_dt

      print*,'nt_in_day=',nt_in_day
      print*,'atmos_nt_in_force=',atmos_nt_in_force

      ratio_it = force_dt / dt
      if (ratio_it < 1) then
         print*,'Forcing dt must be larger than model dt'
         stop
      endif

      atm_force_avg_tmp(:,:) = c0
      atm_force_avg(:,:)     = c0
      print*, 'size(depth) = ', size(depth)

!--------------------------------------------------------------------
! Find output levels
!--------------------------------------------------------------------

      lyr_out(:)=0
      hilyr= vicen(1,1) / (aicen(1,1) * real(nilyr,kind=dbl_kind))
      hslyr= vsnon(1,1) / (aicen(1,1) * real(nslyr,kind=dbl_kind))
      do k=1,nilyr
        depth(k)= hilyr*real(k,kind=dbl_kind) - hilyr*0.5
        if (depth(k) > depth_out1 - puny .and. depth(k) < depth_out1 + puny) &
                                 lyr_out(1)=k
        if (depth(k) > depth_out2 - puny .and. depth(k) < depth_out2 + puny) &
                                 lyr_out(2)=k
        if (depth(k) > depth_out3 - puny .and. depth(k) < depth_out3 + puny) &
                                 lyr_out(3)=k
        if (depth(k) > depth_out4 - puny .and. depth(k) < depth_out4 + puny) &
                                 lyr_out(4)=k
	lyr_out(k) = k
      enddo

      lyr_out_snow = 0
      do k=1,nslyr
        depth_snow(k) = hslyr*real(k,kind=dbl_kind) - hslyr*0.5
        print*, 'k, depth_snow(k), depth_out_snow = ', k, depth_snow(k), depth_out_snow
        if (depth_snow(k) > depth_out_snow - puny .and. depth_snow(k) < depth_out_snow + puny) &
                                 lyr_out_snow = k
      enddo
      print*, 'lyr_out_snow = ', lyr_out_snow      

      if (lyr_out_snow == 0) then
         if (vsnon(1,1)==c0) then
            lyr_out_snow = 1
         else
            print*,'Problem with output layers'
            print*,'lyr_out_snow=', lyr_out_snow
            print*,'Aborting.................'
         stop
         endif
      endif
        
      do k=1,4
        if (lyr_out(k) == 0) then
           print*,'Problem with output layers'
           print*,'k, lyr_out=',k, lyr_out(k)
           print*,'Depths=',depth
           print*,'Aborting.................'
           stop
        endif
      enddo

!---------------------------------------------------------------------
! Calculate temperature profiles
! Complete ncycles cycles to reach equilibrium
!--------------------------------------------------------------------

      open(unit=10,file=filename_d)
      write(10,'(12(e13.6,1x))') c0, c0,c0,c0,depth_snow(1),depth_snow(lyr_out_snow),depth(1),depth(lyr_out),c1
      open(unit=12,file=filename_d2)

      do icycle=1,ncycles

        print*,'In cycle no.',icycle
                                 
        ! Main timestep through day
        do it=1,nt_in_day

            timestep_no=it+icycle*nt_in_day
	    if (mod(timestep_no,min_print) == 0) then
	       print_diags_in_ice_therm = .true.
	    else
	       print_diags_in_ice_therm = .false.
	    endif
	    
            if (mod(it-1,ratio_it) == 0) then

               ! Update forcing

               if (.not.calc_Tsfc) then
                 ! Get most recent top layer T and effective conductivity
                 call  top_layer_Tandk_block (nx_block, ny_block,  & 
                                              aicen,    vicen,     &
                                              vsnon,    eicen,     &
                                              esnon,               &
                                              Tn_top,   keffn_top)
               endif

               atm_force_avg_tmp(1,1) = c0
               do atmos_it = 1, atmos_nt_in_force   ! mimics atmos model
                  time = (icycle-1) * c24 * c60 * c60 + (it-1) * dt + atmos_it * atmos_dt
                  atmos_time = real(atmos_it,kind=dbl_kind)*atmos_dt + atmos_start_time

                  ! Calculate atmos T each timestep
                  Tatmos=Tatmos_a*sin(atmos_time*Tatmos_omega) + Tatmos_offset !deg C
                  potT(1,1) = Tatmos + Tffresh ! K

                  if (calc_Tsfc) then

                    ! Cumulate forcing over forcing timestep
                    atm_force_avg_tmp(1,1) = atm_force_avg_tmp(1,1) + potT(1,1)
                    if (print_diags_in_ice_therm) then
                       write(12,'(5(e13.6,1x))') time,potT-Tffresh,fsurfn,trcrn,Tn_top
                    endif

                  else

                    ! Calculate fcondtop
                    call explicit_calc_Tsfc (nx_block,      ny_block, &
                                             my_task,       icells,   &
                                             indxi,         indxj,    &
                                             dt,                      &
                                             trcrn(1,1,1),  aicen,    &
                                             vicen,         vsnon,    &
                                             rhoa,          flw,      &
                                             potT,          Qa,       &
                                             shcoef,        lhcoef,   &
                                             Tn_top,        keffn_top,&
                                             fswsfc,        flwoutn,  &
                                             fsensn,        flatn,    &
                                             fsurfn,        fcondtopn,&
                                             .false.,1,1)

                    if (print_diags_in_ice_therm) then
                       write(12,'(5(e13.6,1x))') time,potT-Tffresh,fsurfn,trcrn,Tn_top
                    endif
       
                    ! Cummulate forcing over forcing timestep
                    atm_force_avg_tmp(1,1) = atm_force_avg_tmp(1,1) + fcondtopn(1,1)

                 endif  ! calc_Tsfc

               enddo    ! atmos_it

               atmos_start_time=atmos_time

               ! Get average forcing
               atm_force_avg(1,1)= & 
                 atm_force_avg_tmp(1,1)/real(atmos_nt_in_force,kind=dbl_kind)
           
            endif
            time = (icycle-1) * c24 * c60 * c60 + it * dt

            ! Set forcing 
            if (calc_Tsfc) then
              potT(1,1) = atm_force_avg(1,1)
            else
              fcondtopn(1,1) = atm_force_avg(1,1)
            endif

            ! Calculate temperature profile
            call thermo_vertical (nx_block,    ny_block,  &
                                  dt,          icells,    &
                                  indxi,       indxj,     &
                                  aicen,       trcrn,     &
                                  vicen,       vsnon,     &
                                  eicen,       esnon,     &
                                  flw,         potT,      &
                                  Qa,          rhoa,      &
                                  fsnow,                  &
                                  fbot,        Tbot,      &
                                  lhcoef,      shcoef,    &
                                  fswsfc,      fswint,    &
                                  fswthrun,    Tn_top,    &
                                  Sswabs,      Iswabs,    &
                                  fsurfn,      fcondtopn, &
                                  fsensn,      flatn,     &
                                  fswabsn,     flwoutn,   &
                                  evapn,       freshn,    &
                                  fsaltn,      fhocnn,    &
                                  meltt,       melts,     &
                                  meltb,                  &
                                  congel,      snoice,    &
                                  mlt_onset,   frz_onset, &
                                  yday,        l_stop,    &
                                  istop,       jstop,     &
                                  lyr_out,     lyr_out_snow,&
                                  time,      &
				  output_dt,   print_diags_in_ice_therm)

        enddo  ! it

      enddo    ! icycle


      close(10)
      close(11)
      close(12)
      close(100)

      stop
      end
