! 1D model of ice thermodynamics to look at diurnal cycle
! Code taken from CICE 
! Force ice with a sinusoidal atmos T so that it can be compared with
! analytical solution
!
! Alison McLaren
! May 2011
!
!==========================================================================
!
      program ice_model

      use ice_kinds_mod
      use ice_domain_size
      use ice_coupling
      use ice_therm_vertical

      implicit none

      integer (kind=int_kind),parameter :: &
         nx_block = 1    , &
         ny_block = 1    ! block dimensions

      integer (kind=int_kind) :: &
         icells          , &  ! number of cells with ice present
         my_task              ! processor number

      integer (kind=int_kind), dimension (nx_block*ny_block):: &
         indxi, indxj     ! compressed indices for cells with ice

      real (kind=dbl_kind) :: &
         dt               ! model time step

      ! ice state variables
      real (kind=dbl_kind), dimension (nx_block,ny_block) :: &
         aicen , & ! concentration of ice
         vicen , & ! volume per unit area of ice          (m)
         vsnon     ! volume per unit area of snow         (m)

      real (kind=dbl_kind), dimension (nx_block,ny_block,max_ntrcr) :: &
         trcrn

      real (kind=dbl_kind), dimension(nx_block,ny_block,nilyr) :: &
         eicen     ! energy of melting for each ice layer (J/m^2)

      real (kind=dbl_kind), dimension(nx_block,ny_block,nslyr) :: &
         esnon     ! energy of melting for each snow layer (J/m^2)

      ! input from atmosphere
      real (kind=dbl_kind), dimension (nx_block,ny_block) :: &
         flw     , & ! incoming longwave radiation (W/m^2)
         potT    , & ! air potential temperature  (K) 
         Qa      , & ! specific humidity (kg/kg) 
         rhoa    , & ! air density (kg/m^3) 
         fsnow   , & ! snowfall rate (kg m-2 s-1)
         shcoef  , & ! transfer coefficient for sensible heat
         lhcoef      ! transfer coefficient for latent heat

      real (kind=dbl_kind), dimension (nx_block,ny_block) :: &
         fswsfc  , & ! SW absorbed at ice/snow surface (W m-2)
         fswint  , & ! SW absorbed in ice interior, below surface (W m-2)
         fswthrun    ! SW through ice to ocean         (W/m^2)

      real (kind=dbl_kind), dimension (nx_block,ny_block,nslyr) :: &
         Sswabs      ! SW radiation absorbed in snow layers (W m-2)

      real (kind=dbl_kind), dimension (nx_block,ny_block,nilyr) :: &
         Iswabs      ! SW radiation absorbed in ice layers (W m-2)

      ! input from ocean
      real (kind=dbl_kind), dimension (nx_block,ny_block) :: &
         fbot    , & ! ice-ocean heat flux at bottom surface (W/m^2)
         Tbot        ! ice bottom surface temperature (deg C)

      ! coupler fluxes to atmosphere
      real (kind=dbl_kind), dimension (nx_block,ny_block):: &
         fsensn  , & ! sensible heat flux (W/m^2) 
         fswabsn , & ! shortwave flux absorbed in ice and ocean (W/m^2) 
         flwoutn , & ! outgoing longwave radiation (W/m^2) 
         evapn       ! evaporative water flux (kg/m^2/s) 

      ! Note: these are intent out if calc_Tsfc = T, otherwise intent in
      real (kind=dbl_kind), dimension (nx_block,ny_block):: &
         flatn    , & ! latent heat flux   (W/m^2) 
         fsurfn   , & ! net flux to top surface, excluding fcondtopn
         fcondtopn    ! downward cond flux at top surface (W m-2)

      ! coupler fluxes to ocean
      real (kind=dbl_kind), dimension (nx_block,ny_block):: &
         freshn  , & ! fresh water flux to ocean (kg/m^2/s)
         fsaltn  , & ! salt flux to ocean (kg/m^2/s)
         fhocnn      ! net heat flux to ocean (W/m^2) 

      ! diagnostic fields
      real (kind=dbl_kind), dimension(nx_block,ny_block) :: &
         meltt    , & ! top ice melt             (m/step-->cm/day) 
         melts    , & ! snow melt                (m/step-->cm/day) 
         meltb    , & ! basal ice melt           (m/step-->cm/day) 
         congel   , & ! basal ice growth         (m/step-->cm/day) 
         snoice   , & ! snow-ice formation       (m/step-->cm/day) 
         mlt_onset, & ! day of year that sfc melting begins 
         frz_onset    ! day of year that freezing begins (congel or frazil) 

      real (kind=dbl_kind):: &
         yday      ! day of year

      logical (kind=log_kind) :: &
         l_stop          ! if true, print diagnostics and abort on return

      integer (kind=int_kind) :: &
         istop, jstop    ! indices of grid cell where code aborts

      integer (kind=int_kind) :: &
         k, it, icycle, atmos_it         ! indices

      integer (kind=int_kind) :: &
         ncycles         , &  ! no. of day cycles
         ratio_it        , &  ! force_dt / dt
         couple_ratio_it      ! force_dt / atmos_dt

      integer (kind=int_kind), dimension (4) :: &
         lyr_out           ! vertical indices on which to output

      real (kind=dbl_kind) :: &
         force_dt,  & ! forcing time step (i.e. coupling)
         atmos_dt,  & ! 'atmos model' timestep (frequency of calculating Tatmos)
         output_dt    ! frequency of writing output (s)

      integer (kind=int_kind) :: &
         nt_in_year       , &  ! no. of model timesteps in year
         atmos_nt_in_force    ! no. of atmos model timesteps in forcing period

      real (kind=dbl_kind) :: & 
         hilyr        , &  ! thickness of each layer
         depth_out1   , &  ! depths on which to output
         depth_out2   , &  ! depths on which to output
         depth_out3   , &  ! depths on which to output
         depth_out4        ! depths on which to output

      real (kind=dbl_kind) :: & 
         Tatmos_a      , &   ! atmos T cycle amplitude (deg C)
         Tatmos_omega  , &   ! angular freq of atmos T cycle (1/s)
         Tatmos_offset , &   ! offset for atmos T 
                             ! so Tatmos = Tatmos_a*sin(Tatmos_omega*t)+Tatmos_offset
         Tatmos        , &   ! atmos T  (deg C)
         atmos_start_time ,& ! atmos model start time
         atmos_time          ! atmos model time

      real (kind=dbl_kind) :: & 
         Ti            , &   ! initial Ti
         slope               ! vertical gradient of initial T profile


      real (kind=dbl_kind), dimension (nilyr):: &
         depth             ! mid depth of each layer

      real (kind=dbl_kind), dimension (nx_block,ny_block) :: &
         atm_force_avg     ! atmos forcing (either Tatmos or fcondtop depending on calc_Tsfc) 
                           ! averaged over forcing timestep 

      real (kind=dbl_kind), dimension (nx_block,ny_block):: &
         atm_force_avg_tmp    ! working array for atm_force_avg

      real (kind=dbl_kind), dimension (nx_block,ny_block) :: &
         Tn_top    , & ! T of top layer
         keffn_top    ! effective conductivity of top layer

      character (100) :: &
         filename_d   , &    ! output filename (data)
         filename_d2   , &   ! output filename (data on atmos timestep, 
                             !    calc_Tswfc=F only)
         filename_t          ! output filename (txt info)

!=======================================================================

! Things to set...

      filename_d='/data/local/hadal/MULTI_LAYERS/CICE_1D/ANNUAL/data_tatmos_20m_1cm.dat'
      filename_d2='/data/local/hadal/MULTI_LAYERS/CICE_1D/ANNUAL/dataA_tatmos_20m_1cm.dat'
      filename_t='/data/local/hadal/MULTI_LAYERS/CICE_1D/ANNUAL/data_tatmos_20m_1cm.txt'
!      filename_d='test.dat'
!      filename_d2='test2.dat'
!      filename_t='test.txt'

      Tatmos_a=c15
      Tatmos_omega=2.0_dbl_kind*pi/(3600.0_dbl_kind*24.0_dbl_kind*365.0_dbl_kind)
      Tatmos_offset=-c20

      dt = c20*c60         ! model time step (must be > 1s)
      force_dt = c1*dt ! forcing time step (i.e. coupling)
                          ! must be a multiple of atmos_dt and dt
      atmos_dt = c20*c60

      output_dt = dt
      
      ncycles=20          ! no. of cycles

      atmos_start_time = c0 ! can introduce a lag to atmos forcing

      aicen(1,1)=1.0_dbl_kind
      vicen(1,1)=1.0_dbl_kind

      depth_out1=0.125_dbl_kind   ! Depths on which to output T
      depth_out2=0.375_dbl_kind
      depth_out3=0.625_dbl_kind
      depth_out4=0.875_dbl_kind

! ====================================================================

! Write out run info to txt file

      open(unit=11,file=filename_t)
      write(11,*) 'Data filename =',filename_d
      write(11,*) 'dt=',dt
      write(11,*) 'force_dt=',force_dt
      write(11,*) 'atmos_dt=',atmos_dt
      write(11,*) 'nilyr=',nilyr
      write(11,*) 'nslyr=',nslyr
      write(11,*) 'calc_Tsfc=',calc_Tsfc
      write(11,*) 'ncycles=',ncycles
      write(11,*) 'hice=',vicen(1,1)/aicen(1,1)
      write(11,*) 'Tatmos_a=',Tatmos_a
      write(11,*) 'Tatmos_omega=',Tatmos_omega
      write(11,*) 'Tatmos_offset=',Tatmos_offset
      write(11,*) 'atmos_start_time=',atmos_start_time

!=====================================================================

! Initialise fields


! Initialise Tmlt and salin
      call init_thermo_vertical
      write(11,*) 'l_brine=',l_brine

! Initalise ice state

! Surface T
      Tatmos = Tatmos_a*sin(atmos_start_time*Tatmos_omega) + Tatmos_offset
      trcrn(1,1,1) = min(Tsmelt, Tatmos)

! Bottom T
      Tbot(1,1)=-1.8_dbl_kind

! T profile
      do k=1,nilyr 
          ! assume linear temp profile and compute enthalpy
          slope = Tbot(1,1) - trcrn(1,1,1)
             Ti = trcrn(1,1,1) &
                 + slope*(real(k,kind=dbl_kind)-p5) &
                          /real(nilyr,kind=dbl_kind)

           eicen(1,1,k) = &
                      -(rhoi * (cp_ice*(Tmlt(k)-Ti) &
                      + Lfresh*(c1-Tmlt(k)/Ti) - cp_ocn*Tmlt(k))) &
                      * vicen(1,1)/real(nilyr,kind=dbl_kind)
       enddo

! No snow
      vsnon=c0
      esnon=c0

! Only 1 point
      icells=1
      indxi=1
      indxj=1
      my_task=0

      flw=c0
      potT=c0
      Qa=c0
      rhoa=c0
      fsnow=c0
      fbot=c0
      lhcoef=c0
!      shcoef=(1.20e-3_dbl_kind)*cp_air*rhoa(i,j)*wind(i,j)
      shcoef=(1.20e-3_dbl_kind)*cp_air*1.3_dbl_kind*c5
      fswsfc=c0
      fswint=c0
      fswthrun=c0
      Sswabs=c0
      Iswabs=c0
      fsurfn=c0
      fcondtopn=c0
      fsensn=c0
      flatn=c0
      fswabsn=c0
      flwoutn=c0
      evapn=c0
      freshn=c0
      fsaltn=c0
      fhocnn=c0
      meltt=c0
      melts=c0
      meltb=c0
      congel=c0
      snoice=c0
      mlt_onset=c0   
      frz_onset=c0
      yday=c0
      l_stop=.false.
      istop=1
      jstop=1

! Time quantities
      nt_in_year = int(3600.0_dbl_kind*24.0_dbl_kind*365.0_dbl_kind/dt,kind=int_kind)
      atmos_nt_in_force = force_dt / atmos_dt

      print*,'nt_in_year=',nt_in_year
      print*,'atmos_nt_in_force=',atmos_nt_in_force

      ratio_it = force_dt / dt
      if (ratio_it < 1) then
         print*,'Forcing dt must be larger than model dt'
         stop
      endif

      atm_force_avg_tmp(:,:) = c0
      atm_force_avg(:,:)     = c0

!--------------------------------------------------------------------
! Find output levels
!--------------------------------------------------------------------

      lyr_out(:)=0
      hilyr= vicen(1,1) / (aicen(1,1) * real(nilyr,kind=dbl_kind))
      do k=1,nilyr
        depth(k)= hilyr*real(k,kind=dbl_kind) - hilyr*0.5
        if (depth(k) > depth_out1 - puny .and. depth(k) < depth_out1 + puny) &
                                 lyr_out(1)=k
        if (depth(k) > depth_out2 - puny .and. depth(k) < depth_out2 + puny) &
                                 lyr_out(2)=k
        if (depth(k) > depth_out3 - puny .and. depth(k) < depth_out3 + puny) &
                                 lyr_out(3)=k
        if (depth(k) > depth_out4 - puny .and. depth(k) < depth_out4 + puny) &
                                 lyr_out(4)=k
      enddo
      
      do k=1,4
        if (lyr_out(k) == 0) then
           print*,'Problem with output layers'
           print*,'k, lyr_out=',k, lyr_out(k)
           print*,'Depths=',depth
           print*,'Aborting.................'
           stop
        endif
      enddo

!---------------------------------------------------------------------
! Calculate temperature profiles
! Complete ncycles cycles to reach equilibrium
!--------------------------------------------------------------------

      open(unit=10,file=filename_d)
      write(10,'(9(e13.6,1x))') c0,c0,c0,depth(1),depth(lyr_out),c1
      if (.not.calc_Tsfc) open(unit=12,file=filename_d2)

      do icycle=1,ncycles

        print*,'In cycle no.',icycle
                                 
        ! Main timestep through day
        do it=1,nt_in_year

            if (mod(it-1,ratio_it) == 0) then

               ! Update forcing

               if (.not.calc_Tsfc) then
                 ! Get most recent top layer T and effective conductivity
                 call  top_layer_Tandk_block (nx_block, ny_block,  & 
                                              aicen,    vicen,     &
                                              vsnon,    eicen,     &
                                              esnon,               &
                                              Tn_top,   keffn_top)
               endif

               atm_force_avg_tmp(1,1) = c0
               do atmos_it = 1, atmos_nt_in_force   ! mimics atmos model
                  atmos_time = real(atmos_it,kind=dbl_kind)*atmos_dt + atmos_start_time

                  ! Calculate atmos T each timestep
                  Tatmos=Tatmos_a*sin(atmos_time*Tatmos_omega) + Tatmos_offset !deg C
                  potT(1,1) = Tatmos + Tffresh ! K

                  if (calc_Tsfc) then

                    ! Cummulate forcing over forcing timestep
                    atm_force_avg_tmp(1,1) = atm_force_avg_tmp(1,1) + potT(1,1)

                  else

                    ! Calculate fcondtop
                    call explicit_calc_Tsfc (nx_block,      ny_block, &
                                             my_task,       icells,   &
                                             indxi,         indxj,    &
                                             dt,                      &
                                             trcrn(1,1,1),  aicen,    &
                                             vicen,         vsnon,    &
                                             rhoa,          flw,      &
                                             potT,          Qa,       &
                                             shcoef,        lhcoef,   &
                                             Tn_top,        keffn_top,&
                                             fswsfc,        flwoutn,  &
                                             fsensn,        flatn,    &
                                             fsurfn,        fcondtopn,&
                                             .false.,1,1)
                   if( icycle == ncycles ) then ! write output
                     write(12,'(4(e13.6,1x))') potT-Tffresh,fsurfn,trcrn,Tn_top
                   endif
       
                    ! Cummulate forcing over forcing timestep
                    atm_force_avg_tmp(1,1) = atm_force_avg_tmp(1,1) + fcondtopn(1,1)

                 endif  ! calc_Tsfc

               enddo    ! atmos_it

               atmos_start_time=atmos_time

               ! Get average forcing
               atm_force_avg(1,1)= & 
                 atm_force_avg_tmp(1,1)/real(atmos_nt_in_force,kind=dbl_kind)
           
            endif

            ! Set forcing 
            if (calc_Tsfc) then
              potT(1,1) = atm_force_avg(1,1)
            else
              fcondtopn(1,1) = atm_force_avg(1,1)
            endif

            ! Calculate temperature profile
            call thermo_vertical (nx_block,    ny_block,  &
                                  dt,          icells,    &
                                  indxi,       indxj,     &
                                  aicen,       trcrn,     &
                                  vicen,       vsnon,     &
                                  eicen,       esnon,     &
                                  flw,         potT,      &
                                  Qa,          rhoa,      &
                                  fsnow,                  &
                                  fbot,        Tbot,      &
                                  lhcoef,      shcoef,    &
                                  fswsfc,      fswint,    &
                                  fswthrun,               &
                                  Sswabs,      Iswabs,    &
                                  fsurfn,      fcondtopn, &
                                  fsensn,      flatn,     &
                                  fswabsn,     flwoutn,   &
                                  evapn,       freshn,    &
                                  fsaltn,      fhocnn,    &
                                  meltt,       melts,     &
                                  meltb,                  &
                                  congel,      snoice,    &
                                  mlt_onset,   frz_onset, &
                                  yday,        l_stop,    &
                                  istop,       jstop,     &
                                  lyr_out, it*dt, output_dt)

        enddo  ! it

      enddo    ! icycle

      close(10)
      close(11)
      if (.not. calc_Tsfc) close(12)

      stop
      end
