!=======================================================================
!
!BOP
!
! !MODULE: ice_coupling - contains coupling related routines used by Met Office
!
! !DESCRIPTION:
!
!  Contains routines relating to coupling fields used by Met Office
!
! !REVISION HISTORY:
!  SVN:$Id: 
!
!  authors: Alison McLaren, Met Office
!
! !INTERFACE:
!
      module ice_coupling
!
! !USES:
!
      use ice_constants
      use ice_kinds_mod
!
!EOP
!     
      implicit none
      save

! !PUBLIC MEMBER FUNCTIONS:

      public :: explicit_calc_Tsfc, &
                top_layer_Tandk_block
!
!EOP
!
!=======================================================================

      contains

!=======================================================================
!BOP
!
! !ROUTINE: explicit_calc_Tsfc - subroutine to compute sfc fluxes
!
! !DESCRIPTION:
!
! Compute fsurfn and fcondtopn, given temperature, thickness, and 
! conductivity of top ice or snow layer.  This routine is used in 
! runs which are not coupled to an atmosphere model and calc_exp_Tsfc=T.
!
! !REVISION HISTORY:
!
! authors William H. Lipscomb, LANL
!
! !INTERFACE:
!
      subroutine explicit_calc_Tsfc (nx_block,      ny_block, &
                                     my_task,       icells,   &
                                     indxi,         indxj,    &
                                     dt,                      &
                                     Tsfcn,         aicen,    &
                                     vicen,         vsnon,    &
                                     rhoa,          flw,      &
                                     potT,          Qa,       &
                                     shcoef,        lhcoef,   &
                                     T_top,         keff,     &
                                     fswsfcn,       flwoutn,  &
                                     fsensn,        flatn,    &
                                     fsurfn,        fcondtopn,&
                                     example_output,iout,jout)
!
! !USES:
!
      use ice_domain_size, only: nslyr,nilyr
      use ice_therm_vertical
! 
! !INPUT/OUTPUT PARAMETERS:
!
      integer (kind=int_kind), intent(in) :: &
         nx_block, ny_block, & ! block dimensions
         my_task           , & ! process ID
         icells                ! number of cells with ice present

      integer (kind=int_kind), dimension(icells), intent(in) :: &
         indxi, indxj    ! compressed indices for ice cells

      real (kind=dbl_kind), intent(in) :: &
         dt      ! time step

      real (kind=dbl_kind), dimension (nx_block,ny_block), &
         intent(inout) :: &
         Tsfcn     ! temperature of ice/snow top surface  (C)

      real (kind=dbl_kind), dimension (nx_block,ny_block), intent(in) :: &
         aicen , & ! concentration of ice
         vicen , & ! volume per unit area of ice          (m)
         vsnon     ! volume per unit area of snow         (m)

      real (kind=dbl_kind), dimension (nx_block,ny_block), intent(in) :: &
         fswsfcn     , & ! SW absorbed at ice/snow surface (W m-2)
         rhoa        , & ! air density (kg/m^3)
         flw         , & ! incoming longwave radiation (W/m^2)
         potT        , & ! air potential temperature  (K)
         Qa          , & ! specific humidity (kg/kg)
         shcoef      , & ! transfer coefficient for sensible heat
         lhcoef      , & ! transfer coefficient for latent heat
         T_top       , & ! top layer temperature
         keff            ! top layer effective conductivity
         
      real (kind=dbl_kind), dimension (nx_block,ny_block), &
         intent(out) :: &
         flwoutn     , & ! upward LW at surface (W m-2)
         fsensn      , & ! surface downward sensible heat (W m-2)
         flatn       , & ! surface downward latent heat (W m-2)
         fsurfn      , & ! net flux to top surface, excluding fcondtopn
         fcondtopn       ! conductive flux to top surface

      logical (kind=log_kind) ::   &
         example_output  ! if true, write out at example point

      integer (kind=int_kind), intent(in) ::   &
         iout,jout       ! indices for output point
!
!EOP
!
      integer (kind=int_kind) :: &
         isolve    ! number of cells with temps not converged (same as icells)

      integer (kind=int_kind), dimension(icells) :: &
         indxii, indxjj,&  ! compressed indices for cells not converged
         indxij            ! compressed 1D index for cells not converged

      real (kind=dbl_kind), dimension (icells) :: &
         Tsf         , & ! surface temperature
         khis        , & ! limited 2*k/h for top ice or snow layer
         hslyr       , & ! snow layer thickness
         hilyr           ! ice layer thickness 

      real (kind=dbl_kind), dimension (icells) :: &
         dfsens_dT   , & ! deriv of fsens wrt Tsf (W m-2 deg-1)
         dflat_dT    , & ! deriv of flat wrt Tsf (W m-2 deg-1)
         dflwout_dT  , & ! deriv of flwout wrt Tsf (W m-2 deg-1)
         dfsurf_dT       ! derivative of fsurf wrt Tsf

      integer :: i, j, ij, k, ijout   ! indices

      real (kind=dbl_kind) ::  &
         dTsf         , & ! change in Tsf
         khmax        , & ! max allowed value of kh
         ci               ! heat capacity of top ice layer

      logical (kind=log_kind) ::   &
         l_snow           ! true if hsno > hs_min

      logical (kind=log_kind) ::   &
         example_output2  ! if true, write out at example point


      ! Initialize fluxes

      fsurfn   (:,:) = c0
      fcondtopn(:,:) = c0
      flwoutn  (:,:) = c0
      fsensn   (:,:) = c0
      flatn    (:,:) = c0

      ! initialize surface temperature
      do ij = 1, icells	
         i = indxi(ij)
         j = indxj(ij)
         Tsf(ij) = Tsfcn(i,j)
      enddo

      !-----------------------------------------------------------------
      ! Initialize isolve and related indices to be identical to icells
      ! and related indices
      !-----------------------------------------------------------------

      isolve = icells
      do ij = 1, icells	
         indxii(ij) = indxi(ij)
         indxjj(ij) = indxj(ij)
         indxij(ij) = ij
      enddo

      !---------------------------------------------------------------
      ! Ensure conductivity satisfies diffusive CFL condition
      !---------------------------------------------------------------

      do ij = 1, icells
         i = indxi(ij)
         j = indxj(ij)
         k = 1   ! top layer of ice or snow

         ! Check if snow layer thickness hsno > hs_min

         hslyr(ij) = vsnon(i,j) / (aicen(i,j)*real(nslyr,kind=dbl_kind))  
         if (hslyr(ij) > hs_min) then
            l_snow = .true.
         else
            l_snow = .false.
         endif

         ! Calculate max conductivity to satisfy diffusive CFL condition

         if (l_snow) then

            khmax = rhos*cp_ice*hslyr(ij) / dt

         else
            ! Compute heat capacity of the ice layer
            if (l_brine) then
               ci = cp_ice - Lfresh*Tmlt(k) /  (T_top(i,j)*T_top(i,j))
            else
               ci = cp_ice
            endif
            hilyr(ij) = vicen(i,j) / (aicen(i,j)*real(nilyr,kind=dbl_kind))

            khmax = rhoi*ci*hilyr(ij) / dt

         endif

         ! Limit conductivity to satisfy diffusive CFL condition

         !if (keff(i,j) > khmax) then
	 !  print*,'Limiting conductivity!!'
         !  example_output2=.true.
         !endif
         !khis(ij) = min(keff(i,j), khmax)
         khis(ij) = keff(i,j)

      enddo   ! ij

      !-----------------------------------------------------------------
      ! Compute radiative and turbulent fluxes and their derivatives
      ! with respect to Tsf.
      !-----------------------------------------------------------------

      call surface_fluxes (nx_block,    ny_block,           &
                           isolve,      icells,             &
                           indxii,      indxjj,   indxij,   &
                           Tsf,         fswsfcn,            &
                           rhoa,        flw,                &
                           potT,        Qa,                 &
                           shcoef,      lhcoef,             &
                           flwoutn,     fsensn,             &
                           flatn,       fsurfn,             &
                           dflwout_dT,  dfsens_dT,          &
                           dflat_dT,    dfsurf_dT)

      !-----------------------------------------------------------------
      ! Solve for the new surface temperature and fluxes
      !-----------------------------------------------------------------

      do ij = 1, icells
         i = indxi(ij)
         j = indxj(ij)

         dTsf = (fsurfn(i,j) - khis(ij)*(Tsf(ij) - T_top(i,j))) /   &
                (khis(ij) - dfsurf_dT(ij))

         Tsf(ij) = Tsf(ij) + dTsf

         if (Tsf(ij) > c0) then
            dTsf = dTsf - Tsf(ij) 
            Tsf(ij) = c0
         endif

         Tsfcn(i,j) = Tsf(ij)   ! for output

         fsensn (i,j) = fsensn (i,j) + dTsf*dfsens_dT(ij)
         flatn  (i,j) = flatn  (i,j) + dTsf*dflat_dT(ij)
         flwoutn(i,j) = flwoutn(i,j) + dTsf*dflwout_dT(ij)
         fsurfn (i,j) = fsurfn (i,j) + dTsf*dfsurf_dT(ij)

         fcondtopn(i,j) = khis(ij) * (Tsf(ij) - T_top(i,j))

         if (i == iout .and. j == jout) ijout=ij

      enddo

      if (example_output) then
         write(nu_diag,*) 'Output from explicit_calc_Tsfc at diag point 1'
         write(nu_diag,*) 'Tsfcn, Tn_top=',Tsfcn(iout,jout),T_top(iout,jout)
         if (l_snow) then
           write(nu_diag,*) 'Snow thickness=',hslyr(ijout)
         else
           write(nu_diag,*) 'Ice thickness=',hilyr(ijout)
         endif
         write(nu_diag,*) 'keffn_top, limited keffn_top=', & 
                           keff(iout,jout),khis(ijout)
         write(nu_diag,*) 'fswsfcn, flw=',fswsfcn(iout,jout),flw(iout,jout)
         write(nu_diag,*) 'fsurfn, fcondtopn=', & 
                           fsurfn(iout,jout),fcondtopn(iout,jout)
         write(nu_diag,*) 'fsensn, flatn, flwoutn=', & 
                           fsensn(iout,jout),flatn(iout,jout),flwoutn(iout,jout)
      endif

      end subroutine explicit_calc_Tsfc

!=======================================================================
!BOP
!
! !ROUTINE: top_layer_Tandk_block
!
! !DESCRIPTION:
!
! Calculate the top layer temperature and conductivity for passing
! to atmosphere model or calculating Tsfc explicitly.
!
! This routine is only called if calc_imp_Tsfc = F and heat_capacity = T.
!
! !REVISION HISTORY:
!
! authors: Alison McLaren, Met Office
!
! !INTERFACE:

      subroutine top_layer_Tandk_block (nx_block, ny_block,  & 
                                        aicen,    vicen,     &
                                        vsnon,    eicen,     &
                                        esnon,               &
                                        T_top,   keff_top)
!
! !USES:
!
      use ice_domain_size
      use ice_therm_vertical  
!
! !INPUT/OUTPUT PARAMETERS:
!
!EOP
!
      integer (kind=int_kind), intent(in) :: &
         nx_block    , &
         ny_block 

      ! ice state variables
      real (kind=dbl_kind), dimension (nx_block,ny_block), intent(in) :: &
         aicen , & ! concentration of ice
         vicen , & ! volume per unit area of ice          (m)
         vsnon     ! volume per unit area of snow         (m)

      real (kind=dbl_kind), dimension(nx_block,ny_block,nilyr), intent(in) :: &
         eicen     ! energy of melting for each ice layer (J/m^2)

      real (kind=dbl_kind), dimension(nx_block,ny_block,nslyr), intent(in) :: &
         esnon     ! energy of melting for each snow layer (J/m^2)

      real (kind=dbl_kind), dimension (nx_block,ny_block), intent(out) :: &
         T_top    , & ! T of top layer
         keff_top    ! effective conductivity of top layer


      integer (kind=int_kind) :: & 
         i,j             ! horizontal indices

      real (kind=dbl_kind) ::  &
         rnslyr      , & ! real(nslyr)
         rnilyr      , & ! real(nilyr)
         hs1         , & ! top snow layer thickness
         hn1         , & ! top ice layer thickness
         qn1         , & ! top layer enthalpy
         ki              ! top ice layer conductivity


      keff_top(:,:) = c0   ! initialise
      T_top(:,:)    = c0   
      rnslyr = real(nslyr,kind=dbl_kind)      
      rnilyr = real(nilyr,kind=dbl_kind)      

            do j = 1, ny_block
            do i = 1, nx_block

              if (aicen(i,j) > puny) then

                hs1 = vsnon(i,j)/(aicen(i,j)*rnslyr)

                if (hs1 > hs_min/rnslyr) then

                  !snow is top layer
                  qn1 = esnon(i,j,1)*rnslyr/vsnon(i,j)
                  T_top(i,j)    = (Lfresh + qn1/rhos)/cp_ice
                  keff_top(i,j) = c2 * ksno / hs1   

                else

                  !ice is top layer
                  hn1 = vicen(i,j)/(aicen(i,j)*rnilyr)
                  qn1 = eicen(i,j,1)*rnilyr/vicen(i,j)
                  T_top(i,j)    = & 
                            calculate_Tin_from_qin(qn1,Tmlt(1))
                  ki = calculate_ki_from_Tin(T_top(i,j),salin(1))
                  keff_top(i,j) = c2 * ki / hn1
                endif   ! hs1 > hsmin

              endif      ! aice > puny
            enddo        ! i
            enddo        ! j

      end subroutine top_layer_Tandk_block

!======================================================================

      end module ice_coupling

!======================================================================
