! 1D model of ice thermodynamics to look at diurnal cycle
! Code taken from CICE 
! Force ice with a sinusoidal heat flux so that it can be compared with
! analytical solution
!
! Alison McLaren
! April 2011
!
!==========================================================================
!
      program ice_model

      use ice_kinds_mod
      use ice_domain_size
      use ice_therm_vertical

      implicit none

      integer (kind=int_kind),parameter :: &
         nx_block = 1    , &
         ny_block = 1    ! block dimensions

      integer (kind=int_kind) :: &
         icells                ! number of cells with ice present

      integer (kind=int_kind), dimension (nx_block*ny_block):: &
         indxi, indxj     ! compressed indices for cells with ice

      real (kind=dbl_kind) :: &
         dt,        & ! model time step
         force_dt,  & ! forcing time step (i.e. coupling)
         atmos_dt    ! 'atmos model' timestep (freq of calculating Fsurf)

      real (kind=dbl_kind) :: &
         T_initial    ! starting ice temperature
      
      integer (kind=int_kind) :: &
         force_nt_in_day  , &  ! no. of forcing timesteps in day
         atmos_nt_in_day       ! no. of atmos timesteps in day

      ! ice state variables
      real (kind=dbl_kind), dimension (nx_block,ny_block) :: &
         aicen , & ! concentration of ice
         vicen , & ! volume per unit area of ice          (m)
         vsnon     ! volume per unit area of snow         (m)

      real (kind=dbl_kind), dimension (nx_block,ny_block,max_ntrcr) :: &
         trcrn

      real (kind=dbl_kind), dimension(nx_block,ny_block,nilyr) :: &
         eicen     ! energy of melting for each ice layer (J/m^2)

      real (kind=dbl_kind), dimension(nx_block,ny_block,nslyr) :: &
         esnon     ! energy of melting for each snow layer (J/m^2)

      ! input from atmosphere
      real (kind=dbl_kind), dimension (nx_block,ny_block) :: &
         flw     , & ! incoming longwave radiation (W/m^2)
         potT    , & ! air potential temperature  (K) 
         Qa      , & ! specific humidity (kg/kg) 
         rhoa    , & ! air density (kg/m^3) 
         fsnow   , & ! snowfall rate (kg m-2 s-1)
         shcoef  , & ! transfer coefficient for sensible heat
         lhcoef      ! transfer coefficient for latent heat

      real (kind=dbl_kind), dimension (nx_block,ny_block) :: &
         fswsfc  , & ! SW absorbed at ice/snow surface (W m-2)
         fswint  , & ! SW absorbed in ice interior, below surface (W m-2)
         fswthrun    ! SW through ice to ocean         (W/m^2)

      real (kind=dbl_kind), dimension (nx_block,ny_block,nslyr) :: &
         Sswabs      ! SW radiation absorbed in snow layers (W m-2)

      real (kind=dbl_kind), dimension (nx_block,ny_block,nilyr) :: &
         Iswabs      ! SW radiation absorbed in ice layers (W m-2)

      ! input from ocean
      real (kind=dbl_kind), dimension (nx_block,ny_block) :: &
         fbot    , & ! ice-ocean heat flux at bottom surface (W/m^2)
         Tbot        ! ice bottom surface temperature (deg C)

      ! coupler fluxes to atmosphere
      real (kind=dbl_kind), dimension (nx_block,ny_block):: &
         fsensn  , & ! sensible heat flux (W/m^2) 
         fswabsn , & ! shortwave flux absorbed in ice and ocean (W/m^2) 
         flwoutn , & ! outgoing longwave radiation (W/m^2) 
         evapn       ! evaporative water flux (kg/m^2/s) 

      ! Note: these are intent out if calc_Tsfc = T, otherwise intent in
      real (kind=dbl_kind), dimension (nx_block,ny_block):: &
         flatn    , & ! latent heat flux   (W/m^2) 
         fsurfn   , & ! net flux to top surface, excluding fcondtopn
         fcondtopn    ! downward cond flux at top surface (W m-2)

      ! coupler fluxes to ocean
      real (kind=dbl_kind), dimension (nx_block,ny_block):: &
         freshn  , & ! fresh water flux to ocean (kg/m^2/s)
         fsaltn  , & ! salt flux to ocean (kg/m^2/s)
         fhocnn      ! net heat flux to ocean (W/m^2) 

      ! diagnostic fields
      real (kind=dbl_kind), dimension(nx_block,ny_block) :: &
         meltt    , & ! top ice melt             (m/step-->cm/day) 
         melts    , & ! snow melt                (m/step-->cm/day) 
         meltb    , & ! basal ice melt           (m/step-->cm/day) 
         congel   , & ! basal ice growth         (m/step-->cm/day) 
         snoice   , & ! snow-ice formation       (m/step-->cm/day) 
         mlt_onset, & ! day of year that sfc melting begins 
         frz_onset    ! day of year that freezing begins (congel or frazil) 

      real (kind=dbl_kind):: &
         yday      ! day of year

      logical (kind=log_kind) :: &
         l_stop   , & ! if true, print diagnostics and abort on return
         print_diags_in_ice_therm
	 
      integer (kind=int_kind) :: &
         istop, jstop    ! indices of grid cell where code aborts

      integer (kind=int_kind) :: &
         k, it, icycle, nt_in_day, force_it, ncycles, timestep_no, min_print

      integer (kind=int_kind) :: &
         ratio_it       , &  ! force_dt / dt
         atmos_ratio_it      ! force_dt / atmos_dt
 
      integer (kind=int_kind), dimension (4) :: &
         lyr_out           ! vertical indices on which to output

      real (kind=dbl_kind) :: & 
         hilyr        , &  ! thickness of each layer
         depth_out1   , &  ! depths on which to output
         depth_out2   , &  ! depths on which to output
         depth_out3   , &  ! depths on which to output
         depth_out4        ! depths on which to output

      real (kind=dbl_kind) :: & 
         fsurf_a      , &  ! surface heat flux amplitude (m)
         fsurf_omega       ! angular freq of surface heat flux (1/s)

      real (kind=dbl_kind), dimension (nilyr):: &
         depth             ! mid depth of each layer

      real (kind=dbl_kind), dimension (:,:,:), allocatable :: &
         fsurfn_avg        ! averaged over forcing timestep 

      real (kind=dbl_kind), dimension (nx_block,ny_block):: &
         fsurfn_avg_tmp    ! working array for fsurfn_avg

      character (100) :: &
         filename_d   , &    ! output filename (data)
         filename_t          ! output filename (txt info)

!=======================================================================

! Things to set...

      fsurf_a=20.0_dbl_kind
      fsurf_omega=2.0_dbl_kind*pi/(3600.0_dbl_kind*24.0_dbl_kind)

      dt=c1              ! model time step (must be > 1s)
      force_dt = dt       ! forcing time step (i.e. coupling)
                             ! must be a multiple of dt
      atmos_dt=dt     !c20*c60 'atmos model' time step (typically 20min)
      
      ncycles=50          ! no. of cycles

      min_print=60_int_kind   ! on which frequency of timesteps to print data
      aicen(1,1)=1.0_dbl_kind
      vicen(1,1)=1.0_dbl_kind

      T_initial=-1.8_dbl_kind

      depth_out1=0.125_dbl_kind   ! Depths on which to output T
      depth_out2=0.375_dbl_kind
      depth_out3=0.625_dbl_kind
      depth_out4=0.875_dbl_kind
!      depth_out1=0.25_dbl_kind   ! Depths on which to output T
!      depth_out2=0.75_dbl_kind
!      depth_out3=1.25_dbl_kind
!      depth_out4=1.75_dbl_kind

      write(filename_d,fmt='(A44,3(I5.5,A1),I3.3,A4)') &
        '/data/local/hadax/MULTI_LAYERS/CICE_1D/data.', int(dt), '.', int(force_dt), '.', int(atmos_dt), '.', nilyr, '.dat' 

      write(filename_t,fmt='(A44,3(I5.5,A1),I3.3,A4)') &
        '/data/local/hadax/MULTI_LAYERS/CICE_1D/data.', int(dt), '.', int(force_dt), '.', int(atmos_dt), '.', nilyr, '.txt' 


! ====================================================================

! Write out run info to txt file

      print*, 'Data filename = ', filename_d
      open(unit=11,file=filename_t)
      write(11,*) 'Data filename =',filename_d
      write(11,*) 'dt=',dt
      write(11,*) 'force_dt=',force_dt
      write(11,*) 'atmos_dt=',atmos_dt
      write(11,*) 'nilyr=',nilyr
      write(11,*) 'nslyr=',nslyr
      write(11,*) 'calc_Tsfc=',calc_Tsfc
      write(11,*) 'ncycles=',ncycles
      write(11,*) 'hice=',vicen(1,1)/aicen(1,1)
      write(11,*) 'fsurf_a=',fsurf_a
      write(11,*) 'fsurf_omega=',fsurf_omega
      write(11,*) 'T_initial=',T_initial

! Initalise
      do k=1,nilyr 
        eicen(:,:,k)=-rhoi*(-cp_ice*T_initial + Lfresh) *     &
                       vicen(:,:)/real(nilyr,kind=dbl_kind)  
      enddo
      vsnon=c0
      esnon=c0
      trcrn=c0

! Only 1 point
      icells=1
      indxi=1
      indxj=1

      flw=c0
      potT=c0
      Qa=c0
      rhoa=c0
      fsnow=c0
      fbot=c0
      Tbot=-1.8_dbl_kind
      lhcoef=c0
      shcoef=c0
      fswsfc=c0
      fswint=c0
      fswthrun=c0
      Sswabs=c0
      Iswabs=c0
      fsurfn=c0
      fcondtopn=c0
      fsensn=c0
      flatn=c0
      fswabsn=c0
      flwoutn=c0
      evapn=c0
      freshn=c0
      fsaltn=c0
      fhocnn=c0
      meltt=c0
      melts=c0
      meltb=c0
      congel=c0
      snoice=c0
      mlt_onset=c0   
      frz_onset=c0
      yday=c0
      l_stop=.false.
      istop=1
      jstop=1

      nt_in_day = int(3600.0_dbl_kind*24.0_dbl_kind/dt,kind=int_kind)
      force_nt_in_day = int(3600.0_dbl_kind*24.0_dbl_kind/force_dt, & 
                        kind=int_kind)
      atmos_nt_in_day = int(3600.0_dbl_kind*24.0_dbl_kind/atmos_dt,kind=int_kind)

      print*,'nt_in_day=',nt_in_day
      print*,'force_nt_in_day=',force_nt_in_day

!----------------------------------------------------------------------
! Set up forcing.  Forcing is calculated for each second and then averaged
! over a specified time period (force_dt) to mimic what happens in
! the coupled model.
!----------------------------------------------------------------------

      ratio_it = force_dt / dt
      atmos_ratio_it = force_dt / atmos_dt
      if (ratio_it < 1) then
         print*,'Forcing dt must be larger than model dt'
         stop
      endif

      allocate(fsurfn_avg(nx_block,ny_block,nt_in_day/ratio_it))
      fsurfn_avg(:,:,:) = c0

! Calculate heat flux at 'atmos model' timestep and then average over 
! forcing timestep

      do it=1,atmos_nt_in_day ! Loop over each atmos timestep in day

         ! Calculate surface heat flux each second
         fsurfn(1,1)=fsurf_a*cos(real(it,kind=dbl_kind)*atmos_dt*fsurf_omega)

         ! Average over forcing timestep
         fsurfn_avg_tmp(1,1) = fsurfn_avg_tmp(1,1) + fsurfn(1,1)

         if (mod(it,atmos_ratio_it) == 0) then
            force_it = it/atmos_ratio_it
            fsurfn_avg(1,1,force_it)= & 
                 fsurfn_avg_tmp(1,1)/real(atmos_ratio_it,kind=dbl_kind)
            fsurfn_avg_tmp(1,1) = c0 
         endif

      enddo  ! it

!--------------------------------------------------------------------
! Find output levels
!--------------------------------------------------------------------

      lyr_out(:)=0
      hilyr= vicen(1,1) / (aicen(1,1) * real(nilyr,kind=dbl_kind))
      do k=1,nilyr
        depth(k)= hilyr*real(k,kind=dbl_kind) - hilyr*0.5
        if (depth(k) > depth_out1 - puny .and. depth(k) < depth_out1 + puny) &
                                 lyr_out(1)=k
        if (depth(k) > depth_out2 - puny .and. depth(k) < depth_out2 + puny) &
                                 lyr_out(2)=k
        if (depth(k) > depth_out3 - puny .and. depth(k) < depth_out3 + puny) &
                                 lyr_out(3)=k
        if (depth(k) > depth_out4 - puny .and. depth(k) < depth_out4 + puny) &
                                 lyr_out(4)=k
      enddo
      
      do k=1,4
        if (lyr_out(k) == 0) then
           print*,'Problem with output layers'
           print*,'k, lyr_out=',k, lyr_out(k)
           print*,'Depths=',depth
           print*,'Aborting.................'
           stop
        endif
      enddo

!---------------------------------------------------------------------
! Calculate temperature profiles
! Complete ncycles cycles to reach equilibrium
!--------------------------------------------------------------------

      open(unit=10,file=filename_d)
      !write(10,'(8(e13.6,1x))') c0,c0,depth(1),depth(lyr_out),vicen(1,1)/aicen(1,1)
      write(10,'(A15,I2)') 'min_print=', min_print
      write(10,'(8(A13,1x))') 'Fsfc', 'Tsfc', 'T1', 'T.125', 'T.375', 'T.625', 'T.875', &
        'Sum(eicen)'

      do icycle=1,ncycles

        !print*,'In cycle no.',icycle
        force_it = 0             ! This determines lag of forcing
                                 
        ! Main timestep through day
        do it=1,nt_in_day

            timestep_no=it+icycle*nt_in_day
	    if (mod(timestep_no,min_print) == 0) then
	       print_diags_in_ice_therm = .true.
	    else
	       print_diags_in_ice_therm = .false.
	    endif
	    
            if (mod(it-1,ratio_it) == 0) then
               force_it = force_it + 1
               if (force_it > force_nt_in_day) force_it = 1
            endif  

            fsurfn(1,1) = fsurfn_avg(1,1,force_it)
            fcondtopn(1,1) = fsurfn(1,1) ! Surface balance

            ! Initialise column
            call init_thermo_vertical

            if (it == 1 .and. icycle == 1) write(11,*) 'l_brine=',l_brine

            ! Calculate temperature profile
            call thermo_vertical (nx_block,    ny_block,  &
                                  dt,          icells,    &
                                  indxi,       indxj,     &
                                  aicen,       trcrn,     &
                                  vicen,       vsnon,     &
                                  eicen,       esnon,     &
                                  flw,         potT,      &
                                  Qa,          rhoa,      &
                                  fsnow,                  &
                                  fbot,        Tbot,      &
                                  lhcoef,      shcoef,    &
                                  fswsfc,      fswint,    &
                                  fswthrun,               &
                                  Sswabs,      Iswabs,    &
                                  fsurfn,      fcondtopn, &
                                  fsensn,      flatn,     &
                                  fswabsn,     flwoutn,   &
                                  evapn,       freshn,    &
                                  fsaltn,      fhocnn,    &
                                  meltt,       melts,     &
                                  meltb,                  &
                                  congel,      snoice,    &
                                  mlt_onset,   frz_onset, &
                                  yday,        l_stop,    &
                                  istop,       jstop,     &
                                  lyr_out,     print_diags_in_ice_therm)

        enddo  ! it

      enddo    ! icycle

      close(10)
      close(11)

      deallocate(fsurfn_avg)

      stop
      end
