;------------------------------------------------------------------------------- 
;	NAME
;		str
;
;	PURPOSE
;		IDL function to handle strings
;
;	PURPOSE
;		IDL function to handle string
;
;	USAGE
;		r = str(/up,data)			--> return upper case
;		r = str(/low,data)			--> return lower case
;		r = str(data,digit=value)		--> '3' --> '003' if digit=3
;		r = str(data,format=format)		--> format = '(F6.2)'
;		r = str(data,/get_size)			--> return text width in given device mode
;		r = str(data,['method'],/get_format)	--> method 'full' 1.003 --> (F5.3) 'label' 
;		r = str(data,['method'],/auto_format)	--> apply 'get_format'
;		r = str(data,/is_number)		--> 
;		r = str(data,/get_number)
;		r = str(data,/remove_number)
;		r = str(data,/remove_tag[,tag=tag,delimiter=delimiter)
;		r = str(data,'letter',/remove_letter)
;		r = str(data,'word',/remove_word)	
;		r = str(data,/remove_zero_tail)		--> '0.3000' -> '0.3'
;		r = str(/join,data1,data2,data3,...)
;		r = str(/structure,struct1)
;		r = str(data,len=length) /left /center
;		r = str(data,extension=ext)
;		r = str(data,'find','replace',/replace)
;		r = str(data,'string',/pos)
;		r = str(data,/tflag_to_timestamp[,/snaphost|,/period])
;		r = str(data,/title)
;		r = str(data,/systime_to_tflag)
;		r = str(data,/filetime_to_tflag)	: 
;		r = str(data,/tflag_simple)	: '20080101 20080102 20080103' --> '20080101 - 20080103'
;		r = str(data,/tflag_full)	: '20080101 - 20080103' --> '20080101 20080102 20080103'
;		r = str(data,/separate[,/preserve_null])	: '%sdfg%d' --> '%s','d','f','g','%d'
;
;	INPUT
;		data   : string
;		digit  : string length to return
;		format : format to return
;   
;	KEYWORD
;		is_number  : return 1 if input is a number
;		get_number : extact number only
;
;	AUTHOR
;		2006-10-10 Hyun Cheol Kim (hyuncheol.kim@gmail.com, hkim2@mail.uh.edu)
;		2008-07-11 modified /get_format 
;		2008-11-20 added /split
;------------------------------------------------------------------------------- 

	function str,_data,d01,d02,d03,d04,d05,d06,d07,d08,d09,d10,d11,d12,d13,d14,d15,d16,d17,d18,d19,$
	         digit=digit,format=format,up=up,low=low,                              $
	         is_number=is_number,get_number=get_number,full=full,                  $
	         auto_format=auto_format,get_format=get_format,get_size=get_size,      $
	         join=join,structure=structure,delimiter=delimiter,sign=sign,          $
	         len=len,right=right,left=left,center=center,background=background,    $
	         extension=extension,tflag=tflag,snapshot=snapshot,period=period,      $
	         title=title,replace=replace,pos=pos,first=first,                      $
	         remove_number=remove_number,            $
	         remove_tag=remove_tag,                  $					
	         remove_letter=remove_letter,            $
	         remove_word=remove_word,                $
	         remove_zero_tail=remove_zero_tail,      $					
	         filetime_to_tflag=filetime_to_tflag,    $
	         tflag_to_timestamp=tflag_to_timestamp,  $										
	         systime_to_tflag=systime_to_tflag,      $					
	         tflag_simple=tflag_simple,              $
	         month_to_name=month_to_name,            $					 
	         tflag_full=tflag_full,separate=separate,preserve_null=preserve_null,$
	         split=split,group=group,not_used=not_used,fill_space=fill_space,$
	         to_number=to_number,missing=missing,$
		 safe=safe,restore=_restore
 
;------------------------------------------------------------------------------- 

data     = _data
ndata    = n_elements(data)
is_array = (size(data,/n_dim))[0] ne 0 

case 1 of

	keyword_set(up)  : return,str(strupcase(data)) 				
	keyword_set(low) : return,str(strlowcase(data))		
	var_set(digit)   : return,string(data,format='(I'+str(digit)+'.'+str(digit)+')')		
	var_set(format)  : return,string(data,format=format)			
		
	var_set(get_size) : begin
	
		slen      = strlen(_data)
		max_str   = _data[(where(slen eq max(slen)))[0]]
		max_width = d01
		charsize  = 1.0
		
		xyouts,2,2,max_str,charsize=charsize,/normal,width=width
		
		if width gt max_width then begin
		
			while width gt max_width do begin
			
				charsize = charsize - 0.05
				
				xyouts,2,2,max_str,charsize=charsize,/normal,width=width
							
			endwhile
		
		endif else begin
		
			while width lt max_width do begin
			
				charsize = charsize + 0.05
				
				xyouts,2,2,max_str,charsize=charsize,/normal,width=width
							
			endwhile
						
		endelse
		
		return,charsize
	
		end
					
	keyword_set(get_format) : begin
	
		if var_set(d01) then method = d01 else method = 'label'
			
		max_ndata = 30	
		format  = ['(I16)','(F16.1)','(F16.2)','(F16.3)','(F16.4)','(F16.5)','(F16.6)','(F16.7)','(F16.8)']
		nformat = n_elements(format)

		min_value = min(data,max=max_value)
		ndata     = n_elements(data)
				
		if str(/low,method) eq 'label' then if n_elements(data) ne n_elements(array(data,/uniq)) then method = 'range'
		if str(/low,method) eq 'label' then if ndata ge max_ndata then method = 'range'
		if min_value eq max_value then method = 'full'
		
		case str(/low,method) of
		
			'full' : begin
			
				for iformat=0L,nformat-1 do	if total(float(str(data,format=format[nformat-1-iformat])) eq data) ne n_elements(data) then break				
				iformat = nformat-iformat
				end
				
			'label' : begin
			
				for iformat=0L,nformat-1 do if n_elements(str(data,format=format[iformat])) eq 	n_elements(array(str(data,format=format[iformat]),/uniq)) then break
				end	
				
			'range' : begin
				
				data = findgen(max_ndata)/(max_ndata-1)*(max_value-min_value)+min_value
				for iformat=0L,nformat-1 do if n_elements(str(data,format=format[iformat])) eq 	n_elements(array(str(data,format=format[iformat]),/uniq)) then break																
				end
				
			else : message,str(/join,"[USAGE] STR(DATA,'FULL/LABLE/RANGE',/GET_FORMAT)")		
		
		endcase
					
		case 1 of
		
			iformat eq 0 : format = '(I'+str(max(strlen(str(str(data,format=format[iformat])))))+')'
			iformat ge n_elements(format) : format = '(E16.2)'
			else : format = '(F'+str(max(strlen(str(str(data,format=format[iformat])))))+'.'+str(iformat)+')'
		
		endcase	
				
		return,format
								
		end
		
	keyword_set(auto_format) : return,string(data,format=str(data,d01,/get_format))
		  		
	keyword_set(is_number)  : begin
	
		if size(data,/type) ne 7 then return,replicate(1B,ndata)
		
		return,array_equal(str(data,/get_number,sign=sign),strcompress(data,/remove_all))
				                     		  	
		end	
	     
	keyword_set(get_number) : begin
	
		if is_array then result = make_array(size=size(data)) else result = ''		
		if keyword_set(sign) then find_str = '0123456789-+.' else find_str = '0123456789'
	
		for idata=0L,ndata-1 do begin
		
			data2 = byte(str(data[idata]))
			
			x = index(data2,byte(find_str),/contain,count=nx)		
			
			if nx gt 0 then result[idata] = string(data2[x])
		
		endfor
		
		if is_array then return,result else return,result[0]
	
		end
			
	keyword_set(remove_number) : begin
	
		if is_array then result = make_array(size=size(data)) else result = ''		
		if keyword_set(sign) then find_str = '0123456789-+.' else find_str = '0123456789'
	
		for idata=0L,ndata-1 do begin
		
			data2 = byte(data[idata])
			
			x = index(/complement,index(data2,byte(find_str),/contain),total=n_elements(data2),count=nx)		
			
			if nx gt 0 then result[idata] = string(data2[x])
		
		endfor
		
		if is_array then return,result else return,result[0]
			
		end		
			
	var_set(remove_tag) : begin
	
		tag = var_set(remove_tag,/keyword_only) ? '<>' : remove_tag				
		if ~var_set(delimiter) then delimiter = string(96b)
			
		tag1 = (byte(tag))[0]
		tag2 = (byte(tag))[1]
						
		data = byte(data)
		
		itag1 = where(data eq tag1,ntag1)
		itag2 = where(data eq tag2,ntag2)
				
		case 1 of
		
			ntag1 ne ntag2                                                 : method = 2
			check(itag2-itag1,/sign,/negative,/zero)                       : method = 2
			check(itag1[1:ntag1-1]-itag2[0:ntag2-2],/sign,/negative,/zero) : method = 2
			else                                                           : method = 1
		
		endcase
		
		case method of
		
			1 : begin
			
				for itag=0L,ntag1-1 do begin
				
					data[itag1[itag]:itag2[itag]] = 32B
					data[itag1[itag]] = byte(delimiter)
				endfor					
				
			end
			
			2 : begin
			
				for itag=0L,ntag1-1 do begin
		
					if itag eq ntag1-1 then x = where(itag2 gt itag1[itag],nx) $
					                   else x = where(itag2 gt itag1[itag] and itag2 lt itag1[itag+1],nx)
								
					if nx gt 0 then begin
			
						data[itag1[itag]:itag2[x[0]]] = 32B
						data[itag1[itag]] = byte(delimiter)
							
					endif
							
				endfor		
				
			end
					
		endcase
		
		return,string(data)
		end
				
	var_set(remove_letter) : begin
	
		if is_array then result = make_array(size=size(data)) else result = ''
		
		for idata=0L,ndata-1 do result[idata] = strjoin(strsplit(data[idata],d01,/extract))				
	
		if is_array then return,result else return,result[0]		
	  	
		end
					
	keyword_set(remove_word) : return,str(data,d01,'',/replace)
	  										
	keyword_set(remove_zero_tail) : begin
	
		result = strarr(ndata)
		
		if size(data,/type) eq 7 then return,data 
					
		for idata=0L,ndata-1 do begin
		
			if data[idata] eq long(data[idata]) then begin
			
				result[idata] = str(long(data[idata]))
			
			endif else begin
			
				value = reverse(byte(str(data[idata])))
				
				for i=0L,n_elements(value)-1 do if value[i] ne 48 then break

				result[idata] = string(reverse(value[i:*]))

			endelse 
					
		endfor
		
		return,result
	
		end		
		  
	keyword_set(join) : begin
  
		if ~var_set(delimiter) then delimiter = ' '
  
		npar = n_params()
    
		for ipar=0L,npar-1 do begin
    
			case ipar of
      
				0  : data = data
				1  : data = d01
				2  : data = d02
				3  : data = d03
				4  : data = d04
				5  : data = d05
				6  : data = d06
				7  : data = d07
				8  : data = d08
				9  : data = d09
				10 : data = d10
				11 : data = d11
				12 : data = d12
				13 : data = d13
				14 : data = d14
				15 : data = d15
				16 : data = d16
				17 : data = d17
				18 : data = d18
				19 : data = d19

				else : message,'Up to 15 parameters are available'

			endcase	
      
			if ipar eq 0 then result = [str(data,/structure)] $
			             else result = [result,str(data,/structure)] 
    
		endfor
    
		return,strjoin(result,delimiter)
  
		end			
  
	keyword_set(structure) : begin
  
		ntag = n_tags(data)
    
		if ntag ne 0 then begin
    
			for itag=0L,ntag-1 do begin
      
				if itag eq 0 then result = [str(data.(itag),/structure)] $
				             else result = [result,str(data.(itag),/structure)]
      
			endfor
    
		endif else result = str(data)
    
		return,result
  
		end
			  		
	var_set(len) : begin
  
		if ndata eq 1 then result = '' else result = make_array(size=size(data),/string)
    
		if ~var_set(background) then background = ' '
      
		for idata=0L,ndata-1 do begin
    
			backtemp= strjoin(string(replicate((byte(background))[0],len)))
 
			case 1 of
    
				keyword_set(left)   : strput,backtemp,str(data[idata]),0
				keyword_set(right)  : strput,backtemp,str(data[idata]),len-strlen(str(data[idata]))
				keyword_set(center) : strput,backtemp,str(data[idata]),fix((len-strlen(str(data[idata])))/2)
				else                : strput,backtemp,str(data[idata]),len-strlen(str(data[idata]))
							
			endcase
      
			result[idata] = backtemp
       
		endfor      
    
		return,result
  
		end
		
	var_set(extension) : begin
	
		data = str(data)
		
		if is_array then result = make_array(size=size(data)) else result = ''
		
		check = [strlowcase(strsplit(extension,/extract)),strupcase(strsplit(extension,/extract))]
		
		for idata=0L,ndata-1 do begin
		
			ext = strmid(data[idata],strpos(data[idata],'.',/reverse_search)+1,strlen(data[idata]))

			x = where(check eq ext,n)
			
			if n gt 0 then result[idata] = data[idata] else result[idata] = data[idata]+'.'+check[0]

		endfor

		if is_array then return,result else return,result[0]		
	
		end
		
	keyword_set(replace) : begin
	
		if ndata eq 1 then begin
		
			find    = d01
			nfind   = n_elements(find)			
			replace = array(d02,count=nfind)					
			result  = data
			
			for ifind=0L,nfind-1 do result = strjoin(strsplit(result,find[ifind],/extract,/regex),replace[ifind])
							
			return,result
		
		endif else begin
		
			result = strarr(ndata)
			
			for idata=0L,ndata-1 do result[idata] = str(/replace,data[idata],d01,d02)					
			
			return,result
		
		endelse
	
		end
				
	keyword_set(not_used) : begin
	
		not_used = byte('`~!@#$%^&*()_+=-\|;:",.<>/?')
		
		for i=0L,n_elements(not_used)-1 do begin
		
			x = where(strpos(data,not_used[i]) eq -1,nx)
			
			if nx eq ndata then break
		
		endfor
		
		return,string(not_used[i])
	
		end		
	
	keyword_set(pos) : begin
	
		if ndata eq 1 then begin
	
			p      = -1
			result = 0
		
			while strpos(data,d01,p) ne -1 do begin
		
				p = strpos(data,d01,p)+1
				result = [result,p]
		
			endwhile
			
			if n_elements(result) le 1 then return,-1 else return,result[1:*]-1
			
		endif else begin
		
			result = keyword_set(first) ? lonarr(ndata) : strarr(ndata)
			
			for idata=0L,ndata-1 do begin
			
				result[idata] = keyword_set(first) ? (str(/pos,data[idata],d01))[0] : str(/join,str(/pos,data[idata],d01))
			
			endfor
			
			return,result
		
		endelse	
	
		end
				
	keyword_set(split) : begin
		
		delimiter = d01
		
		group_check = strpos(data,group)
		
		for idata=0L,ndata-1 do begin
		
			if group_check[idata] eq -1 then begin
			
				data1 = strsplit(data[idata],delimiter,/extract,/preserve_null)
			
			endif else begin
							
				len = strlen(data[idata])
				
				xdelim = index(str(/pos,data[idata],delimiter),total=len,/mask)
				xgroup = str(/pos,data[idata],group)
				
				pgroup = intarr(len)
				
				for igroup=0L,n_elements(xgroup)-1 do pgroup[xgroup[igroup]:*] = pgroup[xgroup[igroup]:*] + 1
				
				pgroup = pgroup/2. ne fix(pgroup/2.)
				
				xdelim = where(pgroup*xdelim eq 1,nxdelim)
								
				escape = str(data[idata],/not_used)
				data1  = data[idata]
				
				for idelim=0L,nxdelim-1 do begin
				
					data1 = strmid(data1,0,xdelim[idelim]) + escape + delimiter + strmid(data1,xdelim[idelim]+1,len)
					xdelim = xdelim + strlen(escape)
									
				endfor
								
				data1 = strsplit(data1,delimiter,escape=escape,/extract,/preserve_null)
				
				for idata1=0L,n_elements(data1)-1 do begin
				
					data1_temp = str(data1[idata1])
					
					if strmid(data1_temp,0,1) eq group and strmid(data1_temp,0,1,/reverse_offset) eq group then data1[idata1] = strmid(data1_temp,1,strlen(data1_temp)-2)
				
				endfor

			endelse
									
			ndata1 = n_elements(data1)
								
			if not var_set(result) then begin

				result = strarr(ndata1,ndata)
				ncol   = ndata1
										
			endif
				
			if ndata1 gt ncol then begin
							
				result_temp        = result				
				result             = strarr(ndata1,ndata)
				result[0:ncol-1,*] = result_temp[0:ncol-1,*]
				ncol               = ndata1									
				
			endif
				
			result[0:ndata1-1,idata] = data1
			
		endfor
				
		return,result
		
		end		
		
	keyword_set(tflag_to_timestamp) : begin
	
		months = ['','Jan','Feb','Mar','Apr','May','Jun','Jul','Aug','Sep','Oct','Nov','Dec']
		
		case 1 of
		
			keyword_set(snapshot) : begin
			
				if is_array then result = make_array(size=size(data)) else result = ''
			
				for idata=0L,ndata-1 do begin
			
					xx = date(/cut,data[idata],oyear=y1,omonth=m1,oday=d1,ohour=h1)
			
					result[idata] = strcompress(strjoin([months[fix(m1)],d1,y1,h1+':00'],' '))
			
				endfor
			
				if is_array then return,result else return,result[0]
						
				end
				
			keyword_set(period) : begin
			
				if keyword_set(left) then begin
		
  		  			tflag1 = date(/incr,/hour,data[0],-1)
					tflag2 = data[ndata-1]
				
			  	endif else begin
		
		  		  	tflag1 = data[0]						
					tflag2 = date(/incr,/hour,data[ndata-1],1)
			
				endelse
		
				xx = date(/cut,tflag1,oyear=y1,omonth=m1,oday=d1,ohour=h1)
				xx = date(/cut,tflag2,oyear=y2,omonth=m2,oday=d2,ohour=h2)
		
				if y1 eq y2 then y2 = '' 
				if m1 eq m2 then m2 = 0
				if d1 eq d2 then d2 = '' 
	
				return,strcompress(strjoin([months[fix(m1)],d1,y1,h1+':00','~',months[fix(m2)],d2,y2,h2+':00'],' '))
			
				end				
				
			else : begin
			
				if ndata eq 1 then return,str(data,/tflag_to_timestamp,/snapshot) $
				              else return,str(data,/tflag_to_timestamp,/period)
			
				end	
		
		endcase

		end	
		
	keyword_set(month_to_name) : begin
	
		month_name = keyword_set(full) ? ['January','February','March','April','May','June','July','August','Septempber','Octobor','November','December'] $			
		             : ['Jan','Feb','Mar','Apr','May','Jun','Jul','Aug','Sep','Oct','Nov','Dec']
			
		return,month_name[fix(_data)-1]
	
		end		
		
	keyword_set(title) : begin
	
		case str(/up,data) of

			'O3'              : out = 'O!d3'
			'O3_A'            : out = 'O!d3'
			'PM2.5'           : out = 'PM!d2.5'
			'PM2P5'           : out = 'PM!d2.5'
			'PM10'            : out = 'PM!d10!n'						  
			'NO2'             : out = 'NO!d2'
			'NOX'             : out = 'NO!dx'
			'CO'              : out = 'CO'
			'HCHO'            : out = 'HCHO'
			'SO2'             : out = 'SO!d2'
			'NOY'             : out = 'NO!dy'
			'OX'              : out = 'O!dx'
			'CFRAC'           : out = 'Cloud Fraction'
			'WS'		  : out = 'Wind Speed'
			'WSPD10'          : out = 'Wind Speed !d10m'
			'TEMP2'           : out = 'Temperature !d2m'
			'TEMP1P5'         : out = 'Temperature !d1.5m'  
			'MICROGRAMS/M**3' : out = !d.name eq 'PS' ? '!9m!3g/m!u3' : '!4l!3g/m!u3'
			'UG/M3'           : out = !d.name eq 'PS' ? '!9m!3g/m!u3' : '!4l!3g/m!u3'

			else : out = str(data)

		endcase

		return,out+'!n!3'	
	  	
		end
			
	keyword_set(systime_to_tflag) : begin
	
		month_name = ['Jan','Feb','Mar','Apr','May','Jun','Jul','Aug','Sep','Oct','Nov','Dec']
	
		case ndata of
		
			1 : begin
			
				r = strsplit(systime(0,data[0]),' :',/extract)
				return,strjoin([str(r[6],digit=4),str(where(month_name eq r[1])+1,digit=2),$
				       str(r[2],digit=2),str(r[3],digit=2),str(r[4],digit=2),str(r[5],digit=2)])							
			
				end
			
			else : begin
			
				result = strarr(ndata)
				
				for idata=0L,ndata-1 do result[idata] = str(data[idata],/systime_to_tflag)
								
				return,result			
				end
		
		endcase						
		end												
				
	keyword_set(filetime_to_tflag) : begin
			
		data2 = data
		
		if is_array then result = make_array(size=size(data)) else result = ''
	
		find = ['Jan','Feb','Mar','Apr','May','Jun','Jul','Aug','Sep','Oct','Nov','Dec',$
		        'Jan.','Feb.','Mar.','Apr.','May.','Jun.','Jul.','Aug.','Sep.','Oct.','Nov.','Dec.']
		find = [find,strupcase(find)]
		replace = [str(indgen(12)+1,digit=2),str(indgen(12)+1,digit=2),str(indgen(12)+1,digit=2),str(indgen(12)+1,digit=2)]
		
		for ifind=0L,n_elements(find)-1 do data2 = str(data2,find=find[ifind],replace=replace[ifind])
		
		for idata=0L,ndata-1 do begin
		
			line = strsplit(data2[idata],' :',/extract)
	
			case n_elements(line) of
			
				3 : result[idata] = strjoin(line[[2,0,1]])+'0001'
				4 : result[idata] = date(date(/today),/cut,/year)+strjoin(line)
			  			
			endcase
	
		endfor
		
		return,result
				  	
		end		
		
	keyword_set(tflag_simple) : begin
	
		data = str(data,/tflag_full)
	
		x1 = (shift(date(data,1,/incr,/hour),1) eq data) + (shift(date(data,-1,/incr,/hour),-1) eq data)
		x2 = shift(x1,1) eq x1

		x = where(x1 eq 2,nx)		
		if nx gt 0 then data[x] = ' '
		x = where(x1 eq 2 and x2 eq 0,nx)
		if nx gt 0 then data[x] = '-'
		
		return,strjoin(strsplit(strjoin(data,' '),/extract),' ')

		end	
				
	keyword_set(tflag_full) : begin

		if strpos(str(/join,data),'-') eq -1 then return,data
	
		tflag  = strsplit(str(data,'-',' - ',/replace),' ,',/extract)
				
		x = where(tflag eq '-',nx)

		for ix=0L,nx-1 do if x[ix] gt 0 and x[ix] lt n_elements(tflag)-1 then tflag[x[ix]] = str(/join,date(tflag[x[ix]-1],tflag[x[ix]+1],/make,/hour))
					
		result = array(strsplit(str(str(/join,tflag),'-',/remove_word),/extract),/uniq)

		return,result
					
		end		
				
	keyword_set(separate) : begin
	
		if var_set(d01) then sep = d01 else sep = '%'
		
		b = strsplit(data,sep,/extract,/preserve_null)		
		r = b[0]
		
		for i=1L,n_elements(b)-1 do r = [r,sep+strmid(b[i],0,1),strmid(b[i],1)]
				
		x = where(r ne '',nx)
		
		if ~keyword_set(preserve_null) and nx gt 0 then r = r[x]
		
		return,r
			
		end		
				
	keyword_set(fill_space) : begin
			
		if var_set(d01) then fill = d01 else fill = '_'
		
		data_temp = byte(data)
		
		x = where(data_temp eq 32B,nx)
				
		if nx gt 0 then data_temp[x] = byte(fill)
		
		return,string(data_temp)
	
		end		
		
	keyword_set(to_number) : begin
	
		if ~var_set(missing) then missing = -999.
		
		d = str(_data)
		r = make_array(size=size(_data),value=missing)
		
		x = where(d ne '',nx)
		
		if nx gt 0 then r[x] = float(d[x])
		
		return,r
	
		end
		
	keyword_set(safe) : begin
	
		aa = byte(data)
		bb = byte('~`!@#$%^&*()_-+={}|[]\:";<>?,./')		
		xx = idx(aa,bb,/contain,count=nxx)
		yy = idx(bb,aa[xx],/point)
		
		if nxx eq 0 then return,d1
		
		for ixx=0L,nxx-1 do data = strjoin(strsplit(data,'\'+string(bb[yy[ixx]]),/extract,/regex,/preserve_null),'$'+str(yy[ixx],digit=2))

		return,data
		end
		
	keyword_set(_restore) : begin
	
		aa = strsplit(data,'$',/extract,/preserve_null)
		bb = byte('~`!@#$%^&*()_-+={}|[]\:";<>?,./')		
		if n_elements(aa) eq 0 then return,data
				
		for i=0L,n_elements(aa)-1 do cc = var_set(cc) ? cc+string(bb[fix(strmid(aa[i],0,2))])+strmid(aa[i],2) : aa[0]
				
		return,cc
		end	
									
	else : return,strtrim(string(data),2)

endcase

end
