;-------------------------------------------------------------------------------
;	NAME
;		rd_text
;
;	PURPOSE
;		to read ASCII file
;
;	USAGE
;		r = rd_text(data|file[,trim=[1|2],comment='',/column][,oinfo=info])
;
;	INPUT
;		data|file : ASCII file name of string array
;		nheader   : lines of header.If set, returns header content in oinfo
;		comment   : remove comments
;		trim      : remove empty lines in head and tail, If trim=2, remove all empty lines	
;		delimiter : delimiter for column separation. default is null (' ')
;	
;	KEYWORD
;		trim   : remove empty lines
;		column : perform column separation
;
;	OUTOUT
;		oinfo : {nheader,header,ncol,nline}
;
;	AUTHORS
;		2010-12-16 Hyun Cheol Kim (hyun.kim@noaa.gov)
;		2011-08-18 added /structure
;-------------------------------------------------------------------------------

	function rd_text,_data,nheader=nheader,comment=comment,trim=trim,$
	         column=column,delimiter=delimiter,oinfo=info,quiet=quiet,$
	         overwrite=overwrite,structure=structure,should_include=_si
		 
;-------------------------------------------------------------------------------

data = keyword_set(overwrite) ? temporary(_data) : _data
info = {nheader:0L}

if var_set(nheader) then info = struct(/merge,info,{nheader:nheader})

	; read if data is a file

if n_elements(data) eq 1 and file_test(data[0]) then begin
	
	nline  = 1000L
	iline  = 0L
	read   = ''
	result = strarr(nline)
	
	if ~keyword_set(quiet) then message,/info,str(/join,'READ:',data[0])
	
	openr,lun,data[0],/get_lun
	
	while ~eof(lun) do begin
	
		readf,lun,read
		
		if iline ge nline then begin
		
			result = [result,strarr(nline)]
			nline  = nline*2
		
		endif
					
		result[iline] = read
		iline ++

	endwhile
	
	free_lun,lun
		
	if info.nheader gt iline-1 then data = '' else data = result[info.nheader:iline-1]	
	if info.nheader gt 0 then info = struct(/merge,info,{header:result[0:info.nheader-1]})

endif

	; remove comments
	
if var_set(comment) then begin

	pp = strpos(data,str(comment))
	xx = where(pp ne -1,nxx)	
	
	if nxx gt 0 then begin
		
		cc = strmid(data[xx],reform(pp[xx]+1,[1,nxx]))				
		data[xx] = strmid(data[xx],0,reform(pp[xx],[1,nxx]))		
		info = struct_merge(info,{comment:cc})
	
	endif

endif

	; trim
	
if keyword_set(trim) then begin

	case trim of
	
		1 : data = data_check(data,strlen(str(data)) ne 0,/range,if_not_available='')
		2 : data = data_check(data,strlen(str(data)) ne 0,if_not_available='')

	endcase

endif

	; caption
	
if keyword_set(structure) then begin

	if ~var_set(delimiter) then delimiter = ' '
	
	data = str(data)+ ' '
	pp   = strpos(data,delimiter)
	cc   = strmid(data,reform(pp+1,[1,n_elements(data)]))	
	iaa  = idx(strmid(data,0,reform(pp,[1,n_elements(data)])),/self,oself=aa)
	aa   = data_check(aa,str(aa) eq '','NONAME')
			
	for i=0L,n_elements(aa)-1 do rr = var_set(rr) ? struct(/merge,rr,struct(/make,str(aa[i]),cc[where(iaa eq i)])) : struct(/make,str(aa[i]),cc[where(iaa eq i)])
			
	return,rr

endif	
	
	; column
	
if keyword_set(column) then begin

		; remove tab

	data2 = byte(data)
	
	xx = where(data2 eq 9B,nxx)	
	if nxx gt 0 then data2[xx] = 32B	
	
	data = string(data2)
	
		;
	
	ndata = n_elements(data)
	if ~var_set(delimiter) then delimiter = ' '

	for idata=0L,ndata-1 do begin
				
		data1  = strsplit(data[idata],delimiter,/extract,preserve_null=(delimiter ne ' '))				
		ndata1 = n_elements(data1)		
						
		if ~var_set(result2) then begin
		
			ncol = ndata1
			result2 = strarr(ncol,ndata)
			
		endif
		
		if ndata1 gt ncol then begin
		
			result3 = strarr(ndata1,ndata)
			result3[0:ncol-1,*] = result2
			result2 = result3
			ncol = ndata1
		
		endif	
			
		result2[0:ndata1-1,idata] = data1[*]
	
	endfor
	
	data = temporary(reform(result2,[ncol,ndata],/overwrite))
	info = struct(/merge,info,{ncol:ncol,nline:ndata})

endif else begin

	info = struct(/merge,info,{nline:n_elements(data)})

endelse

	;
	
if var_set(_si) then begin

	for isi=0L,n_elements(_si)-1 do rrsi = var_set(rrsi) ? rrsi*(max(strpos(data,_si[isi])) ne -1) : (max(strpos(data,_si[isi])) ne -1)	
	if ~rrsi then return,0B

endif	
	;

return,data

end
