;-------------------------------------------------------------------------------
;	NAME
;		RD_NO2_OMI_TEMIS
;
;	PURPOSE
;		TO READ OMI NO2 VCD FROM TEMIS
;
;	USAGE
;		rr = rd_no2_omi_temis(tflag[,domain=,minfo=][,/show][,/renew])
;
;	INPUT
;		tflag : YYYYMM, YYYYMMDD
;		domain,minfo : domain info
;
;	KEYWORD
;		show : show and save plots
;		renew : regenerate data
;		grd : use GRID data
;
;	AUTHOR
;		20141015 Hyun Cheol Kim (hyun.kim@noaa.gov)	
;-------------------------------------------------------------------------------

	function rd_no2_omi_temis_grid_mon1,mon,domain=domain,minfo=minfo,renew=renew,show=show,_extra=_ex,finfo=finfo

;-------------------------------------------------------------------------------

date_check,mon,/monthly
if struct_set(_ex,'limit') then if mon lt min(_ex.limit) or mon gt max(_ex.limit) then return,0b

odir = path_check(getenv('HOME'),'data','temis','omi','no2','grid_mon',/make)
file = dn_no2_omi_temis_grid_mon(mon)
if ~keyword_set(file) then return,0b
	
minfo = aqm_grid(domain=domain,minfo=minfo,operi=peri,omid=mid)
save = file_name(file,ext=mid,/sdir)

if file_renew(save,renew=renew) then begin

	dd = rd_text(file,nheader=4)
	xx = [where(strpos(dd,'lat=') eq 0),n_elements(dd)]

	nx = 2880
	ny = 1440
	rr = fltarr(nx,ny)

	for ixx=0L,n_elements(xx)-2 do begin

		line = strjoin(dd[xx[ixx]+1:xx[ixx+1]-1])
		line = reform(byte(line),[4,nx])
		line = string(line)

		rr[*,ixx] = float(line)

	endfor

	rr = data_check(rr,rr eq -999,!values.f_nan)/100.
	_lon = findgen(nx)*0.125+0.125/2.-180.
	_lat = findgen(ny)*0.125+0.125/2.-90.

	x0 = min(where(_lon ge min(peri[0,*]) and _lon le max(peri[0,*])),max=x1)
	y0 = min(where(_lat ge min(peri[1,*]) and _lat le max(peri[1,*])),max=y1)

	rr = rr[x0:x1,y0:y1]
	_lon = ary(_lon[x0:x1],ny=(y1-y0+1))
	_lat = ary(_lat[y0:y1],nx=(x1-x0+1))

	tlon = ary(_lon,/tile)
	tlat = ary(_lat,/tile)

	rr = aqm_regrd(tlon,tlat,rr,/avg,minfo=minfo,finfo=finfo)
	rr = data_check(rr,rr eq -999,!values.f_nan)
	
	message,/info,str(/join,'SAVE',save)
	save,filename=save,rr			

endif else begin

	message,/info,str(/join,'READ',save)
	restore,save
		
endelse

if keyword_set(show) then begin	

	odir  = path_check('plot','no2-omi-temis-grid',mid,/make)
	title = str(/join,'NO2!dOMI/temis!n',date(mon,'%b %Y',/format),'13:30 LT ')	
	png   = struct_read(_ex,'png',path_check(odir,str(/join,del='.','no2','omi-temis-grid',mon)))
	unit  = '10!u15!n#/cm!u2!n'
	range = struct_read(_ex,'range',[0,10])
	winn  = struct_read(_ex,'winn',2)
	note  = {type:'note',title:title,unit:unit}

	mir_sp,minfo=minfo,rr,range=range,winn=winn,png=png,_extra=_ex,map=map,quality=2,include=list(note)
	
endif

return,rr
end
;-------------------------------------------------------------------------------

	function rd_no2_omi_temis_grid_mon,mon,domain=domain,minfo=minfo,renew=renew,show=show,_extra=_ex

;-------------------------------------------------------------------------------

for imon=0L,(nmon=n_elements(mon))-1 do begin

	rr1 = rd_no2_omi_temis_grid_mon1(mon[imon],domain=domain,minfo=minfo,renew=renew,show=show,_extra=_ex)
	
	if ~keyword_set(rr1) then continue	
	if ~var_set(rr) then rr = make_array([size(rr1,/dim),nmon],value=!values.f_nan)

	rr[*,*,imon] = rr1

endfor

return,rr
end
;-------------------------------------------------------------------------------

	function rd_no2_omi_temis_file1,file,oinfo=oinfo

;-------------------------------------------------------------------------------

if file_test(file[0]) then message,/info,str(/join,'READ',file[0])

nan = !values.f_nan
res = h5_parse(file,/read_data)
res = res.HDFEOS.SWATHS.DOMINONO2

vcdtrop = res.data_fields.troposphericverticalcolumn._data * res.data_fields.troposphericverticalcolumn.scalefactor._data[0]
a_lev   = res.DATA_FIELDS.TM4PRESSURELEVELA._data
b_lev   = res.DATA_FIELDS.TM4PRESSURELEVELB._data
ndata   = n_elements(vcdtrop)
nlev    = n_elements(a_lev)

rr   = replicate(struct_make(['date','time'],'',['lon','lat','vcdtrop','fltrop','clfrac','psurf','amf','amftrop'],nan,['loncorn','latcorn'],fltarr(4),['kernel'],replicate(nan,nlev)),size(vcdtrop,/dim))

rr.vcdtrop = vcdtrop/1e15
rr.amf     = res.data_fields.airmassfactor._data * res.data_fields.airmassfactor.scalefactor._data[0]
rr.amftrop = res.data_fields.airmassfactortropospheric._data * res.data_fields.airmassfactortropospheric.scalefactor._data[0]
rr.fltrop  = res.data_fields.troposphericcolumnflag._data
rr.lat     = res.geolocation_fields.latitude._data
rr.lon     = res.geolocation_fields.longitude._data
rr.clfrac  = res.data_fields.CloudFraction._data * res.data_fields.CloudFraction.scalefactor._data[0]
rr.psurf   = res.data_fields.tm4surfacepressure._data * res.data_fields.tm4surfacepressure.scalefactor._data[0]*100.

rr.latcorn = transpose(res.geolocation_fields.latitudecornerpoints._data,[2,0,1])
rr.loncorn = transpose(res.geolocation_fields.longitudecornerpoints._data,[2,0,1])
rr.kernel  = transpose(res.data_fields.averagingkernel._data * res.data_fields.averagingkernel.scalefactor._data[0],[2,0,1])

rr = data_check(rr,rr.fltrop eq 0)

oinfo = {a_lev:a_lev,b_lev:b_lev}

return,rr
end
;-------------------------------------------------------------------------------

	function rd_no2_omi_temis_file,cday,renew=renew,lon360=lon360,oinfo=oinfo,_extra=_ex

;-------------------------------------------------------------------------------

sdir = path_check(getenv('HOME'),'data','temis','omi','no2','daily',/make)
save = path_check(sdir,str(/join,del='.','no2vcd','omi','temis',cday,'sav'))

if file_renew(save,renew=renew) then begin

	file = dn_no2_omi_temis(cday)
	if ~keyword_set(file) then return,!null
		
	for ifile=0L,n_elements(file)-1 do begin
	
		rr1 = rd_no2_omi_temis_file1(file[ifile],oinfo=oinfo)
		if ~keyword_set(rr1) then continue
	
		rr = var_set(rr) ? [rr,rr1] : rr1
	
	endfor
	
	if ~var_set(rr) then return,!null
	
	message,/info,str(/join,'SAVE',save)
	save,filename=save,rr,oinfo

endif else begin

	message,/info,str(/join,'READ',save)
	restore,save

endelse

if keyword_set(lon360) then begin
		
	xx = where(rr.lon lt 0.,nxx)
	if nxx gt 0 then rr[xx].lon = (rr[xx].lon+360.) mod 360.				
	if nxx gt 0 then rr[xx].loncorn = (rr[xx].loncorn+360.) mod 360.										
					
endif else begin

	xx = where(rr.lon gt 180.,nxx)					
	if nxx gt 0 then rr[xx].lon = ((rr[xx].lon+180.) mod 360.)-180.
	if nxx gt 0 then rr[xx].loncorn = ((rr[xx].loncorn+180.) mod 360.)-180.	
	
endelse

return,rr
end
;-------------------------------------------------------------------------------

	function rd_no2_omi_temis_d1,cday,opt,domain=domain,minfo=minfo,$
	         renew=renew,show=show,_extra=_ex

;-------------------------------------------------------------------------------

if strlen(str(cday)) ne 8 then message,'CDAY=YYYYMMDD'
if str(cday) lt '20041001' then return,0B
if str(cday) gt date(/today) then return,0B

minfo = aqm_grid(domain=domain,minfo=minfo,omid=mid)
ds    = struct_set(_ex,'downscale')
sdir  = path_check(getenv('HOME'),'data','temis','omi','no2',opt.ver,'l2','daily',/make)
type  = str(/join,del='_',str(fix(opt.max_size)),str(opt.max_cloud,format='(F4.2)')) + (keyword_set(ds) ? '_ds' : '')
save  = path_check(sdir,str(/join,del='.','no2','temis','omi',mid,type,cday))

if file_renew(save,renew=renew) then begin

	if ~keyword_set((rr=rd_no2_omi_temis_file(cday,renew=flag_dn(renew)))) then return,!null		
	if ~keyword_set((rr=data_check(rr,rr.clfrac lt opt.max_cloud,ina=0b))) then return,!null

	limit = aqm_grid_limit(domain=domain,minfo=minfo)
	limit = [min(limit[0,*]),min(limit[1,*]),max(limit[0,*]),max(limit[1,*])]
	
	if ~keyword_set((rr=data_check(rr,rr.lon ge limit[0]-1 and rr.lon le limit[2]+1 and rr.lat ge limit[1]-1 and rr.lat lt limit[3]+1,ina=0b))) then return,!null
				
	pixel_size = map_3points((tlon=(rr.loncorn)[[0,1,3,2,0],*]),(rr.latcorn)[[0,1,3,2,0],*])*2./1e6	
	if struct_set(_ex,'pixel_size') then return,pixel_size
		
	if ~keyword_set((rr=data_check(rr,pixel_size le opt.max_size and max(tlon,dim=1,min=tlon_mn)-tlon_mn lt 180.,ina=0b))) then return,!null
	
	rr = keyword_set(ds) $
	     ? aqm_regrd((rr.loncorn)[[0,1,3,2,0],*],(rr.latcorn)[[0,1,3,2,0],*],rr.vcdtrop,domain=domain,minfo=minfo,weight=_ex.downscale) $
	     : aqm_regrd((rr.loncorn)[[0,1,3,2,0],*],(rr.latcorn)[[0,1,3,2,0],*],rr.vcdtrop,domain=domain,minfo=minfo,/avg)
	
	message,/info,str(/join,'SAVE',save)
	save,filename=save,rr,minfo,mid,cday

endif else begin

	message,/info,str(/join,'READ',save)
	restore,save

endelse

if keyword_set(show) then begin	

	odir  = path_check('plot','no2-omi-temis',mid,/make)
	title = str(/join,'NO2!dOMI/temis!n',cday,date(cday,/weekday,/name,/short),'13:30 LT',keyword_set(ds) ? 'DS' : '')
	png   = struct_read(_ex,'png',path_check(odir,str(/join,'no2','omi-temis',mid,type,cday,del='.')))
	unit  = '10!u15!n#/cm!u2!n'
	range = struct_read(_ex,'range',[0,20])	
	note  = {type:'note',title:title,unit:unit}
	
	mir_sp,minfo=minfo,rr,range=range,map=map,png=png,winn=winn,_extra=_ex,include=list(note),quality=2
	
endif
		
return,rr
end
;-------------------------------------------------------------------------------

	function rd_no2_omi_temis_d,day,opt,domain=domain,minfo=minfo,renew=renew,_extra=_ex

;-------------------------------------------------------------------------------

for iday=0L,(nday=n_elements(day))-1 do begin
	
	rr1 = rd_no2_omi_temis_d1(day[iday],opt,domain=domain,minfo=minfo,renew=renew,_extra=_ex)
	
	if ~keyword_set(rr1) then continue	
	if ~var_set(rr) then rr = make_array([size(rr1,/dim),nday],value=!values.f_nan)

	rr[*,*,iday] = rr1

endfor

return,var_set(rr) ? rr : 0b
end
;-------------------------------------------------------------------------------

	function rd_no2_omi_temis_m1,mon,opt,domain=domain,minfo=minfo,renew=renew,show=show,_extra=_ex

;-------------------------------------------------------------------------------

date_check,mon,/monthly
if str(mon) lt '200410' then return,0B
if struct_set(_ex,'limit') then if mon lt min(_ex.limit) or mon gt max(_ex.limit) then return,0b

minfo = aqm_grid(domain=domain,minfo=minfo,omid=mid)
type  = str(/join,del='_',str(fix(opt.max_size)),str(opt.max_cloud,format='(F4.2)'))
sdir  = path_check(getenv('HOME'),'data','temis','omi','no2',opt.ver,'l2','monthly',/make)
save  = path_check(sdir,str(/join,del='.','no2','temis','omi',mid,type,mon))

if file_renew(save,renew=renew) then begin
	
	rr = rd_no2_omi_temis_d(date(mon,/daily),opt,domain=domain,minfo=minfo,renew=flag_dn(renew),show=flag_dn(show),_extra=_ex)
	rr = keyword_set(rr) ? arys(rr,3,/mean) : 0b
	
	message,/info,str(/join,'SAVE',save)
	save,filename=save,rr
	
endif else begin

	message,/info,str(/join,'READ',save)
	restore,save

endelse	

if keyword_set(show) then begin	

	odir  = path_check('plot','no2-omi-temis',mid,/make)
	title = str(/join,'NO2!dOMI/temis!n',date(mon,'%b %Y',/format),'13:30 LT')
	png   = struct_read(_ex,'png',path_check(odir,str(/join,'no2','omi-temis',mid,type,mon,del='.')))
	unit  = '10!u15!n#/cm!u2!n'
	range = struct_read(_ex,'range',[0,10])	
	note  = {type:'note',title:title,unit:unit}
	
	mir_sp,minfo=minfo,rr,range=range,map=map,png=png,winn=winn,_extra=_ex,include=list(note),quality=2
	
endif
				
return,rr
end
;-------------------------------------------------------------------------------

	function rd_no2_omi_temis_m,mon,opt,domain=domain,minfo=minfo,renew=renew,show=show,_extra=_ex

;-------------------------------------------------------------------------------

for imon=0L,(nmon=n_elements(mon))-1 do begin

	if mon[imon] ge date(date(date(/today),-3,/incr),/cut,/year,/month) then continue

	rr1 = rd_no2_omi_temis_m1(mon[imon],opt,domain=domain,minfo=minfo,renew=renew,show=show,_extra=_ex)
	
	if ~keyword_set(rr1) then continue		
	if ~var_set(rr) then rr = make_array([size(rr1,/dim),nmon],value=!values.f_nan)
			
	rr[*,*,imon] = rr1

endfor

return,var_set(rr) ? rr : 0b
end
;-------------------------------------------------------------------------------

	function rd_no2_omi_temis,tflag,domain=domain,minfo=minfo,$
	         renew=renew,show=show,_extra=_ex
	
;-------------------------------------------------------------------------------

if min(strlen(str(tflag)),max=mx) ne mx then message,'TIME Flag inconsistent'

opt = {max_size:struct_read(_ex,'max_size',1000.),max_cloud:struct_read(_ex,'max_cloud',0.4),ver:struct_read(_ex,'ver','v2')}

case 1 of

	mx eq 6 and struct_set(_ex,'grd') : return,rd_no2_omi_temis_grid_mon(tflag,domain=domain,minfo=minfo,renew=renew,show=show,_extra=_ex)
	mx eq 6 : return,rd_no2_omi_temis_m(tflag,opt,domain=domain,minfo=minfo,renew=renew,show=show,_extra=_ex)		
	mx eq 8 : return,rd_no2_omi_temis_d(tflag,opt,domain=domain,minfo=minfo,renew=renew,show=show,_extra=_ex)				
	else : message,'WRONG TIME FLAG: Use YYYYMM or YYYYMMDD'

endcase

end
