;-------------------------------------------------------------------------------
;	NAME
;		rd_no2_gome2_temis
;
;	PURPOSE
;		to TEMIS GOME2 hdf files
;
;	USAGE
;		rr = rd_no2_gome2_temis_file(file[,nlev][,/lon360][,oinfo=oi])
;
;	INPUT
;		file : TEMIS GOME2 hdf file
;		nlev : no. of level (data-specific)
;
;	KEYWORD
;		lon360 : covert longitude to 0-360
;
;	OUTPUT
;		oinfo : returns additional information
;
;	AUTHOR
;		2015-01-15 Hyun Cheol Kim (hyun.kim@noaa.gov)
;-------------------------------------------------------------------------------

	function rd_no2_gome2_temis_file,file,_nlev,lon360=lon360,oinfo=oinfo

;-------------------------------------------------------------------------------

if ~keyword_set(file) then return,!null
if ~file_test(file) then return,!null
message,/info,str(/join,'READ',file)	

	;
		
file_id = hdf_sd_start(file[0],/read)
hdf_sd_fileinfo,file_id,datasets,attributes
hdf_sd_end,file_id
	
file_id = hdf_open(file,/read)
nvds = n_elements(hdf_vd_lone(file_id))
nlev = var_set(_nlev,ina=34)
rr   = replicate(struct_make(['date','time'],'',['lon','lat','vcdtrop','fltrop','clfrac','psurf','amf','amftrop'],!values.f_nan,['loncorn','latcorn'],fltarr(4),['kernel'],replicate(!values.f_nan,nlev)),30000)
tag  = tag_names(rr)
nn   = 0L

for i=0L,attributes+nvds-1 do begin

	vd_id = hdf_vd_getid(file_id,var_set(vd_id) ? vd_id : -1)		
	vd_handle = hdf_vd_attach(file_id,vd_id,/read)	
	hdf_vd_get,vd_handle,nfields=nfields,name=vd_name,count=count,fields=fields	
	info = [(var_set(info) ? info : !null),vd_name+': '+fields]
	
	if strmid(vd_name,0,4) eq 'pres' then begin
	
		nread = hdf_vd_read(vd_handle,a_lev,fields='a_lev')
		nread = hdf_vd_read(vd_handle,b_lev,fields='b_lev')
	
	endif
						
	if strmid(vd_name,0,4) eq 'NO2_' then begin

		name = ['date','time','lon','lat','vcdtrop','fltrop','psurf']		
		for ii=0L,n_elements(name)-1 do begin
		
			nread = hdf_vd_read(vd_handle,data,fields=name[ii])										
			if nn+nread ge n_elements(rr) then rr = [rr,rr]						
			rr[nn:nn+nread-1].(idx(tag,str(/up,name[ii]),/point)) = data
		
		endfor
				
		name = ['kernel']
		for ii=0L,n_elements(name)-1 do begin
		
			nread = hdf_vd_read(vd_handle,data,fields=name[ii])										
			if nn+nread ge n_elements(rr) then rr = [rr,rr]				
			rr[nn:nn+nread-1].(idx(tag,str(/up,name[ii]),/point)) = data
		
		endfor
						
	endif
		
	if strmid(vd_name,0,4) eq 'GEO_' then begin
	
		name = ['loncorn','latcorn']
		for ii=0L,n_elements(name)-1 do begin
		
			nread = hdf_vd_read(vd_handle,data,fields=name[ii])				
			rr[nn:nn+nread-1].(idx(tag,str(/up,name[ii]),/point)) = data
		
		endfor
		
	endif
	
	if strmid(vd_name,0,4) eq 'ANC_' then begin
	
		name = ['clfrac','amf','amftrop']
				
		for ii=0L,n_elements(name)-1 do begin
		
			nread = hdf_vd_read(vd_handle,data,fields=name[ii])	
			rr[nn:nn+nread-1].(idx(tag,str(/up,name[ii]),/point)) = data
		
		endfor
		
		nn += nread
			
	endif
						
	hdf_vd_detach,vd_handle

endfor

hdf_close,file_id

	;

rr = rr[0:nn-1]
rr.vcdtrop = data_check(rr.vcdtrop,rr.vcdtrop eq -99.99,!values.f_nan)

if keyword_set(lon360) then begin
		
	xx = where(rr.lon lt 0.,nxx)
	if nxx gt 0 then rr[xx].lon = (rr[xx].lon+360.) mod 360.				
	if nxx gt 0 then rr[xx].loncorn = (rr[xx].loncorn+360.) mod 360.										
					
endif else begin

	xx = where(rr.lon gt 180.,nxx)					
	if nxx gt 0 then rr[xx].lon = ((rr[xx].lon+180.) mod 360.)-180.
	if nxx gt 0 then rr[xx].loncorn = ((rr[xx].loncorn+180.) mod 360.)-180.	
	
endelse

oinfo = {a_lev:a_lev,b_lev:b_lev}

return,rr
end
