;-------------------------------------------------------------------------------
;	NAME
;		RD_NO2_CAMQ
;
;	PURPOSE
;		TO READ NO2 VCD from CMAQ runs
;
;	USAGE
;		rr = rd_no2_cmaq(time,platform[,file1,file2][,domain=,minfo][,name=][,/nokernel][,/show,/renew])
;
;	INPUT
;		time : YYYYMM, YYYYMMDD
;		domain,minfo : domain info
;
;	KEYWORD
;		show : show and save plots
;		renew : regenerate data
;
;	AUTHOR
;		20151208 Hyun Cheol Kim (hyun.kim@noaa.gov)	
;-------------------------------------------------------------------------------
		 
	function rd_no2_cmaq_d1,cday,platform,file1,file2,$
	         domain=domain,minfo=minfo,name=name,$
	         nokernel=nokernel,show=show,renew=renew,_extra=_ex
		
;-------------------------------------------------------------------------------

date_check,cday,/daily

case str(/up,var_set(platform,ina='')) of

	'GOME'   : opt = {lhour:[10,11],time_stamp:'1030LT'}
	'GOME2'  : opt = {lhour:[9,10] ,time_stamp:'0930LT'}
	'GOME2A' : opt = {lhour:[9,10] ,time_stamp:'0930LT'}
	'GOME2B' : opt = {lhour:[9,10] ,time_stamp:'0930LT'}	
	'OMI'    : opt = {lhour:[13,14],time_stamp:'1330LT'}
	'SCIA'   : opt = {lhour:[10]   ,time_stamp:'1000LT'}
	
	else  : message,'### USAGE CHECK ### PlATFORM="OMI"/"GOME"/"SCIA"/"GOME2A"/"GOME2B"'

endcase

if ~var_set(name) and var_set(file1) then name = path_check(/uniq,file1)
if ~var_set(domain,minfo,file1,/if_any) then message,'Error'
kn    = keyword_set(nokernel) ? 'nK' : 'AK'
minfo = aqm_grid(file=file1,domain=domain,minfo=minfo,ominfo=minfo,olon=lon,omid=mid)
sdir  = path_check(struct_read(_ex,'sdir','save'),'no2vcd-cmaq',mid+'__'+name,/make)
save1 = path_check(sdir,file_save('no2vcd-cmaq',mid,opt.time_stamp,cday))
save2 = path_check(sdir,file_save('no2vcd-cmaq',mid,opt.time_stamp,kn,cday))

if file_renew(save2,renew=reenw) then begin

	if file_renew(save1,renew=renew) then begin
	
		if ~var_set(file1,file2) then message,'### USAGE CHECK ### FILE1 or FILE2 not specified'

		m   = 28.97e-3
		A   = 6.022e23
		cm2 = 1e-4
	
		ltflag = str(cday)+str(opt.lhour,digit=2)
		tflag  = aqm_lt2utc(ltflag,minfo=minfo)
						
		cc = rd_m3(file1,var='NO2',tflag=tflag,unit='fraction')
		if ~keyword_set(cc) then return,!null	
		dd = rd_m3(file2,var='DENS',tflag=tflag,unit='kg/m3')
		if ~keyword_set(dd) then return,!null	
		
		hh = rd_m3(file2,var='ZF',tflag=tflag,unit='m')	
		hh = [[[hh[*,*,0,*]]],[[(hh-shift(hh,[0,0,1,0]))[*,*,1:*,*]]]]

		nn = arys(cc*dd*hh/m*A*cm2*1e-15,tflag,lon,ltflag,/local_time)		
		pp = arys(rd_m3(file2,var='PRES',tflag=tflag),tflag,lon,ltflag,/local_time)
		
		nn =  n_elements(opt.lhour) gt 1 ? arys(nn,4,/mean) : reform(nn)
		pp =  n_elements(opt.lhour) gt 1 ? arys(pp,4,/mean) : reform(pp)
		
		message,/info,str(/join,'SAVE',save1)
		save,filename=save1,nn,pp
	
	endif else begin

		message,/info,str(/join,'READ',save1)
		restore,save1

	endelse	

		; Averaging Kernel

	if ~keyword_set(nokernel) then begin
	
		case str(/up,platform) of
	
			'GOME'   : message,'NO GOME AK yet'
			'GOME2A' : ak = rd_no2_gome2a_temis_ak(cday,domain=domain,minfo=minfo)
			'GOME2B' : ak = rd_no2_gome2b_temis_ak(cday,domain=domain,minfo=minfo)				
			'OMI'    : ak = rd_no2_omi_temis_ak(cday,domain=domain,minfo=minfo)		
			'SCIA'   : message,'NO SCIA AK yet'
	
		endcase
	
		if ~var_set(ak) then return,!null
	
		dim1 = size(ak.kernel,/dim)
		dim2 = size(nn,/dim)
		
		pp1 = ary(ak.psurf,ny=dim1[2])*ary(ak.info.b_lev,nx=product(dim1[0:1]))+ary(ak.info.a_lev,nx=product(dim1[0:1]))
		pp1 = reform(pp1,dim1,/overwrite)
		kn2 = fltarr(dim2)

		for ix=0L,dim2[0]-1 do begin
		for iy=0L,dim2[1]-1 do begin

			if ak.psurf[ix,iy] eq -999 then continue
			
			kn2[ix,iy,*] = (ak.kernel[ix,iy,*])[(dim1[2]-1)<round(interpol(findgen(dim1[2]),alog10(pp1[ix,iy,*]),alog10(pp[ix,iy,*])))>0]

		endfor
		endfor		
			
		nn = total(kn2*nn,3)*ak.amf/ak.amftrop
		nn = data_check(nn,ak.psurf eq -999,!values.f_nan)
		
	endif else begin

		nn = total(nn,3)

	endelse

	message,/info,str(/join,'SAVE',save2)
	save,filename=save2,nn

endif else begin

	message,/info,str(/join,'READ',save2)
	restore,save2

endelse

if keyword_set(show) then begin

	title = str(/join,'NO2!dCMAQ!n',cday,date(cday,/weekday,/name,/short),opt.time_stamp,kn)
	unit  = '10!u15!n#/cm!u2!n'	
	odir  = struct_read(_ex,'odir',path_check('plot','no2-cmaq',mid,/make))
	png   = path_check(sdir,str(/join,del='.','no2','cmaq',mid,opt.time_stamp,kn,cday))
	range = struct_read(_ex,'range',[0,20])
	note = {type:'note',title:title,unit:unit}
		
	mir_sp,nn,minfo=minfo,range=range,map=map,png=png,include=list(note),_extra=_ex,quality=2

endif

return,nn
end
;-------------------------------------------------------------------------------
		 
	function rd_no2_cmaq_d,cday,platform,file1,file2,$
	         domain=domain,minfo=minfo,name=name,$
	         nokernel=nokernel,show=show,renew=renew,_extra=_ex
		
;-------------------------------------------------------------------------------

for iday=0L,(nday=n_elements(cday))-1 do begin

	rr1 = rd_no2_cmaq_d1(cday[iday],platform,file1,file2,domain=domain,minfo=minfo,name=name,nokernel=nokernel,show=show,renew=renew,_extra=_ex)
	
	if ~keyword_set(rr1) then continue	
	if ~var_set(rr) then rr = make_array([size(rr1,/dim),nday],value=!values.f_nan)

	rr[*,*,iday] = rr1

endfor

return,var_set(rr) ? rr : 0b
end

;-------------------------------------------------------------------------------
	function rd_no2_cmaq_m1,mon,platform,file1,file2,$
	         domain=domain,minfo=minfo,name=name,$
	         nokernel=nokernel,show=show,renew=renew,_extra=_ex
;-------------------------------------------------------------------------------

date_check,mon,/monthly
if struct_set(_ex,'limit') then if mon lt min(_ex.limit) or mon gt max(_ex.limit) then return,0b

case str(/up,var_set(platform,ina='')) of

	'GOME'   : opt = {lhour:[10,11],time_stamp:'1030LT'}
	'GOME2'  : opt = {lhour:[9,10] ,time_stamp:'0930LT'}
	'GOME2A' : opt = {lhour:[9,10] ,time_stamp:'0930LT'}
	'GOME2B' : opt = {lhour:[9,10] ,time_stamp:'0930LT'}	
	'OMI'    : opt = {lhour:[13,14],time_stamp:'1330LT'}
	'SCIA'   : opt = {lhour:[10]   ,time_stamp:'1000LT'}
	
	else  : message,'### USAGE CHECK ### PlATFORM="OMI"/"GOME"/"SCIA"/"GOME2A"/"GOME2B"'

endcase

kn    = keyword_set(nokernel) ? 'nK' : 'AK'
minfo = aqm_grid(file=file1,domain=domain,minfo=minfo,ominfo=minfo,olon=lon,omid=mid)
sdir  = path_check(struct_read(_ex,'sdir','save'),'no2vcd-cmaq',mid+'__'+name,/make)
;save1 = path_check(sdir,file_save('no2vcd-cmaq',mid,opt.time_stamp,cday))
save = path_check(sdir,file_save('no2vcd-cmaq',mid,opt.time_stamp,kn,mon))

if file_renew(save,renew=renew) then begin
	
	rr = rd_no2_cmaq_d(date(mon,/daily),platform,file1,file2,domain=domain,minfo=minfo,name=name,nokernel=nokernel,show=flag_dn(show),renew=flag_dn(renew),_extra=_ex)	
	rr = keyword_set(rr) ? arys(rr,3,/mean) : 0b
	
	message,/info,str(/join,'SAVE',save)
	save,filename=save,rr
	
endif else begin

	message,/info,str(/join,'READ',save)
	restore,save

endelse

if keyword_set(show) then begin

	title = str(/join,'NO2!dCMAQ!n',date(mon,'%b %Y',/format),opt.time_stamp,kn)
	unit  = '10!u15!n#/cm!u2!n'	
	odir  = struct_read(_ex,'odir',path_check('plot','no2-cmaq',mid,/make))
	png   = path_check(sdir,str(/join,del='.','no2','cmaq',mid,opt.time_stamp,kn,mon))
	range = struct_read(_ex,'range',[0,20])
	note = {type:'note',title:title,unit:unit}
		
	mir_sp,rr,minfo=minfo,range=range,map=map,png=png,include=list(note),_extra=_ex,quality=2

endif

return,var_set(rr) ? rr : !null
end
;-------------------------------------------------------------------------------
	function rd_no2_cmaq_m,mon,platform,file1,file2,$
	         domain=domain,minfo=minfo,name=name,$
	         nokernel=nokernel,show=show,renew=renew,_extra=_ex		 
;-------------------------------------------------------------------------------

for imon=0L,(nmon=n_elements(mon))-1 do begin

	rr1 = rd_no2_cmaq_m1(mon[imon],platform,file1,file2,domain=domain,minfo=minfo,name=name,nokernel=nokernel,show=show,renew=renew,_extra=_ex)	

	if ~keyword_set(rr1) then continue
	if ~var_set(rr) then rr = ary_make([size(rr1,/dim),nmon])

	rr[*,*,imon] = rr1

endfor

return,var_set(rr) ? rr : !null
end
;-------------------------------------------------------------------------------

	function rd_no2_cmaq,tflag,platform,file1,file2,$
	         domain=domain,minfo=minfo,name=name,$
	         nokernel=nokernel,show=show,renew=renew,_extra=_ex
				
;-------------------------------------------------------------------------------

date_check,tflag,olen=len

case 1 of

	len eq 6 : return,rd_no2_cmaq_m(tflag,platform,file1,file2,domain=domain,minfo=minfo,name=name,nokernel=nokernel,show=show,renew=renew,_extra=_ex)						
	len eq 8 : return,rd_no2_cmaq_d(tflag,platform,file1,file2,domain=domain,minfo=minfo,name=name,nokernel=nokernel,show=show,renew=renew,_extra=_ex)					
	
	else : message,'WRONG TIME FLAG: Use YYYYMM or YYYYMMDD'

endcase

end
