;-------------------------------------------------------------------------------
;	NAME
;		rd_m3
;
;	PURPOSE
;		To read M3IOAPI format data
;	
;	USAGE
;		rr = rd_m3(file,var=var[,lon=,lat=,layer=,tflag=,alt=][,unit=][,cut=][oinfo=,otflag=,ominfo])
;		
;		rr = rd_m3(file,var='O3',unit='ppb',tflag=[2014010100,2014010200])
;		rr = rd_m3(file,var=var,oinfo=rri,ominfo=minfo,otflag=tflag)  ; help,rri
;		rr = rd_m3(file,var=var,lon=lon,lat=lat)
;		rr = rd_m3(file,var=var,lon=[-100,-90],lat=[30,31],/track)
;		rr = rd_m3(file,var=var,lon=lon,lat=lat,/curtain)
;		rr = rd_m3(file,var=var,lon=lon,lat=lat,alt=alt,tflag=tflag,/flight) ; alt in m & tlfag in string/julday
;		rr = rd_m3(file,var=var,lon=lon,lat=lat,alt=alt,tflag=tflag,/flight,/interpol)
;		rr = rd_m3(file,var=var,lon=lon,lat=lat,layer=layer,tflag=tflag,/flight) ; layer is factions of vertical indices
;		rr = rd_m3(file,var=var,lon=lon,lat=lat,layer=layer,tflag=tflag,/flight,/interpol)
;
;	INPUT
;		file    : M3 file list
;		var     : variable name to read (see ii_m3_vlist for available list)
;		lon,lat : longitude, latitude to read
;		layer   : layer index to read. starts from 0.
;		alt     : altitude in m (used with /flight)
;		cut     : time steps to discard (cut=6, discard the first 6 hours) or to use (cut=[10,20], use only 10-20 time steps)
;		unit    : unit to read (see cv_unit)
;
;	KEYWORD 
;		lon/lat : read points at given lon/lat
;		track/lon/lat : read tracks with lon/lat
;		curtain/lon/lat : read vertical curtains (faster than tack for whole column)
;
;	NOTE
;		Generate a 'vlist' file to add new variable definition
;		
;
;	AUTHOR
;		2014-08-29 Hyun Cheol Kim (hyun.kim@noaa.gov) rewritten
;		2015-03-12 
;-------------------------------------------------------------------------------

	function ii_m3_vlist1,_dd

;-------------------------------------------------------------------------------

dd = byte(str(rd_text(_dd,trim=2,comment='#',/quiet)))
dd = [string(data_check(dd,dd eq 9B,32B))]
rr = replicate(struct_make(['name','list'],''),n_elements(dd))
rr.name = str(strmid(dd,0,transpose((xx=strpos(dd,' ')))))
rr.list = str(strmid(dd,transpose(xx)))

return,rr
end
;-------------------------------------------------------------------------------

	function ii_m3_vlist

;-------------------------------------------------------------------------------

file  = 'vlist'
vlist = file_test(file) ? list(ii_m3_vlist1(file)) : list()

vlist.add,ii_m3_vlist1($	      
	['PM2P5     ASO4IJ ANO3IJ ANH4IJ ANAJ ACLIJ A25J AORGPAIJ AECIJ AORGAJ AORGBJ AORGCJ',$ 
	 'PM10      PM2P5 ACORS ASOIL ANAK ACLK ASO4K ANH4K ANO3K',$
	 'ACLIJ     ACLI ACLJ',$ 
	 'AECIJ     AECI AECJ',$ 
	 'ASO4IJ    ASO4I ASO4J',$ 
	 'ANO3IJ    ANO3I ANO3J',$ 
	 'ANH4IJ    ANH4I ANH4J',$ 
	 'AORGAJ    AXYL1J AXYL2J AXYL3J ATOL1J ATOL2J ATOL3J ABNZ1J ABNZ2J ABNZ3J AALKJ AOLGAJ',$ 
	 'AORGBJ    AISO1J AISO2J AISO3J ATRP1J ATRP2J ASQTJ AOLGBJ',$ 
	 'AORGPAIJ  AORGPAI AORGPAJ',$ 
	 'EC        AECIJ',$ 
	 'OC        AORGAJ AORGBJ',$ 
	 'POC       AORGPAIJ',$ 
	 'SOC       AORGAJ AORGBJ',$ 
	 'PM2.5     PM2P5'])
	               
vlist.add,ii_m3_vlist1($	      
	['PM2P5 ASO4J ASO4I ANO3J ANO3I ANH4J ANH4I AECJ AECI AORGAJ AORGAI AORGPAJ AORGPAI AORGBJ AORGBI A25I A25J',$ 
	 'AEC   AECJ AECI',$ 
	 'AOC   AORGAJ AORGAI AORGPAJ AORGPAI AORGBJ AORGBI',$ 
	 'POC   AORGPAI AORGPAJ',$ 
	 'SOC   AORGAJ AORGAI AORGBJ AORGBI',$ 
	 'NOX   NO NO2',$ 
	 'NOY   NO NO2 HONO NO3 HNO3 N2O5 PAN NTR PNA',$ 
	 'OX    O3 NO2',$ 
	 'ASO4  ASO4J ASO4I',$ 
	 'ANO3  ANO3J ANO3I',$ 
	 'ANH4  ANH4J ANH4I'])
	
return,vlist
end
;-------------------------------------------------------------------------------

	function ii_m3_var1,_var1,_var2,vlist,ina=ina
	
;-------------------------------------------------------------------------------

var1 = strsplit(str(/join,_var1),/extract)
var2 = strsplit(str(/join,_var2),/extract)

if n_elements(var2) eq 1 then begin

	if min(idx(var1,var2,/point)) ne -1 then return,var2		
	if min((xx=idx(str(vlist.name),str(var2[0]),/point))) ne -1 then return,ii_m3_var1(var1,vlist[xx].list,vlist)			
	
	return,'NOT-DEFINED'

endif 

return,[ii_m3_var1(var1,var2[0],vlist),ii_m3_var1(var1,var2[1:*],vlist)]
end
;-------------------------------------------------------------------------------

	function ii_m3_var,ii,_var2

;-------------------------------------------------------------------------------

vlist = ii_m3_vlist()

for ivlist=0L,n_elements(vlist)-1 do begin
		
	xx = where(str((var=ii_m3_var1(ii[0].var[0],_var2,vlist[ivlist]))) eq 'NOT-DEFINED',nxx)
	
	if nxx eq 0 then break

endfor

if nxx ne 0 then message,str(/join,'VARIABLE NOT DEFINED:',_var2)

return,var
end
;-------------------------------------------------------------------------------

	function ii_m3_file_to_read,_ii,tflag,cut=cut
		
;-------------------------------------------------------------------------------

aa = reverse(rd_text(_ii.tflag,/column),2)
rr = replicate(struct_make(['time','file'],''),(dim=size(aa,/dim)))
rr.file = ary(reverse(_ii.file),nx=dim[0])
rr.time = aa

if n_elements(cut) eq 1 then rr = rr[cut:*,*]
if n_elements(cut) eq 2 then rr = rr[min(cut):max(cut),*]

ii = idx(str(rr.time),str(tflag),/point)
rr = data_check(rr[ii],ii ne -1,ina=0b)

if ~keyword_set(rr) then return,0B

aa = idx(rr.file,/self,oself=bb)
cc = strarr(n_elements(bb))

for ibb=0L,n_elements(bb)-1 do cc[ibb]= str(/join,rr[where(aa eq ibb)].time)

rr = replicate(struct_make(['file','tflag','time'],''),n_elements(bb))
rr.file = bb
rr.time = cc
rr.tflag = _ii[idx(_ii.file,rr.file,/point)].tflag

return,rr
end
;-------------------------------------------------------------------------------

	function rd_m3_file2,info,ix=ix,iy=iy,iz=iz,it=it,ip=ip,is=is,method=method
		
;-------------------------------------------------------------------------------

nan = !values.f_nan

case str(/low,var_set(method,ina='plane')) of

	'point' : begin
		
		xs2 = idx(is,/self,oself=is2)
																
		ix = long(strmid(is2,0,6))
		iy = long(strmid(is2,6,6))
		iz = long(strmid(is2,12,6))
		it = long(strmid(is2,18,6))
		rr = make_array((ns=n_elements(is2)),value=nan)
		
		for iis=0L,ns-1 do begin
		
			if ix[iis] eq -1 or iy[iis] eq -1 or iz[iis] eq -1 or it[iis] eq -1 then continue
			
			ncdf_varget1,info.cdf_id,info.var_id,value,offset=[ix[iis],iy[iis],iz[iis],it[iis]]

			rr[iis] = value
		
		endfor
								
		return,rr[xs2]
		end		
		
	'curtain' : begin
					
		rr = reform(make_array([(nx=n_elements(ix)),(nz=info.dim[2]),(nt=n_elements(it))],value=nan),[nx,nz,nt],/overwrite)
					
		for iit=0L,nt-1 do begin		
		for iix=0L,nx-1 do begin
		
			if ix[iix] lt 0 or ix[iix] ge info.dim[0] or iy[iix] lt 0 or iy[iix] ge info.dim[1] or it[iit] lt 0 or it[iit] ge info.dim[3] then continue
					
			ncdf_varget,info.cdf_id,info.var_id,value,count=[1,1,nz,1],offset=[ix[iix],iy[iix],0,it[iit]]			
			rr[iix,*,iit] = value
						
		endfor			
		endfor
		
		if var_set(iz) then rr = rr[*,iz,*]
				
		return,rr			
		end
		
	'plane' : begin
				
		rr = reform(make_array([(nx=info.dim[0]),(ny=info.dim[1]),(nz=n_elements(iz)),(nt=n_elements(it))],value=nan),[nx,ny,nz,nt],/overwrite)
										
		for iit=0L,nt-1 do begin					
		for iiz=0L,nz-1 do begin
		
			if iz[iiz] lt 0 or iz[iiz] ge info.dim[2] or it[iit] lt 0 or it[iit] ge info.dim[3] then continue
				
			case 1 of
							
				array_equal(info.var_info.dim,[5,4,2,0]) : ncdf_varget,info.cdf_id,info.var_id,value,count=[nx,ny,1,1],offset=[0,0,iz[iiz],it[iit]]
				array_equal(info.var_info.dim,[4,2,0])   : ncdf_varget,info.cdf_id,info.var_id,value,count=[nx,ny,1],offset=[0,0,it[iit]]
				else : message,str(/join,'DIMENSION NOT DEFINED:',info.var_info.dim)
						
			endcase
			
			rr[*,*,iiz,iit] = value
				
		endfor
		endfor
		
		if var_set(ip) then begin
		
			rr = reform(rr,[product(info.dim[[0,1]]),nz*nt],/overwrite)
			rr = data_check(rr[ip,*],ip lt 0 or ip ge nx*ny,nan,dim=1)
			rr = reform(rr,[n_elements(ip),nz,nt],/overwrite)
			
		endif
													
		return,rr		
		end		
				
	'obs' : begin
				
		rr = reform(make_array([(nx=info.dim[0]),(nz=n_elements(iz)),(nt=n_elements(it))],value=nan),[nx,nz,nt],/overwrite)
										
		for iit=0L,nt-1 do begin					
		for iiz=0L,nz-1 do begin
		
			if iz[iiz] lt 0 or iz[iiz] ge info.dim[1] or it[iit] lt 0 or it[iit] ge info.dim[2] then continue
			
			ncdf_varget,info.cdf_id,info.var_id,value,count=[nx,1,1],offset=[0,iz[iiz],it[iit]]
						
			rr[*,iiz,iit] = value
				
		endfor
		endfor
													
		return,rr		
		end
			
	'raw' : begin
	
		ncdf_varget,info.cdf_id,info.var_id,value	
		return,value
		end	
		
endcase

end
;-------------------------------------------------------------------------------

	function rd_m3_file1,file,var,ix=ix,iy=iy,iz=iz,it=it,ip=ip,is=is,unit=unit2,method=method

;-------------------------------------------------------------------------------

	;

cdf_id = ncdf_open(file[0])
cdf_info = ncdf_inquire(cdf_id)
if n_elements(var) gt 1 then message,/info,str(/join,'READ',file[0],'/',var)

for dim_id=0L,cdf_info.ndims-1 do begin
	ncdf_diminq,cdf_id,dim_id,dim_name,dim_size1
	dim_size = var_set(dim_size) ? [dim_size,dim_size1] : [dim_size1]
endfor

	;

for ivar=0L,n_elements(var)-1 do begin

	message,/info,str(/join,'READ',file[0],'/',var[ivar])

	var_id = ncdf_varid(cdf_id,var[ivar])
	var_info = ncdf_varinq(cdf_id,var_id)
	for att_id=0L,var_info.natts-1 do att_name = att_id eq 0 ? ncdf_attname(cdf_id,var_id,att_id) : [att_name,ncdf_attname(cdf_id,var_id,att_id)]
						
	if min((xx=idx(str(/low,att_name),['unit','units'],/contain))) ne -1 then begin
	
		ncdf_attget,cdf_id,var_id,att_name(xx[0]),att_value
		unit1 = str(string(att_value))
		if ~var_set(unit2) and ivar eq 0 then unit2 = unit1
		
	endif	
		
	info1 = {cdf_id:cdf_id,var_id:var_id,var_info:var_info,dim:dim_size[var_info.dim]}		
	rr1 = rd_m3_file2(info1,ix=ix,iy=iy,iz=iz,it=it,ip=ip,is=is,method=method)
								
	rr = (ivar eq 0 ? 0 : rr ) + (var_set(unit1,unit2) ? cv_unit(rr1,unit1,unit2) : rr1 )							
	rr = reform(rr,size(rr1,/dim),/overwrite)
										
endfor

ncdf_close,cdf_id

return,rr
end
;-------------------------------------------------------------------------------
	
	function rd_m3_var,_file,var=_var,$
	         lon=lon,lat=lat,layer=layer,alt=alt,tflag=tflag,ix=ix,iy=iy,iz=iz,it=it,ip=ip,$	         
	         unit=unit,cut=cut,info=info,method=method,oinfo=oinfo,_extra=_ex		
	
;-------------------------------------------------------------------------------

ii = var_set(info,ina=ii_m3(_file,ominfo=minfo,otflag=_tflag))
nan = !values.f_nan

case 1 of

	ii[0].ntime eq 0 : begin
	
		var  = ii_m3_var(ii,_var)
			
		if var_set(lon,lat) then gi = aqm_gi(lon,lat,minfo=minfo)
		if var_set(gi) then ip = gi.ixy		
		nz = n_elements((iz=var_set(iz,ina=var_set(layer,ina=lindgen(ii[0].nz)))))
		
		rr = rd_m3_file1(ii[0].file,var,ip=ip,iz=iz,it=0,unit=unit,method='raw')
		
		oinfo  = {unit:unit,minfo:minfo,layer:iz,sigma:float(strsplit(ii[0].sigma,/extract))}

		return,rr
		end
		
	minfo.ny eq 1 : begin
		
		nz = n_elements((iz=var_set(iz,ina=var_set(layer,ina=lindgen(ii[0].nz)))))						
		nt = n_elements((tflag=var_set(tflag,ina=date(min(_tflag),max(_tflag),/make))))		
		var  = ii_m3_var(ii,_var)				
		file = ii_m3_file_to_read(ii,tflag,cut=cut)								
		if ~keyword_set(file) then return,!null
						
		for ifile=0L,n_elements(file)-1 do begin									
		
			it  = idx(strsplit(file[ifile].tflag,/extract),strsplit(file[ifile].time,/extract),/point)																	
			rr1 = rd_m3_file1(file[ifile].file,var,ix=ix,iy=iy,iz=iz,it=it,ip=ip,unit=unit,method='obs')
																															
			if ~var_set(rr) then rr = reform(replicate(nan,(dim=[(size(rr1,/dim))[0:(size(rr1,/n_dim)-2)>0],nt])),dim,/overwrite)
														
			case size(rr,/n_dim) of
			
				1 : rr[      idx(tflag,strsplit(file[ifile].time,/extract),/point)] = rr1
				2 : rr[    *,idx(tflag,strsplit(file[ifile].time,/extract),/point)] = rr1
				3 : rr[  *,*,idx(tflag,strsplit(file[ifile].time,/extract),/point)] = rr1
				4 : rr[*,*,*,idx(tflag,strsplit(file[ifile].time,/extract),/point)] = rr1
					
			endcase
											
		endfor				
												
		oinfo  = {tflag:tflag,unit:unit,minfo:minfo,layer:iz,sigma:float(strsplit(ii[0].sigma,/extract))}
							
		return,rr		
	
		end	
		
	struct_set(_ex,['flight','interpol']) : begin  
	
			; need to update
	
		shift = 0.5d
						
		var   = ii_m3_var(ii,_var)					
		xtime = idx(date(tflag,/hh),/self,oself=time)
		read  = ii_m3_file_to_read(ii,time,cut=cut)								
		if ~keyword_set(read) then return,!null		
				
		gi = aqm_gi(lon,lat,minfo=minfo,map_structure=map_structure)			
		iz = var_set(layer,ina=floor(interpol(findgen(n_elements((alt2=cal_sigma(strsplit(ii[0].sigma,/extract),/height)))),alt2,alt)))
												
		for ifile=0L,n_elements(read)-1 do begin
					
			xx   = idx(xtime,idx(time,strsplit(read[ifile].time,/extract),/contain),/contain)			
			ii1  = ii[idx(ii.file,read[ifile].file,/point)]																																
			ip1  = array_indices_reverse((dim1 = [minfo.nx,minfo.ny,ii1.nz,ii1.ntime]),transpose([[gi.ix[xx]],[gi.iy[xx]],[iz[xx]],[idx(strsplit(read[ifile].tflag,/extract),time[xtime],/point)]]),/dim)						
			xip2 = idx(ip1,/self,oself=ip2)
			
			rr1 = rd_m3_file1(read[ifile].file,var,ip=ip2,unit=unit,method='box-ip')
			rr1 = rr1[*,*,*,xip2]
			
			aqm_fgi,lon[xx],lat[xx],ix1,iy1,minfo=minfo,map_structure=map_structure
						
			ix1 = (ix1=ix1-shift)-floor(ix1)
			iy1 = (iy1=iy1-shift)-floor(iy1)
			iz1 = (iz1=iz-shift)-floor(iz1)								
						
			rr2 = fltarr(n_elements(xx))			
			for i=0L,n_elements(xx)-1 do rr2[i] = interpolate(rr1[*,*,*,i],ix1[i],iy1[i],iz1[i])					
			;rr2 = interpolate(interpolate(transpose(rr1,[3,0,1,2]),ix1,iy1,iz1),findgen(n_elements(xx)),findgen(n_elements(xx)))
														
			rr = [var_set(rr) ? rr : !null,rr2]
											
		endfor
		
			;
			
		time = date(time,1,/incr)	
		read  = ii_m3_file_to_read(ii,time,cut=cut)
		
		for ifile=0L,n_elements(read)-1 do begin
					
			xx   = idx(xtime,idx(time,strsplit(read[ifile].time,/extract),/contain),/contain)
			ii1  = ii[idx(ii.file,read[ifile].file,/point)]																																
			ip1  = array_indices_reverse((dim1 = [minfo.nx,minfo.ny,ii1.nz,ii1.ntime]),transpose([[gi.ix[xx]],[gi.iy[xx]],[iz[xx]],[idx(strsplit(read[ifile].tflag,/extract),time[xtime],/point)]]),/dim)						
			xip2 = idx(ip1,/self,oself=ip2)
			
			rr1 = rd_m3_file1(read[ifile].file,var,ip=ip2,unit=unit,method='box-ip')
			rr1 = rr1[*,*,*,xip2]
			
			aqm_fgi,lon[xx],lat[xx],ix1,iy1,minfo=minfo,map_structure=map_structure
						
			ix1 = (ix1=ix1-shift)-floor(ix1)
			iy1 = (iy1=iy1-shift)-floor(iy1)
			iz1 = (iz1=iz-shift)-floor(iz1)
			
			rr2 = fltarr(n_elements(xx))			
			for i=0L,n_elements(xx)-1 do rr2[i] = interpolate(rr1[*,*,*,i],ix1[i],iy1[i],iz1[i])					
														
			ss = [var_set(ss) ? ss : !null,rr2]
											
		endfor		
		
		rr = interpolate([[rr],[ss]],findgen(n_elements(xx)),float(date(tflag,/cut,/min))/60.)
		
		oinfo  = {tflag:tflag,unit:unit,minfo:minfo,layer:iz,sigma:float(strsplit(ii[0].sigma,/extract))}

		return,rr		
		end
		
	struct_set(_ex,'flight') : begin
	
		var   = ii_m3_var(ii,_var)								
		xtime = idx(date(tflag,/hh),/self,oself=time)
		read  = ii_m3_file_to_read(ii,time,cut=cut)
		
		gi = aqm_gi(lon,lat,minfo=minfo,map_structure=map_structure)		
		iz = var_set(layer,ina=floor(interpol(findgen(n_elements((alt2=cal_sigma(strsplit(ii[0].sigma,/extract),/height)))),alt2,alt)))							
		rr = make_array(size(lon,/dim),value=nan)
		
		for ifile=0L,n_elements(read)-1 do begin
				
			ii1 = ii[idx(ii.file,read[ifile].file,/point)] 			
			xx  = idx(tflag,strsplit(read[ifile].time,/extract),/contain)			
			ix1 = data_check((ix1=gi.ix[xx]),ix1 lt 0 or ix1 gt ii1.minfo.nx-1,-1)
			iy1 = data_check((iy1=gi.iy[xx]),iy1 lt 0 or iy1 gt ii1.minfo.ny-1,-1)
			iz1 = data_check((iz1=iz[xx]),iz1 lt 0 or iz1 gt ii1.nz-1,-1)
			it1 = idx(strsplit(read[ifile].tflag,/extract),tflag[xx],/point)			
			is1 = string(ix1,format='(I6)')+string(iy1,format='(I6)')+string(iz1,format='(I6)')+string(it1,format='(I6)')	
											
			rr[xx] = rd_m3_file1(read[ifile].file,var,is=is1,unit=unit,method='point')
																	
		endfor	
					
		return,rr			
		end
		
	struct_set(_ex,'curtain') : begin
	
		if ~var_set(lon,lat) then message,'CURTAIN requires lon/lat'
	
		gi = aqm_gi(lon,lat,minfo=minfo,/track)						
		aa = idx(gi.ii,/self,oself=bb,oiself=cc)								
		rr = rd_m3_var(_file,var=_var,ix=gi.ix[cc],iy=gi.iy[cc],layer=layer,tflag=tflag,unit=unit,oinfo=oinfo,method='curtain',info=ii)
		rr = data_check(rr,aa,dim=1,/index)											
		oinfo = struct_merge(oinfo,gi)
		
		return,rr		
		end
		
	struct_set(_ex,'track') : begin	
	
		if ~var_set(lon,lat) then message,'TRACK requires lon/lat'
	
		gi = aqm_gi(lon,lat,minfo=minfo,/track)						
		aa = idx(gi.ii,/self,oself=bb,oiself=cc)		
		rr = rd_m3_var(_file,var=_var,ip=bb,layer=layer,tflag=tflag,unit=unit,oinfo=oinfo,info=ii)
		rr = data_check(rr,aa,dim=1,/index)											
		oinfo = struct_merge(oinfo,gi)
		
		return,rr				
		end

	var_set(lon,lat) : begin
	
		gi = aqm_gi(lon,lat,minfo=minfo)										
		rr = rd_m3_var(_file,var=_var,ip=gi.ixy,layer=layer,tflag=tflag,unit=unit,info=ii,oinfo=oinfo)
		return,rr
		end
							
	var_set(_var) : begin
			
		nz   = n_elements((iz=var_set(iz,ina=var_set(layer,ina=lindgen(ii[0].nz)))))						
		nt   = n_elements((tflag=var_set(tflag,ina=date(min(_tflag),max(_tflag),/make))))		
		var  = ii_m3_var(ii,_var)				
		file = ii_m3_file_to_read(ii,tflag,cut=cut)
		
		if ~keyword_set(file) then return,!null
		
		for ifile=0L,n_elements(file)-1 do begin									
		
			it  = idx(strsplit(file[ifile].tflag,/extract),strsplit(file[ifile].time,/extract),/point)			
														
			rr1 = rd_m3_file1(file[ifile].file,var,ix=ix,iy=iy,iz=iz,it=it,ip=ip,unit=unit,method=method)						
																															
			if ~var_set(rr) then rr = reform(replicate(nan,(dim=[(size(rr1,/dim))[0:(size(rr1,/n_dim)-2)>0],nt])),dim,/overwrite)
																				
			case size(rr,/n_dim) of
			
				1 : rr[      idx(tflag,strsplit(file[ifile].time,/extract),/point)] = rr1
				2 : rr[    *,idx(tflag,strsplit(file[ifile].time,/extract),/point)] = rr1
				3 : rr[  *,*,idx(tflag,strsplit(file[ifile].time,/extract),/point)] = rr1
				4 : rr[*,*,*,idx(tflag,strsplit(file[ifile].time,/extract),/point)] = rr1
					
			endcase
											
		endfor				
												
		oinfo  = {tflag:tflag,unit:unit,minfo:minfo,layer:iz,sigma:float(strsplit(ii[0].sigma,/extract))}
							
		return,rr		
		end
		
	else : message,'USAGE rr = r_m3(file,var=var)'	

endcase	

end
;-------------------------------------------------------------------------------

	function rd_m3,_file,var=_var,$
	         lon=lon,lat=lat,layer=layer,alt=alt,tflag=tflag,unit=unit,cut=cut,$
		 oinfo=oinfo,ominfo=ominfo,otflag=otflag,_extra=_ex		

;-------------------------------------------------------------------------------

if total(file_test(var_set(_file,ina=''))) eq 0 then message,str(/join,'NO VALID FILE',_file)
if ~var_set(_var) then message,'NOVAR >> USE rr = rd_m3(file,var=VAR)'

rr = rd_m3_var(_file,var=_var,lon=lon,lat=lat,layer=layer,alt=alt,tflag=tflag,unit=unit,cut=cut,oinfo=oinfo,_extra=_ex)

if struct_set(oinfo,'tflag') then otflag = oinfo.tflag
if struct_set(oinfo,'minfo') then ominfo = oinfo.minfo

return,rr
end

