;-------------------------------------------------------------------------------
;	NAME
;		pos
;
;	PURPOSE
;		IDL function to return plot position
;
;	USAGE
;		r = pos([position,]/colorbar[,dx=dx,dxa=dxa][,dy=dy,dya=dya])
;		r = pos([position,]/multi[,nx=nx,ny=ny,dxa=dxa,dya=dya])
;
;	INPUT
;		position : postion parameter used for plot
;		nx	 : number of plots in x-axis
;		ny	 : number of plots in y-axis;   
;		dx	 : plot width ratio (or colorbar width)
;		dy	 : plot height ratio
;		xa	 : horizontal gap ratio
;		ya       : vertical gap ratio
;
;	KEYWORD
;		colorbar : return position of colorbar
;		multi    : return positions of multi plots
;		left	 : put colorbar at left 
;		right    : put colorbar at right
;		top	 : put colorbar at top
;		bottom   : put colorbar at top
;
;	AUTHOR
;		2007-01-04 Hyun Cheol Kim (hyuncheol.kim@gmail.com, hkim2@mail.uh.edu);
;		2007-11-15 /multi modified
;-------------------------------------------------------------------------------

	function pos,position,colorbar=colorbar,multi=multi,ratio=ratio,box=box,$
	         left=left,right=right,top=top,bottom=bottom,                   $
	         nx=nx,ny=ny,dx=dx,dy=dy,xa=xa,ya=ya,max_length=max_length,     $
	         domain=domain,minfo=minfo,xsize=xsize,ysize=ysize,dot=dot,     $
	         flat=flat,flip=flip,crange=crange,device=device

;-------------------------------------------------------------------------------

if var_set(position)   then p = position else p = [0.05,0.05,0.95,0.95]

case 1 of

	keyword_set(crange) : begin
	
		case 1 of
		
			keyword_set(device) : r = chdmod(!x.crange,!y.crange,/data,/to_device)					
			else                : r = chdmod(!x.crange,!y.crange,/data,/to_normal)				
											
		endcase
	
		return,r[[0,1,3,4]]
	
		end

	keyword_set(colorbar) : begin
	
		if ~var_set(dx) then dx = 0.025
		if ~var_set(dy) then dy = dx
		if ~var_set(xa) then xa = 0.01
		if ~var_set(ya) then ya = 0.01
		
		if ~var_set(max_length) then max_length = px(500)

		
		pp  = [1-p[2],p[1],p[0]]		
		pp2 = (where(pp eq max(pp)))[0]
		
		case 1 of
    
			keyword_set(left)   : r = p + [-dx-xa,0,-p[2]+p[0]-xa,0]
			keyword_set(top)    : r = p + [0,-p[1]+p[3]+ya,0,dy+ya]
			keyword_set(bottom) : r = p + [0,-ya-dy,0,-p[3]+p[1]-ya]	   
			keyword_set(right)  : r = p + [-p[0]+p[2]+xa,0,xa+dx,0]
			pp2 eq 0	    : r = p + [-p[0]+p[2]+xa,0,xa+dx,0]
			pp2 eq 1	    : r = p + [0,-ya-dy,0,-p[3]+p[1]-ya]  
			pp2 eq 2	    : r = p + [-dx-xa,0,-p[2]+p[0]-xa,0]
  		  
			else : message,'OPTIONS (/LEFT /RIGHT /TOP /BOTTOM)'

		endcase
		
		r2 = chdmod(r,/normal,/to_device)
						
		if r2[3]-r2[1] gt max_length then r = chdmod([r2[0],(r2[3]-r2[1])/2-max_length/2.,r2[2],(r2[3]-r2[1])/2+max_length/2.],/device,/to_normal)
		if r2[2]-r2[0] gt max_length then r = chdmod([(r2[2]-r2[0])/2-max_length/2,r2[1],(r2[2]-r2[0])/2+max_length/2,r2[3]],/device,/to_normal)
							
		return,r
		
		end
	
	;.............................................................................
	
	keyword_set(multi) : begin
	
		if ~var_set(nx)     then nx = 2
		if ~var_set(ny)     then ny = 2
		if ~var_set(dx)     then dx = 1
		if ~var_set(dy)     then dy = 1
		if ~var_set(xa)     then xa = 0.1
		if ~var_set(ya)     then ya = 0.1    

		rx =  array(dx,count=nx)
		ry =  array(dy,count=ny)
		rxa = array(xa,count=nx)
		rya = array(ya,count=ny)
		
		xx = ([0,total(array(transpose([[rx],[rxa]]),/flat),/cumulative)])
		xx = (xx/xx[2*nx-1])[0:2*nx-1]*(p[2]-p[0])+p[0]
		yy = ([0,total(array(transpose([[ry],[rya]]),/flat),/cumulative)])
		yy = (yy/yy[2*ny-1])[0:2*ny-1]*(p[3]-p[1])+p[1]
		
		result = fltarr(nx,ny,4)

		for ix=0L,nx-1 do for iy=0L,ny-1 do result[ix,iy,*] = [xx[2*ix],yy[2*iy],xx[2*ix+1],yy[2*iy+1]]
		
		if keyword_set(flip) then result = array(result,/flip,/up) 				
		if keyword_set(flat) then result = reform(result,[nx*ny,4])
		
		return,result
								
		end
		
		;.............................................................................
  
	keyword_set(ratio) : begin
	
		case 1 of

			var_set(domain) : grid = aqm_grid(domain=domain,dot=dot,olon=lon,olat=lat,olx=x,oly=y)        
			var_set(minfo)  : grid = aqm_grid(minfo=minfo,dot=dot,olon=lon,olat=lat,olx=x,oly=y)    
              
			else : message,'OPTION (/DOMAIN /MINFO)'

		endcase
			
		r    = (max(x)-min(x))/(max(y)-min(y))
	    
		case 1 of

			var_set(xsize) : return,round((p[2]-p[0])/(p[3]-p[1])*xsize/r)
			var_set(ysize) : return,round((p[3]-p[1])/(p[2]-p[0])*ysize*r)     
			else  	 : return,r*(p[3]-p[1])/(p[2]-p[0])

		endcase

		end
	
 		;.............................................................................	
	
	else : message,'OPTIONS (/COLORBAR /MULTI /RATIO)'

endcase


end
