;-------------------------------------------------------------------------------
;	NAME
;		plot_circle
;
;	PURPOSE
;		IDL procedure to plot circles 
;
;	USAGE
;		plot_circle,x,y[,radius=radius|,r_device=r_device][,/data|/normal|/device]
;
;	INPUT
;		x        : x points
;		y        : y points
;		radius   : circle radius
;		r_device : circle radius in device mode
;		color    : circle color
;		bcolor   : circle boundary color
;		lcolor   : label color
;		llcolor  : label line color
;		clip     : normal or device window to clip
;		label    : text labels to put along with circles
;		plabel   : position of label
;		ilabel   : apply plabel at some circles only
;
;	KEYWORD
;		fill   : fille circles
;		bcolor : draw boundary of circles
;		auto   : find best label position
;
;	EXAMPLE
;		color = color_load(/basic)
;		!p.color = color.black
;		!p.background = color.white
;		x = randomu(s,100)*10
;		y = randomu(s,100)*10
;		plot,x,y,psym=3
;		plot_circle,x,y,label=str(x,/auto_format),color=['blue','pink','almond','skyblue','orange'],/fill,/bcolor,/auto
;
;	NOTE
;		"GET_CIRCLE" ROUTINE IS FROM "TVCIRCLE.PRO" WRITTEN BY B. PFARR ET AL.
;
;	AUTHOR
;		Hyun Cheol Kim 20080228 (UH-IMAQS,hkim2@mail.uh.edu hyuncheol.kim@gmail.com)
;-------------------------------------------------------------------------------

 pro get_circle,irad,x,y
 
;------------------------------------------------------------------------------- 

x    = 0
y    = irad 
d    = 3 - 2 * irad
path = 0

xhalfquad = make_array(irad+1,/int,/nozero)
yhalfquad = xhalfquad

while x lt y do begin

	xhalfquad[path] = x
	yhalfquad[path] = y
	
	path = path + 1
	
	if d lt 0 then begin
	
		d = d+(4*x)+6 
		
	endif else begin
	
		d = d+(4*(x-y))+10
		y = y-1
	
	endelse
	
	x = x+1 
			
endwhile

if x eq y then begin

	xhalfquad[path] = x
	yhalfquad[path] = y
	
	path = path + 1

endif

xhalfquad = xhalfquad[0:path-1]
yhalfquad = yhalfquad[0:path-1]

xquad = [xhalfquad,rotate(yhalfquad,5)]
yquad = [yhalfquad,rotate(xhalfquad,5)]

xquadrev = rotate(xquad[0:2*path-2],5)
yquadrev = rotate(yquad[0:2*path-2],5)

x = [xquad,xquadrev,-xquad[1:*],-xquadrev]
y = [yquad,-yquadrev,-yquad[1:*],yquadrev]

end

;-------------------------------------------------------------------------------

	pro plot_circle,_xc,_yc,radius=radius,r_device=r_device,$
	    data=data,device=device,normal=normal,fill=fill,auto=auto,$
	    color=color,bcolor=bcolor,lcolor=lcolor,llcolor=llcolor,clip=clip,$
	    thick=thick,bthick=bthick,lthick=lthick,llthick=llthick,lllength=lllength,$	    
	    label=label,plabel=plabel,ilabel=ilabel,charsize=charsize,$
	    nolabel=nolabel,taken=taken,n_auto=n_auto,_extra=_ex

;-------------------------------------------------------------------------------

ndata  = min([n_elements(_xc),n_elements(_yc)])
nlabel = n_elements(label)

if var_set(color)    then color  = array(color_load(color,/get),count=ndata) else color = replicate(!p.color,ndata)
if var_set(bcolor)   then bcolor = var_set(bcolor,/keyword_only) ? array(!p.color,count=ndata) : array(bcolor,count=ndata) else bcolor = color
if var_set(lcolor)   then lcolor = array(lcolor,count=ndata) else lcolor = replicate(!p.color,ndata)
if var_set(llcolor)  then llcolor = array(llcolor,count=ndata) else llcolor = replicate(color_load('gray',/get),ndata)
if var_set(clip)     then clip = max(clip) le 1 ? (clip#[!d.x_size,!d.y_size,!d.x_size,!d.y_size])[index(/diagonal,4)] : clip
if var_set(plabel)   then plabel = array(plabel,count=ndata) else plabel = replicate(0,ndata)
if var_set(ilabel)   then plabel[ilabel[*,0]] = ilabel[*,1]
if~var_set(charsize) then charsize = 1.
if~var_set(thick)    then thick = 1.
if~var_set(bthick)   then bthick = 1.
if~var_set(lthick)   then lthick = 1.
if~var_set(llthick)  then llthick = 1.
if~var_set(lllength) then lllength = [1,3]
if~var_set(n_auto)   then n_auto = 100L


if keyword_set(nolabel) then nlabel = 0

	; data points

case 1 of
		
	keyword_set(data)   : pc = convert_coord(_xc,_yc,/data,/to_device)
	keyword_set(normal) : pc = convert_coord(_xc,_yc,/normal,/to_device)	
	keyword_set(device) : pc = convert_coord(_xc,_yc,/device,/to_device)
	else : pc = convert_coord(_xc,_yc,/data,/to_device)
	
endcase

xc = round(pc[0,*])
yc = round(pc[1,*])

	; get radius

case 1 of

	var_set(r_device)      : irad = array(r_device,count=ndata)	
	var_set(radius,device) : irad = array(r_device,count=ndata)				
	var_set(radius,data) : begin
	
		x1   = convert_coord(radius,replicate(0,n_elements(radius)),/data,/to_device)
		x2   = convert_coord(0,0,/data,/to_device)	
		irad = array(x1[0,*]-x2[0,0],count=ndata)				
		end		
		
	var_set(radius,normal) : begin
	
		x1   = convert_coord(radius,replicate(0,n_elements(radius)),/normal,/to_device)
		x2   = convert_coord(0,0,/normal,/to_device)	
		irad = array(x1[0,*]-x2[0,0],count=ndata)		
		end				
		
	var_set(radius) : begin
		
		x1   = convert_coord(radius,replicate(0,n_elements(radius)),/data,/to_device)
		x2   = convert_coord(0,0,/data,/to_device)	
		irad = array(x1[0,*]-x2[0,0],count=ndata)				
		end
		
	else : irad = array(px2dev(4),count=ndata)

endcase

if ~keyword_set(taken) then taken = bytarr(!d.x_vsize,!d.y_vsize)

qq = finite(_xc) and finite(_yc)

	; circle

for idata=0L,ndata-1 do begin
	
	if qq[idata] eq 0 then continue

	get_circle,irad[idata],x,y
			
	if keyword_set(fill) then polyfill,x+xc[idata],y+yc[idata],color=color[idata],/device,clip=clip,noclip=0,thick=thick
	
	plots,x+xc[idata],y+yc[idata],color=bcolor[idata],/device,clip=clip,noclip=0,thick=bthick
	
	if keyword_set(auto) then begin
	
		if max(x+xc[idata]) ge 0 and  min(x+xc[idata]) le !d.x_vsize-1 and max(y+yc[idata]) ge 0 and  min(y+yc[idata]) le !d.y_vsize-1 then begin
	
			xtaken = polyfillv(x+xc[idata],y+yc[idata],!d.x_vsize,!d.y_vsize)
		
			if xtaken[0] ne -1 then taken[xtaken] = 1
		
		endif
		
	endif	

endfor

	; label

if keyword_set(auto) then begin

	dp = lllength[0]
	aa = lllength[1]
	
	for idata=0L,nlabel-1 do begin

		dx = strlen(label[idata])*!d.x_ch_size*charsize
		dy = !d.y_ch_size*charsize

		for i=0L+(long(randomu(s,1)*4))[0],n_auto-1 do begin
			
			case i mod 4 of
		
				0 : box = [xc[idata]+irad[idata]+dp*fix(i/4.)+aa+[0,dx,dx,0,0],  yc[idata]+irad[idata]+dp*fix(i/4.)+aa+[0,0,dy,dy,0],  0.,0., 1., 1.]
				1 : box = [xc[idata]-irad[idata]-dp*fix(i/4.)+aa+[0,-dx,-dx,0,0],yc[idata]-irad[idata]-dp*fix(i/4.)+aa+[0,0,-dy,-dy,0],1.,1.,-1.,-1.]				
				2 : box = [xc[idata]+irad[idata]+dp*fix(i/4.)+aa+[0,dx,dx,0,0],  yc[idata]-irad[idata]-dp*fix(i/4.)+aa+[0,0,-dy,-dy,0],0.,1., 1.,-1.]
				3 : box = [xc[idata]-irad[idata]-dp*fix(i/4.)+aa+[0,-dx,-dx,0,0],yc[idata]+irad[idata]+dp*fix(i/4.)+aa+[0,0,dy,dy,0],  1.,0.,-1., 1.]
				
			endcase	

			xbox = box[0:4]
			ybox = box[5:9]												
			
			if max(xbox) ge 0 and min(xbox) le !d.x_vsize-1 and max(ybox) ge 0 and min(ybox) le !d.y_vsize-1 then begin
				
				ibox = polyfillv(xbox,ybox,!d.x_vsize,!d.y_vsize)
		
				if n_elements(ibox) lt dx*dy/2 then goto,next_label 
						
				if total(taken[ibox]) eq 0 then begin
						
					plots,[xc[idata]+box[12]*(irad[idata]*sin(45.*!dtor)+1),xbox[0],xbox[0]+dx*box[12]/2.],[yc[idata]+box[13]*(irad[idata]*sin(45.*!dtor)+1),ybox[0],ybox[0]],color=llcolor[idata],/device,clip=clip,noclip=0,thick=llthick	
					xyouts,xbox[0],ybox[0]-box[11]*dy,label[idata],alignment=box[10],color=lcolor[idata],/device,clip=clip,noclip=0,charsize=charsize,charthick=lthick
			
					taken[ibox] = 1B
	
					goto,next_label
	
				endif	
			
			endif else goto,next_label		
	
		endfor
	
		next_label:
				
	endfor
	
endif else begin

		; regular

	for idata=0L,nlabel-1 do begin
		
		L  = fix(plabel[idata]/4.)	
		L1 = irad[idata]*sin(45.*!dtor)+1
		L2 = 5*L
	
		case plabel[idata] mod 4 of
		
			0 : begin
		
				if llcolor[idata] ge 0 and L ne 0 then plots,[xc[idata]+L1,xc[idata]+L1+L2,xc[idata]+L1+2*L2],[yc[idata]+L1,yc[idata]+L1+L2,yc[idata]+L1+L2],/device,color=llcolor[idata],clip=clip,noclip=0							
				xyouts,xc[idata]+L1+2*L2-3,yc[idata]+L1+L2,label[idata],color=lcolor[idata],/device,clip=clip,noclip=0,charsize=charsize,charthick=lthick

				end
			
			1 : begin
			
				if llcolor[idata] ge 0 and L ne 0 then plots,[xc[idata]+L1,xc[idata]+L1+L2,xc[idata]+L2+2*L2],[yc[idata]-L1,yc[idata]-L1-L2,yc[idata]-L1-L2],/device,color=llcolor[idata],clip=clip,noclip=0							
				xyouts,xc[idata]+L1+2*L2-3,yc[idata]-L1-L2-!d.x_ch_size,label[idata],color=lcolor[idata],/device,clip=clip,noclip=0,charsize=charsize,charthick=lthick
												
				end
			
			2 : begin
			
				if llcolor[idata] ge 0 and L ne 0 then plots,[xc[idata]-L1,xc[idata]-L1-L2,xc[idata]-L1-2*L2],[yc[idata]-L1,yc[idata]-L1-L2,yc[idata]-L1-L2],/device,color=llcolor[idata],clip=clip,noclip=0							
				xyouts,xc[idata]-L1-2*L2,yc[idata]-L1-L2-!d.x_ch_size,label[idata],color=lcolor[idata],/device,clip=clip,noclip=0,alignment=1,charsize=charsize,charthick=lthick
			
				end
				
			3 : begin
			
				if llcolor[idata] ge 0 and L ne 0 then plots,[xc[idata]-L1,xc[idata]-L1-L2,xc[idata]-L1-2*L2],[yc[idata]+L1,yc[idata]+L1+L2,yc[idata]+L1+L2],/device,color=llcolor[idata],clip=clip,noclip=0							
				xyouts,xc[idata]-L1-2*L2,yc[idata]+L1+L2,label[idata],color=lcolor[idata],/device,clip=clip,noclip=0,alignment=1,charsize=charsize,charthick=lthick
			
				end
		
		endcase	
			
	endfor

endelse

end
