;-------------------------------------------------------------------------------
;	NAME
;		ncdf_info
;
;	PURPOSE
;		IDL function to check ncdf file
;
;	USAGE
;		r = ncdf_info(file)
;		r = ncdf_info(file,/is_ncdf)
;		r = ncdf_info(file,/check,dim=dim,var=var,gatt=gatt[,/have_all|have_any]
;
;	INPUT
;		file : ncdf file
;		var  : variable list to check
;		dim  : dimension name list to check
;		gatt : global attribute list to check
;		
;	KEYWORD	
;		is_ncdf  : return 1 if given file is ncdf file
;		check		 : check dim,var and gatt anes
;		have_all : return 1 if ncdf file has all of given names for dim,var and gatt
;		have_any : return 1 if ncdf file has any of given names for dim,var and gatt
;
;	AUTHOR
;		2008-02-11Hyun Cheol Kim (hyuncheol.kim@gmail.com, hkim2@mail.uh.edu)
;------------------------------------------------------------------------------- 

	function ncdf_info_is_ncdf,file
 
;------------------------------------------------------------------------------- 

false = 0B
true  = 1B

catch,error_status

if error_status ne 0 then begin

  catch,/cancel
  return,false

endif

cdf_id = ncdf_open(file[0])
    
ncdf_close,cdf_id

catch,/cancel
return,true

end

;-------------------------------------------------------------------------------

	function ncdf_info,_file,check=check,read=read,is_ncdf=is_ncdf,var=var,$
	         dim=dim,gatt=gatt,have_any=have_any,have_all=have_all       

;-------------------------------------------------------------------------------

file = _file[0]

case 1 of

	keyword_set(is_ncdf) : return,ncdf_info_is_ncdf(file)
	
		;...........................................................................

	keyword_set(check) : begin
	
		if not ncdf_info_is_ncdf(file) then return,0B
	
		info = ncdf_info(file)
				
		if var_set(dim) then begin
		
			if n_elements(dim) eq 1 then dim = strsplit(dim,/extract)
		
			result = var_set(result) ? [result,index(str(/up,info.dim),str(/up,dim),/point)] : index(str(/up,info.dim),str(/up,dim),/point)
										
		endif
		
		if var_set(var) then begin
		
			if n_elements(var) eq 1 then var = strsplit(var,/extract)
			
			result = var_set(result) ? [result,index(str(/up,info.var),str(/up,var),/point)] : index(str(/up,info.var),str(/up,var),/point)
		
		endif
		
		if var_set(gatt) then begin
		
			if n_elements(gatt) eq 1 then gatt = strsplit(gatt,/extract)
			
			result = var_set(result) ? [result,index(str(/up,info.gatt),str(/up,gatt),/point)] : index(str(/up,info.gatt),str(/up,gatt),/point)
					
		endif
		
		x = where(result ne -1,nx)
		
		case 1 of
		
			keyword_set(have_any) : return,nx gt 0
			keyword_set(have_all) : return,n_elements(result) eq nx
			else : return,n_elements(result) eq nx
		
		endcase

		end
	
		;...........................................................................
		
	var_set(read,gatt) : begin
	
		if ncdf_info(/check,file,gatt=gatt) then begin
		
			cdf_id = ncdf_open(file)						
			ncdf_attget,cdf_id,/global,gatt[0],gatt_value
			ncdf_close,cdf_id
			
			return,gatt_value
		
		endif else begin
		
			return,0B
		
		endelse
	
		end		
	
		;...........................................................................
	
	else : begin
	
		if ncdf_info_is_ncdf(file) then begin

			cdf_id = ncdf_open(file)
	
			cdf_info = ncdf_inquire(cdf_id)
	
				; dim
	
			for idim=0L,cdf_info.ndims-1 do begin
	
				ncdf_diminq,cdf_id,idim,dim_name,dim_size
		
				result_dim_name = var_set(result_dim_name) ? [result_dim_name,dim_name] : [dim_name]
				result_dim_size = var_set(result_dim_size) ? [result_dim_size,dim_size] : [dim_size]
		
			endfor
	
				; var
	
			for ivar=0L,cdf_info.nvars-1 do begin
	
				var_info = ncdf_varinq(cdf_id,ivar)
			
				result_var_name = var_set(result_var_name) ? [result_var_name,var_info.name] : [var_info.name]
				result_var_size = var_set(result_var_size) ? [result_var_size,str(/join,result_dim_size[var_info.dim])] : [str(/join,result_dim_size[var_info.dim])]
		
			endfor
	
				; global attributes
		
			for igatt=0L,cdf_info.ngatts-1 do begin
	
				gatt_name = ncdf_attname(cdf_id,/global,igatt)									
	
				result_gatt_name = var_set(result_gatt_name) ? [result_gatt_name,gatt_name] : [gatt_name]
	
			endfor	
			
			ncdf_close,cdf_id
	
			return,{dim:result_dim_name,dim_size:result_dim_size,var:result_var_name,var_size:result_var_size,gatt:result_gatt_name,nvar:n_elements(result_var_name)}	

		endif else begin
		
			return,0B		

		endelse	
		
	end

endcase

end
