;-------------------------------------------------------------------------------
;	NAME
;		mir_map_shp
;
;	PURPOSE
;		convert shape file to map data
;
;	USAGE
;		map = mir_map_shap(file[,cut][,domain=,minfo=][,/renew,/glbal])
;
;	INPUT
;		file : shape file
;		cut  : detail (cut=0.1 is to remove detail below 0.1 degree length)
;		domain,minfo : domain 
;		
;	KEYWORD
;		renew : remake data
;		global : save file at data/mir/map. default is ./
;
;	AUTHOR
;		2014-07-09 Hyun Cheol Kim (hyun.kim@noaa.gov)
;-------------------------------------------------------------------------------

	function mir_map_shp,file,_cut,domain=domain,minfo=minfo,renew=renew,global=global

;-------------------------------------------------------------------------------

minfo = aqm_grid(domain=domain,minfo=minfo,operimeter=peri,omid=mid)
db    = [min(peri[0,*]),min(peri[1,*]),max(peri[0,*]),max(peri[1,*])]
cut   = var_set(_cut,ina=cut[value_locate((cut=[0.01,0.03,0.05,0.1,0.3,0.5]),map_2points(((aa=chdmod([0,1],[0,0],/normal,/to_data)))[0],aa[1],aa[3],aa[4],/meters)/50000000)>0])
sdir  = keyword_set(global) ? path_check(getenv('HOME'),'data','mirie','map',/make) : path_check('save',/make)
save  = path_check(sdir,str(/join,del='-','mir_map_shp',file_basename(file),mid,cut))

if file_renew(save,renew=renew) then begin

	cut = float(cut)

	myshape = obj_new('IDLffShape',file)
	myshape->IDLffShape::GetProperty,N_ENTITIES=n_entities
	
	lon = list()
	lat = list()
	
	for ient=0L,n_entities-1 do begin
	
		str_progress,ient,n_entities,message='% mir_map_shp: SHP->MAP'

		ent = myshape->IDLffShape::GetEntity(ient)	
		
		if (check_outside=(ent.bounds[0] gt db[2] or ent.bounds[4] lt db[0] or ent.bounds[1] gt db[3] or ent.bounds[5] lt db[1])) then myshape->DestroyEntity,ent
		if check_outside then continue	
	
		xx = (*ent.vertices)[0,*]
		yy = (*ent.vertices)[1,*]
		pp = [*ent.parts,n_elements(xx)]					
		dd = [0,sqrt((xx-xx[1:*])^2+(yy-yy[1:*])^2)]		
	
		aa = bytarr(n_elements(xx)+1)	
		aa[pp] = 2B
			
		sum = 0.
	
		for idd=0L,n_elements(dd)-1 do begin
	
			sum += dd[idd]
		
			if sum ge cut or aa[idd+1] eq 2 then begin
	
				sum = 0.
				aa[idd] = 1B	
		
			endif
		
		endfor

		for ipp=0L,n_elements(pp)-2 do begin
	
			xx1 = xx[pp[ipp]:pp[ipp+1]-1]
			yy1 = yy[pp[ipp]:pp[ipp+1]-1]
	
			lon.add,xx1[where(aa[pp[ipp]:pp[ipp+1]-1] ne 0)],/extract
			lat.add,yy1[where(aa[pp[ipp]:pp[ipp+1]-1] ne 0)],/extract
			lon.add,!Values.f_nan
			lat.add,!Values.f_nan
	
		endfor	
		
	endfor
	
	OBJ_DESTROY,myshape
	
	map = {lon:lon.toarray(/no_copy),lat:lat.toarray(/no_copy),cut:cut}

	message,/info,str(/join,'SAVE',save)
	save,filename=save,map
	
endif else begin

	message,/info,str(/join,'READ',save)
	restore,save
	
endelse

return,map

end
;-------------------------------------------------------------------------------

	function mir_map,dd,map=map,file=file,domain=domain,minfo=minfo,color=color,thick=thick,noplot=noplot
	
;-------------------------------------------------------------------------------

case 1 of

	var_set(map) :
		
	var_set(dd) : begin
	
		file = dn_gis_gadm()
		nxx  = n_elements((xx=fix(string(reform((xx=byte(strjoin(str(dd)))),[1,n_elements(xx)])))<2))		
		map = list(mir_cl(/get,ary(count=nxx,var_set(color,ina=['gray','black','black']))),ary(count=nxx,var_set(thick,ina=0.3)))
		
		for ixx=0L,nxx-1 do begin
		
			map.add,mir_map_shp(file[xx[ixx]],/global,domain=domain,minfo=minfo)
		
		endfor
		

		end	
		
	
	var_set(file) : begin
		
		map = list(mir_cl(/get,var_set(color,ina=ary('gray',count=(nfile=n_elements(file))))),var_set(thick,ina=ary(0.3,count=nfile)))
		
		for ifile=0L,nfile-1 do map.add,mir_map_shp(file[ifile],/global,domain=domain,minfo=minfo)
			
		end

	~var_set(map) : begin

		minfo = aqm_grid(domain=domain,minfo=minfo)
		file  = dn_gis_gadm()
	
		case str(/low,minfo.proj) of

			'lcc' : bb = floor(sqrt(minfo.nx*minfo.dx*minfo.ny*minfo.dy)/1e4)
			'll'  : bb = floor(sqrt(minfo.nx*minfo.dx*1e5*minfo.ny*minfo.dy*1e5)/1e4)
			'ell'  : bb = floor(sqrt(minfo.nx*minfo.dx*1e5*minfo.ny*minfo.dy*1e5)/1e4)
			
		endcase
	
		ii = 0>floor(interpol(findgen(n_elements((aa=[0,100,300,1000]))),aa,bb))<(n_elements(aa)-1)
		
		case ii of
	
			0 : map = list(mir_cl(/get,['gray','black','black']),[0.3,0.3,0.3],mir_map_shp(file[2],minfo=minfo,/global),mir_map_shp(file[1],minfo=minfo,/global),mir_map_shp(file[0],minfo=minfo,/global))
			1 : map = list(mir_cl(/get,['gray','black','black']),[0.3,0.3,0.3],mir_map_shp(file[2],minfo=minfo,/global),mir_map_shp(file[1],minfo=minfo,/global),mir_map_shp(file[0],minfo=minfo,/global))
			2 : map = list(mir_cl(/get,['gray','black']),[0.3,0.3],mir_map_shp(file[1],minfo=minfo,/global),mir_map_shp(file[0],minfo=minfo,/global))		
			else : 
		
		endcase

		end

endcase
	
if ~keyword_set(noplot) then for i=0L,n_elements((color=map(0)))-1 do oplot,map(i+2).lon,map(i+2).lat,color=(map(0))[i],thick=(map(1))[i]

return,map
end
