;------------------------------------------------------------------------------- 
;	NAME
;		index
;	
;	PURPOSE
;		IDL function to handle array indices
;
;	USAGE
;		ind = index(ind1,/diagonal)
;		ind = index(ind1,ind2,ind3,/expand)
;		ind = index(ind1,total=total,/complement)
;		ind = index(ind1,ind2,/intersection)
;		ind = index(ind1,ind2,/union)
;		ind = index(ind1,ind2,/difference)
;		ind = index(ind1,ind2,/array)
;		ind = index(ind1,total=total,/mask)
;
;	INPUT
;		ind1,2,3,4 : index arrays
;   
;	KEYWORD
;		diagonal       : index(5,/diagonal) = [0,6,12,18,24]
;		expand         : index([2,3],[5,6],/expand) = [[2,2,3,3,],[5,6,5,6])
;		complement     : index([2,4],/complement,total=6) = [0,1,3,5]
;		intersection   : index([1,2,3],[2,3,4],/intersection) = [2,3]
;		union          : index([1,2,3],[2,3,4],/union) = [1,2,3,4]
;		difference     : index([1,2,3],[2,3,4],/difference) = [1]
;		mask	         : index([4,6],/mask,total=8) = [0,0,0,0,1,0,1,0]
;		contain        : index([3,4,5,6,7,8,9],[5,6,6],/contain) = [2,3]
;		point          : index([3,4,5,6,7,8,9],[5,6,6],/point) = [2,3,3]
;		repeated       : index([1,2,2,3,4,5,5,6],/repeated) = [1,2,5,6] 
;		                 -- index of repeated values
;		repeated/label : index([1,2,2,3,4,5,5,6],/repeated) = [-1,0,0,-1,-1,1,1,-1] 
;		                 -- group and label repeated values from 0 (-1 for uniq values)
;		compare        : index([[0,0,1,1],[0,0,0,1],[0,1,0,0]],/compare) = [2,3,1] 
;		                 -- to find earliest(last) match (see path_check.pro)
;		stack          : index(indgen(3,4),12,3) -> [3,4,3] array (see array.pro for usage)
;		to_3d          : index([2,3],
;
;	AUTHOR		
;		2006-09-27 Hyun Cheol Kim (UH-IMAQS,hkim2@mail.uh.edu,hyuncheol.kim@gmail.com)
;		2006-12-20 added /contain /point
;		2007-11-13 added /complement, removed /inverse
;		2007-11-18 added /stack
;		2008-03-06 added /uniq                      
;		2008-04-02 modified /uniq added /repeated /label
;		2009-04-23 added /to_3d
;------------------------------------------------------------------------------- 

	function index,ind1,ind2,ind3,ind4,                                   $
	         diagonal=diagonal,expand=expand,complement=complement,       $
	         intersection=intersection,union=union,difference=difference, $
	         mask=mask,point=point,contain=contain,stack=stack,           $
	         uniq=uniq,repeated=repeated,label=label,compare=compare,     $
	         count=count,total=total,first=first,last=last,$
	         to_3d=to_3d,dimension=dimension					 
							 
;------------------------------------------------------------------------------- 
 
COMPILE_OPT IDL2

npar = n_params()

case 1 of

	keyword_set(diagonal) : begin

		a = indgen(ind1,ind1)
		b = transpose(a)
							
		return,where((a eq b) eq 1,count)
		  
		end	

	keyword_set(expand) : begin
      
		case npar of
    
			2 : begin
  
				n1 = n_elements(ind1)
				n2 = n_elements(ind2)
    
				result = [[reform(ind1#replicate(1,n2),n1*n2)],[reform(replicate(1,n1)#ind2,n1*n2)]]
  
				end
  
			3 : begin
  
				n1 = n_elements(ind1)
				n2 = n_elements(ind2)
				n3 = n_elements(ind3)
    
				result = [[reform(ind1#replicate(1,n2*n3),n1*n2*n3)],                            $
				         [reform(replicate(1,n1)#reform(ind2#replicate(1,n3),n2*n3),n1*n2*n3)],  $
				         [reform(replicate(1,n1*n2)#ind3,n1*n2*n3)]]
    
				end
  
			4 : begin

				n1 = n_elements(ind1)
				n2 = n_elements(ind2)
				n3 = n_elements(ind3)
				n4 = n_elements(ind4)

				result = [[reform(ind1#replicate(1,n2*n3*n4),n1*n2*n3*n4)]				 ,$
				         [reform(replicate(1,n1)#reform(ind2#replicate(1,n3*n4),n2*n3*n4),n1*n2*n3*n4)],$
				         [reform(replicate(1,n1*n2)#reform(ind3#replicate(1,n4),n3*n4),n1*n2*n3*n4)]   ,$
				         [reform(replicate(1,n1*n2*n3)#ind4,n1*n2*n3*n4)]]
  	  
				end		  
  	  
			else : message,'2-4 index arrays can be allowed'  
                      
		endcase
		
		if result[0] eq -1 then count = 0L else count = n_elements(result)
    
		return,transpose(result)
     
		end
		
	var_set(complement,contain) : begin
	
		;return,where(~index(index(ind1,ind2,/contain),total=n_elements(ind1),/mask),count)	
		return,index(index(ind1,ind2,/contain),/complement,total=n_elements(ind1),count=count)
		
		end
      
	keyword_set(complement) : begin

		if npar gt 1 then message,'ONLY 1 INDEX ARRAY ALLOWED'

		if keyword_set(total) then tindex = indgen(total,/long) else tindex = indgen(max(ind1),/long)

		result = match(tindex,ind1,/difference)

		if result[0] eq -1 then count = 0L else count = n_elements(result)

		return,result

		end		
      
	keyword_set(intersection) : begin
  
		case npar of
    
			2 : result = match(ind1,ind2,/intersection)
			3 : result = match(match(ind1,ind2,/intersection),ind3,/intersection)
			4 : result = match(match(match(ind1,ind2,/intersection),ind3,/intersection),ind4,/intersection)

			else : message,'2-4 index array can be allowed' 	  
    
		endcase
		
		if result[0] eq -1 then count = 0L else count = n_elements(result)		
		
		return,result
      
		end	
      
	keyword_set(union) : begin
  
		case npar of
  
			2 : result = match(ind1,ind2,/union)
			3 : result = match(match(ind1,ind2,/union),ind3,/union)
			4 : result = match(match(match(ind1,ind2,/union),ind3,/union),ind4,/union)

			else : message,'2-4 index array can be allowed' 	  

		endcase
		
		if result[0] eq -1 then count = 0L else count = n_elements(result)		
		
		return,result
       
		end	
      
	keyword_set(difference) : begin
 
		case npar of
  
			2 : result = match(ind1,ind2,/difference)
			3 : result = match(match(ind1,ind2,/difference),ind3,/difference)
			4 : result = match(match(match(ind1,ind2,/difference),ind3,/difference),ind4,/difference)

			else : message,' 2-4 index array can be allowed'	  

		endcase
		
		if result[0] eq -1 then count = 0L else count = n_elements(result)		
		
		return,result
       
		end	
	
	keyword_set(mask) : begin
  
		if npar gt 1 then message,'ONLY 1 ARRAY ALLOWED'
		
		if ~var_set(total) then total = max(ind1)+1 > 1
		
		result = bytarr(total)

		x = where(ind1 ge 0 and ind1 le n_elements(result),count)

		if count gt 0 then result[ind1[x]] = 1B
		
		count = total(result)

		return,result
    
		end	
				      
	keyword_set(point) : begin
		
			; uses string array to handle the cases with duplicated ind1 values
		  
		nind2 = n_elements(ind2)
		
		if check(/is_array,ind2) then result = make_array(size=size(ind2),/string) else result = ''
		
		for iind2=0L,nind2-1 do begin		
		
			if keyword_set(first) then result[iind2] = string((where(ind1 eq ind2[iind2]))[0]) $
			                      else result[iind2] = strjoin(where(ind1 eq ind2[iind2]),' ')
		
		endfor
		
		result2 = strsplit(strjoin(result,' '),/extract)
		
		if n_elements(result2) eq nind2 then begin
		
			result = long(result)
			x = where(result ne -1,count)
			
		endif else begin
		
			x = where(result2 ne '-1',count)
		
		endelse			
				
		return,result

		end  
	  
	keyword_set(contain) : begin
		
		result = array(long(strsplit(strjoin(index(ind1,ind2,/point,first=first),' '),/extract)),/uniq)		
				
		x = where(result ne -1,count)		
		
		if count gt 0 then result = result[x]					
		
		return,result		
		
		;return,where(total(array(ind1,nrow=n_elements(ind2)) eq array(ind2,ncol=n_elements(ind1)),2) gt 0)
		
		end
			
	var_set(repeated,label) : begin
	
		if n_elements(ind1) eq 1 then begin & count = 0 & return,-1 & endif
		
		result = make_array(n_elements(ind1),value=-1)
	
		i2 = array(ind1,/sort)
		
		x = where((shift(i2,1) eq i2) eq 1,nx)
		
		if nx gt 0 then begin
		
			i2 = array(i2[x],/uniq)
			
			count = n_elements(i2)
						
			for i=0L,count-1 do result[index(ind1,i2[i],/contain)] = i
			
			return,result
					
		endif else begin 
		
			count = 0L
			return,result
			
		endelse
		
		end		
	
	keyword_set(repeated) : begin
	
		if n_elements(ind1) eq 1 then begin & count = 0 & return,-1 & endif
			
		i2 = array(ind1,/sort)

		x = where((shift(i2,1) eq i2) eq 1,nx)
		
		if nx gt 0 then begin
		
			i2 = array(i2[x],/uniq)
			return,index(ind1,i2,/contain,count=count,first=first)
		
		endif else begin
		
			count = 0		
			return,-1
		
		endelse
					
		end		
				
	keyword_set(uniq) : return,index(index(ind1,/repeated),/complement,total=n_elements(ind1))
								
	keyword_set(compare) : begin
	
		if var_set(ind2) then begin
		
			return,index(array(ind1,nrow=n_elements(ind2)) eq array(ind2,ncol=n_elements(ind1)),/compare)
		
		endif else begin
			
				if size(ind1,/n_dim) ne 2 then message,'[USAGE] R = INDEX(IND1,IND2,/COMPARE) OR IND1 SHOULD BE 2 DIMENSIONAL'
				
				dim = size(ind1,/dim)
				n1  = dim[0]
				n2  = dim[1]
				
				test = ind1 ne 0			
				ind  = array(indgen(n1),nrow=n2)*test

				x = where(test eq 0,nx)		
				
				if nx gt 0 then begin
				
					if keyword_set(last) then begin
					
						ind[x] = -1 
						result = max(ind,dimension=1)
						
					endif else begin
					
						ind[x] = n1		
						result = min(ind,dimension=1)
						
						x = where(result eq n1,nx)
						if nx gt 0 then result[x] = -1
												
					endelse	
				
				endif
				
				return,result				
									
		endelse

		end											
  
	keyword_set(stack) : begin
	
		; ind1 : original array (usually 2D)
		; ind2 : number of original array
		; ind3 : number of stacks
				
		dims = size(ind1,/dim)
		ndim = n_elements(dims)
		
		return,long(rebin(ind1,[dims,ind3])+array(lindgen(ind3)*ind2,ncol=dims[0],nrow=dims[1]))
							
		end
		
	keyword_set(to_3d) : begin
		
		n1 = n_elements(ind1)
		
		if keyword_set(dimension) then n2 = ind2 else n2 = size(ind2,/dim)
		
		result = reform(rebin(transpose(lindgen(product(n2[2:*]))*product(n2[0:1])),[n1,product(n2[2:*])])+rebin(ind1,[n1,product(n2[2:*])]),[n1,n2[2:*]])
	
		return,long(result)
	
		end								
		
  else : message,'OPTIONS (/DIAGONAL,/EXPAND,/COMPLEMENT,/INTERSECTION,/UNION,/DIFFERENCE,/MASK,/POINT,/CONTAIN)'	

endcase
  
end 

