;-------------------------------------------------------------------------------;
;	NAME
;		idx
;	
;	PURPOSE
;		return indices of array A in comparison with array B
;	
;	USAGE
;		ii = idx(a,b,/contain[,|exclusive])
;		ii = idx(a,b,/point)
;		ii = idx(a,/self[,oself=bb][,oiself=cc])
;
;	INPUT
;		d1 : array A
;		d2 : array B
;
;	OUTPUT
;		oself  : return uniq array with /self keyword
;		oiself : indices of array A to make unique set of A (e.g. oself = d1[ioself])
;	
;	KEYWORD
;		contain   : indices of array A containing any of array B
;		point     : indices of array A that each component of array B points
;		            only returns the first element of available indices 
;		            if all indices are needed, use /all 
;		self      : indices of array A pointing to each own uniq array
;		ucontain  : contain function for uniq arrays (array A & B should be uniq)
;		upoint    : point function for uniq arrays
;		all       : with /point, return all available indices in space-separated string
;		exclusive : return indices of array A not including any of array B
;		mask      : convert indices to mask array (0 or 1)
;		           
;	AUTHOR
;		2010-11-05 Hyun Cheol Kim (hyun.kim@noaa.gov) rewritten from index*.pro
;		2013-03-05 Rewritten from idx.pro
;		2013-09-12 Changed idx2.pro to idx.pro
;		
;	NOTE
;		Flag-shift method by JD Smith to find ucontain
;-------------------------------------------------------------------------------

	function idx,d1,d2,contain=contain,point=point,self=self,oself=oself,oiself=oiself,$
	         exclusive=exclusive,ucontain=ucontain,upoint=upoint,all=all,L64=L64,$
	         count=count,mask=mask,intersect=intersect,list=_list,oidx=oidx
	
;-------------------------------------------------------------------------------

case 1 of

	keyword_set(self) : begin
	
		h = histogram([(oiself=uniq(d1[(a=sort(d1,L64=L64))]))],/binsize,min=0,reverse_indices=ri,L64=L64)
		r = (ri[0:n_elements(h)-1]-ri[0])[sort(a,L64=L64)]
		oself  = [d1[uniq(d1,a)]]
		oiself = ((keyword_set(L64) ? l64indgen(n_elements(d1)) : lindgen(n_elements(d1)))[a])[oiself]
		return,r
		end
												
	keyword_set(ucontain) : begin  ; Flag-shift method by JD Smith

		flag = [replicate(0b,n_elements(d1)),replicate(1b,n_elements(d2))]
		s    = [d1,d2]
		srt  = sort(s,L64=L64)
		s    = s[srt] 
		flag = flag[srt]
		
		wh = where(s eq shift(s,-1) and flag ne shift(flag, -1),cnt,L64=L64)
		if cnt eq 0 then return,-1
      
		return,srt[wh] < srt[wh+1]
		end
		
	keyword_set(upoint) : begin
	
		rr = replicate((keyword_set(L64) ? -1LL : -1L),n_elements(d2))
		
		flag = [replicate(0b,n_elements(d1)),replicate(1b,n_elements(d2))]
		s    = [d1,d2]
		srt  = sort(s,L64=L64)
		s    = s[srt]
		flag = flag[srt]
		
		wh = where(s eq shift(s,-1) and flag eq 0 and shift(flag, -1) eq 1,cnt,L64=L64)
				
		if cnt eq 0 then return,rr
		
		rr[idx(d2,d1,/ucontain,L64=L64)] = srt[wh] < srt[wh+1]
										
		return,rr		
		end
								
	keyword_set(contain) : begin
	
		if array_equal(d1,d2) then begin & count = n_elements(d1) & return,lindgen(count) & endif
				
		i1 = idx(d1,/self,oself=dd1,L64=L64)
		i2 = idx(d2,/self,oself=dd2,L64=L64)		
		ii = idx(dd1,dd2,/ucontain,L64=L64)
		
		if ii[0] eq -1 then begin & count=0L & return,-1 & endif
		
		h = histogram(i1,omin=omin,reverse_indices=ri,L64=L64)
		for i=0L,n_elements(ii)-1 do r = var_set(r) ? [r,ri[ri[ii[i]+omin]:ri[ii[i]+omin+1]-1]] : [ri[ri[ii[i]+omin]:ri[ii[i]+omin+1]-1]]
		
		rr    = r[sort(r,L64=L64)]				
		count = n_elements(rr) gt 1 ? n_elements(rr) : (rr[0] eq -1 ? 0 : 1)
						
		return,rr
		end
		
	var_set(point,_list) : begin
	
		if array_equal(d1,d2) then begin & count = n_elements(d1) & return,lindgen(count) & endif
	
		i1 = idx(d1,/self,oself=dd1,L64=L64)
		i2 = idx(d2,/self,oself=dd2,L64=L64)								
		ii = idx(dd1,dd2,/upoint,L64=L64)
				
		if max(ii) eq -1 then begin & count=0L & return,replicate((keyword_set(L64) ? -1LL : -1L),n_elements(d2)) & endif

		h = histogram(i1,omin=omin,reverse_indices=ri,L64=L64)
		
		rr1 = list()
		rr2 = list()
				
		for i=0L,n_elements(ii)-1 do if ii[i] ne -1 then rr1.add,ri[ri[ii[i]+omin]:ri[ii[i]+omin+1]-1]				
		for i=0L,n_elements(i2)-1 do rr2.add,rr1(i)
					
		return,rr2
		end		
		
	keyword_set(point) : begin
	
		if array_equal(d1,d2) then begin & count = n_elements(d1) & return,lindgen(count) & endif
	
		i1 = idx(d1,/self,oself=dd1,L64=L64)
		i2 = idx(d2,/self,oself=dd2,L64=L64)								
		ii = idx(dd1,dd2,/upoint,L64=L64)
				
		if max(ii) eq -1 then begin & count=0L & return,replicate((keyword_set(L64) ? -1LL : -1L),n_elements(d2)) & endif

		h = histogram(i1,omin=omin,reverse_indices=ri,L64=L64)
		
		rr = keyword_set(all) ? replicate('-1',n_elements(ii)) : replicate((keyword_set(L64) ? -1LL : -1L),n_elements(ii))
		for i=0L,n_elements(ii)-1 do if ii[i] ne -1 then rr[i] = keyword_set(all) ? str(/join,ri[ri[ii[i]+omin]:ri[ii[i]+omin+1]-1]) : ri[ri[ii[i]+omin]]
										
		return,rr[i2]			
		end
		
	keyword_set(intersect) : begin
		oidx = idx(d2,d1[(xx=idx(d1,d2,/contain,count=count,L64=L64))],/contain,L64=L64)
		if count eq 0 then oidx = -1L
		return,xx
		end	
		
	keyword_set(exclusive) : return,where(histogram(idx(d1,d2,/contain,L64=L64),/binsize,min=0,max=n_elements(d1)-1L) eq 0,count,L64=L64)
	keyword_set(mask) : return,histogram([d1],binsize=1,min=0,nbins=((var_set(count) ? count : max(d1)+1)>1),L64=L64) ne 0	
					
endcase		

end
