;-------------------------------------------------------------------------------
;	NAME
;		file_name
;
;	PURPOSE
;		to handle files names
;
;	USAGE
;		r = file_name(file[,ext=ext|add=add|,odir=odir][,/sdir][tree=tree][,/distinguish])
;
;		[example]
;	
;		file = '/data/data03/aqf/ncep/5xpm/para2/20110710/aqm.t12z.metcro2d-'+str(indgen(3),digit=2)+'.ncf'
;
;		print,file_name(file)
;		print,file_name(file,add='-2')
;		print,file_name(file,ext='jpg')
;		print,file_name(file,odir='/data/data2')
;
;		file = '/data/data03/aqf/ncep/5xpm/'+['para2','csum']+'/20110710/aqm.t12z.metcro2d.ncf'
;
;		print,file_name(file,/distinguish)
;
;	INPUT
;		file : file name string array
;		ext  : change file extension
;		add  : add string to body (before extension)
;		odir : change directory
;		tree : add directory name to file name (tree=1,2,3,.....)
;
;	KEYWORD
;		sdir        : return with directory
;		distinguish : use /tree till all filenames are unique
;
;	NOTE
;		default option is to return filename only (use /sdir for full file path)
;	
;	AUTHOR
;		2011-08-09 Hyun Cheol Kim (hyun.kim@noaa.gov)
;-------------------------------------------------------------------------------

	function file_name,file,ext=ext,add=add,odir=odir,sdir=sdir,tree=tree,$
	         distinguish=distinguish,_extra=_ex
	
;-------------------------------------------------------------------------------

nfile = n_elements(file)
img_ext = ['ps','png','gif','jpg','jpeg','tif','tiff']

case 1 of

	keyword_set(distinguish) : begin
		
		ofile = file_basename(file)
		ii    = 1L
		
		while n_elements(ofile) gt n_elements(array(ofile,/uniq)) do ofile = file_name(file,tree=ii++)
													
		end
		
	nfile eq 1 : begin		
	
		name = {dir:file_dirname(file),base:file_basename(file),odir:''}
		x    = strpos(name.base,'.',/reverse_search)
		name = struct(/merge,name,{body:x[0] eq -1 ? name.base : strmid(name.base,0,x),ext:x[0] eq -1 ? '' : strmid(name.base,x+1)})

		if var_set(add)  then name.body = name.body+str(add)
		if var_set(ext)  then name.ext  = str(ext)		
		if struct_read(_ex,'png') then name.ext = name.ext eq '' ? 'png' : (idx(str(/low,name.ext),img_ext,/contain))[0] eq -1 ? name.ext+'.png' : 'png'
		if struct_read(_ex,'ps')  then name.ext = name.ext eq '' ? 'ps'  : (idx(str(/low,name.ext),img_ext,/contain))[0] eq -1 ? name.ext+'.ps'  : 'ps'
		if struct_read(_ex,'gif') then name.ext = name.ext eq '' ? 'gif' : (idx(str(/low,name.ext),img_ext,/contain))[0] eq -1 ? name.ext+'.gif' : 'gif'								
		if var_set(odir) then name.odir = str(odir)		
		if var_set(sdir) then name.odir = name.dir

		ofile = path_check(name.odir,name.ext eq '' ? name.body : strjoin([name.body,name.ext],'.'))

		if keyword_set(tree) then ofile = strjoin([reverse((reverse(strsplit(name.dir,path_sep(),/extract)))[0:tree-1]),ofile],'-')
		
		end
		
	else : begin		
	
		ofile = strarr(nfile)
	
		for ifile=0L,nfile-1 do ofile[ifile] = file_name(file[ifile],ext=ext,add=add,odir=odir,sdir=sdir,tree=tree)
	
		end

endcase

return,ofile

end
