;------------------------------------------------------------------------------- 
;	NAME
;		date
;
;	PURPOSE
;		IDL function to handle date string(s)
;
;	USAGE 
;		r = date(date,/cut[,/year,/month,/day,/hour,/min,/sec][oyear=oyear,omonth=omonth,oday=oday,ohour=ohour,omin=omin,osec=osec)
;		r = date(stime,etime,/count[,/month,/day,/hour,/min,/sec])
;		r = date(stime,etime,/make[,/year,/month,/day,/hour,/min,/sec])
;		r = date(stime,n,/incr[,/year,/month,/day,/hour,/min,/sec])
;
;		EXAMPLE	
;		print,date(/today)
;		print,date(/today,/year,/month,/day,/hour)
;		print,date(2014070100,20140723,/make)
;		aa = date(2014070100,10,/make)
;		print,aa
;		print,date(aa)
;		print,date(aa,/iso)
;		print,date(aa,/hh)
;		print,date(aa,/cut,/year,/month)
;		print,date(aa,'y m d h i',/cut)
;		print,date(aa,'%Y-xx-%m-%d',/format)
;		print,date(aa,3,/incr)
;		print,date(aa,[3,4],/incr)
;		print,date(aa,/julian)
;		print,date(201407,/nday_of_month)
;		print,date(201407,/eday_of_month)
;		print,date(201407,/daily)
;		print,date(20140702,/hourly)
;		print,date(aa,/timestamp)
;		print,date(aa,/tflag_to_julday)
;		print,date(aa,/t2j)
;		print,date(date(aa,/t2j),/j2t)
;		print,date(aa,'utc','est',/timezone)
;		print,date(aa,-100,/local_hour)
;		print,date(aa,[-100,-90,-80],/local_hour)
;		print,date(1,[-100,-90,-80],/lon_to_timezone)
;		print,date(20140701,20140731,/count)
;		print,date(20140701,20140731,/count,/hour)
;		print,date(date(20140702,20,/make),/weekday)
;		print,date(date(20140702,20,/make),/weekday,/name)
;
;	INPUT
;		data1 : stime
;		data2 : etime or ntime or timezone string
;		data3 : timezone string or nth day number of month
;   
;	OUTPUT
;		oyear/omonth/oday/ohour/omin/osec/ojday
;
;	KEYWORD
;		cut	      : cut date
;		make	      : make consecutive date string array
;		incr	      : increase time step(s)
;		count         : count time
;		today         : return today
;		julian        : return julian day
;		nday_of_month : return total day's number of given month
;		weekday       : return weekday name of given date
;		nth_weekday   : return nth weekday of a month
;		dst	      : check day light saving time
;		timezone      : convert timezone
;		name	      : with /weekday, return weekday name
;		short         : with /weekday, return short weekday name 
;		format        : %Y:YYYY %y:YY %m:month %b:month (name) %B: month (full) %d: day %H: hour %M: min %S: sec %J: Julday (YYYYJJJ) %j: Julday (JJJ)
;
;	AUTHOR
;		
;		2013-06-06 rewritten from date.pro
;		2014-01-17 added /tflag_of_month[,/day]
;		2014-08-06 minor changes
;		2015-04-06 added /beef
;------------------------------------------------------------------------------- 

	function date,_dd1,_dd2,_dd3,  $
	         year=year,month=month,day=day,hour=hour,min=min,sec=sec,$
		 oyear=oyear,omonth=omonth,oday=oday,ohour=ohour,omin=omin,osec=osec,ojday=ojday, $ 
	         _extra=_ex 
		 
;------------------------------------------------------------------------------- 
   
if var_set(_dd1) then begin
	
	data_is_array = size(_dd1,/n_dim) ne 0 

		; check if input is julday or tflag
	
	case size(_dd1,/type) of
	
		5 : begin
		
			caldat,_dd1,imo,idy,iyr,ihr,imi,ise
			
			iyr = str(iyr,digit=4)
			imo = str(imo,digit=2)
			idy = str(idy,digit=2)
			ihr = str(ihr,digit=2)
			imi = str(imi,digit=2)
			ise = str(fix(ise),digit=2)
			ijd = _dd1 			
			dd  = iyr+imo+idy+ihr+imi+ise
			len = strlen(dd)
			end
			
		else : begin
		
			dd  = str(_dd1)
			len = strlen(dd)
									
			xjul = where(len eq 7,nxjul)
			if nxjul gt 0 then begin
				dd[xjul] = date(/incr,/day,strmid(dd[xjul],0,4)+'0101',fix(strmid(dd[xjul],4,3))-1)    
				len[xjul] = 8    
			endif
			
			dd  = dd+string(0,format='(I16.16)')						
			iyr = strmid(dd,0,4)
			imo = data_check((imo=strmid(dd,4,2)),imo eq '00','01')			
			idy = data_check((idy=strmid(dd,6,2)),idy eq '00','01')
			ihr = strmid(dd,8,2)
			imi = strmid(dd,10,2)
			ise = strmid(dd,12,2) 
			ijd = julday(data_check(imo,imo eq '00','01'),data_check(idy,idy eq '00','01'),iyr,ihr,imi,ise)				

			end	
	
	endcase
  
endif  

	; main
      
case 1 of
	
	struct_set(_ex,'weekday_names') : return,['Sunday','Monday','Tuesday','Wednesday','Thursday','Friday','Saturday']
	struct_set(_ex,'month_names') : return,['January','February','March','April','May','June','July','August','Septempber','Octobor','November','December']
	
	struct_set(_ex,'format') : begin

		result = strarr(n_elements(dd))
		format = str(var_set(_dd2) ? _dd2 : format,/separate)
		
		for iformat=0L,n_elements(format)-1 do begin
		
			case format[iformat] of
			
				'%Y' : result += iyr
				'%y' : result += strmid(iyr,2,2)
				'%m' : result += imo
				'%b' : result += str(fix(imo),/month_to_name)
				'%B' : result += str(fix(imo),/month_to_name,/full)
				'%d' : result += idy  
				'%H' : result += ihr
				'%M' : result += imi
				'%S' : result += ise
				'%J' : result += date(_dd1,/julian)
				'%j' : result += date(_dd1,/julian,/day)				
				else : result += format[iformat]
			
			endcase
		
		endfor
		
		if data_is_array or n_elements(result) gt 1 then return,result else return,result[0]

		end		
			
	struct_set(_ex,'iso') : return,iyr+'-'+imo+'-'+idy+' '+ihr+':'+imi+':'+ise							
	struct_set(_ex,'hh')  : return,iyr+imo+idy+ihr
	
	struct_set(_ex,'cut') : begin
  
		result = strarr(n_elements(dd))
								
		if var_set(_dd2) then begin
			
			if n_elements(_dd2) eq 1 then cut = strsplit(_dd2,/extract) else cut = _dd2
		
		endif else begin
		
			if var_set(_ex.cut,/keyword_only) then cut = ''					
			if keyword_set(year)  then cut += ' year'
			if keyword_set(month) then cut += ' month' 
			if keyword_set(day)   then cut += ' day' 
			if keyword_set(hour)  then cut += ' hour' 
			if keyword_set(min)   then cut += ' min' 
			if keyword_set(sec)   then cut += ' sec' 
						
			cut = strsplit(cut,/extract)
				
		endelse

		for icut=0L,n_elements(cut)-1 do begin
		
			case str(/low,cut[icut]) of
			
				'year'  : result += iyr
				'month' : result += imo
				'day'   : result += idy
				'hour'  : result += ihr
				'min'   : result += imi
				'sec'   : result += ise
				'y'     : result += iyr
				'm'     : result += imo
				'd'     : result += idy
				'h'     : result += ihr
				'i'     : result += imi
				's'     : result += ise
		
				else : ;message,/info,'[OPTIONS] year/y month/m day/d hour/h min/i sec/s'
			
			endcase
		
		endfor

		oyear  = iyr
		omonth = imo
		oday   = idy
		ohour  = ihr
		omin   = imi
		osec   = ise
    
		if data_is_array or n_elements(result) gt 1 then return,result else return,result[0]
      
		end
	  
	struct_set(_ex,'count') : begin
  
		lens2	= strlen(_dd2) 
		data2	= str(_dd2)+string(0,format='(I16.16)')
		ndata2  = n_elements(data2) 

		iyr2 = strmid(data2,0,4)
		imo2 = strmid(data2,4,2)
		idy2 = strmid(data2,6,2)
		ihr2 = strmid(data2,8,2)
		imi2 = strmid(data2,10,2)
		ise2 = strmid(data2,12,2) 
    	    
		maxlen  = max([strlen(str(_dd1)),strlen(str(_dd2))])

		case 1 of

			keyword_set(year)  : rr = long64(iyr2)-long64(iyr)
			keyword_set(month) : rr = (long64(iyr2)-long64(iyr))*12+(long64(imo2)-long64(imo))
			keyword_set(day)   : rr = long64(julday(imo2,idy2,iyr2)-long64(julday(imo,idy,iyr)))			  
			keyword_set(hour)  : rr = date(_dd1,_dd2,/count,/day )*24L+(long64(ihr2)-long64(ihr))
			keyword_set(min)   : rr = date(_dd1,_dd2,/count,/hour)*60L+(long64(imi2)-long64(imi))
			keyword_set(sec)   : rr = date(_dd1,_dd2,/count,/min )*60L+(long64(ise2)-long64(ise))
			maxlen eq 4	   : rr = long64(iyr2)-long64(iyr)
			maxlen eq 6	   : rr = (long64(iyr2)-long64(iyr))*12+(long64(imo2)-long64(imo))
			maxlen eq 8	   : rr = long64(julday(imo2,idy2,iyr2)-long64(julday(imo,idy,iyr)))			
			maxlen eq 10	   : rr = date(_dd1,_dd2,/count,/day )*24L+(long64(ihr2)-long64(ihr))
			maxlen eq 12	   : rr = date(_dd1,_dd2,/count,/hour)*60L+(long64(imi2)-long64(imi))
			maxlen eq 14	   : rr = date(_dd1,_dd2,/count,/min )*60L+(long64(ise2)-long64(ise))												  
						
			else : message,str(/join,'[USAGE] N = DATE(/COUNT,DATE1,DATE2[,/YEAR,/MONTH,/DAY,/HOUR,/MIN,/SEC')
    
		endcase
    
		if data_is_array or n_elements(rr) gt 1 then return,rr else return,rr[0]  
		end
	  
	struct_set(_ex,'make') : begin
	
		stime = str(_dd1[0])
		etime = long64(str(_dd2[0]))
		step  = var_set(_dd3) ? _dd3 : 1
										
		case 1 of
	
			keyword_set(year)   : ntime = etime gt 9999 ? date(stime,etime,/count,/year) +1 : abs(etime)
			keyword_set(month)  : ntime = etime gt 9999 ? date(stime,etime,/count,/month)+1 : abs(etime)
			keyword_set(day)    : ntime = etime gt 9999 ? date(stime,etime,/count,/day)  +1 : abs(etime)
			keyword_set(hour)   : ntime = etime gt 9999 ? date(stime,etime,/count,/hour) +1 : abs(etime)
			keyword_set(min)    : ntime = etime gt 9999 ? date(stime,etime,/count,/min)  +1 : abs(etime)
			keyword_set(sec)    : ntime = etime gt 9999 ? date(stime,etime,/count,/sec)  +1 : abs(etime)			
			strlen(stime) eq 4  : ntime = etime gt 9999 ? date(stime,etime,/count,/year) +1 : abs(etime)
			strlen(stime) eq 6  : ntime = etime gt 9999 ? date(stime,etime,/count,/month)+1 : abs(etime)
			strlen(stime) eq 8  : ntime = etime gt 9999 ? date(stime,etime,/count,/day)  +1 : abs(etime)
			strlen(stime) eq 10 : ntime = etime gt 9999 ? date(stime,etime,/count,/hour) +1 : abs(etime)											
			strlen(stime) eq 12 : ntime = etime gt 9999 ? date(stime,etime,/count,/min)  +1 : abs(etime)
			strlen(stime) eq 14 : ntime = etime gt 9999 ? date(stime,etime,/count,/sec)  +1 : abs(etime)		     
								
			else : message,'[OPTIONS] /YEAR /MONTH /DAY /HOUR /MIN /SEC'
						
		endcase
				
		
		if step eq 1 then result = date(stime,lindgen(ntime)*(etime/abs(etime)),/incr,year=year,month=month,day=day,hour=hour,min=min,sec=sec) $
		             else result = date(stime,lindgen(ntime/step+1)*(etime/abs(etime))*step,/incr,year=year,month=month,day=day,hour=hour,min=min,sec=sec)
	
		xx = date(/cut,result,oyear=oyear,omonth=omonth,oday=oday,ohour=ohour,omin=omin,osec=osec)
	
		return,result					                 	  
		end
				
	struct_set(_ex,'incr') : begin
	
		len = max(strlen(str(_dd1)))
								
		case 1 of
		
			keyword_set(year)  : rr = str(long(iyr)+_dd2)+imo+idy+ihr+imi+ise
			keyword_set(month) : rr = str(fix((aa=(long(iyr)*12+long(imo)-1)+long(_dd2))/12),digit=4)+str((aa mod 12)+1,digit=2)+idy+ihr+imi+ise
			keyword_set(day)   : rr = date_j2t(date_t2j(_dd1)+_dd2)
			keyword_set(hour)  : rr = date_j2t(date_t2j(_dd1)+_dd2/(24d))
			keyword_set(min )  : rr = date_j2t(date_t2j(_dd1)+_dd2/(24d*60d))
			keyword_set(sec )  : rr = date_j2t(date_t2j(_dd1)+_dd2/(24d*60d*60d))
			len eq 4           : rr = str(long(iyr)+_dd2)+imo+idy+ihr+imi+ise
			len eq 6           : rr = str(fix((aa=(long(iyr)*12+long(imo)-1)+long(_dd2))/12),digit=4)+str((aa mod 12)+1,digit=2)+idy+ihr+imi+ise
			len eq 8           : rr = date_j2t(date_t2j(_dd1)+_dd2)
			len eq 10          : rr = date_j2t(date_t2j(_dd1)+_dd2/(24d))
			len eq 12          : rr = date_j2t(date_t2j(_dd1)+_dd2/(24d*60d))
			len eq 14          : rr = date_j2t(date_t2j(_dd1)+_dd2/(24d*60d*60d))
					
		endcase				
			
		xx = date(/cut,rr,oyear=oyear,omonth=omonth,oday=oday,ohour=ohour,omin=omin,osec=osec)				
		rr = strmid(rr,0,max(len))																														     
		if data_is_array or n_elements(rr) gt 1 then return,rr else return,rr[0] 		
		end
		
	struct_set(_ex,'beef') : return,date(date(min(_dd1),-((beef=_ex.beef)),/incr,/day),date(max(_dd1),beef,/incr,/day),/make)
						  
	struct_set(_ex,'today') : begin
  
		today  = bin_date(systime(utc=struct_read(_ex,'utc')))
			
		oyear  = str(today[0],digit=4)
		omonth = str(today[1],digit=2)
		oday   = str(today[2],digit=2)
		ohour  = str(today[3],digit=2)
		omin   = str(today[4],digit=2)
		osec   = str(today[5],digit=2)

		result = ''
		if keyword_set(year)  then result = result + oyear
		if keyword_set(month) then result = result + omonth
		if keyword_set(day)   then result = result + oday
		if keyword_set(hour)  then result = result + ohour
		if keyword_set(min)   then result = result + omin
		if keyword_set(sec)   then result = result + osec

		if result eq '' then return,oyear+omonth+oday else return,result[0]
      
		end
	
		;.............................................................................
  
	struct_set(_ex,'julian') : begin
  
		jday  = str(date(/count,/day,iyr+'0101',iyr+imo+idy)+1,digit=3)
		oyear = iyr
		ojday = jday
				    
		if keyword_set(day) then return,jday else return,iyr+jday    
		end
	  			
	struct_set(_ex,'nday_of_month') : return,date(iyr+imo+'01',date(iyr+imo,1,/incr,/month)+'01',/count,/day)
	struct_set(_ex,'eday_of_month') : return,date(date(iyr+imo,1,/incr,/month)+'01',-1,/incr,/day)

	struct_set(_ex,'daily') : begin
	
		for i=0L,n_elements(_dd1)-1 do begin
	
			case strlen(str(_dd1[i])) of

				4 : rr = [var_set(rr) ? rr : !null,date(str(_dd1[i])+'0101',str(_dd1[i])+'1231',/make)]
				6 : rr = [var_set(rr) ? rr : !null,date(str(_dd1[i])+'01',date(date(_dd1[i],1,/incr)+'01',-1,/incr),/make)]
				else : rr = [var_set(rr) ? rr : !null, ary(date(_dd1[i],/cut,/year,/month,/day),/uniq)]
	
			endcase	
		
		endfor
		
		return,rr		
		end
		
	struct_set(_ex,'hourly') : begin
	
		for i=0L,n_elements(_dd1)-1 do begin
			
			case strlen(str(_dd1[i])) of
		
				4 : rr = [var_set(rr) ? rr : !null,date(str(_dd1[i])+'010100',str(_dd1[i])+'123123',/make)]
				6 : rr = [var_set(rr) ? rr : !null,date(str(_dd1[i])+'0100',date(date(_dd1[i],1,/incr)+'0123',-1,/incr,/day),/make,/hour)]
				8 : rr = [var_set(rr) ? rr : !null,date(str(_dd1[i])+'00',str(_dd1[i])+'23',/make)]
				else : rr = [var_set(rr) ? rr : !null, ary(date(_dd1[i],/hh),/uniq)]
			
			endcase
		
		endfor
		
		return,rr
		end
				
	struct_set(_ex,'weekday') : begin
	
		weekday_names = date(/weekday_names)
		iweekday = (julday(imo,idy,iyr)+1) mod 7
		
		case 1 of
		
			struct_set(_ex,['name','short']) : return,strmid(weekday_names[iweekday],0,3)
			struct_set(_ex,'name')           : return,weekday_names[iweekday]
			else : return,iweekday
		
		endcase		
		end
			
	struct_set(_ex,'nth_weekday') : begin
	
		weekday_names = date(/weekday_names)
		cmonth = date(iyr+imo+'01',date(iyr+imo,/nday_of_month),/make)
	
		iw  = where(date(cmonth,/weekday) eq idx(strupcase(strmid(weekday_names,0,3)),strupcase(str(_dd2)),/point))
					
		if keyword_set(backward) then iw = reverse(iw)
		
		return,cmonth[iw[_dd3-1]]									
		end
		  
	struct_set(_ex,'dst') : begin
	
		; Daylight Saving Time at 2:00 a.m. on the second Sunday in March and reverts to 
		; standard time on the first Sunday in November
		; http://webexhibits.org/daylightsaving/b.html	
					
		dst0 = long(date(iyr+'03','Sun',2,/nth_weekday)+'02')
		dst1 = long(date(iyr+'11','Sun',1,/nth_weekday)+'02')		
		
		before_2007 = where(long(iyr) lt 2007,nv2)
		
		if nv2 gt 0 then begin
		
	  	dst0[before_2007] = long(date(iyr+'04','Sun',1,/nth_weekday)+'02')
	  	dst1[before_2007] = long(date(iyr+'10','Sun',1,/nth_weekday,/backward)+'02')		
		
		endif
		
		at_2007 = where(long(iyr) eq 2007,nv3)
		
		if nv3 gt 0 then begin
		
	  	dst0[at_2007] = long(date(iyr+'04','Sun',1,/nth_weekday)+'02')
	  	dst1[at_2007] = long(date(iyr+'11','Sun',1,/nth_weekday)+'02')		
				
		endif
												
		ctime = long(iyr+imo+idy+ihr)
		
		return,ctime ge dst0 and ctime lt dst1			
		end	

	struct_set(_ex,'timezone') : begin
	
		zone = {name:['UTC','GMT','Z','AST','ADT','EST','EDT','CST','CDT','MST','MDT','PST','PDT','HST','AKST','AKDT','BST','IST','WET','WEST','CET','CEST','EET','EEST','MSK','MSD','GST','CHST','KST','MYT'],$
		        diff:[    0,    0,  0,   -4,   -4,   -5,   -5,   -6,   -6,   -7,   -7,   -8,   -8,  -10,    -9,    -9,    0,    0,    0,     0,    1,     1,    2,     2,    3,   3,   10,    10 ,   9,    8],$
		        dst :[    0,    0,  0,    0,    1,    0,    1,    0,    1,    0,    1,    0,    1,    0,     0,     1,    1,    1,    0,     1,    0,     1,    0,     1,    0,   1,    0,    0  ,   0,    0 ]}
												
		ix1 = idx(zone.name,strupcase(str(_dd2)),/point)
		ix2 = idx(zone.name,strupcase(str(_dd3)),/point)
		
		diff1 = zone.diff[ix1[0]]+(zone.dst[ix1[0]] eq 0 ? 0 : date(_dd1,/dst))
		diff2 = zone.diff[ix2[0]]+(zone.dst[ix2[0]] eq 0 ? 0 : date(_dd1,/dst))

		return,strmid(date(date(_dd1,/tflag_to_julday)-diff1/(24d)+diff2/(24d),/julday_to_tflag),0,max(len))	
		end
		
	struct_set(_ex,'lon_to_timezone') : return,round(((lon=var_set(_dd2,ina=_dd1) mod 360)*(lon le 180)+(lon-360)*(lon gt 180))/15.)
	struct_set(_ex,'local_hour') : return,(fix(ihr)+round(((lon=_dd2 mod 360)*(lon le 180)+(lon-360)*(lon gt 180))/15.)+24 ) mod 24
		
	struct_set(_ex,'timestamp') : begin
		
		len = max(strlen(str(_dd1)))
	
		case strlen(str(_dd1[0])) of
		
			keyword_set(year)  : return,_dd1
			keyword_set(month) : return,(date(/month_names))[long(imo)-1]+' '+iyr
			keyword_set(day)   : return,idy+' '+ (date(/month_names))[long(imo)-1]+' '+iyr
			keyword_set(hour)  : return,idy+' '+ (date(/month_names))[long(imo)-1]+' '+iyr+' '+ihr+':'+imi
			keyword_set(min)   : return,idy+' '+ (date(/month_names))[long(imo)-1]+' '+iyr+' '+ihr+':'+imi

			len eq 4  : return,_dd1
			len eq 6  : return,(date(/month_names))[long(imo)-1]+' '+iyr
			len eq 8  : return,idy+' '+ (date(/month_names))[long(imo)-1]+' '+iyr
			len eq 10 : return,idy+' '+ (date(/month_names))[long(imo)-1]+' '+iyr+' '+ihr+':'+imi
			len eq 12 : return,idy+' '+ (date(/month_names))[long(imo)-1]+' '+iyr+' '+ihr+':'+imi
		endcase			
		end	
			
	struct_set(_ex,'tflag_to_julday') : return,julday(data_check(imo,imo eq '00','01'),data_check(idy,idy eq '00','01'),iyr,ihr,imi,ise)
	struct_set(_ex,'t2j') : return,julday(data_check(imo,imo eq '00','01'),data_check(idy,idy eq '00','01'),iyr,ihr,imi,ise)
	
	struct_set(_ex,'julday_to_tflag') : return,dd
	struct_set(_ex,'j2t') : return,dd
	  
	else : if data_is_array then return,strmid(dd,0,8) else return,(strmid(dd,0,8))[0]
        
endcase 
  
end 
