;-------------------------------------------------------------------------------
;	NAME
;		unit : IDL function to convert units
;
;	USAGE
;		r = unit(data,unit1,unit2)
;
;	INPUT
;		data  : data array
;		unit1 : original unit
;		unit2 : target unit
;
;	AUTHOR
;		2007-09-20 Hyun Cheol Kim (hyun.kim@noaa.gov)
;-------------------------------------------------------------------------------

 function cv_unit_constant,unit

;-------------------------------------------------------------------------------

COMPILE_OPT IDL2

case strupcase(str(unit)) of

	'C'		  : c = 1.	      ; temperature

	'M'		  : c = 1.	      ; length
	'CM'		  : c = 1e-2
	'MM'		  : c = 1e-3
	'INCH'  	  : c = 0.0254
	'KM'		  : c = 1e3
	'FT'		  : c = 0.3048
	'FEET'  	  : c = 0.3048
	'FOOT'  	  : c = 0.3048
	'YARD'  	  : c = 0.9144 
	'MILE'  	  : c = 1609.344
	'FURLONGS'	  : c = 201.168
	'CHAINS'	  : c = 20.1167999

	'1/M'		  : c = 1.	      ; inverse length
	'1/MILE'	  : c = 6.21375e-4
	'1/KM'  	  : c = 1e-3
  
	'KG'		  : c = 1.	      ; mass
	'G'		  : c = 1e-3
	'MG'		  : c = 1e-6
	'TON'		  : c = 1016.064
	'CWT'		  : c = 50.8032
	'LB'		  : c = 0.4536
	'USTON' 	  : c = 907.2
	'QUARTERS'	  : c = 12.7008
	'STONE' 	  : c = 6.35
	'OZ'		  : c = 0.0283495
	'DRAMS' 	  : c = 0.00177184375
	'GRAINS'	  : c = 0.0000648
	'TONNE' 	  : c = 1e3
  
	'KM2'		  : c = 1.	      ; area
	'ACRE'  	  : c = 4046.8564224
	'HECTARES'	  : c = 1e4
	'ARES'  	  : c = 1e2
  
	'M3'		  : c = 1.	      ; volume
	'GALLONS'	  : c = 0.0045461
	'QUARTS'	  : c = 0.001136525
	'PINTS' 	  : c = 0.0005682625
	'LITER' 	  : c = 1e-3
	'ML'		  : c = 1e-6
  
	'PPM'		  : c = 1e-6	      ; count
	'PARTS PER MILLION' : c = 1e-6
	
	'PPMV'  	  : c = 1e-6
	'PPB'		  : c = 1e-9
	'PARTS PER BILLION' : c = 1e-9
	
	'PPBV'  	  : c = 1e-9  
	'PPT'             : c = 1e-12	 
	'PPTV'            : c = 1e-12
	'PERCENT'         : c = 1e-2
	'%'		  : c = 1e-2

	'KG/KG' 	  : c = 1.
	'G/KG'  	  : c = 1e-3
	'MG/KG' 	  : c = 1e-6
	'MOL/MOL'	  : c = 1.
    
	'MICROGRAMS/M**3' : c = 1e-9	      ; concentration
	'MICROGRAMS/M3'   : c = 1e-9
	'MG/M3' 	  : c = 1e-6
	'UG/M3' 	  : c = 1e-9
	'UG/M**3'	  : c = 1e-9
	'KG/M3'           : c = 1.
	'KG/M**3'         : c = 1.
	      
	'PASCAL'	  : c = 1.	      ; pressure
	'NEWTON/M2'	  : c = 1.
	'N/M2'  	  : c = 1.
	'PA'		  : c = 1.
	'MB'		  : c = 1e2
	'HPA'		  : c = 1e2
	'INCHHG'	  : c = 3386.05395
	'MMHG'  	  : c = 133.32
	'TORR'  	  : c = 133.3

	'M/S'		  : c = 1.	      ; velocity
	'KNOT'  	  : c = 0.5144  
	'MILE/H'	  : c = 0.45	
	'MPH'		  : c = 0.45
	'KM/H'  	  : c = 0.28

	'KG/M/S2'	  : c = 1.	      ; force
	'NEWTON'	  : c = 1.
	'DYNE'  	  : c = 1e-5

	'J'		  : c = 1.	      ; power
	'JOULE' 	  : c = 1.
	'CALORIES'	  : c = 4.186

	'J/S'		  : c = 1.	      ; power rate
	'WATT'  	  : c = 1.
	'HP'		  : c = 746.

	'W/M2'  	  : c = 1.	      ; light intensity
	'LANGLEYS/MIN'    : c = 697.5
	'LY/MIN'	  : c = 697.5

	'RAD'		  : c = 1.	      ; degree
	'RADIAN'	  : c = 1.  
	'DEG'		  : c = !dtor	   
	'DEGREE'	  : c = !dtor
	'PI'		  : c = !pi

	'UNITLESS'	  : c = 1.	      ; unitless
	'FRACTION'	  : c = 1.
	'DIMENSIONLESS'   : c = 1.	      ; 

	else : message,str(/join,'NOT DEFINED',unit)
    
endcase  

  ; return result

return,c

end

;-------------------------------------------------------------------------------

	 function cv_unit,_data,unit1,unit2,missing=missing

;-------------------------------------------------------------------------------

if ~var_set(missing) then missing = -999.
if ~var_set(unit2)   then return,_data
if unit1 eq unit2    then return,_data

data = data_check(_data,_data eq missing,!values.f_nan)

	;
  
case str(/up,unit1) of

	'F'  : data = (data-32.)*5./9.
	'K'  : data = data - 273.15
	else : data = data*cv_unit_constant(unit1)

endcase

	;

case str(/up,unit2) of

	'F'  : data = 32.+9./5.*data
	'K'  : data = data + 273.15
	else : data = data/cv_unit_constant(unit2)

endcase

return,size(_data,/n_dim) eq 0 ? data[0] : data
end
