;-------------------------------------------------------------------------------
;
;	color_index3
;	color_index2
;	color_index
;
;-------------------------------------------------------------------------------
;	NAME
;		color_index3
;
;	PURPOSE
;		to convert data to color index for color bar
;
;	USAGE
;		ct = color_table()
;		ci = color_index(data[,range=][,ct=|div1=,div2])
;
;	INPUT
;		data   : data
;		ct     : output from "color_table"
;		range  : fix data ragne
;		div1   : major divisions
;		div2   : minor divisions
;		option : diff   - use difference plot indexing.
;		         mcolor - missing color (default is 'white')
;		         raw    - allow expolate
;		         group  - use group indexing
;
;	KEYWORD
;		diff   : use difference plot indexing. label will be at the middle of ct div
;		group  : group data by its major division
;		raw    : if set, color index will expolate to outside of color range
;
;	AUTHOR
;		2010-05-14 Hyun Cheol Kim (hyun.kim@noaa.gov)
;		           rewritten from color_index2  
;-------------------------------------------------------------------------------

	function color_index3,data,ct=ct,div1=div1,div2=div2,range=range, $
	         missing=missing,diff=diff,group=group,raw=raw,option=_opt
	
;-------------------------------------------------------------------------------

COMPILE_OPT IDL2

data_dim   = size(data,/dim)
data_max   = abs(max(data,min=data_min)) > abs(data_min)
data_range = data_max - data_min

opt = {diff:1,raw:0,mcolor:'white',group:1}

if var_set(missing) then opt = struct(/merge,opt,{missing:missing})
if var_set(diff)    then opt = struct(/merge,opt,{diff:diff})
if var_set(group)   then opt = struct(/merge,opt,{group:group})
if var_set(raw)     then opt = struct(/merge,opt,{raw:raw})

if struct(/set,_opt) then opt = struct(/merge,opt,_opt)

	; check ct
	
if struct(/set,ct) then ct_div = keyword_set(opt.diff) ? [min(ct.div),ct.div + (ct.div[1:*]-ct.div)/2.,max(ct.div)] : ct.div	
if var_set(ct_div) then ndiv   = keyword_set(opt.diff) ? n_elements(ct_div)-2 : n_elements(ct_div)	

	; set div

if struct(/set,opt,'div')   then div  = opt.div
if struct(/set,opt,'div1')  then div1 = opt.div1
if struct(/set,opt,'div2')  then div2 = opt.div2

if struct(/set,opt,'range') then range = float(opt.range)
if var_set(div) then ndiv = n_elements(div)
if ~var_set(ndiv) then ndiv = 5

if ~var_set(range) then range = data_min eq data_max ? [data_min-1,data_max+1] : [-data_max,data_max]

if ~var_set(div) then div = var_set(div1) ? var_set(div2) ? array([div1,div2],/uniq) : div1 : keyword_set(opt.diff) $
	? [range[0],(findgen(ndiv+1)/(ndiv)*(range[1]-range[0])+range[0]-(range[1]-range[0])/(ndiv)/2.)[1:*],range[1]] $
	: findgen(ndiv)/(ndiv-1)*(range[1]-range[0])+range[0]
	
	; interpol
		
ci = keyword_set(opt.raw) ? interpol(ct_div,div,data) : min(ct_div) > interpol(ct_div,div,data) < max(ct_div)

	; handle missing

if struct(/set,opt,'missing') then begin

	xx = where(data eq opt.missing,nxx)
	
	if nxx gt 0 then ci[xx] = color_load(/get,opt.mcolor)

endif

	; set result
	
ndiv = keyword_set(opt.diff) ? n_elements(div)-2 : n_elements(div)
div  = keyword_set(opt.diff) ? div[1:ndiv] : div

result = {index:reform(ci,data_dim),div1:div,ndiv1:ndiv,div:div,ndiv:ndiv}

if var_set(div2) then result = struct(/merge,result,{div2:div2})

	; group
	
if keyword_set(opt.group) and var_set(div1,div2) then begin

	if n_elements(data_dim) ne 2  then message,'2D DATA NEEDED FOR /GROUP'

	gr = fix(interpol(indgen(n_elements(div1)),ct.div1,ci))+1
	gi = fix(gr ne array(gr,shift=[1,0]))+2*fix(gr ne array(gr,shift=[0,1]))

	result = struct(/merge,result,{gi:gi})

endif	

	;

return,result

end



;-------------------------------------------------------------------------------
;	NAME
;		color_index2 
;
;	PURPOSE
;		IDL function to return color index for color_bar
; 
;	USAGE
;		ct = color_table2()
;		ci = color_index2(data[,range=range][,ct=ct|div1=div1,div2=div2])
;   
;	INPUT
;		data   : data 
;		ct     : output from color_table
;		range  : data range
;		div1   : major divisions
;		div2   : minor divisions
;		option : additional option
;
;	KEYWORD
;		group  : generate group index. see "aqm_plot_tile"
;
;	AUTHOR
;		2007-11-15 Hyun Cheol Kim 
;		2007-12-06 /OPTION ADDED;			   
;		2008-07-28 rewritten to ver 2
;		2008-08-14 div1/div2 improved
;-------------------------------------------------------------------------------

	function color_index2,data,ct=ct,range=range,div1=div1,div2=div2,    $
	         group=group,file=file,raw=raw,missing=missing,mcolor=mcolor,$
	         outside=outside,option=option
 
;-------------------------------------------------------------------------------

COMPILE_OPT IDL2

dim        = size(data,/dim)
;data_min   =  min(data,max=data_max)
data_min   = aqm_array(data,/min,/quiet)
data_max   = aqm_array(data,/max,/quiet)
data_range = data_max - data_min

if struct(/set,option,'range')   then range = option.range
if struct(/set,option,'file')    then file  = option.file
if struct(/set,option,'div')     then div   = option.div
if struct(/set,option,'div1')    then div1  = option.div1
if struct(/set,option,'div2')    then div2  = option.div2
if struct(/set,option,'missing') then missing = option.missing

	; from option
	
case 1 of

	var_set(div,div1,div2) :
	var_set(div,div1)      :
	var_set(div,div2)      : message,str(/join,'[USAGE] ci = color_index2(div=div|div1=div1,div2=div2)')
	var_set(div1,div2)     : div = array([div1,div2],/uniq)
	var_set(div)           :
	var_set(div1)          : div = div1
	var_set(div2)          : message,str(/join,'[USAGE] ci = color_index2(div=div|div1=div1,div2=div2)')
	else : 

endcase	

	; from color_table

if struct(/set,ct) then begin
		
	ndiv = ct.ndiv+1

	if ~var_set(range) then if data_min eq data_max then range = [data_min-1,data_max+1] else range = [data_min,data_max]
	if ~var_set(div)   then div  = findgen(ndiv)/(ndiv-1)*(range[1]-range[0])+range[0]
	if ~var_set(div1)  then div1 = div[index(ct.div,ct.div1,/contain)]
	if ~var_set(div2)  then if struct(/set,ct,'div2') then div2 = div[index(ct.div,ct.div2,/contain)]     
		
	ct_div   = ct.div
	ct_div1  = ct.div1
	ct_range = ct.range

endif

	; from file
		
if var_set(file) then begin
	
	if file_test(file[0]) then begin

		file = struct(/file,file[0])
	
		div  = float(file.i)
		div1 = float(file.i[where(file.d eq 1)])
		div2 = float(file.i[where(file.d ne 1)])
		ndiv = n_elements(div)
					
	endif else message,str(/join,'[NO FILE]',file)				
	
endif

	; get index
	
	
	

if ~var_set(ndiv)     then ndiv = n_elements(div)
if ~var_set(ct_range) then ct_range = [40,255]
if ~var_set(ct_div)   then ct_div = round(findgen(ndiv)/(ndiv-1)*(ct_range[1]-ct_range[0])+ct_range[0])
if ~var_set(ct_div1)  then ct_div1 = ct_div[index(div,div1,/contain)]

if keyword_set(raw) then begin

	ci = interpol(ct_div,div,data)

endif else begin
	
	ci = min(ct_div) > interpol(ct_div,div,data) < max(ct_div)
	
	if var_set(missing) then begin
	
		if var_set(missing,/keyword_only) then missing = -999.
		if var_set(mcolor) then mi = color_load(/get,mcolor) else mi = color_load(/get,'white')
		
		xm = where(data eq missing,nm)
		
		if nm gt 0 then ci[xm] = mi
	
	endif
	
	if var_set(outside) then begin
	
		if var_set(outside,/keyowrd_only) then outside = [color_load(/get,'white'),color_load(/get,'black')] 
		
		xo0 = where(ci lt ct_range[0],no0)
		xo1 = where(ct gt ct_range[1],no1)
		
		if no0 gt 0 then ci[xo0] = outside[0]
		if no1 gt 0 then ci[xo1] = outside[1]
			
	endif

end

	; result
	
result = {index:reform(ci,dim),div:div,div1:div1}

if var_set(div2)  then result = struct(/merge,result,{div2:div2})

	; check group id

if keyword_set(group) then begin

	if n_elements(dim) ne 2  then message,'2D DATA NEEDED FOR /GROUP'

	gr = fix(interpol(indgen(n_elements(div1)),ct_div1,ci))+1
	gi = fix(gr ne array(gr,shift=[1,0]))+2*fix(gr ne array(gr,shift=[0,1]))

	result = struct(/merge,result,{gi:gi})

endif

return,result

end

;-------------------------------------------------------------------------------

	function color_index,data,ct=ct,middle=middle,diff=diff,_extra=_ex

;-------------------------------------------------------------------------------

COMPILE_OPT IDL2

case 1 of

	keyword_set(middle)                        : return,color_index3(data,ct=ct,_extra=_ex)
	keyword_set(struct(/read,option,'middle')) : return,color_index3(data,ct=ct,_extra=_ex)	
	keyword_set(diff)                          : return,color_index3(data,ct=ct,_extra=_ex)
	keyword_set(struct(/read,option,'diff'))   : return,color_index3(data,ct=ct,_extra=_ex)
	struct(/read,ct,'type') eq 2               : return,color_index2(data,ct=ct,_extra=_ex)
	struct(/read,ct,'type') eq 3               : return,color_index3(data,ct=ct,_extra=_ex)
	else                                       : return,color_index2(data,_extra=_ex)

endcase

end
