;-------------------------------------------------------------------------------
;
;	color_bar_div
;	color_bar
;
;-------------------------------------------------------------------------------
;	NAME
;		color_bar_div
;
;	PURPOSE
;		subroutine of "color_bar". add line and label
;
;	USAGE
;		color_bar_div,x,y,label=label
;
;
;	INPUT
;		x      : x-position in device mode
;		y      : y-position in device mode
;		option : labeling    - label direction (0-3, anti-clockwise from bottom)
;		         color       - line color  
;		         fill        - line fraction (both)
;		         fill1       - line fraction (left/bottom)
;		         fill2       - line fraction (right/top)
;		         lcolor      - label color 
;		         shift       - label shift [dx,dy]
;		         shift_fix   - label shift without vertical adjustment to character height
;		         charsize    - label size
;		         orientation - label orientation
;		         alignment   - label alignment
;		         trim        - remove space (front/tail)
;		         nolabel     - do not label	
;
;	AUTHOR
;		2010-05-14 Hyun Cheol Kim (hyun.kim@noaa.gov)
;-------------------------------------------------------------------------------

	pro color_bar_div,_x,_y,label=_label,option=option,oopt=oopt

;-------------------------------------------------------------------------------

COMPILE_OPT IDL2

opt = {labeling:0,color:color_load(/get,'black'),fill1:0.5,fill2:0.5,lcolor:color_load(/get,'black'),shift:[0,0],charsize:1.0,orientation:0,alignment:0.5,trim:1,nolabel:0}

nx = n_elements(_x)
ny = n_elements(_y)

x = array(_x,nrow=ny)
y = array(_y,ncol=nx)

if struct(/set,option) then opt = struct(/merge,opt,option)

if struct(/set,opt,'fill') then opt.fill1 = opt.fill
if struct(/set,opt,'fill') then opt.fill2 = opt.fill

case opt.labeling of

	0 : begin
	
		if struct(/set,option) then opt = struct(/merge,opt,option)
	
		for ix=0L,nx-1 do begin
		
			plots,[x[ix,0],x[ix,1]],[y[ix,0],y[ix,0]+opt.fill1*(y[ix,1]-y[ix,0])],/device,color=opt.color
			plots,[x[ix,0],x[ix,1]],[y[ix,1]-opt.fill2*(y[ix,1]-y[ix,0]),y[ix,1]],/device,color=opt.color
		
		endfor
		
		if keyword_set(_label) then begin
		
			label = array(_label,count=nx)
			
			if struct(/read,opt,'format') then label = str(label,format=opt.format) 
			if struct(/read,opt,'trim')   then label = str(label)
						
			opt.shift = opt.shift + [0,-!d.y_ch_size*opt.charsize]
			
			if struct(/set,opt,'shift_fix') then opt.shift = opt.shift_fix

			if ~keyword_set(opt.nolabel) then xyouts,x[*,0]+opt.shift[0],y[*,0]+opt.shift[1],label,$
			alignment=opt.alignment,charsize=opt.charsize,orientation=opt.orientation,color=opt.lcolor,/device
		
		endif
	
		end
		
	1 : begin
	
		opt = struct(/merge,opt,{alignment:0.5,shift:[5,0]})
		
		if struct(/set,option) then opt = struct(/merge,opt,option)
	
		for iy=0L,ny-1 do begin
		
			plots,[x[0,iy],x[0,iy]+opt.fill1*(x[1,iy]-x[0,iy])],[y[0,iy],y[1,iy]],/device,color=opt.color
			plots,[x[1,iy]-opt.fill1*(x[1,iy]-x[0,iy]),x[1,iy]],[y[0,iy],y[1,iy]],/device,color=opt.color
								
		endfor
		
		if keyword_set(_label) then begin
		
			label = array(_label,count=ny)
			
			if struct(/read,opt,'format') then label = str(label,format=opt.format) 
			if struct(/read,opt,'trim')   then label = str(label)
			
			if opt.alignment eq 0.5 then begin
							
				xyouts,0,0,label[where(strlen(label) eq max(strlen(label)))],charsize=-opt.charsize,width=w,/normal
								
				str_len = w*!d.x_vsize
				
				if str_len gt (!d.x_size-max(x)-opt.shift[0]-3) then begin
									
					opt.charsize = opt.charsize * (!d.x_size-max(x)-opt.shift[0]-3) / str_len 
										
					xyouts,0,0,label[where(strlen(label) eq max(strlen(label)))],charsize=-opt.charsize,width=w,/normal
				
					str_len = w*!d.x_vsize
				
				endif
				
				opt.shift = opt.shift + [str_len/2.,0]
				
				if struct(/set,opt,'shift_fix') then opt.shift = opt.shift_fix
								
				if ~keyword_set(opt.nolabel) then xyouts,x[1,*]+opt.shift[0],y[1,*]-opt.shift[1]-!d.y_ch_size*opt.charsize*0.3,label,$
				alignment=opt.alignment,charsize=opt.charsize,orientation=opt.orientation,color=opt.lcolor,/device

			endif else begin
						
				if ~keyword_set(opt.nolabel) then xyouts,x[1,*]+opt.shift[0],y[1,*]-opt.shift[1]-!d.y_ch_size*opt.charsize*0.3,label,$
				alignment=opt.alignment,charsize=opt.charsize,orientation=opt.orientation,color=opt.lcolor,/device
				
			endelse	
		
		endif
				
		end	
			
	2 : begin
	
		opt = struct(/merge,opt,{alignment:0.5,shift:[0,5]})

		if struct(/set,option) then opt = struct(/merge,opt,option)
	
		for ix=0L,nx-1 do begin
		
			plots,[x[ix,0],x[ix,1]],[y[ix,0],y[ix,0]+opt.fill1*(y[ix,1]-y[ix,0])],/device,color=opt.color
			plots,[x[ix,0],x[ix,1]],[y[ix,1]-opt.fill2*(y[ix,1]-y[ix,0]),y[ix,1]],/device,color=opt.color
		
		endfor
		
		if keyword_set(_label) then begin
		
			label = array(_label,count=nx)
			
			if struct(/read,opt,'format') then label = str(label,format=opt.format) 
			if struct(/read,opt,'trim')   then label = str(label)
			if struct(/set,opt,'shift_fix') then opt.shift = opt.shift_fix

			if ~keyword_set(opt.nolabel) then xyouts,x[*,0]+opt.shift[0],y[*,1]+opt.shift[1],label,$
			alignment=opt.alignment,charsize=opt.charsize,orientation=opt.orientation,color=opt.lcolor,/device
		
		endif
		
		end
		
	3 : begin
	
		opt = struct(/merge,opt,{alignment:0.5,shift:[-5,0]})
		if struct(/set,option) then opt = struct(/merge,opt,option)
	
		for iy=0L,ny-1 do begin
		
			plots,[x[0,iy],x[0,iy]+opt.fill1*(x[1,iy]-x[0,iy])],[y[0,iy],y[1,iy]],/device,color=opt.color
			plots,[x[1,iy]-opt.fill1*(x[1,iy]-x[0,iy]),x[1,iy]],[y[0,iy],y[1,iy]],/device,color=opt.color
								
		endfor
		
		if keyword_set(_label) then begin
		
			label = array(_label,count=ny)
			
			if struct(/read,opt,'format') then label = str(label,format=opt.format) 
			if struct(/read,opt,'trim')   then label = str(label)
			
			if opt.alignment eq 0.5 then begin
							
				xyouts,0,0,label[where(strlen(label) eq max(strlen(label)))],charsize=-opt.charsize,width=w,/normal
				
				str_len = !d.x_vsize
				
				if str_len gt (min(x)+opt.shift[0]-3) then begin
									
					opt.charsize = opt.charsize * (min(x)+opt.shift[0]-3) / str_len 
										
					xyouts,0,0,label[where(strlen(label) eq max(strlen(label)))],charsize=-opt.charsize,width=w,/normal
				
					str_len = w*!d.x_vsize
				
				endif
				
				opt.shift = opt.shift - [str_len/2.,0]
				
				if struct(/set,opt,'shift_fix') then opt.shift = opt.shift_fix
				
				if ~keyword_set(opt.nolabel) then xyouts,x[0,*]+opt.shift[0],y[1,*]-opt.shift[1]-!d.y_ch_size*opt.charsize*0.3,label,$
				alignment=opt.alignment,charsize=opt.charsize,orientation=opt.orientation,color=opt.lcolor,/device

			endif else begin
						
				if ~keyword_set(opt.nolabel) then xyouts,x[1,*]+opt.shift[0],y[1,*]-opt.shift[1]-!d.y_ch_size*opt.charsize*0.3,label,$
				alignment=opt.alignment,charsize=opt.charsize,orientation=opt.orientation,color=opt.lcolor,/device
				
			endelse	
		
		endif
		end								
		
endcase

oopt = opt

end

;-------------------------------------------------------------------------------
;	NAME
;		color_bar
;
;	PURPOSE 
;		
;		IDL function to draw a colorbar
;
;	USAGE
;		cb = color_bar([,position=a,range=a,labeling=][,ct=str,ci=str]
;		     [div1=div1,div2=div2][,/right,/left,/top,/bottom][,title=s/str]
;
;	INPUT
;		position : normal mode position to draw a colorbar
;		range    : color index range (range=[40,255])
;		labeling : labeling direction (0:bottom 1:right 2:top 3:left)
;		ct       : from color_table
;		ci       : from color_index
;		div1     : major division information structure
;		           {index,color,label,lcolor,shift,format,charsize,alignment,orientation,trim}
;		div2     : minor division information structure
;		           {index,color,label,lcolor,shift,format,charsize,alignment,orientation,trim}
;		title    : title legend
;		           opt3 = {text,labeling,shift,alignment,charsize,color,orientation}
;
;	KYEWORD
;		right    : vertical color bar
;		left     : put label left 
;		top      : put label up
;		bottom   : put label at the bottom
;
;	AUTHOR
;		Hyun Cheol Kim 20080725 rewritten from color_bar
;
;-------------------------------------------------------------------------------

	function color_bar,ci=ci,ct=ct,position=p,color=color,range=range,labeling=labeling,$
	         title=title,right=right,left=left,top=top,bottom=bottom,$
	         div1=div1,div2=div2,format=format,option=_opt
				 
;-------------------------------------------------------------------------------

COMPILE_OPT IDL2
				 				
opt  = {p:[0.1,0.05,0.9,0.08],range:[0,!d.table_size],color:color_load(/get,'black'),labeling:1,scalablepixels:(!D.flags AND 1) NE 0}

if var_set(p)      then opt = struct(/merge,opt,{p:p})
if var_set(range)  then opt = struct(/merge,opt,{range:range})
if var_set(color)  then opt = struct(/merge,opt,{color:color})
if var_set(div1)   then opt = struct(/merge,opt,{div1:div1})
if var_set(div2)   then opt = struct(/merge,opt,{div2:div2})
if var_set(title)  then opt = struct(/merge,opt,{title:title})
if var_set(format) then opt = struct(/merge,opt,{format:format})

if (opt.p[2]-opt.p[0])/2.+opt.p[0] lt 0.5 then opt.labeling = 3
if (opt.p[2]-opt.p[0]) gt (opt.p[3]-opt.p[1]) then if (opt.p[3]-opt.p[1])/2.+opt.p[1] gt 0.5 then opt.labeling = 2 else opt.labeling = 0

if keyword_set(bottom) then opt.labeling = 0
if keyword_set(right)  then opt.labeling = 1
if keyword_set(top)    then opt.labeling = 2
if keyword_set(left)   then opt.labeling = 3

if var_set(labeling) then opt = struct(/merge,opt,{labeling:labeling})

if struct(/set,ct,'range') then opt = struct(/merge,opt,{range:ct.range})

if struct(/set,_opt)       then opt = struct(/merge,opt,_opt)
 
  ; construct and display bar
  
if check(opt.labeling,[0,2],/have_any) then begin

	bar = bindgen(!d.table_size) # replicate(1B,20) 
	bar = bytscl(bar,top=(!d.table_size-1 < (opt.range[1]-opt.range[0]))) + opt.range[0]
	
	if opt.scalablepixels then begin
	
		tv,bar,opt.p[0],opt.p[1],xsize=(opt.p[2]-opt.p[0]),ysize=(opt.p[3]-opt.p[1]),/normal 
		
	endif else begin
			
		bar = congrid(bar,ceil((opt.p[2]-opt.p[0])*!d.x_vsize),ceil((opt.p[3]-opt.p[1])*!d.y_vsize),/interp)
		tv,bar,opt.p[0],opt.p[1],/normal
	
	endelse

endif else begin

	bar = replicate(1B,20) # bindgen(!d.table_size)
	bar = bytscl(bar,top=(!d.table_size-1 < (opt.range[1]-1-opt.range[0]))) + opt.range[0]

	if opt.scalablepixels then begin
	
		tv,bar,opt.p[0],opt.p[1],xsize=(opt.p[2]-opt.p[0]),ysize=(opt.p[3]-opt.p[1]),/normal 
		
	endif else begin
			
		bar = congrid(bar,ceil((opt.p[2]-opt.p[0])*!d.x_vsize),ceil((opt.p[3]-opt.p[1])*!d.y_vsize),/interp)
		tv,bar,opt.p[0],opt.p[1],/normal
	
	endelse
			
	;bar = congrid(bar,ceil((opt.p[2]-opt.p[0])*!d.x_vsize),ceil((opt.p[3]-opt.p[1])*!d.y_vsize));,/interp)
	
endelse

	; line & label

case 1 of
	
	struct(/set,ct) : begin
				
		if ~struct(/set,opt,'format') then opt = struct(/merge,opt,{format:struct(/set,ci) ? str(ci.div,/get_format) : ''})
		
		if ~struct(/set,ct,'div1') then break
		
		ndiv1 = n_elements(ct.div1)
		
		if check(/have_any,opt.labeling,[0,2]) then begin
							
			x = (opt.p[0]+float(index(bar[*,0],ct.div1,/point,/first))/n_elements(bar[*,0])*(opt.p[2]-opt.p[0]))*!d.x_vsize
			y = opt.p[[1,3]]*!d.y_vsize	
																
			x[n_elements(x)-1] = opt.p[2]*!d.x_vsize	 
				
			if struct(/set,ci,'div1') then if ndiv1 eq n_elements(ci.div1)+1 then x = ((findgen(ndiv1)/(ndiv1-1)*(opt.p[2]-opt.p[0])+opt.p[0]-(opt.p[2]-opt.p[0])/ndiv1/2.)*!d.x_vsize)[1:*]
						
		endif else begin
		
			x = opt.p[[0,2]]*!d.x_vsize			
			y = (opt.p[1]+float(index(bar[0,*],ct.div1,/point,/first))/n_elements(bar[0,*])*(opt.p[3]-opt.p[1]))*!d.y_vsize	
				
			y[n_elements(y)-1] = opt.p[3]*!d.y_vsize	
				
			if struct(/set,ci,'div1') then if ndiv1 eq n_elements(ci.div1)+1 then y = ((findgen(ndiv1)/(ndiv1-1)*(opt.p[3]-opt.p[1])+opt.p[1]-(opt.p[3]-opt.p[1])/ndiv1/2.)*!d.y_vsize)[1:*]
								
		endelse
					
		opt1   = {labeling:opt.labeling,format:opt.format,fill:struct(/set,ct,'div2') ? 1 : 0.1}
		label1 = struct(/set,ci) ? struct(/read,opt1,'nolabel') ? 0 : struct(/set,ci,'div1') ? ci.div1 : 0 : 0

		if struct(/set,opt,'div1') then opt1 = struct(/merge,opt1,opt.div1)
				
		color_bar_div,x,y,label=label1,option=opt1,oopt=oopt
						
			;
								
		if ~struct(/set,ct,'div2') then break
		
		ndiv2 = n_elements(ct.div2)
										
		if check(/have_any,opt.labeling,[0,2]) then begin
		
			;x = opt.p[0]*!d.x_vsize+index(bar[*,0],ct.div2,/point,/first)
			;x = (opt.p[0]+float(index(bar[*,0],ct.div2,/point,/first))/n_elements(bar[*,0])*(opt.p[2]-opt.p[0]))*!d.x_vsize
			x = (opt.p[0]+float(idx(bar[*,0],ct.div2,/point))/n_elements(bar[*,0])*(opt.p[2]-opt.p[0]))*!d.x_vsize
		
			y = opt.p[[1,3]]*!d.y_vsize	
																				
			if struct(/set,ci,'div2') then if ndiv2 eq n_elements(ci.div2)+1 then x = ((findgen(ndiv2)/(ndiv2-1)*(opt.p[2]-opt.p[0])+opt.p[0]-(opt.p[2]-opt.p[0])/ndiv2/2.)*!d.x_vsize)[1:*]
				
		endif else begin
		
			x = opt.p[[0,2]]*!d.x_vsize			
			;y = opt.p[1]*!d.y_vsize+index(bar[0,*],ct.div2,/point,/first)	
			y = (opt.p[1]+float(index(bar[0,*],ct.div2,/point,/first))/n_elements(bar[0,*])*(opt.p[3]-opt.p[1]))*!d.y_vsize	
						
			if struct(/set,ci,'div2') then if ndiv2 eq n_elements(ci.div2)+1 then y = ((findgen(ndiv2)/(ndiv2-1)*(opt.p[3]-opt.p[1])+opt.p[1]-(opt.p[3]-opt.p[1])/ndiv2/2.)*!d.y_vsize)[1:*]
								
		endelse
												
		opt2   = {labeling:opt.labeling,shift_fix:oopt.shift,format:opt.format,charsize:0.8,lcolor:color_load(/get,'slategray'),fill:0.1}										
		label2 = struct(/set,ci) ? struct(/read,opt2,'nolabel') ? 0 : struct(/set,ci,'div2') ? ci.div2 : 0 : 0
				
		if struct(/set,opt,'div2') then opt2 = struct(/merge,opt2,opt.div2)
		
		color_bar_div,x,y,label=label2,option=opt2
		
		end
			
	else : begin
			
		ndiv1 = struct(/set,opt,'div1.ndiv') ? opt.div1.ndiv :6		
		x1 = check(/have_any,opt.labeling,[0,2]) ? (findgen(ndiv1+1)/ndiv1*(opt.p[2]-opt.p[0])+opt.p[0])*!d.x_size : opt.p[[0,2]]*!d.x_vsize
		y1 = check(/have_any,opt.labeling,[0,2]) ? opt.p[[1,3]]*!d.y_vsize : (findgen(ndiv1+1)/ndiv1*(opt.p[3]-opt.p[1])+opt.p[1])*!d.y_size
				
		opt1 = {labeling:opt.labeling}
								
		if struct(/set,opt,'div1') then opt1 = struct(/merge,opt1,opt.div1)															
		
		color_bar_div,x1,y1,label=struct(/read,opt1,'label'),option=opt1,oopt=oopt
		
			;
						
		ndiv2 = struct(/set,opt,'div2.ndiv') ? opt.div2.ndiv : 12														
		x2 = check(/have_any,opt.labeling,[0,2]) ? index((findgen(ndiv2+1)/ndiv2*(opt.p[2]-opt.p[0])+opt.p[0])*!d.x_vsize,x1,/difference) : opt.p[[0,2]]*!d.x_vsize
		y2 = check(/have_any,opt.labeling,[0,2]) ? opt.p[[1,3]]*!d.y_vsize : index((findgen(ndiv2+1)/ndiv2*(opt.p[3]-opt.p[1])+opt.p[1])*!d.y_size,y1,/difference)	
						
		opt2 = {labeling:opt.labeling,shift_fix:oopt.shift,charsize:0.8,lcolor:color_load(/get,'slategray'),fill:0.1}
								
		if struct(/set,opt,'div2') then opt2 = struct(/merge,opt2,opt.div2)	
				
		color_bar_div,x2,y2,label=struct(/read,opt2,'label'),option=opt2
							
		end		

endcase

plots,opt.p[[0,2,2,0,0]],opt.p[[1,1,3,3,1]],color=opt.color,/normal

	; title

if struct(/set,opt,'title') then begin

	opt3 = {text:'',labeling:5,shift:[0,0],alignment:0,charsize:0.8,color:color_load(/get,'slategray'),orientation:0.}
	opt3 = struct(/update,opt3,opt.title)
	
	case opt3.labeling of
	
		0 : opt3 = struct(/merge,opt3,{x:opt.p[0]*!d.x_size,y:opt.p[1]*!d.y_size-!d.y_ch_size*opt3.charsize,shift:[0,-2]})
		1 : opt3 = struct(/merge,opt3,{x:(opt.p[0]+opt.p[2])/2.*!d.x_size,y:opt.p[1]*!d.y_size-!d.y_ch_size*opt3.charsize,alignment:0.5,shift:[0,-2]})
		2 : opt3 = struct(/merge,opt3,{x:opt.p[2]*!d.x_size,y:opt.p[1]*!d.y_size-!d.y_ch_size*opt3.charsize,shift:[0,-2]})
		3 : opt3 = struct(/merge,opt3,{x:opt.p[2]*!d.x_size,y:(opt.p[3]+opt.p[1])/2.*!d.y_size,shift:[30,0],alignment:0.5,orientation:90})
		4 : opt3 = struct(/merge,opt3,{x:opt.p[2]*!d.x_size,y:opt.p[3]*!d.y_size,shift:[0,5]})		
		5 : opt3 = struct(/merge,opt3,{x:(opt.p[0]+opt.p[2])/2.*!d.x_size,y:opt.p[3]*!d.y_size,shift:[0,5],alignment:0.5})
		6 : opt3 = struct(/merge,opt3,{x:opt.p[0]*!d.x_size,y:opt.p[3]*!d.y_size,shift:[0,5]})
		7 : opt3 = struct(/merge,opt3,{x:opt.p[0]*!d.x_size,y:(opt.p[1]+opt.p[3])/2.*!d.y_size,shift:[-30,0],alignment:0.5,orientation:90})
	
	endcase
	
	opt3 = struct(/update,opt3,opt.title)
		
	xyouts,opt3.x+opt3.shift[0],opt3.y+opt3.shift[1],opt3.text,alignment=opt3.alignment,$
	charsize=opt3.charsize,orientation=opt3.orientation,color=opt3.color,/device
	
endif

end
