;-------------------------------------------------------------------------------
;	NAME
; 		check
;
;	PURPOSE
;		IDL function to check... (see usage)
;
;	USAGE
;		if check(/valid,data1[,index=index,value=value][,more_than=n])  then 
;		if check(/dim,data1,data2,data3,data4[,/type,/value])           then
;		if check(/sign,data1[,/positive,/negetive,/zero][,more_than=n]) then
; 	if check(/inside,x,y,px,py,[index=index])                       then
;
;	INPUT
;		data1-4   : data array
; 	more_than : least number to pass
;
;	OUTPUT
;		value : selected data
;		index : index of selected data
;   
;	KEYWORD
;		valid    : check array 
;		dim      : check arrays' dimentions
;		type	   : check arrays' type (with /dim)
;		value    : check arrays' value (with /dim)
;		sign	   : check sign
;		positive : return 1 if positive
;		negative : return 1 if negative
;		zero	   : return 1 if zero
;		inside   : check if points are inside polygon
; 
;	AUTHOR
;		2007-01-09 Hyun Cheol Kim (hyuncheol.kim@gmail.com, hkim2@mail.uh.edu)
; 
;-------------------------------------------------------------------------------

	function check,data1,data2,data3,data4,data5,                     $                   
	         valid=valid,dim=dim,sign=sign,inside=inside,             $
	         more_than=more_than,value=value,type=type,               $
	         index=index,ci=ci,nci=nci,                               $
	         missing=missing,min_value=min_value,max_value=max_value, $
	         positive=positive,negative=negative,zero=zero,           $
	         lcc=lcc,rot=rot,complement=complement,count=count,       $
	         is_array=is_array,is_number=is_number,                   $
	         have_any=have_any,have_all=have_all,same=same,$
	         n_elements=n_elements,option=option

;-------------------------------------------------------------------------------

COMPILE_OPT IDL2

if ~var_set(missing)   then missing   = -999.
if ~var_set(min_value) then min_value = -900.
if ~var_set(max_value) then max_value = 1e37
if ~var_set(more_than) then more_than = 0

if struct(/set,option,'missing')   then missing = option.missing
if struct(/set,option,'min_value') then min_value = option.min_value
if struct(/set,option,'max_value') then max_value = option.max_value

case 1 of

	keyword_set(valid) : begin
	
		case n_params() of
						
			1 : index = where(data1 ne missing and data1 gt min_value and data1 le max_value,n,complement=ci,ncomplement=nci)
			2 : index = where(data1 ne missing and data1 gt min_value and data1 le max_value and $
			                  data2 ne missing and data2 gt min_value and data2 le max_value,n,complement=ci,ncomplement=nci)
			3 : index = where(data1 ne missing and data1 gt min_value and data1 le max_value and $
			                  data2 ne missing and data2 gt min_value and data2 le max_value and $
						            data3 ne missing and data3 gt min_value and data3 le max_value,n,complement=ci,ncomplement=nci)
			4 : index = where(data1 ne missing and data1 gt min_value and data1 le max_value and $
			                  data2 ne missing and data2 gt min_value and data2 le max_value and $
						            data3 ne missing and data3 gt min_value and data3 le max_value and $
						            data4 ne missing and data4 gt min_value and data4 le max_value,n,complement=ci,ncomplement=nci)
			5 : index = where(data1 ne missing and data1 gt min_value and data1 le max_value and $
			                  data2 ne missing and data2 gt min_value and data2 le max_value and $
						            data3 ne missing and data3 gt min_value and data3 le max_value and $
						            data4 ne missing and data4 gt min_value and data4 le max_value and $
						            data5 ne missing and data5 gt min_value and data5 le max_value,n,complement=ci,ncomplement=nci)
												
			else : message,'USE data1,...,data5'

		endcase		
		
			; get index,nindex, and valid/unvalid values
		
		if keyword_set(complement) then begin
		
			index = ci
			count = nci		
			if nci gt 0 then value = data1[ci]
			
		endif else begin
		
			count = n
			if n gt 0 then value = data1[index]

		endelse
		
		if count gt more_than then return,1B else return,0B		
			
		end
	
	;.............................................................................
	
	keyword_set(dim) : begin
			
		case n_params() of
		
			2 : begin
			
				case 1 of
				
					keyword_set(value) and keyword_set(type) : return,array_equal(data1,data2,/no_typeconv)
					keyword_set(value) : return,array_equal(data1,data2)
					else               : return,array_equal(size(data1,/dim),size(data2,/dim))
				
				endcase
			
			end
			
			3 : result = check(data1,data2,/dim,type=type,value=value)*check(data2,data3,/dim,type=type,value=value)									 
			4 : result = check(data1,data2,data3,/dim,type=type,value=value)*check(data3,data4,/dim,type=type,value=value)									 
			5 : result = check(data1,data2,data3,data4,/dim,type=type,value=value)*check(data4,data5,/dim,type=type,value=value)
									 									 
			else : message,'/DIM AVAILABLE UP TO 5D'
		
		endcase
		
		return,result
	
		end
	
	;.............................................................................
	
	keyword_set(sign) : begin
	
		case 1 of
		
			keyword_set(positive) and keyword_set(zero) : index = where(data1 ge 0,n)
			keyword_set(negative) and keyword_set(zero) : index = where(data1 le 0,n)		
			keyword_set(positive)                       : index = where(data1 gt 0,n)
			keyword_set(negative)                       : index = where(data1 lt 0,n)
			keyword_set(zero)                           : index = where(data1 eq 0,n)
			
			else : message,'/SIGN OPTIONS (/POSITIVE /NEGATIVE /ZERO)'
		
		endcase
		
		if n gt more_than then begin
		
			value = data1[index]
			return,1B
		
		endif else return,0B
	
		end
	
	;.............................................................................
	
	keyword_set(inside) : begin
	
		case n_params() of

			2 : begin
  
				x   = reform(data1[0,*])
				y   = reform(data1[1,*])  
				px  = reform(data2[0,*])
				py  = reform(data2[1,*])
  
				end
  
			4 : begin
 
				x   = reform(data1)
				y   = reform(data2)
				px  = reform(data3)
				py  = reform(data4)  

				end 
  
			else : message,'USE CHECK(/INSIDE,X,Y,PX,PY) OR CHECK(/INSIDE,[X,Y],[PX,PY])'

		endcase
		
			; convert if needed
		
		case 1 of
		
			keyword_set(lcc) : begin
			
				r1 = aqm_map_proj( x, y,/ll,/to_lcc,out1=temp1,out2=temp2)
				r2 = aqm_map_proj(px,py,/ll,/to_lcc,out1=temp3,out2=temp4)
			
				x  = temp1
				y  = temp2
				px = temp3
				py = temp4
			
				end
			
			keyword_set(rot) : begin
			
				r1 = aqm_map_proj( x, y,/ll,/to_rot,out1=temp1,out2=temp2)
				r2 = aqm_map_proj(px,py,/ll,/to_rot,out1=temp3,out2=temp4)
			
				x  = temp1
				y  = temp2
				px = temp3
				py = temp4
										
				end
			
			else : 
		
		endcase
		
		  ; get hull
			
		triangulate,px,py,triangles,hull

		px = [px[hull],px[hull[0]]]
		py = [py[hull],py[hull[0]]]
	
			; check inside
	
		nx  = n_elements(x)
		npx = n_elements(py)
		
		ie  = indgen(npx,/long)
		io  = indgen(npx,/long)+1
		
		x1 = px[ie] # replicate(1,nx) - replicate(1,npx) # reform([x],nx)
		y1 = py[ie] # replicate(1,nx) - replicate(1,npx) # reform([y],nx)
		x2 = px[io] # replicate(1,nx) - replicate(1,npx) # reform([x],nx)
		y2 = py[io] # replicate(1,nx) - replicate(1,npx) # reform([y],nx)
		
		dp = x2*x1 + y1*y2
		cp = x1*y2 - y1*x2
		
		theta = atan(cp,dp)
		
		result = replicate(0L,nx)
		
		index = where(abs(total(theta,1)) gt 0.01,count)
		
		if count gt 0 then result[index] = 1
		
		return,total(result) gt 0
	
		end
	
		;...........................................................................
	
	keyword_set(is_array) : return,(size(data1,/n_dim))[0] ne 0		
	
		;...........................................................................
		
	keyword_set(is_number) : return,total(str(data1,/is_number,/sign)) eq n_elements(data1)
	
		;...........................................................................	
	
	keyword_set(have_any) : begin
	
		x = index(data1,data2,/point,count=n)		
		return,n gt more_than
		
		end
	
		;...........................................................................
		
	keyword_set(have_all) : begin

		x = index(data1,data2,/point,count=n)
		return,n eq n_elements(data2)

		end
		
		;...........................................................................
		
	keyword_set(same) : begin
	
		if var_set(data1) then r = array(data1,/flat)
		if var_set(data2) then r = [r,array(data2,/flat)]
		if var_set(data3) then r = [r,array(data3,/flat)]
		if var_set(data4) then r = [r,array(data4,/flat)]
		if var_set(data5) then r = [r,array(data5,/flat)]
		
		if n_elements(r) eq total(r eq shift(r,1)) then return,1B else return,0B
		
		end		
		
		;...........................................................................
				
	keyword_set(n_elements) : begin
	
		if var_set(data1) then r = n_elements(data1)
		if var_set(data2) then r = [r,n_elements(data2)]
		if var_set(data3) then r = [r,n_elements(data3)]
		if var_set(data4) then r = [r,n_elements(data4)]
		if var_set(data5) then r = [r,n_elements(data5)]
		
		if n_elements(r) eq total(r eq shift(r,1)) then return,1B else return,0B
		
		end		
		
	else : message,str(/join,'[USAGE] /VALID,/DIM,/SIGN,/INSIDE,/IS_ARRAY,/IS_NUMBER,/HAVE_ANY,/HAVE_ALL')
  
endcase

end
