;-------------------------------------------------------------------------------
;	NAME
;		chdmod
;
;	PURPOSE
;		IDL function to convert display modes
;
;	USAGE
;		r = chdmod([p|x,y[,z]][,/data,/normal,/device][,/to_data,/to_normal,/to_device][,out1=out1,out2=out2,out3=out3])
;
;	INPUT
;		data1 : position or x value
;		data2 : y value
;		data3 : z value;			
;
;	KEYWORD
;		data      : input is in data mode
;		normal    : input is in normal mode
;		device    : input is in device mode
;		to_data   : output is in data mode
;		to_normal : output is in normal mode
;		to_device : output is in device mode
;
;	OUTPUT
;		out1 : result x value
;		out2 : result y value
;		out3 : result z value
;
;	AUTHOR
;		2008-11-08 Hyun Cheol Kim (Hyun.Kim@noaa.gov)
;		2009-10-08 changed function name
;		2011-09-15 added /x,/y,/cm2d
;-------------------------------------------------------------------------------

	function chdmod,data1,data2,data3,$
	         data=data,normal=normal,device=device,$
	         to_data=to_data,to_normal=to_normal,to_device=to_device,$
		 x=x,y=y,cm2d=cm2d,$
	         out1=out1,out2=out2,out3=out3

;-------------------------------------------------------------------------------

case 1 of

	var_set(data1,data2,data3) : begin
	
		r1 = convert_coord([data1],[data2],[data3],data=data,normal=normal,device=device,/to_normal)
		r2 = convert_coord(r1,/normal,to_data=to_data,to_normal=to_normal,to_device=to_device)
				
		out1 = reform(r2[0,*],size(data1,/dim))
		out2 = reform(r2[1,*],size(data2,/dim))
		out3 = reform(r2[2,*],size(data3,/dim))
				
		return,r2	
		end

	var_set(data1,data2) : begin
		
		r1 = convert_coord([data1],[data2],data=data,normal=normal,device=device,/to_normal)
		r2 = convert_coord(r1,/normal,to_data=to_data,to_normal=to_normal,to_device=to_device)
				
		out1 = reform(r2[0,*],size(data1,/dim))
		out2 = reform(r2[1,*],size(data2,/dim))
		
		return,r2	
		end
		
	var_set(data1) : begin
	
		case 1 of
		
			n_elements(data1) eq 4 : begin
			
				r1 = convert_coord([data1[[0,2]]],[data1[[1,3]]],data=data,normal=normal,device=device,/to_normal)
				r2 = convert_coord(r1,/normal,to_data=to_data,to_normal=to_normal,to_device=to_device)
				
				out1 = reform(r2[0,*])
				out2 = reform(r2[1,*])
				
				return,[out1[0],out2[0],out1[1],out2[1]]

				end
				
			n_elements(data1) eq 1 : begin
			
				case 1 of
				
					var_set(cm2d,x) : return,data1*!d.x_px_cm
					var_set(cm2d,y) : return,data1*!d.y_px_cm
					var_set(cm2d)   : return,data1*sqrt((!d.x_px_cm^2+!d.y_px_cm^2)/2.)
					else : 
								
				endcase
										
				r1 = convert_coord(!x.crange,!y.crange,/data,to_normal=normal,to_device=device,to_data=data)				
				r2 = convert_coord([r1[0],r1[0]+data1,r1[0]+data1,r1[0]],[r1[1],r1[1],r1[1]+data1,r1[1]+data1],data=data,normal=normal,device=device,/to_normal)
				r3 = convert_coord(r2,/normal,to_data=to_data,to_normal=to_normal,to_device=to_device)
								
				case 1 of
				
					keyword_set(x) : rr = r3[0,1]-r3[0,0]
					keyword_set(y) : rr = r3[1,3]-r3[1,0]
					else : rr = sqrt(((r3[0,1]-r3[0,0])^2+(r3[1,3]-r3[1,0])^2)/2.)
				
				endcase
				
				return,rr
			
				end
				
			else : begin
					
				r1 = convert_coord([data1],data=data,normal=normal,device=device,/to_normal)
				r2 = convert_coord(r1,/normal,to_data=to_data,to_normal=to_normal,to_device=to_device)
				
				return,r2
				
				end
				
		
		endcase
				
		end
		
	else : message		

endcase

end
