;-------------------------------------------------------------------------------
;	NAME
;		aryn
;
;	PURPOSE
;		to perform array operation with NaN
;
;	USAGE
;		r = aryn(dd,n[,/total,/mean,/max,/min,/descend,/ascend,/percentile,
;		          /daily_average,/daily_max,/diurnal,/local_time,/moving_average])
;
;		dd = randomu(s,[4,5,6])
;			
;		r = aryn(dd,2,/total[,count=nn])  ; total through dimension 2
;		r = aryn(dd,2,/mean[,count=nn])   ; mean     "
;		r = aryn(dd,2,/max[,count=nn])    ; max      "
;		r = aryn(dd,2,/min[,count=nn])    ; min      "
;		r = aryn(dd,2,/descend)           ; sort through dim 2 from max value to NaN
;		r = aryn(dd,2,/ascend)            ; sort through dim 2 from min value to NaN
;		r = aryn(dd,2,75,/percentile)     ; find 75% percentile in dim 2
;		r = aryn(dd,tflag,/daily_average) ; find daily average (the last dimension should be tflag dimension)
;		r = aryn(dd,tflag,/daily_max)     ; find daily max
;		r = aryn(dd,tflag,/diurnal)       ; find average diurnal cycle
;		r = aryn(dd,nmove,/moving_average) ; find moving average (nmove=8, nmove=[-5,6], default is nmove=[0,8] ; 8h forward moving average)
;		r = aryn(dd,tflag,lon[,ltime],/local_time[,/backward]) ; convert utc-based data to local time. /backward convert local time to utc
;		r = aryn(dd,tflag,timezone[,ltime],/local_time,/timezone[,/backward]) ; local time conversion using timezone

;
;	INPUT
;		dd : data with multi dimension
;		n1 : dimensison (n1=1,2,3,...) or tflag or nmove
;		n2 : percentile or lon
;		n3 : local time
;
;	AUTHOR
;		2014-08-01 Hyun Cheol Kim (hyun.kim@noaa.gov) rewritten from aqm_array
;-------------------------------------------------------------------------------

	function aryn,_dd,n1,n2,n3,count=count,otflag=otflag,oidx=oidx,_extra=_ex

;-------------------------------------------------------------------------------

nan  = typename(_dd) eq 'COUBLE' ? !values.d_nan : !values.f_nan
ndim = n_elements((dim=size(_dd,/dim)))
dd   = data_check(_dd,~finite(_dd) or _dd eq (missing=struct_read(_ex,'missing',-999.)),nan)

case 1 of
		
	struct_set(_ex,'total')  : rr = var_set(n1) ? total(dd,n1,/nan) : total(dd,/nan)		
	struct_set(_ex,'mean')   : rr = var_set(count) ? data_check(mean(dd,dim=n1,/nan),total(finite(dd),n1) lt (count gt 1 ? count : count*dim[n1-1]),nan) : mean(dd,dim=n1,/nan)
	struct_set(_ex,'max')    : rr = var_set(count) ? data_check(max(dd,dim=n1,/nan),total(finite(dd),n1) lt (count gt 1 ? count : count*dim[n1-1]),nan) : max(dd,dim=n1,/nan)		
	struct_set(_ex,'min')    : rr = var_set(count) ? data_check(min(dd,dim=n1,/nan),total(finite(dd),n1) lt (count gt 1 ? count : count*dim[n1-1]),nan) : min(dd,dim=n1,/nan)	
	struct_set(_ex,'stddev') : rr = struct_set(_ex,'zero') ? data_check(stddev(dd,dim=n1,/nan),aryn(finite(dd),n1,/total) eq 1,0) : stddev(dd,dim=n1,/nan)
									
	struct_set(_ex,'ascend') : begin
						
		dim2 = size((dd=ary(dd,n1,/forward)),/dim)												
		rr = dd[(oidx=ary(ss[transpose((idx(fix((ss=sort(dd))/dim2[0]),lindgen(dim2[1]),/point,/list)).toarray(/no_copy))],n1,dim,/forward,/restore))]
		end
		
	struct_set(_ex,'descend') : begin
	
		dim2 = size((dd=ary(dd,n1,/forward)),/dim)						
		dd = dd[ss[transpose((idx(fix((ss=sort(dd))/dim2[0]),lindgen(dim2[1]),/point,/list)).toarray(/no_copy))]]								
		oidx = reverse(ary((rebin(lindgen(dim2[0])+dim2[0],dim2)+rebin(transpose(-total(~finite(dd),1)),dim2)) mod dim2[0] + rebin(transpose(lindgen(dim2[1])*dim2[0]),dim2),n1,dim,/forward,/restore),n1)
		rr = dd[oidx]	
		end
		
	struct_set(_ex,'whisker') : begin
		
		dim2 = size((dd=ary(dd,n1,/forward)),/dim)		
		ss = ss[transpose((idx(fix((ss=sort(dd))/dim2[0]),lindgen(dim2[1]),/point,/list)).toarray(/no_copy))]						
		rr1 = dd[(oidx1=reform(ss[ceil((total(finite(dd[ss]),1)-1.)*(percentile=0.25))+lindgen(dim2[1])*dim2[0]],dim[where(~idx([n1-1],/mask,count=ndim))]))]
		rr2 = dd[(oidx2=reform(ss[ceil((total(finite(dd[ss]),1)-1.)*(percentile=0.5 ))+lindgen(dim2[1])*dim2[0]],dim[where(~idx([n1-1],/mask,count=ndim))]))]
		rr3 = dd[(oidx3=reform(ss[ceil((total(finite(dd[ss]),1)-1.)*(percentile=0.75))+lindgen(dim2[1])*dim2[0]],dim[where(~idx([n1-1],/mask,count=ndim))]))]
				
		rr = replicate(struct_make(['min','p25','median','p75','max'],!values.f_nan),size(rr1,/dim))
		
		rr.min    = min(_dd,dim=n1,/nan)
		rr.p25    = rr1
		rr.median = rr2
		rr.p75    = rr3
		rr.max    = max(_dd,dim=n1,/nan)
		
		end	
				
	struct_set(_ex,'percentile') : begin
	
		dim2 = size((dd=ary(dd,n1,/forward)),/dim)		
		ss = ss[transpose((idx(fix((ss=sort(dd))/dim2[0]),lindgen(dim2[1]),/point,/list)).toarray(/no_copy))]						
		rr = dd[(oidx = reform(ss[ceil((total(finite(dd[ss]),1)-1.)*(percentile=n2/100.))+lindgen(dim2[1])*dim2[0]],dim[where(~idx([n1-1],/mask,count=ndim))]))]			
		end
		
	struct_set(_ex,'interpol') : begin
	
		dim2 = size((dd=ary(dd,(n1=var_set(n1,ina=ndim)),/backward)),/dim)		
		for i=0L,dim2[0]-1 do if total((xx=finite(dd[i,*]))) ne dim2[1] then dd[i,*] = interpol(dd[i,where(xx)],where(xx),findgen(dim2[1]))
					
		rr = ary(temporary(dd),n1,dim,/backward,/restore)	
		end		
	
	struct_set(_ex,'daily_average'): begin
	
		if ~var_set(n1) then message,'USAGE: rr = aryn(dd,tflag,/daily_average)
			
		dd = ary(dd,ndim,/backward,odim=dim2)		
		if (x1=fix(date(min((tflag=n1)),/cut,/hour))) ne 0 then dd = [[replicate(nan,[dim2[0],x1])],[dd]]									
		if (x2=23-fix(date(max(tflag),/cut,/hour))) ne 0 then dd = [[dd],[replicate(nan,[dim2[0],x2])]]
		
		otflag = var_set(n2,ina=(tflag1=date(date(min(tflag)),date(max(tflag)),/make)))												
		nn3 = (nn=max(var_set(n3,ina=0))-23) gt 0 ? n3-nn : var_set(n3,ina=0)
		if nn gt 0 then dd = shift(dd,[0,-nn])
																	
		dd = reform(dd,[((dim3=size(dd,/dim)))[0],24,dim3[1]/24.],/overwrite)								
		rr = aryn(keyword_set(nn3) ? ary(dd,2,nn3,/extract) : dd,2,/mean,count=var_set(count,ina=24))								
		rr = reform(var_set(n2) ? data_check(rr[*,(xx=idx(tflag1,otflag,/point))],xx eq -1,nan,dim=2) : rr,[dim[0:ndim-2],n_elements(otflag)],/overwrite)	
		end
		
	struct_set(_ex,'daily_max') : begin
	
		if ~var_set(n1) then message,'USAGE: rr = aryn(dd,tflag,/daily_max)
				
		dd = ary(dd,ndim,/backward,odim=dim2)		
		if (x1=fix(date(min((tflag=n1)),/cut,/hour))) ne 0 then dd = [[replicate(nan,[dim2[0],x1])],[dd]]									
		if (x2=23-fix(date(max(tflag),/cut,/hour))) ne 0 then dd = [[dd],[replicate(nan,[dim2[0],x2])]]

		otflag = var_set(n2,ina=(tflag1=date(date(min(tflag)),date(max(tflag)),/make)))												

		dd = aryn(reform(dd,[((dim3=size(dd,/dim)))[0],24,dim3[1]/24.],/overwrite),2,/max,count=var_set(count,ina=24))
		rr = reform(var_set(n2) ? data_check(dd[*,(xx=idx(tflag1,otflag,/point))],xx eq -1,nan,dim=2) : dd,[dim[0:ndim-2],n_elements(otflag)],/overwrite)		
		end
		
	struct_set(_ex,'diurnal') : begin

		dd = ary(dd,ndim,/backward,odim=dim2)		
		if (x1=fix(date(min((tflag=n1)),/cut,/hour))) ne 0 then dd = [[replicate(nan,[dim2[0],x1])],[dd]]									
		if (x2=23-fix(date(max(tflag),/cut,/hour))) ne 0 then dd = [[dd],[replicate(nan,[dim2[0],x2])]]
				
		if struct_set(_ex,'raw') then return,reform(dd,[((dim3=size(dd,/dim)))[0],24,dim3[1]/24.],/overwrite)
		
		rr = reform(mean(reform(dd,[((dim3=size(dd,/dim)))[0],24,dim3[1]/24.],/overwrite),dim=3,/nan),[dim[0:ndim-2],24],/overwrite)								
		end
		
	struct_set(_ex,'local_time') : begin
		
		nd1 = ndim eq 1 ? 1 : product(dim[0:ndim-2])		
		nt1 = n_elements((tflag1=n1))
		nt2 = n_elements((tflag2=var_set(n3,ina=n1)))		
		dt  = struct_read(_ex,'timezone',0) ? n2 : -round(((lon = n2 mod 360.)*(lon le 180)+(lon-360)*(lon gt 180))/15.)*(struct_read(_ex,'backward',0) ? -1 : 1)

		tflag1 = [date(date(min(tflag1),-1,/incr,/day),24,/make,/hour),tflag1,date(date(max(tflag1),1,/incr,/hour),24,/make,/hour)]
		dim2   = size((dd=[[replicate(nan,[nd1,24])],[reform(dd,[nd1,nt1],/overwrite)],[replicate(nan,[nd1,24])]]	),/dim)		
		
		dd = data_check(dd[[(ii=lindgen(dim2)+rebin(ary(dt,count=nd1),dim2)*nd1)]],ii lt 0 or ii ge product(dim2),nan)		
		dd = dd[*,[(ii=idx(tflag1,tflag2,/point))]]		
		rr = reform(data_check(temporary(dd),rebin(transpose(ii),[nd1,nt2]) eq -1,nan),[dim[0:ndim-2],nt2])					
		end
				
	struct_set(_ex,'moving_average') : begin
	
		nmove = n_elements((nmove=var_set(n1,ina=[0,7]))) eq 1 ? [0,nmove-1] : nmove
												
		dm = size((dd = reform(dd,[ndim eq 1 ? 1 : product(dim[0:ndim-2]),dim[ndim-1]],/overwrite)),/dim)		
		dd = [[replicate(nan,[dm[0],(nn=max(abs(nmove)))])],[temporary(dd)],[replicate(nan,[dm[0],nn])]]
		dd = data_check(dd,~(ii=finite(dd)),0.)
		
		for i=nmove[0],nmove[1] do begin
				
			ddd = var_set(ddd) ? ddd + shift(dd,[0,-i]) : shift(dd,[0,-i])
			iii = var_set(iii) ? iii + shift(ii,[0,-i]) : shift(ii,[0,-i])
				
		endfor
				
		rr = data_check(ddd/iii,iii lt var_set(count,ina=5),nan)		
		rr = reform(rr[*,nn:nn+dm[1]-1],dim,/overwrite)					
		end
				
	else : rr = temporary(dd)

endcase

return,rr
end
