;-------------------------------------------------------------------------------
;	NAME
;		ary
;
;	PURPOSE
;		to perform array operation(s)
;
;	USAGE
;		rr = ary(data,/uniq)
;		rr = ary(dtaa,/sort)
;		rr = ary(data,/flat)
;		rr = ary(data,count=n)
;		rr = ary(data,ndim,/reverse) (ndim=1,2,...)
;		rr = ary(data,/boundary) ;
;		rr = ary(data,/plane)
;		rr = ary(data,/plane,/flat)
;
;		rr = ary(data,nx=5)
;		rr = ary(data,ny=3)
;		rr = ary(data,nx=3,ny=2)
;		rr = ary(data,ix=ix,iy=iy)
;		rr = ary(data,ix=ix,iy=iy,/grid)
;		rr = ary(data,n1,n2,n3,n4,/track)
;		rr = ary(data,n1,n2,n3,/curtain)
;		rr = ary(data,/add[,/right,/left,/up,/down])
;		rr = ary(dtaa,/cut[,/right,/left,/up,/down])
;		rr = ary(data,/extend[,/right,/left,/up,/down])
;		rr = ary(data,/tile)
;
;		rr = ary(data,ndim,/forward) 
;		rr = ary(data,ndim,/backward)   
;		rr = ary(data,ndim,dim,/backward,/restore)
;		rr = ary(data,ndim,dim,/forward,/restore)
;
;		
;	KEYWORD 
;		uniq       : return uniq elements
;		sort       : return sorted elements
;		flat       : return 1D array
;		count      : reform data to given element number (repeats if count is larger)
;		reverse    : reverse data for given dimensions
;		boundary   : cut boundary data (counter-clockwise)
;		forward    : move "ndim" dimension to first
;		backward   : move "ndim" dimension to last
;		restore    : with /forward or /backward, resotre original dimension
;		plane      : convert to 3D array
;		nx         : convert to (nx,ndata) array
;		ny         : convert to (ndata,ny) array
;		nx,ny      : convert to (nx,ny,ndata) array
;		ix,iy      : read data at [ix,iy] 
;		ix,iy,grid : read data at [ix,iy,*]
;		track      : read data at [ix,iy,iz,it]
;		curtain    : read data at [ix,iy,*,it]
;		add        : add same column or row for given direction (/right,/left,/up,/down)
;		cut        : cut column or row fo given direction (/right,/left,/up,/down)
;		extend     : similar to add, but extrapolated (e.g. x3=x2+(x2-x1))
;		tile       : return 4+1 corners of tile (counter-clockwise)
;
;	AUTHOR	
;		2014-01-14 Hyun Cheol Kim (hyun.kim@noaa.gov) rewritten from arry.pro & array2.pro
;		2014-07-01 added /dominant
;		2014-07-16 added /count,/last
;-------------------------------------------------------------------------------

	function ary,_dd,n1,n2,n3,n4,overwrite=overwrite,               $
	         uniq=_uniq,sort=_sort,dominant=dominant,flat=flat,reverse=_reverse,      $
	         nx=nx,ny=ny,count=count,last=last,first=first,plane=plane,boundary=boundary, $
		 ix=ix,iy=iy,grid=grid,                                 $
		 track=track,curtain=curtain,                           $
		 add=add,cut=cut,extend=extend,slide=slide,tile=tile,   $
		 cro2dot=cro2dot,dot2cro=dot2cro,                       $
		 up=up,down=down,right=right,left=left,                 $
		 forward=forward,backward=backward,restore=_restore,extract=_extract,    $
		 odim=odim,oidx=oidx,oinfo=oinfo
	         	
;-------------------------------------------------------------------------------

ndim = n_elements((dim=size(_dd,/dim)))
nd   = n_elements(_dd)
dd   = keyword_set(overwrite) ? temporary(_dd) : _dd
ff   = [keyword_set(down),keyword_set(right),keyword_set(up),keyword_set(left)]

case 1 of

	keyword_set(_uniq) : return,dd[uniq(dd,sort(dd))]
	keyword_set(_sort) : return,dd[sort(dd)]
	keyword_set(dominant) : begin
		
		xx = max(histogram((aa=idx(dd,/self,oself=bb)),omin=min),ind)
		oidx = where(aa eq ind)		
		return,bb[ind]		
		end
		
	keyword_set(_reverse) : return,reverse(dd,n1,/overwrite)

	keyword_set(boundary) : begin	
	
		dd = ary(dd,/plane,/overwrite,odim=dim1)	
		ni = n_elements((ii=[lindgen(dim[0]),lindgen(dim[1]-1)*dim[0]+2*dim[0]-1,dim[0]*dim[1]-2-lindgen(dim[0]-1),dim[0]*(dim[1]-1)-lindgen(dim[1]-1)*dim[0]-dim[0]]))
		ii = rebin(ii,[ni,dim1[2]],/sample)+ary(lindgen(dim1[2])*product(dim1[0:1]),nx=ni)	
		odim = size((rr=(ndim gt 2 ? reform(dd[ii],[ni,dim[2:*]],/overwrite) : dd[ii])),/dim)
		oidx = temporary(ii)
		end
		
	var_set(forward,_restore)  : odim = size((rr=transpose(reform(dd,n2[[shift(indgen(n1),1),n1 eq n_elements(n2) ? !null : indgen(n_elements(n2)-n1)+n1]],/overwrite),[shift(indgen(n1),-1),n1 eq n_elements(n2) ? !null : indgen(n_elements(n2)-n1)+n1])),/dim)
	var_set(backward,_restore) : odim = size((rr=transpose(reform(dd,n2[[n1 eq 1 ? !null : indgen(n1-1),shift((indgen(n_elements(n2)))[n1-1:*],-1)]]),[n1 eq 1 ? !null : indgen(n1-1),shift((indgen(n_elements(n2)))[n1-1:*],1)])),/dim)		
	keyword_set(forward)  : odim = size((rr = ndim eq 1 ? reform(dd,[nd,1]) : reform(transpose(dd,[(n1=1>var_set(n1,ina=1)<ndim)-1,(xx1=where(~idx([n1-1],/mask,count=n_elements(dim))))]),[dim[n1-1],product(dim[xx1])],/overwrite)),/dim)			
	keyword_set(backward) : odim = size((rr=ndim eq 1 ? reform(dd,[1,nd],/overwrite) : reform(transpose(dd,[(xx1=where(~idx([(n1=1>var_set(n1,ina=ndim)<ndim)-1],/mask,count=n_elements(dim)))),n1-1]),[product(dim[xx1]),dim[n1-1]],/overwrite)),/dim)
	
	keyword_set(_extract) : begin
	
		dim[n1-1] = n_elements(n2)
		rr = ary((ary(temporary(dd),n1,/forward))[n2,*],n1,dim,/forward,/restore)			
		end

	var_set(plane,flat) : odim = size((rr=ndim eq 1 ? reform(dd,[nd,1]) : ndim eq 2 ? reform(dd,[product(dim),1]) : reform(temporary(dd),[product(dim[0:1]),product(dim)/product(dim[0:1])])),/dim)			
	keyword_set(plane)  : odim = size((rr=ndim eq 1 ? reform(dd,[nd,1,1]) : ndim eq 2 ? reform(dd,[dim,1]) : reform(temporary(dd),[dim[0:1],product(dim)/product(dim[0:1])])),/dim)			
	keyword_set(flat)   : return,reform(dd,nd,/overwrite)
						
	var_set(nx,ny)       : return,dd[reform(rebin(transpose(lindgen(nd)),[long(ny)*long(nx),nd],/sample),[nx,ny,nd],/overwrite)]			
	var_set(ny)          : return,dd[rebin(lindgen(nd),nd,ny,/sample)]
	var_set(nx)          : return,dd[rebin(transpose(lindgen(nd)),nx,nd,/sample)] 	
	var_set(ix,iy,grid)  : return,reform(dd[ary((lindgen(dim[0:1]))[ix,iy,*],ny=product(dim[2:*]))+ary(lindgen(product(dim[2:*]))*product(dim[0:1]),nx=n_elements(ix)*n_elements(iy))],[n_elements(ix),n_elements(iy),dim[2:*]],/overwrite)		
	var_set(ix,iy)       : return,reform(dd[ary(ix+iy*dim[0],ny=product(dim[2:*]))+ary(lindgen(product(dim[2:*]))*product(dim[0:1]),nx=n_elements(ix))],[n_elements(ix),dim[2:*]],/overwrite)

	var_set(count,last)  : return,dd[lindgen(count)<(n_elements(dd)-1)]			
	var_set(count)       : return,dd[(array(lindgen(nd),ny=long(count[0]/nd)+1))[0:count[0]-1]]	
	keyword_set(track)   : return,interpolate(interpolate(transpose(dd,[3,0,1,2]),n1,n2,n3),n4,indgen(n_elements(n1)))
	keyword_set(curtain) : return,transpose(interpolate(transpose(dd,[2,0,1,3]),n1,n2,n3))
	
	keyword_set(dot2cro) : return,reform((((dd=ary(dd,/plane))+shift(dd,[1,0,0])+shift(dd,[0,1,0])+shift(dd,[1,1,0]))/4d)[1:*,1:*,*],[dim[0:1]-1,ndim gt 2 ? dim[2:*] : !null])
	keyword_set(cro2dot) : return,ary(ary(dd,/extend),/dot2cro)
	
	keyword_set(cut) : begin
	
		dd = ary(dd,/plane,/overwrite,odim=dim1)		
	
		if total(ff) eq 0 then ff = [1,1,1,1]		
		if keyword_set(ff[1]) then dd = dd[0:dim[0]-2,*,*]
		if keyword_set(ff[2]) then dd = dd[*,0:dim[1]-2,*]		
		if keyword_set(ff[0]) then dd = dd[*,1:*,*]
		if keyword_set(ff[3]) then dd = dd[1:*,*,*]
											
		odim = size((rr=reform(dd,[(size(dd,/dim))[0:1],ndim gt 2 ? dim[2:*] : !null],/overwrite)),/dim)			
		end
		
	keyword_set(add) : begin
	
		dd = ary(dd,/plane,/overwrite,odim=dim1)		
		ii = lindgen(dim[0:1])
		
		if total(ff) eq 0 then ff = [1,1,1,1]		
		if keyword_set(ff[0]) then ii = [[ii[*,0]],[ii]]				
		if keyword_set(ff[1]) then ii = [ii,ii[(size(ii,/dim))[0]-1,*]]		
		if keyword_set(ff[2]) then ii = [[ii],[ii[*,(size(ii,/dim))[1]-1]]]										
		if keyword_set(ff[3]) then ii = [ii[0,*],ii]
		
		ii = rebin(ii,[(dim2=size(ii,/dim)),dim1[2]],/sample)+ary(lindgen(dim1[2])*product(dim[0:1]),nx=dim2[0],ny=dim2[1],/overwrite)										
		odim = size((rr=reform(dd[ii],[dim2[0:1],ndim gt 2 ? dim[2:*] : !null],/overwrite)),/dim)
		end
						
	keyword_set(extend) : begin
	
		dd = ary(dd,/plane,/overwrite,odim=dim1)
		
		if total(ff) eq 0 then ff = [1,1,1,1]						
		if keyword_set(ff[0]) then dd = [[[2*dd[*,0,*]-dd[*,1,*]]],[dd]]
		if keyword_set(ff[1]) then dd = [dd,2*dd[(size(dd,/dim))[0]-1,*,*]-dd[(size(dd,/dim))[0]-2,*,*]]
		if keyword_set(ff[2]) then dd = [[dd],[[2*dd[*,(size(dd,/dim))[1]-1,*]-dd[*,(size(dd,/dim))[1]-2,*]]]]		
		if keyword_set(ff[3]) then dd = [2*dd[0,*,*]-dd[1,*,*],dd]
		
		odim = size((rr=reform(dd,[(size(dd,/dim))[0:1],ndim gt 2 ? dim[2:*] : !null],/overwrite)),/dim)
		end
		
	keyword_set(tile) : begin
			
		dd1 = ary(ary((dd= ary(ary(dd,/cro2dot),/plane)),/cut,/right,/up),/flat)
		dd2 = ary(ary(shift(dd,-1,0,0),/cut,/right,/up),/flat)
		dd3 = ary(ary(shift(dd,-1,-1,0),/cut,/right,/up),/flat)
		dd4 = ary(ary(shift(dd,0,-1,0),/cut,/right,/up),/flat)
			
		return,transpose([[dd1],[dd2],[dd3],[dd4],[dd1]])				
		end
											
endcase

return,rr
end
