;-------------------------------------------------------------------------------
;	NAME
;		array
;
;	PURPOSE
;		IDL functions to handle array
;
;	USAGE
;		see KEYWORD section		
;
;	INPUT
;		data      : data array
;		nrow      : row number of output array
;		ncol      : column number of output array
;		count     : expand array to match given array numbers 
;		more_than : least number to handle (/mean,/min,/max)
;		missing   : missing value
;		min_value : min value
;		max_value : max value
;
;	KEYWORD
;		uniq    : return uniq values only
;		sort    : return sorted array 
;		flat    : make array flat (1 dimensional)
;		cross   : reduce array with x,y indices (take out regular grid)
;		dots    : reduce array with x,y indices (irregular points. index can be given as [ix,iy] or [i2d]) 
;		track   :
;		curtain :
;		flip    : flip array vertically
;		add     : add one column or row (/right,/left,/up,/down,/all)
;		cut     : cut one column or row (/right,/left,/up,/down,/all)
;		extend  : add one column or row with extend values (/right,/left,/up,/down,/all)
;		shift   : shift array (/right,/left,/up,/down)
;		slide   : shift and fill zero at remaining part (/right,/left,/up,/down)
;		cro2dot : convert cro grid data to dot grid
;		dot2cro : convert dot grid data to cro grid
;		tile    : generate tile array
;		nrow    : generate 2D array with given row number
;		ncol    : generate 2D array with given column number
;		nx      : same with ncol
;		ny      : same with nrow
;		nstack  : generate 3D array with given stack (height or depth) number
;		count   : cut or repeat array elements to fix total number of array count. 
;		mean    : return array mean
;		_min    : return array min
;		_max    : return array max
;		moving_average : return moving average. default is 8hr
;		redim   : reduce array dimension
;		divide  : increase array resolution 
;		interpolate : increase array resolution (faster but cannot handle missing)
;   
;	AUTHOR
;		2006-09-29 Hyun Cheol kim (hyuncheol.kim@gmail.com, hkim2@mail.uh.edu)
;		2007-10-10 added cor2dot/dot2cro
;		2007-11-30 added /mean /min /max /moving_average
;		2007-12-11 added /cross
;		2008-10-29 added /dots
;		2009-06-10 added diurnal average, local_time
;-------------------------------------------------------------------------------
 
	function array,_data,n1,n2,n3,n4,                                              $
	         uniq=_uniq,sort=_sort,flat=flat,flip=flip,tile=tile,dominant=dominant,$
	         nrow=nrow,ncol=ncol,nstack=nstack,count=count,nx=nx,ny=ny,            $
	         add=add,cut=cut,extend=extend,slide=slide,shift=shift,                $
	         cro2dot=cro2dot,dot2cro=dot2cro,nthick=nthick,boundary=boundary,      $
	         mean=_mean,_min=_min,_max=_max,moving_average=moving_average,         $
	         daily_average=daily_average,otflag=otflag,daily_max=daily_max,        $
	         diurnal_average=diurnal_average,local_time=local_time,                $
	         redim=redim,missing=missing,min_value=min_value,max_value=max_value,  $
	         more_than=more_than,absolute=absolute,dimension=dimension,nmove=nmove,$
	         up=up,down=down,left=left,right=right,all=all,mm5=mm5,                $
	         vertical=vertical,no_vertical=no_vertical,has_tflag=has_tflag,        $
	         cross=cross,dots=dots,track=track,curtain=curtain,                    $
	         divide=divide,ndiv=ndiv,                                              $
	         stack=stack,interpolate=interpolate,test=test

;-------------------------------------------------------------------------------

COMPILE_OPT IDL2

data  = reform(_data)
ndata = n_elements(data)
dims  = size(data,/dim)
ndims = n_elements(dims)
dtype = size(data,/type)

case 1 of

	keyword_set(_uniq) : return,data[uniq(data,sort(data))]	
	keyword_set(_sort) : return,data[sort(data)]	
	keyword_set(flat)  : return,reform(data,ndata)
	
	keyword_set(dominant) : begin
	
		if check(/have_any,dtype,[1,2,3,13,14,15]) then begin
		
			xx = max(histogram(data,omin=min),ind)		
			return,ind+min 
			
		endif else begin
	
			du = array(data,/uniq)				
			xx  = max(histogram(idx(du,data,/point),min=0),ind)				
			return,du[ind]
		
		endelse
			
		end
			
	keyword_set(cross) : begin
	
		case ndims of
		
			1 : message,'/CROSS REQUIRES 2 OR MORE DIMENSIONS'
			2 : return,reform(data[where(index(n1,/mask,total=dims[0])#index(n2,/mask,total=dims[1]) eq 1)],[n_elements(array(n1,/uniq)),n_elements(array(n2,/uniq))])									 		
			else : return,data[index(array(lindgen(dims[0:1]),n1,n2,/cross),product(dims[0:1]),product(dims[2:*]),/stack)]

		endcase		
	
		end
				
	keyword_set(dots) : begin
	
		case 1 of
		
			var_set(n1,n2) : begin
			
				ndot = n_elements(n1)
	
				case ndims of
		
					1 : message,'/DOTS REQUIRES 2 OR MORE DIMENSIONS'
					2 : return,data[[n1],[n2]]									
					else : return,reform(data[rebin((lindgen(dims[0:1]))[n1,n2],[ndot,product(dims[2:*])])+array(lindgen(product(dims[2:*]))*product(dims[0:1]),ncol=ndot)],[ndot,dims[2:*]])
													
				endcase
				
				end
				
			var_set(n1) : return,data[index(n1,data,/to_3d)]
		
		endcase
	
		end		
		
	keyword_set(track) : return,interpolate(interpolate(transpose(_data,[3,0,1,2]),n1,n2,n3),n4,indgen(n_elements(n1)))

	keyword_set(curtain) : return,transpose(interpolate(transpose(_data,[2,0,1,3]),n1,n2,n3))
					
	keyword_set(flip) : begin
	
		case ndims of
		
			1 : message,'/FLIP REQUIRES 2 OR MORE DIMENSIONS'			
			2 : begin
			
				if keyword_set(up)    then return,reverse(data,2) 
				if keyword_set(down)  then return,reverse(data,2) 
				if keyword_set(left)  then return,reverse(data,1) 
				if keyword_set(right) then return,reverse(data,1) 
				if keyword_set(mm5)   then return,transpose(data)
				
			end
			
			else : begin
			
				if keyword_set(up)    then return,reform(data[index(array(lindgen(dims[[0,1]]),/flip,/up   ),product(dims[[0,1]]),product(dims[2:*]),/stack)],dims)
				if keyword_set(down)  then return,reform(data[index(array(lindgen(dims[[0,1]]),/flip,/down ),product(dims[[0,1]]),product(dims[2:*]),/stack)],dims)
				if keyword_set(left)  then return,reform(data[index(array(lindgen(dims[[0,1]]),/flip,/left ),product(dims[[0,1]]),product(dims[2:*]),/stack)],dims)
				if keyword_set(right) then return,reform(data[index(array(lindgen(dims[[0,1]]),/flip,/right),product(dims[[0,1]]),product(dims[2:*]),/stack)],dims)
				if keyword_set(mm5)   then begin
				
					if ndims eq 3 then begin
	
								case 1 of
								
									keyword_set(vertical)    : return,transpose(reverse(data,3),[[1,0],indgen(ndims-2)+2])	
									keyword_set(no_vertical) : return,transpose(data,[[1,0],indgen(ndims-2)+2])	
									keyword_set(has_tflag)   : return,transpose(data,[[1,0],indgen(ndims-2)+2])	
									
									else : message,str(/join,'If data is 3D, use r = array(3D_data,/flip,/mm5,/vertical[/no_vertical])')
								
								endcase
																							
					endif else return,transpose(reverse(data,3),[[1,0],indgen(ndims-2)+2])	
					
				endif	
			
			end
		
		endcase
	
		end		
	
	keyword_set(add) : begin
	
		result = data
		
		case ndims of
		
			1 : message,'/ADD REQUIRES 2 OR MORE DIMENSIONS'
			2 : begin
			
				if keyword_set(up)    then result = [[result],[result[*,dims[1]-1]]]
				if keyword_set(down)  then result = [[result[*,0]],[result]]  
				if keyword_set(left)  then result = transpose(array(transpose(result),/add,/down))
				if keyword_set(right) then result = transpose(array(transpose(result),/add,/up))								
				if keyword_set(all)   then result = array(data,/add,/up,/down,/left,/right) 
				
				return,result
			
			end
			
			else : return,reform(data[index(array(lindgen(dims[[0,1]]),/add,right=right,left=left,up=up,down=down,all=all),product(dims[[0,1]]),product(dims[2:*]),/stack)],$
							(keyword_set(all) ? dims+[[2,2],replicate(0,ndims-2)] : dims+[[keyword_set(left)+keyword_set(right),keyword_set(up)+keyword_set(down)],replicate(0,ndims-2)]))
		
		endcase 
					
		end
			
	keyword_set(cut) : begin
	
		result = data 
		
		case ndims of
		
			1 : message,'/CUT REQUIRES 2 OR MORE DIMENSIONS'
			2 : begin
		
				if keyword_set(up)    then result = result[*,0:dims[1]-2]
				if keyword_set(down)  then result = result[*,1:dims[1]-1]
				if keyword_set(left)  then result = result[1:dims[0]-1,*]
				if keyword_set(right) then result = result[0:dims[0]-2,*]
				if keyword_set(all)   then result = data[1:dims[0]-2,1:dims[1]-2]
		
				return,result
			
			end
			
			else : return,reform(data[index(array(lindgen(dims[[0,1]]),/cut,right=right,left=left,up=up,down=down,all=all),product(dims[[0,1]]),product(dims[2:*]),/stack)], $
							(keyword_set(all) ? dims-[[2,2],replicate(0,ndims-2)] : dims-[[keyword_set(left)+keyword_set(right),keyword_set(up)+keyword_set(down)],replicate(0,ndims-2)]))
						
		endcase
				
		end
					
	keyword_set(extend) : begin
	
		result = data
		
		case ndims of
		
			1 : message,'/EXTEND REQUIRES 2 OR MORE DIMENSIONS'
			2 : begin
	
				if keyword_set(up)    then result = [[result],[2.*result[*,dims[1]-1]-result[*,dims[1]-2]]]
				if keyword_set(down)  then result = [[2.*result[*,0]-result[*,1]],[result]]  
				if keyword_set(left)  then result = transpose(array(transpose(result),/extend,/down))
				if keyword_set(right) then result = transpose(array(transpose(result),/extend,/up))
				if keyword_set(all)   then result = array(array([[2.*data[*,0]-data[*,1]],[data],[2.*data[*,dims[1]-1]-data[*,dims[1]-2]]],/extend,/left),/extend,/right)
	
				return,result
				
			end
			
			else : begin			
			
				if keyword_set(all) then return,array(data,/extend,/up,/down,/right,/left)
											
				if keyword_set(up) then begin
				
					result = array(result,/add,/up)
					dims2  = size(result,/dim)
					data2  = array([[lonarr(dims2[[0,1]]-[0,3])],[replicate(1,dims2[0])],[replicate(2,dims2[0])],[replicate(3,dims2[0])]],nstack=dims[2:*])
																													
					x1 = where(data2 eq 1)
					x2 = where(data2 eq 2)
					x3 = where(data2 eq 3)														
					
					result[x3] = 2.*result[x2]-result[x1]			
						
				endif
				
				if keyword_set(down) then begin
				
					result = array(result,/add,/down)
					dims2  = size(result,/dim)
					data2  = array([[replicate(3,dims2[0])],[replicate(2,dims2[0])],[replicate(1,dims2[0])],[lonarr(dims2[[0,1]]-[0,3])]],nstack=dims[2:*])		
					
					x1 = where(data2 eq 1)
					x2 = where(data2 eq 2)
					x3 = where(data2 eq 3)
																			
					result[x3] = 2.*result[x2]-result[x1]			
																		
				endif				
				
				if keyword_set(right) then begin
				
					result = array(result,/add,/right)
					dims2  = size(result,/dim)
					data2  = array(transpose([[lonarr(dims2[[1,0]]-[0,3])],[replicate(1,dims2[1])],[replicate(2,dims2[1])],[replicate(3,dims2[1])]],[1,0]),nstack=dims[2:*])
					
					x1 = where(data2 eq 1)
					x2 = where(data2 eq 2)
					x3 = where(data2 eq 3)														
					
					result[x3] = 2.*result[x2]-result[x1]			
																
				endif
				
				if keyword_set(left) then begin
				
					result = array(result,/add,/left)
					dims2  = size(result,/dim)
					data2  = array(transpose([[replicate(3,dims2[1])],[replicate(2,dims2[1])],[replicate(1,dims2[1])],[lonarr(dims2[[1,0]]-[0,3])]],[1,0]),nstack=dims[2:*])
					
					x1 = where(data2 eq 1)
					x2 = where(data2 eq 2)
					x3 = where(data2 eq 3)														
					
					result[x3] = 2.*result[x2]-result[x1]			
																	
				endif								
				
				return,result

			end
			
		endcase				
		
		end
					
	keyword_set(slide) : begin
	
		if ndims eq 1 then message,'/SLIDE REQUIRES 2 AND MORE DIMENSION ARRAY'
		
		case 1 of
	
			var_set(up,left)    : return,array(array(data,/slide,/up),  /slide,/left)
			var_set(up,right)   : return,array(array(data,/slide,/up),  /slide,/righ)
			var_set(down,left)  : return,array(array(data,/slide,/down),/slide,/left)
			var_set(down,right) : return,array(array(data,/slide,/down),/slide,/right)   
			keyword_set(up)     : return,array(array(data,/extend,/up),   /cut,/down)
			keyword_set(down)   : return,array(array(data,/extend,/down), /cut,/up)
			keyword_set(left)   : return,array(array(data,/extend,/left), /cut,/right)
			keyword_set(right)  : return,array(array(data,/extend,/right),/cut,/left)
			
		endcase

		end
			
	var_set(shift) : begin
	
		result  = shift(data,shift)
		missing = 0
		
		for ishift=0L,n_elements(shift)-1 do begin
		
			if shift[ishift] ne 0 then begin
			
				shift2         = indgen(ndims)
				shift2[0]      = ishift
				shift2[ishift] = 0

				result = transpose(result,shift2)
				dims2 = dims[shift2]

				if shift[ishift] gt 0 then x = [0 ,(shift[ishift]-1) < (dims2[0]-1)]
				if shift[ishift] lt 0 then x = [(dims2[0]-abs(shift[ishift])) > 0,dims2[0]-1]
			
				result[x[0]:x[1],*,*,*,*] = missing
																					
				result = transpose(result,shift2)
				
			endif	
		
		endfor
		
		return,result
	
		end
							
	keyword_set(cro2dot) : begin
	
		if keyword_set(mm5) then begin
		
	  	case n_elements(dims) of
		
				2     : return,array(data,/add,/right,/up)				
				else  : return,reform(data[index(array(lindgen(dims[[0,1]]),/add,/right,/up),product(dims[[0,1]]),$
								product(dims[2:*]),/stack)],[dims[[0,1]]+[1,1],dims[2:*]])
						
			endcase		
		
		endif else begin
		
			return,array(array(data,/dot2cro),/extend,/all)
		
		endelse
		  			
		end
	
	keyword_set(dot2cro) : begin
	
		if keyword_set(mm5) then return,data[0:dims[0]-2,0:dims[1]-2,*,*] $
		                    else return,array((data+shift(data,([-1,0,0,0])[0:ndims-1])+shift(data,([0,-1,0,0])[0:ndims-1])+shift(data,([-1,-1,0,0])[0:ndims-1]))/4.,/cut,/right,/up)
												
		end		
							
	keyword_set(tile) : begin

		data1 = array(data,/cro2dot)
		
		data2 = array(array(shift(data1,-1,0),/cut,/up,/right),/flat)
		data3 = array(array(shift(data1,-1,-1),/cut,/up,/right),/flat)
		data4 = array(array(shift(data1,0,-1),/cut,/up,/right),/flat)
		data1 = array(array(data1,/cut,/up,/right),/flat)
				
		result = transpose([[data1],[data2],[data3],[data4],[data1]])
		
		return,result
					
		end
	
	keyword_set(nthick) : begin

		case ndims of
		
			1 : message,'INOUT SHOULD BE 2D OR MORE'
			2 : begin
	
				ind = lindgen(dims[[0,1]])
				ind = [array(ind[nthick:*,0:nthick-1],/flat),array(ind[dims[0]-nthick:*,nthick:*],/flat),array(ind[0:dims[0]-nthick-1,dims[1]-nthick:*],/flat),array(ind[0:nthick-1,0:dims[1]-nthick-1],/flat)]

				return,data[ind]
				
				end
				
			else : begin
			
				ind = lindgen(dims[[0,1]])
				ind = [array(ind[nthick:*,0:nthick-1],/flat),array(ind[dims[0]-nthick:*,nthick:*],/flat),array(ind[0:dims[0]-nthick-1,dims[1]-nthick:*],/flat),array(ind[0:nthick-1,0:dims[1]-nthick-1],/flat)]
				ni1 = n_elements(ind)
				ind = array(ind,ny=product(dims[2:*]))+array(lindgen(product(dims[2:*]))*product(dims[0:1]),nx=ni1)
				
				return,reform(data[ind],[ni1,dims[2:*]])
			
				end
			
		endcase			

		end
		
	keyword_set(boundary) : begin
	
		case ndims of
		
			1 : message,'INOUT SHOULD BE 2D OR MORE'
			2 : begin
	
				ind = lindgen(dims[[0,1]])
				ind = [array(ind[*,0],/flat),array(ind[dims[0]-1,1:*],/flat),reverse(array(ind[0:dims[0]-2,dims[1]-1],/flat)),reverse(array(ind[0,1:dims[1]-2],/flat))]
		
				return,data[ind]
				
				end
				
			else : begin
			
				ind = lindgen(dims[[0,1]])
				ind = [array(ind[*,0],/flat),array(ind[dims[0]-1,1:*],/flat),reverse(array(ind[0:dims[0]-2,dims[1]-1],/flat)),reverse(array(ind[0,0:dims[1]-2],/flat))]
				ni1 = n_elements(ind)
				ind = array(ind,ny=product(dims[2:*]))+array(lindgen(product(dims[2:*]))*product(dims[0:1]),nx=ni1)
				
				return,reform(data[ind],[ni1,dims[2:*]])
			
				end
			
		endcase		
					
		end		
			
	var_set(nrow,ncol) : return,data[reform(rebin(transpose(lindgen(ndata)),[nrow*ncol,ndata]),[ncol,nrow,ndata])]
	var_set(nrow)      : return,data[rebin(lindgen(ndata),ndata,nrow)]			
	var_set(ncol)      : return,data[rebin(transpose(lindgen(ndata)),ncol,ndata)] 
	var_set(nx,ny)     : return,data[reform(rebin(transpose(lindgen(ndata)),[ny*nx,ndata]),[nx,ny,ndata])]
	var_set(ny)        : return,data[rebin(lindgen(ndata),ndata,ny)]
	var_set(nx)        : return,data[rebin(transpose(lindgen(ndata)),nx,ndata)]   
	var_set(nstack)    : return,data[rebin(lindgen(dims[[0,1]]),[dims[[0,1]],[nstack]])]
	var_set(count)     : return,data[(array(lindgen(ndata),nrow=long(count[0]/ndata)+1))[0:count[0]-1]]
					
	keyword_set(_mean) : begin
	
		if not var_set(n1) then message,'[USAGE] R = ARRAY(DATA,N,/MEAN) N:DIMENSION INDEX (1-N)'
		if not var_set(more_than) then more_than = 1
		if ndims eq 1 then return,_data
	
		if var_set(missing) then begin
		
			ncount = make_array(size=size(data),value=1.)
								
			if check(/valid,data,missing=missing,min_value=min_value,max_value=max_value,more_than=more_than,/complement,index=mi) then begin
								 
				data[mi]   = dtype eq 5 ? !values.d_nan : !values.f_nan
				ncount[mi] = 0
				 
			endif 
									
			result = total(data,n1,/nan)/total(ncount,n1)
			
			x = where(finite(result,/nan),nx)
			
			if nx gt 0 then result[x] = missing

			return,result
					
		endif else begin
		
			return,float(total(data,n1))/dims[n1-1]
		
		endelse
	
		end
			
	keyword_set(_min) : begin
	
		if dtype eq 5 then data = double(data) else data = float(data)
					
		if var_set(missing) then begin
		
			if check(/valid,data,missing=missing,min_value=min_value,max_value=max_value,/complement,index=mi) then data[mi] = dtype eq 5 ? !values.d_nan : !values.f_nan
			
			result = min(data,dimension=n1,/nan,absolute=absolute)
			
			x = where(finite(result,/nan),nx)
			
			if nx gt 0 then result[x] = missing

			return,result
		
		endif else begin
		
			return,min(data,dimension=n1,/nan,absolute=absolute)
		
		endelse
			
		end
	
	keyword_set(_max) : begin
	
		if dtype eq 5 then data = double(data) else data = float(data)
					
		if var_set(missing) then begin
		
			if check(/valid,data,missing=missing,min_value=min_value,max_value=max_value,/complement,index=mi) then data[mi] = dtype eq 5 ? !values.d_nan : !values.f_nan
			
			result = min(data,dimension=n1,/nan,absolute=absolute)
			
			x = where(finite(result,/nan),nx)
			
			if nx gt 0 then result[x] = missing

			return,result
		
		endif else begin
		
			return,max(data,dimension=n1,/nan,absolute=absolute)
		
		endelse	
	
		end
		
	keyword_set(moving_average) : begin
	
		if not var_set(nmove)     then nmove = 8
		if not var_set(dimension) then dimension = ndims
		if not var_set(more_than) then more_than = (abs(nmove)-3) > 1
		
		message,/info,str(/join,'moving average ( nmove =',nmove,')')		
		
		shift = replicate(0,ndims)
		
		result1 = data
		result2 = make_array(size=size(data))
		ncount1 = make_array(size=size(data),value=1.)
		ncount2 = make_array(size=size(data))
														
		if var_set(missing) then if check(/valid,data,missing=missing,min_value=min_value,max_value=max_value,/complement,index=mi) then begin

			result1[mi] = 0
			ncount1[mi] = 0
			
		endif 
																	
		for imove=1L,abs(nmove)-1 do begin
						
			shift[dimension-1] = -imove*nmove/abs(nmove)
			result2 = result2 + array(result1,shift=shift)  
			ncount2 = ncount2 + array(ncount1,shift=shift) 
						
		endfor

		result2 = result2/ncount2
				
		x = where(ncount2 lt more_than,nx)

		if nx gt 0 then if var_set(missing) then result2[x] = missing else result2[x] = !values.f_nan				
			
		return,result2					
	
		end
				
	keyword_set(daily_max) : begin
	
		message,/info,str(/join,'daily max')
	
		tflag = n1
		days  = array(date(tflag,/cut,/year,/month,/day),/uniq)
		nday  = n_elements(days)
		
		if not var_set(missing) then missing = -999.
				
			; error check
		
		if ndims eq 1 then result = fltarr(nday) else result = make_array([dims[0:ndims-2],nday],value=missing)

			;

		for iday=0L,nday-1 do begin
		
			xtime = index(date(tflag,/cut,/year,/month,/day),days[iday],/contain,count=nxtime)
						
			if nxtime gt 1 then begin
			
				case ndims of
			
					1 : result[iday] = max(data[xtime],dimension=1)
					2 : result[*,iday] = max(data[*,xtime],dimension=2)
					3 : result[*,*,iday] = max(data[*,*,xtime],dimension=3)
					4 : result[*,*,*,iday] = max(data[*,*,*,xtime],dimension=4)
			
				endcase
			
			endif
		
		endfor 
		
		otflag = days
		
		return,result
	
		end	

	keyword_set(daily_average) : begin
	
		message,/info,str(/join,'daily average')
			
		tflag = n1
		days  = array(date(tflag,/cut,/year,/month,/day),/uniq)
		nday  = n_elements(days)
								
		if n_elements(n1) ne dims[ndims-1] then message,str(/join,'[USAGE] r = array(data,tflag,/daily_average) ; data=data[*,*,*,n], tflag=tflag[n]')
		
		if ndims eq 1 then result = fltarr(nday) else result = fltarr([dims[0:ndims-2],nday])
									
		for iday=0L,nday-1 do begin
		
			xtime = index(date(tflag,/cut,/year,/month,/day),days[iday],/contain,count=nxtime)
			
			case ndims of
			
				1 : result[iday] = nxtime eq 1 ? data[xtime] : array(data[xtime],1,/mean,missing=missing)												
				2 : result[*,iday] = nxtime eq 1 ? data[*,xtime] : array(data[*,xtime],2,/mean,missing=missing)
				3 : result[*,*,iday] = nxtime eq 1 ? data[*,*,xtime] : array(data[*,*,xtime],3,/mean,missing=missing)
				4 : result[*,*,*,iday] = nxtime eq 1 ? data[*,*,*,xtime] : array(data[*,*,*,xtime],4,/mean,missing=missing)

			endcase
															
		endfor
		
		otflag = days
								
		return,result
		
		end	
						
	var_set(diurnal_average,local_time) : begin
	
		message,/info,str(/join,'diurnal average with local time')
										
		lon   = n2 mod 360
		dt    = round((lon*(lon le 180)+(lon-360)*(lon gt 180))/15.)
		ntime = 24
						
		ltime  = (reform(rebin(transpose(indgen(ntime)),[product(dims[0:ndims-2]),ntime]),[dims[0:ndims-2],ntime])+rebin(dt,[dims[0:ndims-2],ntime])+ntime) mod ntime
		data   = array(_data,n1,/diurnal_average)				
		result = fltarr([dims[0:ndims-2],ntime])
		
		for itime=0L,ntime-1 do begin
		
			case ndims of
			
				2 : result[*,itime] = total(data*(ltime eq itime),2)
				3 : result[*,*,itime] = total(data*(ltime eq itime),3)
				4 : result[*,*,*,itime] = total(data*(ltime eq itime),4)
			
			endcase
				
		endfor
					
		return,result
					
		end	
			
	keyword_set(diurnal_average) : begin		
	
		message,/info,str(/join,'diurnal average')
		
		tflag = n1
		ntime = 24
		
		if not var_set(more_than) then more_than = 1		
		if not var_set(missing) then missing = -999.				
		
		result = make_array([dims[0:ndims-2],ntime],value=missing)
				
		for itime=0L,ntime do begin
		
			xtime = index(date(tflag,/cut,/hour),str(itime,digit=2),/contain,count=nxtime)
			
			if nxtime ge more_than then begin
			
				case ndims of
				
					1 : result[itime] = nxtime eq 1 ? data[xtime] : array(data[xtime],1,/mean,missing=missing)
					2 : result[*,itime] = nxtime eq 1 ? data[*,xtime] : array(data[*,xtime],2,/mean,missing=missing)
					3 : result[*,*,itime] = nxtime eq 1 ? data[*,*,xtime] : array(data[*,*,xtime],3,/mean,missing=missing)
					4 : result[*,*,*,itime] = nxtime eq 1 ? data[*,*,*,xtime] : array(data[*,*,*,xtime],4,/mean,missing=missing)							
				
				endcase
									
			endif			
		
		endfor
		
		return,result
			
		end	
	
	keyword_set(redim) : begin
	
		result = fltarr(n1)
	
		redim = dims < n1
				
		case n_elements(redim) of
		
			2 : result[0:redim[0]-1,0:redim[1]-1] = data[0:redim[0]-1,0:redim[1]-1]
			3 : result[0:redim[0]-1,0:redim[1]-1,0:redim[2]-1] = data[0:redim[0]-1,0:redim[1]-1,0:redim[2]-1]
			4 : result[0:redim[0]-1,0:redim[1]-1,0:redim[2]-1,0:redim[3]-1] = data[0:redim[0]-1,0:redim[1]-1,0:redim[2]-1,0:redim[3]-1]
			
			else : message,'AVAILABLE 2-4 DIMENSIONS'
		
		endcase
		
		return,result
	
		end
			
	keyword_set(divide) : begin
	
		; works similar to /interpolate
		; can handle missing points
		; designed for sst analysis (aqm_sst.pro)
	
		nx = dims[0]
		ny = dims[1]
	
		if not var_set(ndiv) then ndiv = 2
		
		if var_set(missing) then begin
					
			im = where(data eq missing,nm)
			
			if nm gt 0 then data[im] = !values.f_nan

			data2 = congrid(data,(nx-1)*ndiv+1,(ny-1)*ndiv+1,/minus_one,/interp)
			
			inan = where(finite(data2,/nan),nnan)
			
			if nnan gt 0 then data2[inan] = missing
		
		endif else begin
		
			data2 = congrid(data,(nx-1)*ndiv+1,(ny-1)*ndiv+1,/minus_one,/interp)
				
		endelse 
		
		return,data2
	
		end			
					
	keyword_set(interpolate) : begin

		data = array(data,/stack)
		
		dims2 = size(data,/dim) 
	
		case 1 of
		
			;var_set(n1,n2,n3) : return,interpolate(data,findgen(dims2[0]*n1-1)/n1,findgen(dims2[1]*n2-1)/n2,findgen(dims2[2]*n3-1)/n3,/grid)
			;var_set(n1,n2)    : return,interpolate(data,findgen(dims2[0]*n1-1)/n1,findgen(dims2[1]*n2-1)/n2,findgen(dims2[2]),/grid)
			;var_set(n1)       : return,interpolate(data,findgen(dims2[0]*n1-1)/n1,findgen(dims2[1]*n1-1)/n1,findgen(dims2[2]),/grid)
			
			var_set(n1,n2,n3) : return,interpolate(data,findgen((dims2[0]-1)*n1+1)/n1,findgen((dims2[1]-1)*n2+1)/n2,findgen((dims2[2]-1)*n3+1)/n3,/grid)
			var_set(n1,n2)    : return,interpolate(data,findgen((dims2[0]-1)*n1+1)/n1,findgen((dims2[1]-1)*n2+1)/n2,findgen(dims2[2]),/grid)
			var_set(n1)       : return,interpolate(data,findgen((dims2[0]-1)*n1+1)/n1,findgen((dims2[1]-1)*n1+1)/n1,findgen(dims2[2]),/grid)
			
								
		endcase		
	
		end		
								
	keyword_set(stack) : begin		
	
		case ndims of
		
			1    : return,reform(data,[ndata,1,1])
			2    : return,reform(data,[dims,1])
			else : return,reform(data,[dims[0:1],product(dims[2:*])])
		
		endcase
			
		end	
			
	else : message,'OPTIONS (/UNIQ /SORT /FLAT /EXPAND /SHRINK /CRO2DOT /DOT2CRO /NROW /NCOL /NSTACK /COUNT)'

endcase

end

