;-------------------------------------------------------------------------------
;	NAME
;		aqm_regrd
;
;	PURPOSE
;		to regrid pixeled data into target domain
;
;	USAGE
;		rr = aqm_regrd(tlon,tlat,data[,domain=,minfo=][,min_fraction=][,finfo=][,/sum,/avg,/total,/mean,/average][,missing=][,save=][,/show,winn=])
;
;	INPUT
;		tlon/tlat : tiled lat/lon (rf. tlon = array(lon,/tile))
;		data : raw data
;		domain/minfo : target domain setting
;		finfo : fraction information (could be recycling for better speed)
;		min_fraction : minimum fraction for each grid cell to use
;		missing : missing number
;		save : save/use finto to file
;		winn : set window with /show
;		weight : array for weighting
;		
;	KEYWORD
;		sum/total : get total value
;		avg/mean/average : get weighted average
;		weight : if set, using weighting method
;		show : show plot
;
;	AUTHOR
;		2013-04-04 Hyun Cheol Kim (hyun.kim@noaa.gov)
;		           Rewritten from aqm_rgdf.pro using list for accurate calculation
;		2013-04-22 Added weighting option
;-------------------------------------------------------------------------------

	function aqm_regrd,tlon,tlat,data,domain=domain,minfo=minfo,$
	         finfo=finfo,min_fraction=mf,min_size=min_size,max_size=max_size,$
		 sum=sum,avg=avg,weighting=wt,$
	         missing=missing,save=save,show=show,winn=winn
	
;-------------------------------------------------------------------------------

if ~var_set(mf) then mf = 0.
if ~var_set(missing) then missing = !values.f_nan

minfo = aqm_grid_minfo(domain=domain,minfo=minfo)
dim   = size(data,/dim)

	; get fraction
	
if file_test(var_set(save) ? save : '') then restore,save

if ~var_set(finfo) then begin

	tt = systime(1)
		
	data_indx = strarr(minfo.nx,minfo.ny)
	frac_indx = strarr(minfo.nx,minfo.ny)		
	frac_px = list()
	frac_gr = list()
	frac_wt = list()	
	frac_cont = 0LL

	for itile=0L,product(dim)-1 do begin
		
		aqm_fgi,tlon[*,itile],tlat[*,itile],x,y,minfo=minfo,map_structure=map_structure
		cell = aqm_polyfillaa(x,y,minfo.nx,minfo.ny,areas=area)	

		if cell[0] eq -1 then continue
		
		px_size = poly_area(x,y)*minfo.dx*minfo.dy/1e6
						
		if var_set(max_size) then if px_size gt max_size then continue
		if var_set(min_size) then if px_size gt min_size then continue
				
		data_indx[cell] += str(itile)+' '
		frac_indx[cell] += str(lindgen(n_elements(cell))+frac_cont)+' '
						
		if var_set(/if_any,sum) then frac_px.add,area/poly_area(x,y),/extract
		if var_set(/if_any,avg,wt) then frac_gr.add,area,/extract				
		if var_set(/if_any,wt) then frac_wt.add,(xxx=data[itile]*wt[cell]/total(wt[cell]))/mean(xxx)*data[itile],/extract
						
		frac_cont += n_elements(cell)
						
	endfor
	
	finfo = {data_indx:data_indx,frac_indx:frac_indx,$
	         frac_px:var_set((frac_px=frac_px.toarray(/no_copy))) ? frac_px : -1,$
		 frac_gr:var_set((frac_gr=frac_gr.toarray(/no_copy))) ? frac_gr : -1,$		 
	         frac_wt:var_set((frac_wt=frac_wt.toarray(/no_copy))) ? frac_wt : -1}
	
	if var_set(save) then save,filename=save,finfo
		
	message,/info,str(/join,'FINFO in',str(systime(1)-tt,format='(F8.2)'),'s')
		
endif 

	; combine fractions
		
rr = make_array(minfo.nx,minfo.ny,value=missing)
	
for igrid=0L,minfo.nx*minfo.ny-1 do begin

	if (finfo.data_indx[igrid])[0] eq '' then continue
	
	dd1 = data(long(strsplit(finfo.data_indx[igrid],/extract)))
	dd1 = data_check(dd1,dd1 ne missing,if_not_available=0b,oxx=xx)	
		
	if ~keyword_set(dd1) then continue
	
	if var_set(/if_any,sum,total) then frac_px = (finfo.frac_px[strsplit(finfo.frac_indx[igrid],/extract)])[xx]
	if var_set(/if_any,avg,mean,average,wt) then frac_gr = (finfo.frac_gr[strsplit(finfo.frac_indx[igrid],/extract)])[xx]
	if var_set(/if_any,wt) then frac_wt = (finfo.frac_wt[strsplit(finfo.frac_indx[igrid],/extract)])[xx]
	
	if total(frac_gr) lt mf then continue
									
	case 1 of
	
		var_set(/if_any,sum,total)        : rr[igrid] = total(dd1*frac_px)
		var_set(/if_any,avg,mean,average) : rr[igrid] = total(dd1*frac_gr)/total(frac_gr)
		var_set(/if_any,wt)               : rr[igrid] = total(frac_wt*frac_gr)/total(frac_gr)
		
		else : message,str(/join,'[USAGE] /sum,/avg,/weight')
		
	endcase
		
endfor

if keyword_set(show) then aqm_plot_tile,rr,minfo=minfo,missing=missing,winn=winn

return,rr
end
