;-------------------------------------------------------------------------------
;	NAME
;		aqm_proj
;
;	PURPOSE
;		IDL function to convert between map projections
;
;	USAGE
;		r = aqm_proj(lon,lat[,minfo,minfo=minfo,domain=domain,file=file][,/ll,/lcc,/rll][,/to_ll,/to_lcc,/to_rll],olon=,olat=,olx=,oly=,orlon=,orlat)
;
;	INPUT
;		data1 : lon,lx,rlon
;		data2 :	lat,ly,rlat
;		minfo :
;		
;	KEYWORD
;		ll  : input is in lat/lon
;		lcc : input is in lambert conformal conic
;		rll : input is in rotated lat/lon
;
;	OUTPUT
;		olon/olat   : output in lat/lon
;		olx/oly     : output in lcc		
;		orlon/orlat : output in rotated lat/lon
;
;	NOTE
;		rewritten from aqm_map_proj.pro
;	
;	AUTHOR
;		2009-12-16 Hyun Cheol Kim (hyun.kim@noaa.gov)
;		2015-08-13 added Equatorial Lat/Lon 
;-------------------------------------------------------------------------------

	function aqm_proj,_data1,_data2,minfo=_minfo,domain=_domain,file=_file,$
	         ll=ll,lcc=lcc,rll=rll,ell=ell,$
		 to_ll=to_ll,to_lcc=to_lcc,to_rll=to_rll,to_ell=to_ell,$
	         olon=olon,olat=olat,olx=olx,oly=oly,orlon=orlon,orlat=orlat,oelon=oelon,oelat=oelat	
		 
;-------------------------------------------------------------------------------		 	 

	; check dimension
	
if var_set(_data2) then begin

	dim1 = size(_data1,/dim)
	dim2 = size(_data2,/dim)	
	data1 = ary(_data1,/flat)
	data2 = ary(_data2,/flat)
	
endif else begin	

	dim1 = size(_data1,/dim)	
	data1 = _data1[0,*]
	data2 = _data1[1,*]

endelse

	; get minfo
	
minfo = aqm_grid_minfo(minfo=_minfo,domain=_domain,file=_file)

	; input to ll
	
case 1 of

	keyword_set(lcc) : begin
		
		map_structure = map_proj_init(3,sphere_radius=minfo.r,standard_par1=minfo.lat1,standard_par2=minfo.lat2,center_longitude=minfo.lon0,center_latitude=minfo.lat0)
		r = map_proj_inverse(data1,data2,map_structure=map_structure)
    
		olon = r[0,*]
		olat = r[1,*]  
		end	
		
	keyword_set(rll) : begin

		olon = ((minfo.lon0*!dtor + atan(cos(data2*!dtor)*sin(data1*!dtor),(cos(minfo.lat0*!dtor)*cos(data2*!dtor)*cos(data1*!dtor)-sin(minfo.lat0*!dtor)*sin(data2*!dtor))))/!dtor + 540.) mod 360 - 180.
		olat = (asin(sin(minfo.lat0*!dtor)*cos(data2*!dtor)*cos(data1*!dtor)+cos(minfo.lat0*!dtor)*sin(data2*!dtor)))/!dtor		    				
		
		; abnormal results when used RLL2NLL 
		;olat = asin(1. < sin(minfo.lat0*!dtor)*cos(data2*!dtor)*cos(data1*!dtor)+cos(minfo.lat0*!dtor)*sin(data2*!dtor) > (-1.))/!dtor
		;olon = (minfo.lon0 + asin(1. < sin(data1*!dtor)*cos(data2*!dtor)/cos(olat*!dtor) > (-1.))/!dtor + 540 ) mod 360 - 180

		end
		
	keyword_set(ell) : begin
			
		lon_0 = minfo.lon0+180.

		xx = where(data1 gt 180.,nxx)
		if nxx gt 0 then data1[xx] = data1[xx] - 360.0
		xx = where(data1 le -180.,nxx)
		if nxx gt 0 then data1[xx] = data1[xx] + 360.0

		olat = ASIN(-1.>(COS(minfo.lat0*!dtor)*COS(data1*!dtor)*COS(data2*!dtor)+SIN(data2*!dtor)*SIN(minfo.lat0*!dtor))<1.)/!dtor
		olon = abs(ACOS(-1.>((COS(data2*!dtor)*COS(data1*!dtor)*SIN(minfo.lat0*!dtor)-SIN(data2*!dtor)*COS(minfo.lat0*!dtor))/COS(olat*!dtor))<1.)/!dtor)*abs(data1)/data1+lon_0

		xx = where(olon gt 180.,nxx)
		if nxx gt 0 then olon[xx] = olon[xx] - 360.
		xx = where(olon le -180.,nxx)
		if nxx gt 0 then olon[xx] = olon[xx] + 360.
		
		end	
		
	keyword_set(ll) : begin
	
		olon = data1
		olat = data2
	
		end
		
	else : message,str(/join,'[USAGE] Choose /ll,/lcc,/rll')		

endcase	

	; ll to output
	
case 1 of

	keyword_set(to_lcc) : begin
	
		map_structure = map_proj_init(3,sphere_radius=minfo.r,standard_par1=minfo.lat1,standard_par2=minfo.lat2,center_longitude=minfo.lon0,center_latitude=minfo.lat0)
		out = map_proj_forward(olon,olat,map_structure=map_structure)
		
		olx = var_set(dim2) ? reform(out[0,*],dim1) : ary(out[0,*],/flat)
		oly = var_set(dim2) ? reform(out[1,*],dim2) : ary(out[1,*],/flat)
				
		return,[transpose(ary(olx,/flat)),transpose(ary(oly,/flat))]
			
		end
              
	keyword_set(to_rll) : begin
		     		
		out1 = (atan(cos(data2*!dtor)*sin(data1*!dtor-minfo.lon0*!dtor),cos(minfo.lat0*!dtor)*cos(data2*!dtor)*cos(data1*!dtor-minfo.lon0*!dtor)+sin(minfo.lat0*!dtor)*sin(data2*!dtor)))/!dtor
		out2 = (asin(cos(minfo.lat0*!dtor)*sin(data2*!dtor)-sin(minfo.lat0*!dtor)*cos(data2*!dtor)*cos(data1*!dtor-minfo.lon0*!dtor)))/!dtor
		
		; from NLL2RLL (working OK)	
		;out1 = atan(cos(olat*!dtor)*sin(olon*!dtor-minfo.lon0*!dtor),cos(minfo.lat0*!dtor)*cos(olat*!dtor)*cos(olon*!dtor-minfo.lon0*!dtor)+sin(minfo.lat0*!dtor)*sin(olat*!dtor))/!dtor		
		;out2 = asin(1 < cos(minfo.lat0*!dtor)*sin(olat*!dtor)-sin(minfo.lat0*!dtor)*cos(olat*!dtor)*cos(olon*!dtor-minfo.lon0*!dtor) > (-1))/!dtor		
				
		orlon = keyword_set(dim2) ? reform(out1,dim1) : out1
		orlat = keyword_set(dim2) ? reform(out2,dim2) : out2
		
		return,[transpose(ary(out1,/flat)),transpose(ary(out2,/flat))]		
					
		end
		
	keyword_set(to_ell) : begin
	
		lon_a = olon-(minfo.lon0+180.)

		xx = where(lon_a gt 180.0,nxx)
		if nxx gt 0 then lon_a[xx] = lon_a[xx] - 360.0
		xx = where(lon_a le -180.0,nxx)
		if nxx gt 0 then lon_a[xx] = lon_a[xx] + 360.0

		oelat = (-1.>ASIN(-cos(minfo.lat0*!dtor)*COS(lon_a*!dtor)*COS(olat*!dtor)+SIN(olat*!dtor)*SIN(minfo.lat0*!dtor))< 1.)/!dtor
		oelon = abs(ACOS(-1.>(COS(olat*!dtor)*COS(lon_a*!dtor)*sin(minfo.lat0*!dtor)+ SIN(olat*!dtor)*COS(minfo.lat0*!dtor))/COS(oelat*!dtor)< 1.)/!dtor)*lon_a/abs(lon_a)

		xx = where(oelon gt 180.0,nxx)
		if nxx gt 0 then oelon[xx] = oelon[xx] - 360.0
		xx = where(oelon le -180.0,nxx)
		if nxx gt 0 then oelon[xx] = oelon[xx] + 360.0
		
		return,[transpose(ary(oelon,/flat)),transpose(ary(oelat,/flat))]
		end	
		
	keyword_set(to_ll) : begin
	
		olon = keyword_set(dim2) ? reform(olon,dim1) : olon
		olat = keyword_set(dim2) ? reform(olat,dim2) : olat

		return,[transpose(ary(olon,/flat)),transpose(ary(olat,/flat))]
		
		end
					  
	else : message,str(/join,'[USAGE] Choose /to_ll,/to_lcc,/to_rll')

endcase		



end
