;-------------------------------------------------------------------------------
;	NAME
;		aqm_plot_wind
;
;	PURPOSE
;		to plot wind arrow or wind barb at aqm domain
;
;	USAGE
;		aqm_plot_wind,u,v[,lon=,lat=|,domain=|,minfo=][,/barb][,position=,option=]
;
;	INPUT
;		u,v      : uwind/vwind 
;		lon/lat  : lon/lat
;		domain   : domain nick name
;		minfo    : map information
;		position : area limit of wind arrow plot for clipping
;
;	KEYWORD
;		barb : plot wind barb
;
;	NOTE
;		to use wind speed/direction use option = {ws:ws,wd:wd}
;		can provide more detailed options using option=opt
;
;	AUTHOR
;		2010-01-08 Hyun Cheol Kim (hyun.kim@noaa.gov)
;-------------------------------------------------------------------------------

	pro aqm_plot_wind,u,v,lon=_lon,lat=_lat,domain=domain,minfo=minfo,$
	    barb=barb,position=p,option=option,$
	    npx=npx,npy=npy,nix=nix,niy=niy	    

;-------------------------------------------------------------------------------	    

opt = {p:[0,0,1,1],npx:15,base:3,factor:1.,hsize:5,color:!p.color,legend:1,barb:0}

if ~struct(/set,opt,'npy') then opt = struct(/merge,opt,{npy:opt.npx})
if var_set(barb) then opt = struct(/merge,opt,{barb:barb})

opt = struct(/merge,opt,option)

	; map information
	
if var_set(_lon)   then lon = reform(_lon)
if var_set(_lat)   then lat = reform(_lat)
if var_set(minfo)  then gr = aqm_grid(minfo=minfo,olon=lon,olat=lat)
if var_set(domain) then gr = aqm_grid(domain=domain,olon=lon,olat=lat)	
if struct(/set,opt,'lon') then lon = opt.lon
if struct(/set,opt,'lat') then lat = opt.lat	

	;
	
if struct(/set,opt,'u')  then u = opt.u
if struct(/set,opt,'v')  then v = opt.v
if struct(/set,opt,'ws') then ws = opt.ws
if struct(/set,opt,'wd') then wd = opt.wd
	
	; reduce dimension
	
dim = var_set(u) ? size(u,/dim) : size(ws,/dim)

if n_elements(dim) eq 2 then begin

	nx = float(!d.x_size)*(opt.p[2]-opt.p[0])/opt.npx
	ix = round(findgen(nx)/(nx-1)*(dim[0]-1))

	ny = float(!d.y_size)*(opt.p[3]-opt.p[1])/opt.npy
	iy = round(findgen(ny)/(ny-1)*(dim[1]-1))

	rr = ceil(float(dim[0])/n_elements(ix))

	if rr le 3 then ix = indgen(dim[0]/rr)*rr
	if rr le 3 then iy = indgen(dim[1]/rr)*rr
	
	if struct(/set,opt,'nix') then ix = indgen(dim[0]/opt.nix)*opt.nix
	if struct(/set,opt,'niy') then iy = indgen(dim[1]/opt.niy)*opt.niy

	lon = lon[ix,iy,*]
	lat = lat[ix,iy,*]
	
	xy = chdmod(lon,lat,/data,/to_device)
	
	if var_set(u)  then u  = u[ix,iy,*]	
	if var_set(v)  then v  = v[ix,iy,*]
	if var_set(ws) then ws = ws[ix,iy,*]
	if var_set(wd) then wd = wd[ix,iy,*]

endif else begin

	xy = chdmod(lon,lat,/data,/to_device)

endelse

	; arrow or wind barb
	
clip = convert_coord(opt.p[[0,2]],opt.p[[1,3]],/normal,/to_device)
		
if keyword_set(opt.barb) then begin

	if ~var_set(ws) then ws = cvdat(u,v,/wind_speed,owd=wd)
	
	plot_wind_barb,ws/0.5147965,wd+180,xy[0,*],xy[1,*],noclip=0,color=color_load(/get,opt.color)

endif else begin

	if ~var_set(u) then u = cvdat(ws,wd,/uwind,ovwind=v)

	arrow2,xy[0,*],xy[1,*],xy[0,*]+u*opt.base*opt.factor,xy[1,*]+v*opt.base*opt.factor,hsize=opt.hsize,noclip=0,color=color_load(/get,opt.color),resize=5

endelse	

	; legend
	
if keyword_set(opt.legend) then begin

	opt2 = struct(/update,{legend_value:5,legend_position:opt.p[[2,3]]+[0,0.01],legend_unit:'m/s',$
	        legend_color:color_load('slategray',/closest),legend_pixel:5,legend_alignment:1.,legend_charsize:0.6},opt)
				 				 
	arrow2,clip[3],clip[4]+opt2.legend_pixel,clip[3]-opt2.legend_value*opt.base*opt.factor,$	
		clip[4]+opt2.legend_pixel,hsize=opt.hsize,color=color_load(/get,opt2.legend_color)
		
	xyouts,clip[3],clip[4]+opt2.legend_pixel*2,str(/join,str(opt2.legend_value),opt2.legend_unit),$
		/device,alignment=opt2.legend_alignment,charsize=opt2.legend_charsize,color=opt2.legend_color


endif	
	
end	    
