;-------------------------------------------------------------------------------	
;	NAME
;		aqm_plot_tile
;
;	PURPOSE
;		to generate spatial (tile) plot 
;
;	USAGE
;		aqm_plot_tile,data[,lon=,lat=|,tile=|,domain=|,minfo=|,file=]
;
;	INPUT
;		data     : 2D data
;		lon      : longitude
;		lat      : lattitude
;		tile     : ll locations of each cell [5,nx*ny]
;		domain   : domain nick name
;		mfino    : map information
;		file     : data file name (to obtain minfo)
;		var      : variable name (used to get predefined color table)
;		tflag    : time tflag string (to plot time stamp)
;		position : plot position
;		xsize    : plot width
;		ysize    : plot height 
;		range    : color index range (pass to color_index)
;		unit     : data unit (color_bar labeling)
;		timezone : convert tflag to given timezone
;		limit    : map_set limit (for zoom in)
;		title    : plot title
;		wind     : add wind arrow (wind = {u:u,v:v,lon:lon,lat:lat}
;		obs      : plot observation circles
;		mapdata  : pass map data (see aqm_plot_map with /skip_plot)
;		missing  : missing value
;		color    : pass color_load
;		gif      : save gif 
;		png      : save png
;		jpg      : save jpg
;		winn     : window position in X display (0~3)
;		option   : any additional options
;
;
;	KEYWORD
;		congrid : use congrid/tv method in tile plotting (default is using polyfill)
;		          congrid is fater, but polyfill is more accurate
;		hide    : do not show plot
;		pause   : pause
;
;	AUTHOR
;		2010-01-13 Hyun Cheol Kim (hyun.kim@noaa.gov)                   
;		2010-05-14 updated cl,ct,ci,cb
;-------------------------------------------------------------------------------	

	pro aqm_plot_tile,_data, $
	    lon=lon,lat=lat,tile=tile, $
	    domain=domain,minfo=minfo,file=file,$
	    var=var,tflag=tflag,position=position,xsize=xsize,ysize=ysize,range=range,$
	    unit=unit,timezone=timezone,limit=limit,congrid=congrid,title=title,$
	    wind=wind,obs=obs,front=front,mapdata=mapdata,missing=missing,color=color,$
	    gif=gif,png=png,jpg=jpg,update=update,$
	    winn=winn,hide=hide,pause=pause,option=option,_extra=_ex
	
;-------------------------------------------------------------------------------	

;COMPILE_OPT IDL2
;ON_ERROR,2

opt = {missing:-999.,min_value:-900.,max_value:1e37,p:[0.03,0.03,0.91,0.9],timezone:'UTC',winn:2,congrid:0,cb:{title:{shift:[10,10],alignment:0.5}}}

if var_set(var) then opt = struct(/merge,opt,aqm_plot_info(var=var))	

if var_set(missing)  then opt.missing = missing
if var_set(position) then opt.p = position
if var_set(winn)     then opt.winn = winn
if var_set(timezone) then opt.timezone  = timezone

if var_set(xsize)    then opt = struct(/merge,opt,{xsize:xsize})
if var_set(ysize)    then opt = struct(/merge,opt,{ysize:ysize})
if var_set(unit)     then opt = struct(/merge,opt,{cb:{title:{text:str(unit,/title)}}})
if var_set(mapdata)  then opt = struct(/merge,opt,{mapdata:mapdata})
if var_set(range)    then opt = struct(/merge,opt,{ci:{range:range}})
if var_set(limit)    then opt = struct(/merge,opt,{limit:limit})
if var_set(congrid)  then opt = struct(/merge,opt,{congrid:congrid})
if var_set(range)    then opt = struct(/merge,opt,{ci:{range:range}})

if var_set(title) then if struct(/set,title) then opt = struct(/merge,opt,{title:title}) else opt = struct(/merge,opt,{title:{text:title}})
if var_set(mesh)  then if struct(/set,mesh)  then opt = struct(/merge,opt,{mesh:mesh})   else opt = struct(/merge,opt,{mesh:{mesh:mesh}})
if var_set(note)  then if struct(/set,note)  then opt = struct(/merge,opt,{note:note})   else opt = struct(/merge,opt,{note:{text:note}})

if struct(/set,option) then opt = struct(/merge,opt,option)

	;
	
data = reform(_data)
dim  = size(data,/dim)
ndim = size(data,/n_dim)

	; map information
	
case 1 of 

	var_set(domain) : minfo = aqm_grid_minfo(domain=domain)
	var_set(file)   : minfo = aqm_grid_minfo(file=file)
	var_set(minfo)  : 
	else : message,str(/join,'USAGE! /minfo,/domain,/file')
	
endcase	

	; get limit if necessary

if ~struct(/set,opt,'limit') then begin

	gr  = aqm_grid(minfo=minfo,ominfo=minfo,olimit=limit)
	opt = struct(/merge,opt,{limit:limit})
	
endif	

	; get tile
	
case 1 of

	var_set(tile) : 

	var_set(lon,lat) : begin
	
		xx = aqm_proj(lon,lat,/ll,/to_lcc,minfo=minfo,olx=lx,oly=ly)
		tile = {lon:array(lx,/tile),lat:array(ly,/tile)}
	
		end
		
	else : gr = aqm_grid(minfo=minfo,otile=tile)				
			
endcase

	; set xsize and ysize
	
	
case 1 of

	struct(/set,opt,'xsize') and struct(/set,opt,'ysize') : 
	struct(/set,opt,'xsize') : opt = struct(/merge,opt,{ysize:round((opt.p[2]-opt.p[0])/(opt.p[3]-opt.p[1])*opt.xsize/(float(minfo.nx)/minfo.ny))})	
	struct(/set,opt,'ysize') : opt = struct(/merge,opt,{xsize:round((opt.p[3]-opt.p[1])/(opt.p[2]-opt.p[0])*opt.ysize*(float(minfo.nx)/minfo.ny))})				
	else : opt = minfo.nx gt minfo.ny ? struct(/merge,opt,{xsize:800,ysize:round((opt.p[2]-opt.p[0])/(opt.p[3]-opt.p[1])*800/(float(minfo.nx)/minfo.ny))}) $
	                                  : struct(/merge,opt,{ysize:800,xsize:round((opt.p[3]-opt.p[1])/(opt.p[2]-opt.p[0])*800*(float(minfo.nx)/minfo.ny))})

endcase	 

	; load colors, if necessary
	; start z-buffer

if keyword_set(hide) then begin
	
	set_plot,'z',/copy
	device,set_resolution=[opt.xsize,opt.ysize],z_buffer=0  
	
	if ~var_set(color) then color = color_load(/basic)
	
endif else begin

	if ~var_set(color) then color = color_load(/basic)
	
	set_plot,'z',/copy
	device,set_resolution=[opt.xsize,opt.ysize],z_buffer=0  
	
endelse	

	; set map / rll / lcc		
				
case str(/low,minfo.proj) of	
	
	'll'  : begin
		limit[[0,2,4,6]] = [-75,-999,-999,-75] > limit[[0,2,4,6]] < [999,75,999,75]		
		lat0 = round(mean(limit[[0,2,4,6]])/30.)*30.
		lon0 = round(mean(limit[[1,3,5,7]])/30.)*30.
		
		map_set,/merc,lat0,lon0,position=opt.p,/noerase,_extra=_ex,limit=limit		
		;map_set,/merc,minfo.nx/2d*minfo.dx+minfo.x0,minfo.ny/2d*minfo.dy+minfo.y0,position=opt.p,/noerase,_extra=_ex,limit=limit
		
		end
		
	'rll' : map_set,/merc,minfo.lat0,minfo.lon0,limit=opt.limit,position=opt.p,/noerase,_extra=_ex
	'ell' : map_set,/merc,mean(opt.limit[[0,2,4,6]]),mean(opt.limit[[1,3,5,7]]),limit=opt.limit,position=opt.p,/noerase,_extra=_ex			
	'lcc' : map_set,/conic,minfo.lat0,minfo.lon0,standard_parallels=[minfo.lat1,minfo.lat2],limit=opt.limit,position=opt.p,/noerase,_extra=_ex

endcase

	; color table & index

ct = color_table(option=struct(/read,opt,'ct'),diff=struct(/read,opt,'use_diff'))
ci = color_index(data,ct=ct,option=struct(/read,opt,'ci'),diff=struct(/read,opt,'use_diff'),group=struct(/set,opt,'mesh'))

!p.color      = color.black
!p.background = color.white

	; update
	
if struct(/read,opt,'update') then begin

	if var_set(png) then begin
	
		ofile = str(png,extension='png PNG')
		
		if file_test(ofile) then begin
		
			message,/info,str(/join,'UPDATE',ofile)
			read_png,ofile,image
			
			if array_equal(image[0:3],byte('done')) then return			
			if array_equal(size(image,/dim),[opt.xsize,opt.ysize]) then tv,image
											
		endif
	
	endif
	
	if var_set(gif) then begin
	
		ofile = str(png,extension='gif GIF')
		
		if file_test(ofile) then begin
		
			message,/info,str(/join,'UPDATE',ofile)
			read_gif,ofile,image
			
			if array_equal(image[0:3],byte('done')) then return			
			if array_equal(size(image,/dim),[opt.xsize,opt.ysize]) then tv,image
											
		endif
	
	endif
	
endif

	; plot tiles
		
if check(/valid,data,index=ind,count=nv,missing=opt.missing,min_value=opt.min_value,max_value=opt.max_value) and ~var_set(image)then begin

		; tile

	if keyword_set(opt.congrid) then begin
	
		pp1 = round(chdmod([tile.lon[0]],[tile.lat[0]],/data,/to_device))
		pp2 = round(chdmod([tile.lon[n_elements(tile.lon)-3]],[tile.lat[n_elements(tile.lon)-3]],/data,/to_device))
		data = congrid(reform(ci.index,dim),pp2[0]-pp1[0],pp2[1]-pp1[1])		
		tv,data,pp1[0],pp1[1],/device
		
	endif else begin
	
		for iv=0L,nv-1 do polyfill,tile.lon[*,ind[iv]],tile.lat[*,ind[iv]],color=ci.index[ind[iv]],/fill,/data				
	
	endelse

		; mesh		

	if struct(/set,opt,'mesh') then begin

		opt_mesh = struct(/update,{mesh:1,color:color.gray,linestyle:0},opt.mesh)
		
		mesh_color = color_load(opt_mesh.color,/get)
		
		for iv=0L,nv-1 do begin
																			
			case ci.gi[ind[iv]] of
		
				1 : plots,tile.lon[[3,4],ind[iv]],tile.lat[[3,4],ind[iv]],    color=mesh_color,/data,linestyle=opt_mesh.linestyle
				2 : plots,tile.lon[[0,1],ind[iv]],tile.lat[[0,1],ind[iv]],    color=mesh_color,/data,linestyle=opt_mesh.linestyle
				3 : plots,tile.lon[[3,0,1],ind[iv]],tile.lat[[3,0,1],ind[iv]],color=mesh_color,/data,linestyle=opt_mesh.linestyle
				else : 
		
			endcase
			
		endfor	
		
	endif 

endif
	
		; map data
		
if struct(/set,opt,'mapdata') then begin
	
	if struct(/set,opt.mapdata) then aqm_plot_map,mapdata=opt.mapdata else aqm_plot_map,opt.mapdata,minfo=minfo

endif else begin

	case str(/low,minfo.proj) of
		
		'll'  : map_set,/merc,lat0,lon0,position=opt.p,/noerase,_extra=_ex,limit=limit,/continent
		'rll' : map_set,/merc,minfo.lat0,minfo.lon0,limit=opt.limit,position=opt.p,/noerase,_extra=_ex,/usa,/continent
		'ell' : map_set,/merc,mean(opt.limit[[0,2,4,6]]),mean(opt.limit[[1,3,5,7]]),limit=opt.limit,position=opt.p,/noerase,_extra=_ex,/usa,/continent		
		'lcc' : map_set,/conic,minfo.lat0,minfo.lon0,standard_parallels=[minfo.lat1,minfo.lat2],limit=opt.limit,position=opt.p,/noerase,_extra=_ex,/usa,/continent

	endcase
	
endelse

	; observatons

if struct(/set,obs) then begin

	if struct(/set,obs,'missing') then if check(/valid,obs.data,missing=obs.missing,index=iv) then obs = struct(/merge,obs,{data:obs.data[iv],lon:obs.lon[iv],lat:obs.lat[iv]}) else skip = 1
	if var_set(image) and keyword_set(skip) then return

	obs_ci  = color_index2(obs.data,ct=ct,option=struct(/read,opt,'ci')) 		       
		
	obs_opt = struct(/merge,{r:4,charsize:0.8,bcolor:color.black,lcolor:color.black,llcolor:color.slategray,format:str(ci.div,/get_format)},obs)
	obs_opt = struct(/merge,{r:4,charsize:0.8,bcolor:array(color.black,count=n_elements(obs.data)),lcolor:array(color.black,count=n_elements(obs.data)),llcolor:array(color.slategray,count=n_elements(obs.data)),format:str(ci.div,/get_format)},obs)
		
	if struct(/set,obs,'label_max') then obs_opt.lcolor  = obs_opt.lcolor* (obs.data lt obs.label_max)+(struct(/set,obs,'lable_max_color') ? obs.label_max_color:color.red)*(obs.data ge obs.label_max)
	if struct(/set,obs,'label_max') then obs_opt.llcolor = obs_opt.llcolor*(obs.data lt obs.label_max)+(struct(/set,obs,'lable_max_color') ? obs.label_max_color:color.red)*(obs.data ge obs.label_max)
			        								       
	if ~keyword_set(skip) then begin
	
		plot_circle,obs.lon,obs.lat,r_device=obs_opt.r,/fill,color=obs_ci.index,bcolor=obs_opt.bcolor,$
		label=str(str(round(obs.data),format=obs_opt.format)),/auto,charsize=obs_opt.charsize,lcolor=obs_opt.lcolor,llcolor=obs_opt.llcolor,$
		nolabel=struct(/read,obs_opt,'nolabel')
		
	endif 	
        	       	
endif	

	; put wind 

case 1 of

	~keyword_set(wind)  :
	struct(/set,wind)   : aqm_plot_wind,option=struct(/merge,wind,{p:opt.p,legend:1})
	var_set(wind,tflag) : aqm_plot_wind,tflag=tflag,domain=domain,position=opt.p,/legend
	else :

endcase

	; contour
	
if struct_set(_ex,'contour') then begin
	
	minfo = aqm_grid(domain=domain,minfo=minfo,olon=lon,olat=lat)
				
	c_data   = struct_read(_ex,'c_data',data)
	c_lon    = struct_read(_ex,'c_lon',lon)
	c_lat    = struct_read(_ex,'c_lat',lat)  				
	c_levels = struct_read(_ex,'c_levels',ci.div1)
	c_labels = struct_read(_ex,'c_labels',replicate(1B,n_elements(c_levels)))
	c_colors = struct_read(_ex,'c_colors',replicate(color_load(/get,'blue'),n_elements(c_levels)))
		
	contour,c_data,c_lon,c_lat,/overplot,levels=c_levels,c_labels=c_labels,c_colors=c_colors
	
endif	

	; add frontal analysis (HPC gif)

if struct_set(_ex,'wmap') then plot_wmap,'us',_ex.wmap
if struct_set(_ex,'wmap_kma') then plot_wmap,'kma',_ex.wmap_kma,symsize=0.1,psym=1
	
	; frame again 

plots,opt.p[[0,2,2,0,0]],opt.p[[1,1,3,3,1]],thick=2,/normal	

	; title

if struct(/set,opt,'title') then begin

	opt_title = struct(/update,{text:'',charsize:1.5,color:!p.color,orientation:0,alignment:0.5},opt.title)
	xyouts,(opt.p[0]+opt.p[2])/2.,opt.p[3]+0.02,opt_title.text,alignment=opt_title.alignment,charsize=opt_title.charsize,orientation=opt_title.orientation,/normal

endif	

	; note

notes = {text:'',color:color.red,charsize:0.8}

if struct(/set,opt,'note.text') then begin

	ntext = n_elements(opt.note.text)
		
	notes = [replicate({text:'',color:color.black,charsize:0.8},ntext),notes]

	if struct(/set,opt,'note.text')     then notes[0:ntext-1].text     = opt.note.text
	if struct(/set,opt,'note.color')    then notes[0:ntext-1].color    = array(color_load(/get,opt.note.color),count=ntext)
	if struct(/set,opt,'note.charsize') then notes[0:ntext-1].charsize = array(opt.note.charsize,count=ntext)

endif	

if var_set(tflag) then notes = [{text:str(/join,str(date(/timezone,tflag,'utc',opt.timezone),/tflag_to_timestamp),opt.timezone),color:color.black,charsize:0.8},notes]

dychar = float(!d.y_ch_size)/!d.y_size/3.

for inote=0L,n_elements(notes)-1 do begin

	xyouts,opt.p[0],opt.p[3]+dychar,notes[inote].text,color=notes[inote].color,charsize=notes[inote].charsize,/normal

	dychar = dychar + float(!d.y_ch_size)/!d.y_size*notes[inote].charsize*1.2

endfor
	
	; put colorbar 

cb = color_bar(ct=ct,ci=ci,position=pos(opt.p,/colorbar,max_length=400),option=struct(/read,opt,'cb'))

	; capture

snapshot = tvrd()

if struct(/read,opt,'update') then if ~keyword_set(skip) then snapshot[0:3] = byte('done')

	;save gif/png/jpg

if keyword_set(gif) then begin

	tvlct,r,g,b,/get
	ofile = str(gif,extension='gif GIF')
	message,/info,str(/join,'WRITE!',ofile) 

	write_gif,ofile,snapshot,r,g,b
	
endif	

if var_set(png) then begin

	tvlct,r,g,b,/get

	ofile = str(png,extension='png PNG')
	message,/info,str(/join,'WRITE!',ofile)
	write_png,ofile,snapshot,r,g,b

endif

if var_set(jpg) then begin

	tvlct,r,g,b,/get
	ofile = str(jpg,extension='jpg JPG')

	message,/info,str(/join,'WRITE!',ofile)
	
	image24 = bytarr(3,opt.xsize,opt.ysize)
	image24[0,*,*] = r[snapshot]
	image24[1,*,*] = g[snapshot]
	image24[2,*,*] = b[snapshot]
	
	write_jpeg,ofile,image24,true=1,quality=100

endif

	; close z-buffer

device,z_buffer=1
set_plot,'x'

	; display in X

if ~keyword_set(hide) then begin
	
	if !d.window ne opt.winn or !d.x_size ne opt.xsize or !d.y_size ne opt.ysize then window,opt.winn,xsize=opt.xsize,ysize=opt.ysize	
	tv,snapshot

endif  

	;

if keyword_set(pause) then pause

end
