;-------------------------------------------------------------------------------				 
;	NAME
;		aqm_grid
;
;	PURPOSE 
;		IDL function to return grid information
;
;	USAGE
;		aqm_grid([file=file][domain=domain],[/dot,/boundary][olon=lon,olat=lat][,tile=tile])
;
;	INPUT
;		file     : file name
;		domain   : domain name
;  
;	OUTPUT
;		olon     : output longitude
;		olat     : output latitude
;		tile     : file information for tile plot
;
;	KEYWORD
;		dot      : return dot
;		boundary : include boundary
;  
;
;	AUTHOR
;		2007-01-09 Hyun Cheol Kim 
; 		2009-12-16 modified minfo as a structure
;		2010-01-11 added zoom
;		2012-09-20 separated aqm_grid_minfo.pro
;-------------------------------------------------------------------------------				 

	function aqm_grid,input,file=_file,domain=_domain,minfo=_minfo,simple=simple,  $  
	         dot=dot,boundary=boundary,radius=radius,nthick=nthick,projection=proj,$
	         olon=olon,olat=olat,olimit=olimit,otile=otile,odomain=odomain,$
	         olx=olx,oly=oly,ominfo=ominfo,onx=onx,ony=ony,omid=omid,$
		 orlon=orlon,orlat=orlat,operimeter=operi,zoom=zoom
					
;-------------------------------------------------------------------------------

	; minfo

minfo  = aqm_grid_minfo(input,file=_file,domain=_domain,minfo=_minfo,radius=radius,proj=proj,zoom=zoom,omid=omid,onx=onx,ony=ony)
ominfo = minfo
	
if keyword_set(simple) then return,minfo	

	; construct array

case str(minfo.proj,/low) of

	'rlle' : begin
		
		lon = ary(dindgen(minfo.nx)*2*minfo.dx+minfo.x0,ny=minfo.ny)
		lat = ary(dindgen(minfo.ny)*minfo.dy+minfo.y0,dx=minfo.dx)
    
		if keyword_set(dot) then ind = indgen(fix(minfo.ny/2))*2+1 else ind = indgen(fix(minfo.ny/2)+1)*2
    
		lon[*,ind] = lon[*,ind] + minfo.dx
    
		tlon = transpose([[ary(lon-minfo.dx,/flat)],[ary(lon,/flat)],[ary(lon+minfo.dx,/flat)],[ary(lon,/flat)],[ary(lon-minfo.dx,/flat)]])
		tlat = transpose([[ary(lat,/flat)],[ary(lat-minfo.dy,/flat)],[ary(lat,/flat)],[ary(lat+minfo.dy,/flat)],[ary(lat,/flat)]])
	               	               
		r  = aqm_map_proj( lon, lat,/rot,/to_ll,out1=olon, out2=olat, rlon=minfo.x0,rlat=minfo.y0)
		r2 = aqm_map_proj(tlon,tlat,/rot,/to_ll,out1=otlon,out2=otlat,rlon=minfo.x0,rlat=minfo.y0)
	 	
		result =  transpose([[ary(olon,/flat)],[ary(olat,/flat)]])

		otile  = {lon:otlon,lat:otlat}	
		olimit = [olat[0,0],olon[0,0],olat[0,nrows-1],olon[0,nrows-1],olat[ncols-1,nrows-1],olon[ncols-1,nrows-1],olat[ncols-1,0],olon[ncols-1,0]]
		omid   = aqm_grid_mid(minfo)
		ominfo = minfo
			
		return,result			
		
		end
		
	'rll' : begin

		if keyword_set(dot) then begin
		
			minfo.x0 -= minfo.dx/2.
			minfo.y0 -= minfo.dy/2.
			minfo.nx ++
			minfo.ny ++

		endif
	
		if keyword_set(boundary) then begin
		
			if ~var_set(nthick) then nthick = 1
			
			minfo.x0 -= minfo.dx*nthick
			minfo.y0 -= minfo.dy*nthick
			minfo.nx += 2*nthick
			minfo.ny += 2*nthick
	
		endif
		
		orlon  = ary(minfo.x0+indgen(minfo.nx)*minfo.dx+minfo.dx/2.,ny=minfo.ny)
		orlat  = ary(minfo.y0+indgen(minfo.ny)*minfo.dy+minfo.dy/2.,nx=minfo.nx)
		
		otrlon = transpose([[ary(orlon-minfo.dx/2,/flat)],[ary(orlon+minfo.dx/2,/flat)],[ary(orlon+minfo.dx/2,/flat)],[ary(orlon-minfo.dx/2,/flat)],[ary(orlon-minfo.dx/2,/flat)]])
		otrlat = transpose([[ary(orlat-minfo.dy/2,/flat)],[ary(orlat-minfo.dy/2,/flat)],[ary(orlat+minfo.dy/2,/flat)],[ary(orlat+minfo.dy/2,/flat)],[ary(orlat-minfo.dy/2,/flat)]])
						
		xx = aqm_proj(orlon,orlat,/rll,/to_ll,minfo=minfo,olon=olon,olat=olat)
		xx = aqm_proj(otrlon,otrlat,/rll,/to_ll,minfo=minfo,olon=otlon,olat=otlat)
				
		olon = reform(olon,[minfo.nx,minfo.ny])
		olat = reform(olat,[minfo.nx,minfo.ny])
						
		result = transpose([[ary(olon,/flat)],[ary(olat,/flat)]])
				
		otile  = {lon:otlon,lat:otlat}
		olimit = [olat[0,0],olon[0,0],olat[0,minfo.ny-1],olon[0,minfo.ny-1],olat[minfo.nx-1,minfo.ny-1],olon[minfo.nx-1,minfo.ny-1],olat[minfo.nx-1,0],olon[minfo.nx-1,0]]
		
		ominfo = minfo
		onx    = minfo.nx
		ony    = minfo.ny
		omid   = aqm_grid_mid(minfo)
				
		return,minfo	
		end
		
	'ell' : begin

		if keyword_set(dot) then begin
		
			minfo.x0 -= minfo.dx/2.
			minfo.y0 -= minfo.dy/2.
			minfo.nx ++
			minfo.ny ++

		endif
	
		if keyword_set(boundary) then begin
		
			if ~var_set(nthick) then nthick = 1
			
			minfo.x0 -= minfo.dx*nthick
			minfo.y0 -= minfo.dy*nthick
			minfo.nx += 2*nthick
			minfo.ny += 2*nthick
	
		endif
		
		orlon  = ary(minfo.x0+indgen(minfo.nx)*minfo.dx+minfo.dx/2.,ny=minfo.ny)
		orlat  = ary(minfo.y0+indgen(minfo.ny)*minfo.dy+minfo.dy/2.,nx=minfo.nx)
		
		otrlon = transpose([[ary(orlon-minfo.dx/2,/flat)],[ary(orlon+minfo.dx/2,/flat)],[ary(orlon+minfo.dx/2,/flat)],[ary(orlon-minfo.dx/2,/flat)],[ary(orlon-minfo.dx/2,/flat)]])
		otrlat = transpose([[ary(orlat-minfo.dy/2,/flat)],[ary(orlat-minfo.dy/2,/flat)],[ary(orlat+minfo.dy/2,/flat)],[ary(orlat+minfo.dy/2,/flat)],[ary(orlat-minfo.dy/2,/flat)]])
						
		xx = aqm_proj(orlon,orlat,/ell,/to_ll,minfo=minfo,olon=olon,olat=olat)
		xx = aqm_proj(otrlon,otrlat,/ell,/to_ll,minfo=minfo,olon=otlon,olat=otlat)
				
		olon = reform(olon,[minfo.nx,minfo.ny])
		olat = reform(olat,[minfo.nx,minfo.ny])
						
		result = transpose([[ary(olon,/flat)],[ary(olat,/flat)]])
				
		otile  = {lon:otlon,lat:otlat}
		olimit = [olat[0,0],olon[0,0],olat[0,minfo.ny-1],olon[0,minfo.ny-1],olat[minfo.nx-1,minfo.ny-1],olon[minfo.nx-1,minfo.ny-1],olat[minfo.nx-1,0],olon[minfo.nx-1,0]]
				
		if arg_present(operi) then begin
		
			operi = transpose([[minfo.x0+[lindgen(minfo.nx)*minfo.dx,replicate(minfo.nx*minfo.dx,minfo.ny),reverse((lindgen(minfo.nx)+1)*minfo.dx),replicate(0,minfo.ny)]],$
			         [minfo.y0+[replicate(0,minfo.nx),lindgen(minfo.ny)*minfo.dy,replicate(minfo.ny*minfo.dy,minfo.nx),reverse((lindgen(minfo.ny)+1)*minfo.dy)]]])		
			operi = aqm_proj(operi[0,*],operi[1,*],/ell,/to_ll,minfo=minfo)
																				
		endif
		ominfo = minfo
		onx    = minfo.nx
		ony    = minfo.ny
		omid   = aqm_grid_mid(minfo)
				
		return,minfo	
		end
		
		
	'lcc' : begin
		
		if keyword_set(dot) then begin
		
			minfo.x0 -= minfo.dx/2.
			minfo.y0 -= minfo.dy/2.
			minfo.nx ++
			minfo.ny ++

		endif
	
		if keyword_set(boundary) then begin
		
			if ~var_set(nthick) then nthick = 1
			
			minfo.x0 -= minfo.dx*nthick
			minfo.y0 -= minfo.dy*nthick
			minfo.nx += 2*nthick
			minfo.ny += 2*nthick
	
		endif
		
		if arg_present(olon) or arg_present(olat) then begin
		
			x = var_set(x,ina=ary(minfo.x0+indgen(minfo.nx)*minfo.dx+minfo.dx/2.,ny=minfo.ny))
			y = var_set(y,ina=ary(minfo.y0+indgen(minfo.ny)*minfo.dy+minfo.dy/2.,nx=minfo.nx))
			m = var_set(m,ina=map_proj_init(3,sphere_radius=minfo.r,standard_par1=minfo.lat1,standard_par2=minfo.lat2,center_longitude=minfo.lon0,center_latitude=minfo.lat0))
			r = map_proj_inverse(x,y,map_structure=m)
			if arg_present(olon) then olon = reform(r[0,*],minfo.nx,minfo.ny,/overwrite)
			if arg_present(olat) then olat = reform(r[1,*],minfo.nx,minfo.ny,/overwrite)
			if arg_present(olx) then olx = x
			if arg_present(oly) then oly = y			
			
		endif
		
		if arg_present(otile) then begin
					
			x = var_set(x,ina=ary(minfo.x0+indgen(minfo.nx)*minfo.dx+minfo.dx/2.,ny=minfo.ny))
			y = var_set(y,ina=ary(minfo.y0+indgen(minfo.ny)*minfo.dy+minfo.dy/2.,nx=minfo.nx))
			m = var_set(m,ina=map_proj_init(3,sphere_radius=minfo.r,standard_par1=minfo.lat1,standard_par2=minfo.lat2,center_longitude=minfo.lon0,center_latitude=minfo.lat0))
							
			tx = transpose([[ary(x-minfo.dx/2,/flat)],[ary(x+minfo.dx/2,/flat)],[ary(x+minfo.dx/2,/flat)],[ary(x-minfo.dx/2,/flat)],[ary(x-minfo.dx/2,/flat)]])
			ty = transpose([[ary(y-minfo.dy/2,/flat)],[ary(y-minfo.dy/2,/flat)],[ary(y+minfo.dy/2,/flat)],[ary(y+minfo.dy/2,/flat)],[ary(y-minfo.dy/2,/flat)]])
								
			r2 = map_proj_inverse(tx,ty,map_structure=m)
			otile = {lon:reform(r2[0,*],5,minfo.nx*minfo.ny,/overwrite),lat:reform(r2[1,*],5,minfo.nx*minfo.ny,/overwrite)}

		endif
		
		if arg_present(olimit) then begin
								
			olimit = transpose([[minfo.x0+[0,0,1,1]*minfo.nx*minfo.dx],[minfo.y0+[0,1,1,0]*minfo.ny*minfo.dy]])			
			m = var_set(m,ina=map_proj_init(3,sphere_radius=minfo.r,standard_par1=minfo.lat1,standard_par2=minfo.lat2,center_longitude=minfo.lon0,center_latitude=minfo.lat0))						
			r = map_proj_inverse(olimit,map_structure=m)			
			olimit = r[[1,0,3,2,5,4,7,6]]
				
		endif
		
		if arg_present(operi) then begin
		
			operi = transpose([[minfo.x0+[lindgen(minfo.nx)*minfo.dx,replicate(minfo.nx*minfo.dx,minfo.ny),reverse((lindgen(minfo.nx)+1)*minfo.dx),replicate(0,minfo.ny)]],$
			         [minfo.y0+[replicate(0,minfo.nx),lindgen(minfo.ny)*minfo.dy,replicate(minfo.ny*minfo.dy,minfo.nx),reverse((lindgen(minfo.ny)+1)*minfo.dy)]]])
			m = var_set(m,ina=map_proj_init(3,sphere_radius=minfo.r,standard_par1=minfo.lat1,standard_par2=minfo.lat2,center_longitude=minfo.lon0,center_latitude=minfo.lat0))						
			operi = map_proj_inverse(operi,map_structure=m)
																				
		endif
										
		ominfo = minfo
		onx    = minfo.nx
		ony    = minfo.ny
		omid   = aqm_grid_mid(minfo)
				
		return,minfo
		end
		
	'll' : begin
	
		if keyword_set(dot) then begin
		
			minfo.x0 -= minfo.dx/2.
			minfo.y0 -= minfo.dy/2.
			minfo.nx ++
			minfo.ny ++

		endif
	
		if keyword_set(boundary) then begin
		
			if not var_set(nthick) then nthick = 1
			
			minfo.x0 -= minfo.dx*nthick
			minfo.y0 -= minfo.dy*nthick
			minfo.nx += 2*nthick
			minfo.ny += 2*nthick
	
		endif
		
		x  = ary(minfo.x0+indgen(minfo.nx)*minfo.dx+minfo.dx/2.,ny=minfo.ny)
		y  = ary(minfo.y0+indgen(minfo.ny)*minfo.dy+minfo.dy/2.,nx=minfo.nx)		
		tx = transpose([[ary(x-minfo.dx/2.,/flat)],[ary(x+minfo.dx/2.,/flat)],[ary(x+minfo.dx/2.,/flat)],[ary(x-minfo.dx/2.,/flat)],[ary(x-minfo.dx/2.,/flat)]])
		ty = transpose([[ary(y-minfo.dy/2.,/flat)],[ary(y-minfo.dy/2.,/flat)],[ary(y+minfo.dy/2.,/flat)],[ary(y+minfo.dy/2.,/flat)],[ary(y-minfo.dy/2.,/flat)]])

		olon = x
		olat = y
		otlon = tx
		otlat = ty
			
		result = transpose([[ary(olon,/flat)],[ary(olat,/flat)]])
		otile  = {lon:otlon,lat:otlat}
		olimit = [olat[0,0],olon[0,0],olat[0,minfo.ny-1],olon[0,minfo.ny-1],olat[minfo.nx-1,minfo.ny-1],olon[minfo.nx-1,minfo.ny-1],olat[minfo.nx-1,0],olon[minfo.nx-1,0]]
		olx    = x
		oly    = y
		ominfo = minfo
		onx    = minfo.nx
		ony    = minfo.ny
		omid   = aqm_grid_mid(minfo)
		
		if arg_present(operi) then begin
		
			operi = transpose([[minfo.x0+[lindgen(minfo.nx)*minfo.dx,replicate(minfo.nx*minfo.dx,minfo.ny),reverse((lindgen(minfo.nx)+1)*minfo.dx),replicate(0,minfo.ny)]],$
			         [minfo.y0+[replicate(0,minfo.nx),lindgen(minfo.ny)*minfo.dy,replicate(minfo.ny*minfo.dy,minfo.nx),reverse((lindgen(minfo.ny)+1)*minfo.dy)]]])
																		
		endif		
		
		return,minfo		
		end	
		
	else : message,'MAP_PROJECTION needed (LCC/RLL/RLLe/LL)'

endcase

end
