;-------------------------------------------------------------------------------
;	NAME
;		aqm_array
;
;	PURPOSE
;		calculate various min/max/mean calculation with missing value handling and dimensional operation
;
;	USAGE
;		see example	
;
;	INPUT
;		data      : data array
;		n1        : dimension (min/max/mean) or tflag (daily_)
;		ltflag    : local time flag
;		count_min : least data number
;		missing   : missing value
;		nth       : if used with /max, returns n-th maximum values
;
;	KEYWORD 
;		mean            : calculate mean
;		max             : calculate max
;		min             : calculate min
;		daily_max       : calculate daily max
;		daily_min       : calculate daily min
;		daily_average   : calculate daily average;		
;		ltflag          : return local time adjested data array
;		diurnal_average : return diurnal average
;		finite          : convert NAN/Inf to missing value
;		nan             : convert missing value to NAN
;		quiet           : no verse display
;
;	OUTPUT
;		otflag : return date strings for "daily" calculation
;		oindex : indices of returning results from original data array
;
;	EXAMPLE
;		r = aqm_array(data,/mean[,/min,/max])
;		r = aqm_array(data,1,/mean) (data = [3,4,5] --> r = [4,5])
;		r = aqm_array(data,tflag1,lon,tflag2,/local_time)
;
;	AUTHOR
;		2010-06-29 Hyun Cheol Kim (hyun.kim@noaa.gov)
;		2011-01-12 updated for better handling of NAN/INF
;		2011-03-02 added /add,/subtract, and /double keyword
;		2011-04-15 added /local_time,/backward
;		2011-09-21 added /sort,/nth
;		2011-11-03 added /percentile
;		2013-01-07 modified /moving_average, nmove can have two components
;		2013-08-12 modified /sort,/nth,/percentile
;-------------------------------------------------------------------------------

	function aqm_array,_data,n1,n2,n3,double=double,$
	         mean=_mean,max=_max,min=_min,sort=_sort,stddev=stddev,total=_total,$
	         local_time=local_time,backward=backward,nth=nth,percentile=percentile,$
	         add=add,subtract=subtract,operation=operation,$
	         daily_max=daily_max,daily_average=daily_average,$
		 moving_average=moving_average,nmove=nmove,timezone=timezone,ltflag=ltflag,$
	         diurnal_average=diurnal_average,odim=odim,$
	         finite=finite,nan=nan,overwrite=overwrite,$		 
	         count_min=count_min,missing=missing,guess=guess,$
		 otflag=otflag,oindex=oindex,ocount=ocount,quiet=quiet,option=_opt,test=_test
	
;-------------------------------------------------------------------------------

COMPILE_OPT IDL2

opt = {missing:-999.,count_min:1L}
opt = keyword_set(guess) ? struct(/update,opt,data_info_missing(_data)) : opt
opt = struct(/merge,opt,struct(/initiate,'missing',missing,'count_min',count_min))
opt = struct(/merge,opt,_opt)

data = keyword_set(overwrite) ? (temporary(keyword_set(double) ? double(_data) : float(_data))) : keyword_set(double) ? double(_data): float(_data)
data = data_check(data,data eq opt.missing,keyword_set(double) ? !values.d_nan : !values.f_nan)
dim  = size(data,/dim)
ndim = n_elements(dim)

case 1 of

	var_set(operation) : begin
	
		operation = var_set(n2) ? n2 : operation
	
		case operation of
		
			'+' : r = data+data_check(keyword_set(double) ? double(n1) : float(n1),n1 eq opt.missing,keyword_set(double) ? !values.d_nan : !values.f_nan)
			'-' : r = data-data_check(keyword_set(double) ? double(n1) : float(n1),n1 eq opt.missing,keyword_set(double) ? !values.d_nan : !values.f_nan)
			'*' : r = data*data_check(keyword_set(double) ? double(n1) : float(n1),n1 eq opt.missing,keyword_set(double) ? !values.d_nan : !values.f_nan)
			'/' : r = data/data_check(keyword_set(double) ? double(n1) : float(n1),n1 eq opt.missing,keyword_set(double) ? !values.d_nan : !values.f_nan)
		
		endcase
		
		return,data_check(r,~finite(r),opt.missing)					
		end

	var_set(add)          : return,data_check((r=data+data_check(keyword_set(double) ? double(n1) : float(n1),n1 eq opt.missing,keyword_set(double) ? !values.d_nan : !values.f_nan)),~finite(r),opt.missing)				
	var_set(subtract)     : return,data_check((r=data-data_check(keyword_set(double) ? double(n1) : float(n1),n1 eq opt.missing,keyword_set(double) ? !values.d_nan : !values.f_nan)),~finite(r),opt.missing)			
	var_set(n2,value='+') : return,data_check((r=data+data_check(keyword_set(double) ? double(n1) : float(n1),n1 eq opt.missing,keyword_set(double) ? !values.d_nan : !values.f_nan)),~finite(r),opt.missing)
	var_set(n2,value='-') : return,data_check((r=data-data_check(keyword_set(double) ? double(n1) : float(n1),n1 eq opt.missing,keyword_set(double) ? !values.d_nan : !values.f_nan)),~finite(r),opt.missing)
	var_set(n2,value='*') : return,data_check((r=data*data_check(keyword_set(double) ? double(n1) : float(n1),n1 eq opt.missing,keyword_set(double) ? !values.d_nan : !values.f_nan)),~finite(r),opt.missing)
	var_set(n2,value='/') : return,data_check((r=data/data_check(keyword_set(double) ? double(n1) : float(n1),n1 eq opt.missing,keyword_set(double) ? !values.d_nan : !values.f_nan)),~finite(r),opt.missing)
			
	var_set(_mean) : begin
	
		if ~var_set(quiet) then message,/info,str(/join,'MEAN (missing =',opt.missing,')')	
		if ~var_set(n1) then return,mean(data_check(data,(data ne opt.missing)*finite(data)))
		
		count  = total(finite(data),n1,/nan)		
		result = total(data,n1,/nan)/count
		result = data_check(result,(~finite(result))*(count lt opt.count_min),opt.missing)
	
		return,result
		end
		
	var_set(_total) : return,data_check((rr = var_set(n1) ? total(data,n1,/nan) : total(data,/nan)),~finite(rr),opt.missing)		
		
	var_set(stddev) : begin
	
		data = transpose(data,[n1-1,idx(indgen(size(data,/n_dim)),[n1-1],/exclusive)])
		dim2 = size(data,/dim)

		data = reform(data,[dim2[0],product(dim2[1:*])])
		rr   = make_array(dim2[1:*],value=!values.f_nan)
		
		for i=0L,product(dim2[1:*])-1 do begin
							
			case total(finite((s = data[*,i]))) of
	
				0 : 
				1 : rr[i] = 0
				else : rr[i] = stddev(s[where(finite(s))])
	
			endcase
	
		endfor
		
		return,data_check(rr,~finite(rr),opt.missing)		
		end
				
	var_set(_sort) : begin
		
		if ~var_set(quiet) then message,/info,str(/join,'NTH (missing =',opt.missing,')')						
		if (ndim eq 1) or ~var_set(n1) then begin & oindex = keyword_set(_max) ? reverse(shift(sort(data),total(~finite(data)))) : sort(data) & return,data[oindex] & endif

		data = reform(transpose(data,(xx=[n1-1,(xx1=where(~idx([n1-1],/mask,count=ndim)))])),[dim[n1-1],product(dim[xx1])],/overwrite)
		ind2 = lonarr((dim2=size(data,/dim)),/nozero)		
		
		for i=0L,dim2[1]-1 do ind2[*,i] = (keyword_set(_max) ? reverse(shift(sort(data[*,i]),total(~finite(data[*,i])))) : sort(data[*,i]))+dim2[0]*i			
				
		if var_set(percentile) then nth = total(finite(data),1)*percentile/100.
						
		data = var_set(nth) ? data[(ind2=transpose(reform(ind2[(dim2[0]-1)<array2(nth-1,count=product(dim[xx1]))>0,lindgen(product(dim[xx1]))],[1,dim[xx1]],/overwrite),[shift(indgen(n1),-1),n1 eq ndim ? !null : indgen(ndim-n1)+n1]))] $
				    : data[(ind2=transpose(reform(ind2,[dim[n1-1],dim[xx1]],/overwrite),[shift(indgen(n1),-1),n1 eq ndim ? !null : indgen(ndim-n1)+n1]))]	
					    								
		if arg_present(oindex) then oindex = (transpose(lindgen(dim),xx))[ind2]
						
		return,data_check(data,~finite(data),opt.missing)		
										
		; note for sort
		;aa = array2(rr,n,/backward,odim=dim2)
		;aa = transpose(reform(reform(aa[((xx1=sort(aa)))[sort((rebin(lindgen(dim2[0]),dim2))[xx1])]],reverse(dim2)),[dim[n-1],dim[(xx2= where(~idx([n-1],/mask,count=n_elements(dim))))]]),[n-1,xx2])
		end
			
	var_set(_max) : begin
	
		if ~var_set(quiet) then message,/info,str(/join,'MAX (missing =',opt.missing,')')	
		if ~var_set(n1) then return,max(data_check(data,(data ne opt.missing)*finite(data)))
		
		count  = total(finite(data),n1,/nan)		
		result = max(data,oindex,dimension=n1,/nan)
		result = data_check(result,(~finite(result))*(count lt opt.count_min),opt.missing)
			
		return,result		
		end				
		
	var_set(_min) : begin
	
		if ~var_set(quiet) then message,/info,str(/join,'MIN (missing =',opt.missing,')')
		if ~var_set(n1) then return,min(data_check(data,(data ne opt.missing)*finite(data)))
		
		count  = total(finite(data),n1,/nan)		
		result = min(data,oindex,dimension=n1,/nan)
		result = data_check(result,(~finite(result))*(count lt opt.count_min),opt.missing)
	
		return,result		
		end
											
	var_set(local_time) : begin
	
		if ~var_set(quiet) then message,/info,str(/join,'LOCAL TIME (missing= ',opt.missing,')')
	
		ndata1 = ndim eq 1 ? 1 : product(dim[0:ndim-2])		
		tflag1 = n1
		tflag2 = var_set(n3) ? n3 : n1
		nt1    = n_elements(tflag1)
		nt2    = n_elements(tflag2)
		
		lon = n2 mod 360
		dt  = keyword_set(timezone) ? n2 : -round((lon*(lon le 180)+(lon-360)*(lon gt 180))/15.)*(keyword_set(backward) ? -1 : 1)
		
		data   = [[make_array([ndata1,24],value=opt.missing)],[reform(data,[ndata1,nt1],/overwrite)],[make_array([ndata1,24],value=opt.missing)]]
		tflag1 = [date(date(min(tflag1),-1,/incr,/day),24,/make,/hour),tflag1,date(date(max(tflag1),1,/incr,/hour),24,/make,/hour)]
		dim2   = size(data,/dim)
		
		ii = lindgen(dim2)+rebin(ary(dt,count=ndata1),dim2)*ndata1				
		rr = data_check(data[[ii]],ii lt 0 or ii ge product(dim2),opt.missing)
		
		ii = idx(tflag1,tflag2,/point)
		rr = rr[*,[ii]]
		rr = data_check(rr,rebin(transpose(ii),[ndata1,nt2]) eq -1,opt.missing)
		rr = reform(rr,[dim[0:ndim-2],nt2])
		
		return,data_check(rr,~finite(rr),opt.missing)	
		end		
		
	var_set(daily_max,_test) : begin
		
		day1 = ary(date(n1),/uniq)
		day2 = var_set(n2,ina=day1)
		
		data = ary(data,ndim,/forward,odim=dim2,/overwrite)		
		data = data[(xx=ary(idx(n1,date(min(day1)+'00',max(day1)+'23',/make),/point),ny=dim2[1]))+ary(lindgen(dim2[1])*dim2[0],nx=n_elements(day1)*24L)]
		data = reform(data_check(data,xx eq -1,!values.f_nan),[24,n_elements(day1),dim2[1]],/overwrite)				
		nn   = finite(data)							
		data = reform(transpose(data_check(max(data,dim=1,/nan),idx(day1,day2,/point),/index,dim=1)),[dim[0:ndim-2],n_elements(day2)],/overwrite)		
		nn   = reform(transpose(data_check(total(nn,1),idx(day1,day2,/point),/index,dim=1)),[dim[0:ndim-2],n_elements(day2)],/overwrite)		
		data = data_check(data,~finite(data) or (nn lt var_set(count_min,ina=24)),-999.)
				
		return,data	
		end	
																	
	var_set(daily_max) : begin
						
		if ~var_set(quiet) then message,/info,str(/join,'DAILY MAX (missing= ',opt.missing,')')
		
		ndata1 = ndim eq 1 ? 1 : product(dim[0:ndim-2])						
		tflag  = n1							
		day1   = array2(date(tflag,/cut,/year,/month,/day),/uniq)
		day2   = var_set(n2) ? array(date(n2,/cut,/year,/month,/day),/uniq) : day1				
		nday   = n_elements(day1)
		count_min = var_set(count_min) ? count_min : 24
			
		data   = reform(data,[ndata1,dim[ndim-1]],/overwrite)
		result = make_array([ndata1,nday],value=opt.missing)
		ocount = lonarr(nday)
										
		for iday=0L,nday-1 do begin
		
				; xtime here is index (don't be confused with daily_average)
		
			xtime = idx(date(tflag,/cut,/year,/month,/day),day1[iday],/contain,count=nxtime)
						
			if nxtime ge count_min then begin
														
				if total(total(finite(data[*,xtime]),1) ne 0) ge count_min then begin
																		
					result[*,iday] = aqm_array(data[*,xtime],2,/max,/quiet,missing=opt.missing)				
					ocount[iday] = total(finite(data[*,xtime]))
				
				endif
															
			endif				
			
		endfor
						
		ii = idx(day1,day2,/point)		
		rr = result[*,[ii]]			
		rr = data_check(rr,rebin(transpose(ii),[ndata1,n_elements(ii)]) eq -1,opt.missing)
		rr = reform(rr,[ndim eq 1 ? 1 : dim[0:ndim-2],n_elements(day2)],/overwrite)
				
		otflag = day2
		
		return,rr
		end
						
	var_set(daily_average) : begin
				
			; usages			
			; r = aqm_array(data,tflag,/daily_average,otflag=dtflag)
			; r = aqm_array(data,tflag,[2,3,4],/daily_average,otflag=dtflag) ; 2-4AM average
			; r = aqm_array(data,tflag,dtflag,/daily_average)
			; r = aqm_array(data,tflag,[2,3,4],dtflag,/daily_average,otflag=dtflag) 
	
		if ~var_set(quiet) then message,/info,str(/join,'DAILY AVERAGE (missing= ',opt.missing,')')
		
		if  var_set(n2) then if max(long(n2)) le 100 then hour = n2 else day2 = array(date(n2,/cut,/year,/month,/day),/uniq)
		if  var_set(n3) then if max(long(n3)) le 100 then hour = n3 else day2 = array(date(n3,/cut,/year,/month,/day),/uniq)		
		if ~var_set(hour) then hour = lindgen(24)
				
		tflag = n1
		
		if max(hour) ge 24 then xx = where(date(tflag,/cut,/hour) lt 12,nxx)
		if keyword_set(nxx) then tflag[xx] = date(date(tflag[xx],-1,/incr,/day),/cut,/year,/month,/day)+str(fix(date(tflag[xx],/cut,/hour))+24,digit=2)
												
		day1  = array(date(tflag,/cut,/year,/month,/day),/uniq)
		nday  = n_elements(day1)
				
		if ~var_set(day2) then day2 = day1
				
		count_min = var_set(count_min) ? count_min : n_elements(hour)
				
		data   = reform(data,[ndim eq 1 ? 1 : product(dim[0:ndim-2]),dim[ndim-1]],/overwrite)
		rr     = make_array([ndim eq 1 ? 1 : product(dim[0:ndim-2]),nday],value=opt.missing)
		ocount = lonarr(nday)
													
		for iday=0L,nday-1 do begin
		
				; xtime is not an index, but a mask
		
			xtime = idx(idx(date(tflag,/cut,/year,/month,/day),day1[iday],/contain,count=nxtime),/mask,count=n_elements(tflag)) and $
			        idx(idx(date(tflag,/cut,/hour),str(hour,digit=2),/contain),/mask,count=n_elements(tflag))
									
			if total(xtime) ge count_min then begin
									
				if total(total(finite(data[*,where(xtime)]),1) ne 0) ge count_min then rr[*,iday] = total(xtime) eq 1  ? data[*,where(xtime)] : $
				aqm_array(data[*,where(xtime)],2,/mean,/quiet,missing=opt.missing,count_min=opt.count_min)
				
				ocount[iday] = total(finite(data[*,where(xtime)]))
											
			endif				
													
		endfor
								
			;
			
		if ~array_equal(day1,day2) then begin
		
			rr     = data_check(rr[*,[(xx = idx(day1,day2,/point))]],xx eq -1,opt.missing,dim=2)
			ocount = data_check(ocount[[xx]],xx eq -1,opt.missing)													
					
		endif	
						
		rr = reform(rr,[ndim eq 1 ? 1 : dim[0:ndim-2],n_elements((otflag = day2))],/overwrite)
			
		return,data_check(rr,~finite(rr),opt.missing)
	
		end
		
	var_set(moving_average) and (n_elements(nmove) eq 2) : begin
		
		if ~var_set(count_min) then count_min = 5L
		if ~var_set(quiet) then message,/info,str(/join,'[TEST] moving average (nmove =',nmove,')')
		
		nn    = max(abs(nmove))				
		data  = reform(data,[ndim eq 1 ? 1 : product(dim[0:ndim-2]),dim[ndim-1]],/overwrite)
		count = finite(data)*(data ne opt.missing)
		data  = data_check(data,~count,0)
										
			;

		dim2 = size(data,/dim)
		
		data   = [[fltarr(dim2[0],nn)],[temporary(data)],[fltarr(dim2[0],nn)]]
		count  = [[fltarr(dim2[0],nn)],[temporary(count)],[fltarr(dim2[0],nn)]]
		tdata  = fltarr(size(data,/dim))
		tcount = lonarr(size(count,/dim))
				
		for imove=min(nmove),max(nmove) do begin
					
			tdata  += shift(data,[0,-imove])
			tcount += shift(count,[0,-imove])										
		endfor
		
		result = temporary(tdata)/tcount
		result = (temporary(result))[*,nn:nn+dim2[1]-1]
		tcount = (temporary(tcount))[*,nn:nn+dim2[1]-1]
		result = data_check(result,~finite(result) or tcount lt opt.count_min,opt.missing)
		
		return,reform(result,dim,/overwrite)			
	
		end		
				
	var_set(moving_average) : begin
	
		if ~var_set(nmove) then nmove = 8
		if nmove eq 0 then nmove = 1
		if keyword_set(backward) then nmove = -nmove
		if ~var_set(count_min) then count_min = 5	
		if ~var_set(quiet) then message,/info,str(/join,'MOVING_AVERAGE ( nmove =',nmove,')')
		
		data  = reform(data,[ndim eq 1 ? 1 : product(dim[0:ndim-2]),dim[ndim-1]],/overwrite)
		count = finite(data)*(data ne opt.missing)
		data  = data_check(data,~count,0)
		
			; extend array
		
		dim2  = size(data,/dim) 								
		
		data   = nmove ge 0 ? [[temporary(data)],[intarr(dim2[0],abs(nmove))]] : [[intarr(dim2[0],abs(nmove))],[temporary(data)]]
		count  = nmove ge 0 ? [[temporary(count)],[intarr(dim2[0],abs(nmove))]] : [[intarr(dim2[0],abs(nmove))],[temporary(count)]]				
		tcount = count
		tdata  = data
		
			; calculate sum and average
							
		for imove=0L,abs(nmove)-2 do begin
		
			data   = shift(temporary(data),[0,-abs(nmove)/nmove])
			count  = shift(temporary(count),[0,-abs(nmove)/nmove])		
			tcount += count
			tdata  += data 
					
		endfor				
		
		result = temporary(tdata)/tcount
		result = nmove ge 0 ? (temporary(result))[*,0:dim2[1]-1] : (temporary(result))[*,-nmove:*]
		tcount = nmove ge 0 ? (temporary(tcount))[*,0:dim2[1]-1] : (temporary(tcount))[*,-nmove:*]
		result = data_check(result,~finite(result) or tcount lt opt.count_min,opt.missing)
		
		return,reform(result,dim,/overwrite)	
		end
		
	var_set(diurnal_average) : begin
	
		if ~var_set(quiet) then message,/info,str(/join,'DIURNAL_AVEAGE ( missing =',opt.missing,')')
		
		tflag = n1
		ntime = 24L
							
		data   = ndim eq 1 ? data : reform(data,[product(dim[0:ndim-2]),dim[ndim-1]],/overwrite)
		result = make_array([ndim eq 1 ? 1 : product(dim[0:ndim-2]),ntime],value=opt.missing)
										
		for itime=0L,ntime-1 do begin
		
			xtime = idx(date(tflag,/cut,/hour),str(itime,digit=2),/contain,count=nxtime)
						
			if nxtime gt 1 then result[*,itime] = aqm_array(data[*,xtime],2,/mean,/quiet,missing=opt.missing)
							
		endfor
		
		if var_set(n2) then begin
		
			if ~var_set(quiet) then message,/info,str(/join,'also LOCAL_TIME shift')
					
			lon    = n2 mod 360
			dt     = round((lon*(lon le 180)+(lon-360)*(lon gt 180))/15.)			
			result = result[(lindgen(n_elements(result))-rebin(n_elements(lon)*dt,[size(lon,/dim),n_elements(result)/n_elements(lon)])) mod n_elements(result)]
						
		endif
		
		data   = reform(data,dim,/overwrite)
		result = ndim eq 1 ? result : reform(result,[dim[0:ndim-2],ntime],/overwrite)
		
		return,result		
		end		
		
	keyword_set(finite) : return,data_check(data,~finite(data),opt.missing)
		
	keyword_set(nan) : return,data_check(data,data eq opt.missing,keyword_set(double) ? !values.d_nan : !values.f_nan)
		
endcase

end
