!
! Here, the main body of REcoM2 is located:
!
!	- subroutine REcoM_sms
!	- subroutine Cobeta
! 
! And the functions:
!	- recom_limiter
!	- iron_chemistry
!
!===============================================================================
! Subroutine REcoM_sms
!===============================================================================
subroutine REcoM_sms(cosAI,Nn,state,dzF,recipdzF,SurfSW,dt_s,sms,Temp,SinkVel,zF)

  Use REcoM_params
  Use REcoM_constants
  Use REcoM_declarations
  use REcoM_LocVar
  Implicit none

  Real(kind=8),                      intent(in)    :: cosAI  !
  Integer,                           intent(in)    :: Nn                   ! Total number of nodes in the vertical
  Real(kind=8),dimension(Nn,bgc_num),intent(inout) :: state                ! ChlA conc in phytoplankton [mg/m3]
  Real(kind=8),dimension(Nn),        intent(in)    :: dzF                  ! [m] Thickness of control volumes
  Real(kind=8),dimension(Nn),        intent(in)    :: recipdzF             ! [1/m] 
  Real(kind=8),                      intent(in)    :: SurfSW               ! [W/m2] ShortWave radiation at surface
  Real(kind=8),                      intent(in)    :: dt_s                 ! [s] Size of time steps
  Real(kind=8),dimension(Nn,bgc_num),intent(inout) :: sms                  ! Source-Minus-Sinks term
  Real(kind=8),dimension(Nn)        ,intent(in)    :: Temp                 ! [degrees C] Ocean temperature
  Real(kind=8),dimension(Nn+1,3)    ,intent(in)    :: SinkVel
  Real(kind=8),dimension(Nn+1)      ,intent(in)    :: zF                   ! [m] Depth of control volume borders
  Real(kind=8)                                     :: dt                   ! Size of time steps [day]
  Real(kind=8),dimension(Nn)                       :: Sink
  Real(kind=8)                                     :: dt_sink              ! Size of local time step
  Integer                                          :: k,step,ii
  Real(kind=8)                                     :: & 
    DIN,     & ! [mmol/m3] Dissolved Inorganic Nitrogen 	
    DIC,     & ! [mmol/m3] Dissolved Inorganic Carbon
    Alk,     & !? [mmol/m3] Total Alkalinity
    PhyN,    & ! [mmol/m3] Intracellular conc of Nitrogen in small phytoplankton
    PhyC,    & ! [mmol/m3] Intracellular conc of Carbon in small phytoplankton
    Chl,     & ! [mg/m3] Current intracellular ChlA conc.
    DetN,    & ! [mmol/m3] Conc of N in Detritus
    DetC,    & ! [mmol/m3] Conc of C in Detritus
    HetN,    & ! [mmol/m3] Conc of N in heterotrophs
    HetC,    & ! [mmol/m3] Conc of C in heterotrophs
    DON,     & ! [mmol/m3] Dissolved organic N in the water
    EOC,     & ! [mmol/m3] Dissolved Organic C conc
    DiaN,    &
    DiaC,    &
    DiaChl,  &
    DiaSi,   &
    DetSi,   &
    Si,      &
    Fe,      &
    PhyCalc, &
    DetCalc, &
    FreeFe
	
  sms = zero
!-------------------------------------------------------------------------------
! Size of REcoM time steps are calculated [day]
!-------------------------------------------------------------------------------

  rTref =  real(one)/recom_Tref
  
  dt	 =	dt_s/SecondsPerDay ! Size of fysics time step [day]
  dt	 =	dt/real(biostep)   ! Size of REcoM time step [day]

!-------------------------------------------------------------------------------
!Main time loop starts
  do step  = one,biostep

    kdzUpper	= 0.d0	            ! Upper light attenuation of top cell is set to zero
    do k  = one,Nn
      do ii = one,bgc_num
        if (abs(sms(k,ii)) .le. tiny) sms(k,ii) = zero
      end do
    end do 

!-------------------------------------------------------------------------------
! Main vertical loop starts
  do k = one,Nn

    DIN    = max(tiny,state(k,idin)   + sms(k,idin  )) ! Avoids division by zero
    DIC    = max(tiny,state(k,idic)   + sms(k,idic  )) ! and updates Conc between
    ALK    = max(tiny,state(k,ialk)   + sms(k,ialk  )) ! local steps in REcoM when
    PhyN   = max(tiny,state(k,iphyn)  + sms(k,iphyn )) ! biostep > 1
    PhyC   = max(tiny,state(k,iphyc)  + sms(k,iphyc ))
    Chl    = max(tiny,state(k,ipchl)  + sms(k,ipchl ))
    DetN   = max(tiny,state(k,idetn)  + sms(k,idetn ))
    DetC   = max(tiny,state(k,idetc)  + sms(k,idetc ))
    HetN   = max(tiny,state(k,ihetn)  + sms(k,ihetn ))
    HetC   = max(tiny,state(k,ihetc)  + sms(k,ihetc ))
    DON    = max(tiny,state(k,idon)   + sms(k,idon  ))
    EOC    = max(tiny,state(k,idoc)   + sms(k,idoc  ))
    DiaN   = max(tiny,state(k,idian)  + sms(k,idian ))
    DiaC   = max(tiny,state(k,idiac)  + sms(k,idiac ))
    DiaChl = max(tiny,state(k,idchl)  + sms(k,idchl ))
    DiaSi  = max(tiny,state(k,idiasi) + sms(k,idiasi))
    DetSi  = max(tiny,state(k,idetsi) + sms(k,idetsi))
    Si     = max(tiny,state(k,isi)    + sms(k,isi   ))
    Fe     = max(tiny,state(k,ife)    + sms(k,ife   ))
    FreeFe = zero
#ifdef REcoM_calcification		
    PhyCalc= max(tiny,state(k,iphycal)+ sms(k,iphycal))
    DetCalc= max(tiny,state(k,idetcal)+ sms(k,idetcal))

    calc_diss      = calc_diss_rate * SinkVel(k,ivdet) /20.d0 ! Dissolution rate of CaCO3 scaled by the sinking velocity at the current depth
#endif

    quota          =  PhyN / PhyC
    recipquota     =  real(one) / quota
    Chl2C          =  Chl  / PhyC
    Chl2N          =  Chl  / PhyN

    quota_dia      =  DiaN / DiaC
    recipQuota_dia =  real(one)/quota_dia
    Chl2C_dia      =  DiaChl / DiaC
    Chl2N_dia      =  DiaChl / DiaN
    qSiC           =  DiaSi / DiaC
    qSiN           =  DiaSi / DiaN

    recipQZoo      =  HetC / HetN

!-------------------------------------------------------------------------------
! Temperature dependence of rates
!-------------------------------------------------------------------------------
    rTloc          = real(one)/(Temp(k) + C2K)
    arrFunc        = exp( -Ae * ( rTloc - rTref))
!-Silicate temperature dependence
    reminSiT       = reminSi * arrFunc
    if (useReminSiT) reminSiT = &
      min(1.32e16 * exp(-11200.d0 * rTloc),reminSi)
!-------------------------------------------------------------------------------
! Photosynthesis section, light parameters and rates
!-------------------------------------------------------------------------------
! Small phytoplankton

    qlimitFac     = recom_limiter(NMinSlope,NCmin,quota)
    if (Felimit) then
      feLimitFac  = Fe/(k_Fe + Fe)
      qlimitFac   = min(qlimitFac,feLimitFac)
    endif

    pMax          = P_cm * qlimitFac * arrFunc

!-------------------------------------------------------------------------------
! Diatoms

    qlimitFac     = recom_limiter(NMinSlope,NCmin_d,quota_dia)
    qlimitFacTmp  = recom_limiter(SiMinSlope,SiCmin,qSiC)
    qlimitFac     = min(qLimitFac,qlimitFacTmp)
    if (Felimit) then
      feLimitFac  = Fe/(k_Fe_d + Fe)
      qlimitFac   = min(qlimitFac,feLimitFac)
    endif

    pMax_dia      = P_cm_d * qlimitFac * arrFunc

!-------------------------------------------------------------------------------
! Light

    kappa          =  k_w + a_chl * (Chl + DiaChl)
    kappastar      =  kappa / cosAI
    kdzLower       =  kdzUpper + kappastar*dzF(k)
    Upperlight     =  SurfSW * exp(-kdzUpper)
    Lowerlight     =  SurfSW * exp(-kdzLower)
    PARave         =  (Upperlight - Lowerlight) &
     						 / kappastar * recipdzF(k)
    PARave         =  max(tiny,PARave)
    kdzUpper       =  kdzLower

!-------------------------------------------------------------------------------
! Small phytoplankton photosynthesis rate

    if ( pMax .lt. tiny) then
      Cphot       = zero
    else
      Cphot       = pMax*( real(one) &
                   - exp(-alfa * Chl2C * PARave / pMax))
    end if
    if ( Cphot .lt. tiny) Cphot = zero
    
 !------------------------------------------------------------------------------
 ! Diatom photosynthesis rate

    if ( pMax_dia .lt. tiny ) then
      Cphot_dia   = zero
    else
      Cphot_dia   = pMax_dia * (real(one) &
           	- exp( -alfa_d * Chl2C_dia * PARave / pMax_dia))
    end if
    if (Cphot_dia .lt. tiny) Cphot_dia = zero

!-------------------------------------------------------------------------------
! Assimilation section
!-------------------------------------------------------------------------------
! Compute assimilation from Geider et al 1998

    V_cm           = V_cm_fact
    limitFacN      = recom_limiter(NMaxSlope,quota,NCmax)
    N_assim        = V_cm * pMax * NCuptakeRatio &
                      * limitFacN * (DIN/(DIN + k_din))

    V_cm           = V_cm_fact_d
    limitFacN_dia  = recom_limiter(NMaxSlope,quota_dia,NCmax_d)
    N_assim_dia    = V_cm * pMax_dia * NCUptakeRatio_d &
                      * limitFacN_dia * DIN/(DIN + k_din_d)

    limitFacSi     = recom_limiter(SiMaxSlope,qSiC,SiCmax)  &
                      * limitFacN_dia
    Si_assim       = V_cm * P_cm_d * arrFunc * SiCUptakeRatio &
                      * limitFacSi * Si/(Si + k_si)

!-------------------------------------------------------------------------------
! Iron chemistry
 
    if (Felimit) then 
      freeFe      = iron_chemistry(Fe,totalligand,ligandStabConst)
    end if

!-------------------------------------------------------------------------------
! Chlorophyll synthesis

    chlSynth       = zero
    if (PARave .ge. tiny) chlSynth = N_assim * Chl2N_max &
          * min( real(one),Cphot/(alfa*Chl2C*PARave))

    ChlSynth_dia   = zero
    if (PARave .ge. tiny) ChlSynth_dia = N_assim_dia     &
          * Chl2N_max_d * min(real(one),                 &
            Cphot_dia /(alfa_d * Chl2C_dia * PARave))

!-------------------------------------------------------------------------------
! Phytoplankton respiraion rate
    phyRespRate     = res_phy * limitFacN + biosynth * N_assim
    phyRespRate_dia = res_phy_d * limitFacN_dia +        &
            biosynth * N_assim_dia + biosynthSi * Si_assim
!-------------------------------------------------------------------------------
! Zooplankton grazing on small phytoplankton and diatoms
! 

    if (REcoM_2Classes) then
      if (REcoM_Grazing_Variable_Preference) then
        DiaNsq        = DiaN * DiaN
        varpzDia      = pzDia * DiaNsq /(sDiaNsq + DiaNsq)
        fDiaN         = varpzDia * DiaN
      else
        fDiaN         = pzDia * DiaN
      end if
      food            = PhyN + fDiaN
      foodsq          = food * food
      grazingFlux     = (Graz_max * foodsq)/(epsilon + foodsq) * HetN * arrFunc
      grazingFlux_phy = grazingFlux * phyN / food
      grazingFlux_Dia = grazingFlux * fDiaN / food
    else ! Only grazing on phytoplankton
      foodsq          = phyN * phyN
      grazingFlux_phy = ( Graz_max * foodsq)/ &
                      ( epsilon + foodsq) * HetN * arrFunc
      grazingFlux     = grazingFlux_phy
    end if
!-------------------------------------------------------------------------------
! Heterotrophic respiration is assumed to drive zooplankton back to Redfield C:N
! if their C:N becomes higher than Redfield

    HetRespFlux    = recip_res_het * arrFunc * recipQZoo * HetC &
                   - recip_res_het * arrFunc * redfield  * HetC

! Next part changes results, but is needed: Otherwise heterotrophs take up 
! inorganic C when their C:N becomes lower than Redfield

    HetRespFlux    = max(zero,HetRespFlux)
!-------------------------------------------------------------------------------
! Quadratic zooplanton mortality
    hetLossFlux    = loss_het * HetN * HetN
!-------------------------------------------------------------------------------
! Phytoplankton and detritus aggregation
    aggregationrate = agg_PD * DetN + agg_PP * PhyN &
                    + agg_PP * DiaN
    if (TEPaggregation) aggregationRate &
       = aggregationRate * f_TEP * EOC/(45.d0 + f_TEP*EOC)
!-------------------------------------------------------------------------------
! Terms required for the formation and dissolution of CaCO3
#ifdef REcoM_calcification
    calcification = calc_prod_ratio * Cphot * PhyC   ! Z in equations
    calc_loss_agg = aggregationRate * PhyCalc
    calc_loss_gra = grazingFlux_phy *              &
      recipQuota/(PhyC + tiny)              * PhyCalc
#endif
		
!-------------------------------------------------------------------------------
! Sources minus sinks are calculated
!-------------------------------------------------------------------------------
! DIN
    sms(k,idin)      = (                       &
      - N_assim                      * PhyC    &
      - N_assim_Dia                  * DiaC    &
      + rho_N * arrFunc              * DON     &
                                             ) * dt + sms(k,idin)  
!-------------------------------------------------------------------------------
! DIC
    sms(k,idic)      = (                       &
     - Cphot                         * PhyC    &
     + phyRespRate                   * PhyC    &
     - Cphot_Dia                     * DiaC    &
     + phyRespRate_Dia               * DiaC    &
     + rho_C1 * arrFunc              * EOC     &
     - rho_C1 * arrFunc * f_TEP      * EOC     &
     + HetRespFlux                             & 
     + dflux * recipdzF(k) * max( 2-k, 0 )     &
#ifdef REcoM_calcification                     
     + calc_diss                     * DetCalc &
     + calc_loss_gra * calc_diss_guts          &
     - calcification                           &
#endif
                                             ) * dt + sms(k,idic)
!-------------------------------------------------------------------------------
! Alkalinity (Assumes that N:P follows a constant Redfield ratio
! N_assimC: 1.0625 = 1/16 + 1
    sms(k,ialk)      = (                       &
      + 1.0625 * N_assim             * PhyC    &
      + 1.0625 * N_assim_Dia         * DiaC    &
      - 1.0625 * rho_N * arrFunc     * DON     &
#ifdef REcoM_calcification
	  + 2.d0 * calc_diss             * DetCalc &
      + 2.d0 * calc_loss_gra * calc_diss_guts  &
      - 2.d0 * calcification                   &
#endif 
                                             ) * dt + sms(k,ialk) 
!-------------------------------------------------------------------------------
! Phytoplankton N
    sms(k,iphyn)      = (                      &
      + N_assim                      * PhyC    &
      - lossN * limitFacN            * PhyN    &
      - aggregationRate              * phyN    &
      - grazingFlux_phy                        &
                                             ) * dt + sms(k,iphyn)
!-------------------------------------------------------------------------------
! Phytoplankton C
    sms(k,iphyc)      = (                      &
      + Cphot                        * PhyC    &
      - lossC * limitFacN            * PhyC    &
      - phyRespRate                  * PhyC    &
      - aggregationRate              * PhyC    &
      - grazingFlux_phy * recipQuota           &
                                             ) * dt + sms(k,iphyc)
!-------------------------------------------------------------------------------
! Phytoplankton ChlA
    sms(k,ipchl)       = (                       &
     	+ chlSynth                     * phyC    &
     	- deg_Chl                      * Chl     &
     	- aggregationRate              * Chl     &
     	- grazingFlux_phy * Chl2N                &
                                               ) * dt + sms(k,ipchl)
!-------------------------------------------------------------------------------
! Detritus N
    sms(k,idetn)       = (                       &
     	+ grazingFlux_phy                        &
     	+ grazingFlux_dia                        &
     	- grazingFlux * grazEff                  &
     	+ aggregationRate              * PhyN    & 
     	+ aggregationRate              * DiaN    &
     	+ hetLossFlux                            &
     	- reminN * arrFunc             * DetN    &
                                               ) * dt + sms(k,idetn)
!-------------------------------------------------------------------------------
! Detritus C
    sms(k,idetc)       = (                           &
     	+ grazingFlux_phy * recipQuota               &
     	- grazingFlux_phy * recipQuota * grazEff     &
     	+ grazingFlux_Dia * recipQuota_Dia           &
     	- grazingFlux_Dia * recipQuota_Dia * grazEff &
     	+ aggregationRate              * phyC        &
     	+ aggregationRate              * DiaC        &
     	+ hetLossFlux * recipQZoo                    &
     	- reminC * arrFunc             * DetC        &
     	+ rho_c2 * f_TEP               * EOC         &
                                             )   * dt + sms(k,idetc)
!-------------------------------------------------------------------------------
! Heterotrophic N
    sms(k,ihetn)       = (                       &
     	+ grazingFlux * grazEff                  &
     	- hetLossFlux                            &
     	- lossN_z                      * HetN    &
                                               ) * dt + sms(k,ihetn)
!-------------------------------------------------------------------------------
! Heterotrophic C
    sms(k,ihetc)      = (                            &
     	+ grazingFlux_phy * recipQuota * grazEff     &
     	+ grazingFlux_Dia * recipQuota_Dia * grazEff &
     	- hetLossFlux * recipQZoo                    &
     	- lossC_z                      * HetC        &
     	- hetRespFlux                                & 
                                               ) * dt + sms(k,ihetc)
!-------------------------------------------------------------------------------
! DON (Extracellular organic N)
    sms(k,idon)      = (                        &
      + lossN * limitFacN              * phyN   &
      + lossN_d * limitFacN_Dia        * DiaN   &
      + reminN * arrFunc               * DetN   &
      + lossN_z                        * HetN   &
      - rho_N * arrFunc                * DON    &
                                              ) * dt + sms(k,idon)
!-------------------------------------------------------------------------------
! EOC
    sms(k,idoc)       = (                       &
      + lossC * limitFacN              * phyC   &
      + lossC_d * limitFacN_dia        * DiaC   &
      + reminC * arrFunc               * DetC   &
      + lossC_z                        * HetC   &
      - rho_c1 * arrFunc               * EOC    &
      + rho_c1 * arrFunc * f_TEP       * EOC    &
      - rho_c2 * f_TEP                 * EOC    &
                                              ) * dt + sms(k,idoc)	
!-------------------------------------------------------------------------------
! Diatom N
    sms(k,idian)      = (                      &
      + N_assim_dia                    * DiaC  &
      - lossN_d * limitFacN_dia        * DiaN  &
      - aggregationRate                * DiaN  &
      - grazingFlux_Dia                        &
                                             ) * dt + sms(k,idian)
!-------------------------------------------------------------------------------
! Diatom C
    sms(k,idiac)      = (                      &
      + Cphot_dia                      * DiaC  &
      - lossC_d * limitFacN_dia        * DiaC  &
      - phyRespRate_dia                * DiaC  &
      - aggregationRate                * DiaC  &
      - grazingFlux_dia * recipQuota_dia       &
     	                                     ) * dt + sms(k,idiac)
!-------------------------------------------------------------------------------
! Diatom Chl
    sms(k,idchl)      = (                       &
      + chlSynth_dia                   * DiaC   &
      - deg_chl_d                      * DiaChl &
      - aggregationRate                * DiaChl &
      - grazingFlux_dia * Chl2N_dia             &
                                             ) * dt + sms(k,idchl)
!-------------------------------------------------------------------------------
! Diatom Si
    sms(k,idiasi)        = (                    &
      + Si_assim                        * DiaC  &
      - lossN_d * limitFacN_dia         * DiaSi &
      - aggregationRate                 * DiaSi &
      - grazingFlux_dia * qSiN                  &
                                             ) * dt + sms(k,idiasi)
!-------------------------------------------------------------------------------
! Detritus Si
    sms(k,idetsi)     = (                       &
      + aggregationRate                 * DiaSi &
      + lossN_d * limitFacN_dia         * DiaSi &
      + grazingFlux_dia * qSiN                  &
      - reminSiT                        * DetSi &
                                             ) * dt + sms(k,idetsi)
!-------------------------------------------------------------------------------
! Silicate
    sms(k,isi)        = (                       &
      - Si_assim                        * DiaC  &
      + reminSiT                        * DetSi &
                                             ) * dt + sms(k,isi)
!-------------------------------------------------------------------------------
! Fe
    if (Felimit) then
      if (FeShort) then
        sms(k,ife)      = ( &
          - Fe2C * Cphot                  * PhyC     &
          - Fe2C * Cphot_dia              * DiaC     &
          + Fe2C * phyRespRate            * PhyC     &
          + Fe2C * phyRespRate_dia        * DiaC     &
          + Fe2C * lossC*limitFacN        * phyC     &
          + Fe2C * lossC_d*limitFacN_dia  * diaC     &
          + Fe2C * reminC * arrFunc       * detC     &
          + Fe2C * lossC_z                * hetC     &
          + Fe2C * hetRespFlux                       &
          - kScavFe                * DetC * FreeFe   & 
                                            ) * dt + sms(k,ife)
      else ! If not Feshort
        sms(k,ife)      = (                          &
          - Fe2C * Cphot                    * PhyC   &
     	  - Fe2C * Cphot_dia                * DiaC   &
     	  + Fe2C * phyRespRate              * PhyC   &
     	  + Fe2C * phyRespRate_dia          * DiaC   &
     	  + Fe2C * rho_C1 * arrFunc         * EOC    &
     	  - Fe2C * rho_C1 * arrFunc * f_TEP * EOC    &
     	  + Fe2C * hetRespFlux                       &
          - kScavFe                  * DetC * FreeFe &
                                            ) * dt + sms(k,ife)
      end if  
    else ! if not felimit
      sms(k,ife)      = zero
    end if

!-------------------------------------------------------------------------------
! Small phytoplankton calcite
#ifdef REcoM_calcification

    sms(k,iphycal)    = (           &
      + calcification               &
      - lossC * limitFacN * phyCalc &
      - phyRespRate       * phyCalc &
      - calc_loss_agg               &
      - calc_loss_gra               &
                                            ) * dt + sms(k,iphycal)
!-------------------------------------------------------------------------------
! Detritus calcite
    sms(k,idetcal)   = (               &
      + lossC * limitFacN * phyCalc    &
      + phyRespRate       * phyCalc    &
      + calc_loss_agg                  &
      + calc_loss_gra                  &
      - calc_loss_gra * calc_diss_guts &
      - calc_diss     * DetCalc        &
                                           ) * dt + sms(k,idetcal)
#endif

!-------------------------------------------------------------------------------
! Diagnostics: Averaged rates
	
	recipbiostep    = 1.d0/real(biostep)

!*** Net primary production [mmol C /(m3 * day)]
	Diags3D(k,1) = Diags3D(k,1) + (   &
     	+ Cphot                   * PhyC  &
     	- PhyRespRate             * PhyC  &
     	) * recipbiostep

	Diags3D(k,2) = Diags3D(k,2) + (   &
     	+ Cphot_dia               * DiaC  &
     	- PhyRespRate_dia         * DiaC  &
     	) * recipbiostep

!*** Gross primary production [mmol C /(m3 * day)]
	Diags3D(k,3) = Diags3D(k,3) + (   &
     	+ Cphot                   * PhyC  &
     	) * recipbiostep

	Diags3D(k,4) = Diags3D(k,4) + (   &
     	+ Cphot_dia               * DiaC  &
     	) * recipbiostep

!*** Net N-assimilation [mmol N/(m3 * day)]
	Diags3D(k,5) = Diags3D(k,5) + (   &
     	+ N_assim                 * PhyC  &
     	- lossN * limitFacN       * PhyN  &
     	) * recipbiostep

	Diags3D(k,6) = Diags3D(k,6) + (   &
     	+ N_assim_dia             * DiaC  &
     	- lossN * limitFacN_dia   * DiaN  &
     	) * recipbiostep

!*** Gross N-assimilation
	Diags3D(k,7) = Diags3D(k,7) + (   &
     	+ N_assim                 * PhyC  &
     	) * recipbiostep

	Diags3D(k,8) = Diags3D(k,8) + (   &
     	+ N_assim_dia             * DiaC  &
     	) * recipbiostep

  end do ! Main vertikal loop ends
  
!-------------------------------------------------------------------------------
! Remineralization from the sediments into the bottom layer
!		kLoc = Nn      
  if (BenthicLayer) then
!*** DIN ***		
    decayBenthos(1) = decayRateBenN * LocBenthos(1)
    LocBenthos(1)      = LocBenthos(1)   - decaybenthos(1) * dt 
    sms(Nn,idin)    = sms(Nn,idin) + decayBenthos(1) * dt  &
                      * recipdzF(Nn)

!*** DIC ***
    decayBenthos(2) = decayRateBenC * LocBenthos(2)
    LocBenthos(2)      = LocBenthos(2)   - decaybenthos(2) * dt 
    sms(Nn,idic)    = sms(Nn,idic) + decayBenthos(2) * dt  &
                      * recipdzF(Nn)

!*** Si ***
    decayBenthos(3) = decayRateBenSi * LocBenthos(3)
    LocBenthos(3)   = LocBenthos(3)   - decaybenthos(3) * dt 
    sms(Nn,isi)     = sms(Nn,isi)  + decayBenthos(3) * dt  &
                      * recipdzF(Nn)

!*** Calc: DIC, Alk ***
#ifdef REcoM_calcification
    decayBenthos(4) = calc_diss * LocBenthos(4)
    LocBenthos(4)      = LocBenthos(4)   - decayBenthos(4) * dt 
    sms(Nn,idic)    = sms(Nn,idic) + decayBenthos(4) * dt  &
                      * recipdzF(Nn)
    sms(Nn,ialk)    = sms(Nn,ialk) +  &                                             ! From DIN and calcite
                      (1.0625 *decayBenthos(1)+ 2.d0 * decayBenthos(4)) * dt &
    			      * recipdzF(Nn)
#endif

  Ironflux          = decayRateBenC * LocBenthos(2) * Fe2C_benthos
  sms(Nn,ife)       = sms(Nn,ife) + Ironflux * recipdzF(Nn) * dt
  
  end if ! BenthicLayer
  
  end do ! Main time loop ends

!-------------------------------------------------------------------------------
! Sinking

  dt_sink = dt * real(biostep)	

  if (VPhy .gt. 0.1) then
    call REcoM_sinking(dt,Nn,SinkVel(:,ivphy),dzF,recipDzF,state(:,iphyn),sink,zF)
    sms(:,iphyn) = sms(:,iphyn) + sink

    call REcoM_sinking(dt,Nn,SinkVel(:,ivphy),dzF,recipDzF,state(:,iphyc),sink,zF)
    sms(:,iphyc) = sms(:,iphyc) + sink	

    call REcoM_sinking(dt,Nn,SinkVel(:,ivphy),dzF,recipDzF,state(:,ipchl),sink,zF)
    sms(:,ipchl) = sms(:,ipchl) + sink

#ifdef REcoM_calcification
    
    call REcoM_sinking(dt,Nn,SinkVel(:,ivphy),dzF,recipDzF,state(:,iphycal),sink,zF)
    sms(:,iphycal) = sms(:,iphycal) + sink

#endif
  
  end if		

  if (VDia .gt. 0.1) then
    call REcoM_sinking(dt,Nn,SinkVel(:,ivdia),dzF,recipDzF,state(:,idian),sink,zF)
    sms(:,idian) = sms(:,idian) + sink

    call REcoM_sinking(dt,Nn,SinkVel(:,ivdia),dzF,recipDzF,state(:,idiac),sink,zF)
    sms(:,idiac) = sms(:,idiac) + sink	
    
    call REcoM_sinking(dt,Nn,SinkVel(:,ivdia),dzF,recipDzF,state(:,idchl),sink,zF)
    sms(:,idchl) = sms(:,idchl) + sink
  
    call REcoM_sinking(dt,Nn,SinkVel(:,ivdia),dzF,recipDzF,state(:,idiasi),sink,zF)
    sms(:,idiasi) = sms(:,idiasi) + sink
  end if		

  if (VDet .gt. 0.1) then
    call REcoM_sinking(dt,Nn,SinkVel(:,ivdet),dzF,recipDzF,state(:,idetn),sink,zF)
    sms(:,idetn) = sms(:,idetn) + sink

    call REcoM_sinking(dt,Nn,SinkVel(:,ivdet),dzF,recipDzF,state(:,idetc),sink,zF)
    sms(:,idetc) = sms(:,idetc) + sink	

    call REcoM_sinking(dt,Nn,SinkVel(:,ivdet),dzF,recipDzF,state(:,idetsi),sink,zF)
    sms(:,idetsi) = sms(:,idetsi) + sink

#ifdef REcoM_calcification

    call REcoM_sinking(dt,Nn,SinkVel(:,ivdet),dzF,recipDzF,state(:,idetcal),sink,zF)
    sms(:,idetcal) = sms(:,idetcal) + sink
#endif
  
  end if		

!-------------------------------------------------------------------------------
! Benthic layer: Detritus and phytoplankton sink into the benthic layer and are 
! lost from the water column
! (but remineralized and re-released in dissolved form later

  if (BenthicLayer) then
  
  if (allow_var_sinking) then
    Vben_det = Vdet_a * abs(zF(Nn)) + VDet
    Vben_phy = Vdet_a * abs(zF(Nn)) + VPhy
    Vben_dia = Vdet_a * abs(zF(Nn)) + VDia
  else
    Vben_det = VDet
    Vben_phy = VPhy
    Vben_dia = VDia
  endif
  
!*** Det *** (index: 1=N,2=C,3=Si,4=calc)
    wFluxDet(1)    = Vben_det * state(Nn,idetn)
    sms(Nn,idetn)  = sms(Nn,idetn)  - wFluxDet(1) * dt  &
                     * recipDzF(Nn)
    LocBenthos(1)     = LocBenthos(1)     + wFluxDet(1) * dt 

    wFluxDet(2)    = Vben_det * state(Nn,idetc)
    sms(Nn,idetc)  = sms(Nn,idetc)  - wFluxDet(2) * dt  &
                     * recipDzF(Nn)
    LocBenthos(2)     = LocBenthos(2)     + wFluxDet(2) * dt 
    
    wFluxDet(3)    = Vben_det * state(Nn,idetsi)
    sms(Nn,idetsi) = sms(Nn,idetsi) - wFluxDet(3) * dt  &
                     * recipDzF(Nn)
    LocBenthos(3)     = LocBenthos(3)     + wFluxDet(3) * dt 
	  
#ifdef REcoM_calcification 
    wFluxDet(4)     = Vben_det * state(Nn,idetcal)
    sms(Nn,idetcal) = sms(Nn,idetcal) - wFluxDet(4) * dt &
                      * recipDzF(Nn) 
    LocBenthos(4)      = LocBenthos(4)      + wFluxDet(4) * dt 
#endif

!*** Phy *** (index: 1=N,2=C,3=calc)
    wFluxPhy(1)        = Vben_phy * state(Nn,iphyn)
    sms(Nn,iphyn)      = sms(Nn,iphyn)   - wFluxPhy(1) * dt  &
                         * recipDzF(Nn)
    LocBenthos(1)         = LocBenthos(1)      + wFluxPhy(1) * dt 

    wFluxPhy(2)        = Vben_phy * state(Nn,iphyc)
    sms(Nn,iphyc)      = sms(Nn,iphyc)   - wFluxPhy(2) * dt  &
                         * recipDzF(Nn)
    LocBenthos(2)         = LocBenthos(2)      + wFluxPhy(2) * dt 
#ifdef REcoM_calcification
    wFluxPhy(3)        = Vben_phy * state(Nn,iphycal)
    sms(Nn,iphycal)    = sms(Nn,iphycal) - wFluxPhy(3) * dt  &
                         * recipDzF(Nn)
    LocBenthos(4)         = LocBenthos(4)      + wFluxPhy(3) * dt 
#endif

    wFluxPhy(4)        = Vben_phy * state(Nn,ipchl)
	  sms(Nn,ipchl)      = sms(Nn,ipchl)   - wFluxPhy(4) * dt  &
     	* recipDzF(Nn)

!*** Dia *** (index: 1=N,2=C,3=Si)
    wFluxDia(1)        = Vben_dia * state(Nn,idian)
    sms(Nn,idian)      = sms(Nn,idian)   - wFluxDia(1) * dt  &
                         * recipDzF(Nn)
    LocBenthos(1)         = LocBenthos(1)      + wFluxDia(1) * dt 
 
    wFluxDia(2)        = Vben_dia * state(Nn,idiac)
    sms(Nn,idiac)      = sms(Nn,idiac)   - wFluxDia(2) * dt  &
                         * recipDzF(Nn)
    LocBenthos(2)         = LocBenthos(2)      + wFluxDia(2) * dt 

    wFluxDia(3)        = Vben_dia * state(Nn,idiasi)
    sms(Nn,idiasi)     = sms(Nn,idiasi)  - wFluxDia(3) * dt  &
                         * recipDzF(Nn)
    LocBenthos(3)         = LocBenthos(3)      + wFluxDia(3) * dt 
    
    wFluxDia(4)        = Vben_dia * state(Nn,idchl)
	  sms(Nn,idchl)      = sms(Nn,idchl)   - wFluxDia(4) * dt  &
     	* recipDzF(Nn)
  end if


end subroutine REcoM_sms

!-------------------------------------------------------------------------------
! Function for calculating limiter
!-------------------------------------------------------------------------------

function recom_limiter(slope,qa,qb)
  Use REcoM_params
  Implicit None
  Real(kind=8) :: recom_limiter
  Real(kind=8) :: slope, qa, qb
  Real(kind=8) :: dq
	
  dq = qa - qb
  if (REcoM_Geider_limiter) then
    recom_limiter = max(min( -slope*dq, 1.d0),0.d0)
  else
    recom_limiter = 1.d0 - exp( -slope*( abs(dq)-dq )**2)
  endif
  return
  end

!-------------------------------------------------------------------------------
! Function for iron chemistry
!-------------------------------------------------------------------------------

function iron_chemistry(Fe, totalLigand, ligandStabConst)
  implicit none

  Real(kind=8) :: iron_chemistry
  Real(kind=8) :: Fe, totalLigand, ligandStabConst ! Input
  Real(kind=8) :: FreeFe                          ! Output
  Real(kind=8) :: ligand,FeL,a,b,c,discrim

! Abbrevations
  a = ligandstabConst
  b = ligandstabConst * (Fe - totalLigand) + 1.d0
  c = -totalLigand
  discrim = b*b - 4.d0 * a * c
	
  if (a .ne. 0.d0 .and. discrim .ge. 0.d0) then
    ligand = ( -b + sqrt(discrim) ) / (2.d0 * a)
    FeL    = totalLigand - ligand
    freeFe = Fe - FeL
  else ! No free iron
    freeFe = 0.d0
  end if

  iron_chemistry = freeFe

  return
  end

!===============================================================================
! Subroutine for calculating cos(AngleOfIncidence)
!===============================================================================
subroutine Cobeta(cosAngleOfIncidence,Latr)
  use g_clock	! Provides daynumber: daynew, and number of days pr year: ndpyr
  Implicit none

! Values from FESOM
  Real(kind=8),intent(in)    :: Latr
	
! Declarations
  Real(kind=8)               :: yearfrac              ! The fraction of the year that has passed [0 1]
  Real(kind=8)               :: yDay                  ! yearfrac in radians [0 2*pi]
  Real(kind=8)               :: declination   = 0.d0  ! Declination of the sun at present lat and time
  Real(kind=8)               :: CosAngleNoon  = 0.d0  ! Cos(Angle of Incidence) at Noon ?
  Real(kind=8),intent(out)   :: cosAngleOfIncidence   ! Cos(Angle of incidence)
! Constants
  Real(kind=8)		     :: nWater        = 1.33

! find day (****NOTE for year starting in winter*****)  
! Paltridge, G. W. and C. M. R. Platt, Radiative Processes in Meteorology and Climatology, Developments in Atmospheric Sciences, vol. 5, Elsevier Scientific Publishing Company, Amsterdam, Oxford, New York, 1976, ISBN 0-444-41444-4.
  yearfrac    = mod(real(daynew),real(ndpyr))/real(ndpyr)
  yDay        = 2 * pi * yearfrac
  declination = 0.006918           &
     	- 0.399912 * cos(yDay)     &
     	+ 0.070257 * sin(yDay)     &
     	- 0.006758 * cos(2 * yDay) &
     	+ 0.000907 * sin(2 * yDay) &
     	- 0.002697 * cos(3 * yDay) &
     	+ 0.001480 * sin(3 * yDay) 
  cosAngleNoon   = sin(LatR) * sin(declination) &
     				 + cos(LatR) * cos(declination)
  cosAngleOfIncidence &
                  = sqrt(1-(1-cosAngleNoon**2)/nWater**2)

end subroutine Cobeta