!
! This file contains all subroutines necessary to create output files and save values:
!
!	- subroutine REcoM_init_output 
!	- subroutine REcoM_control_output 
!	- subroutine REcoM_write_snapshots
!	- subroutine REcoM_write_means 
!	- subroutine REcoMadd2mean
!
!=======================================================================================================
!
!          Subroutine to initialize netcdf files for REcoM diagnostics
!
!=======================================================================================================
subroutine REcoM_init_output
  use o_param
  use o_mesh
  use g_clock
  use g_PARFE
  use REcoM_params
  implicit none
  
#include "netcdf.inc" 

  integer                   :: status, ncid, dimid_rec, j
  integer                   :: dimid_2d, dimid_3d, dimids(2)
  integer                   :: time_varid, iter_varid, tottra_varid, totvol_varid, benthos_varid(4)
  integer                   :: aCO2_varid, aFeIn_varid, pCO2surf_varid, CO2flux_varid, Hplus_varid
  integer                   :: diags1_varid, diags2_varid, diags3_varid, diags4_varid
  integer                   :: diags5_varid, diags6_varid, diags7_varid, diags8_varid
  character(100)            :: longname
  character(100)            :: filename
  
  if(yearnew==yearold) return
  if (mype/=0) return
  
!--  Initializing output files for snap-shots  -------------------------------------------------

  filename=trim(ResultPath)//runid//'.'//cyearnew//'.bio.nc'
  
  ! create a file
  status = nf_create(filename, nf_clobber, ncid)
  if (status.ne.nf_noerr) call handle_err(status)
  
  ! Define the dimensions
  status = nf_def_dim(ncid, 'nodes_2d', nod2d, dimid_2d)
  if (status .ne. nf_noerr) call handle_err(status)
  status = nf_def_dim(ncid, 'nodes_3d', nod3d, dimid_3d)
  if (status .ne. nf_noerr) call handle_err(status)
  status = nf_def_dim(ncid, 'T', NF_UNLIMITED, dimid_rec)
  if (status .ne. nf_noerr) call handle_err(status)

  ! Define the time and iteration variables
  status = nf_def_var(ncid, 'time', NF_DOUBLE, 1, dimid_rec, time_varid)
  if (status .ne. nf_noerr) call handle_err(status)
  status = nf_def_var(ncid, 'iter', NF_INT, 1, dimid_rec, iter_varid)
  if (status .ne. nf_noerr) call handle_err(status)
  
  status = nf_def_var(ncid, 'AtmCO2pp', NF_DOUBLE, 1, dimid_rec, aCO2_varid)
  if (status .ne. nf_noerr) call handle_err(status)
    
  ! Define variables for conservation check
  if (Test_conservation) then
    status = nf_def_var(ncid, 'TotTra', NF_DOUBLE, 1, dimid_rec, TotTra_varid)
    if (status .ne. nf_noerr) call handle_err(status)
    status = nf_def_var(ncid, 'TotVol', NF_DOUBLE, 1, dimid_rec, TotVol_varid)
    if (status .ne. nf_noerr) call handle_err(status)
  end if ! Test_conservation

  ! Define the netCDF variables for 2D fields.
  ! In Fortran, the unlimited dimension must come
  ! last on the list of dimids.
  dimids(1) = dimid_2d
  dimids(2) = dimid_rec
  
  ! Benthos
  status = nf_def_var(ncid, 'BenN', NF_DOUBLE, 2, dimids, benthos_varid(1))
  if (status .ne. nf_noerr) call handle_err(status)
  status = nf_def_var(ncid, 'BenC', NF_DOUBLE, 2, dimids, benthos_varid(2))
  if (status .ne. nf_noerr) call handle_err(status)
  status = nf_def_var(ncid, 'BenSi', NF_DOUBLE, 2, dimids, benthos_varid(3))
  if (status .ne. nf_noerr) call handle_err(status)
  status = nf_def_var(ncid, 'BenCalc', NF_DOUBLE, 2, dimids, benthos_varid(4))
  if (status .ne. nf_noerr) call handle_err(status)
  
  ! CO2 and Fe
  status = nf_def_var(ncid, 'AtmFeInput', NF_DOUBLE, 2, dimids, aFeIn_varid)
  if (status .ne. nf_noerr) call handle_err(status)
  status = nf_def_var(ncid, 'pCO2surf', NF_DOUBLE, 2, dimids, pCO2surf_varid)
  if (status .ne. nf_noerr) call handle_err(status)
  status = nf_def_var(ncid, 'CO2flux', NF_DOUBLE, 2, dimids, CO2flux_varid)
  if (status .ne. nf_noerr) call handle_err(status)
  status = nf_def_var(ncid, 'Hplus', NF_DOUBLE, 2, dimids, Hplus_varid)
  if (status .ne. nf_noerr) call handle_err(status)
  
  ! Diagnostics
  status = nf_def_var(ncid, 'diags_bio1', NF_DOUBLE, 2, dimids, diags1_varid)
  if (status .ne. nf_noerr) call handle_err(status)
  
  status = nf_def_var(ncid, 'diags_bio2', NF_DOUBLE, 2, dimids, diags2_varid)
  if (status .ne. nf_noerr) call handle_err(status)
  
  status = nf_def_var(ncid, 'diags_bio3', NF_DOUBLE, 2, dimids, diags3_varid)
  if (status .ne. nf_noerr) call handle_err(status)
  
  status = nf_def_var(ncid, 'diags_bio4', NF_DOUBLE, 2, dimids, diags4_varid)
  if (status .ne. nf_noerr) call handle_err(status)
  
  status = nf_def_var(ncid, 'diags_bio5', NF_DOUBLE, 2, dimids, diags5_varid)
  if (status .ne. nf_noerr) call handle_err(status)
  
  status = nf_def_var(ncid, 'diags_bio6', NF_DOUBLE, 2, dimids, diags6_varid)
  if (status .ne. nf_noerr) call handle_err(status)
  
  status = nf_def_var(ncid, 'diags_bio7', NF_DOUBLE, 2, dimids, diags7_varid)
  if (status .ne. nf_noerr) call handle_err(status)
  
  status = nf_def_var(ncid, 'diags_bio8', NF_DOUBLE, 2, dimids, diags8_varid)
  if (status .ne. nf_noerr) call handle_err(status)
  
  ! Assign long_name and units attributes to variables.
  longname='Partial pressure of atmospheric CO2'
  status = nf_put_att_text(ncid, aCO2_varid, 'description', len_trim(longname), trim(longname)) 
  if (status .ne. nf_noerr) call handle_err(status)
  status = nf_put_att_text(ncid, aCO2_varid, 'units', 4, 'uatm')
  if (status .ne. nf_noerr) call handle_err(status)
  
  if (Test_conservation) then
  longname='Total amount of tracer'
  status = nf_put_att_text(ncid, TotTra_varid, 'description', len_trim(longname), trim(longname)) 
  if (status .ne. nf_noerr) call handle_err(status)
  status = nf_put_att_text(ncid, TotTra_varid, 'units', 4, 'mmol')
  if (status .ne. nf_noerr) call handle_err(status)
  
  longname='Total volume of ocean'
  status = nf_put_att_text(ncid, TotVol_varid, 'description', len_trim(longname), trim(longname)) 
  if (status .ne. nf_noerr) call handle_err(status)
  status = nf_put_att_text(ncid, TotVol_varid, 'units', 2, 'm3')
  if (status .ne. nf_noerr) call handle_err(status)
  end if 
  
  ! Benthos
  longname='Benthos Nitrogen'
  status = nf_put_att_text(ncid, benthos_varid(1), 'description', len_trim(longname), trim(longname)) 
  if (status .ne. nf_noerr) call handle_err(status)
  status = nf_put_att_text(ncid, benthos_varid(1), 'units', 7, 'mmol/m3')
  if (status .ne. nf_noerr) call handle_err(status)

  longname='Benthos Carbon'
  status = nf_put_att_text(ncid, benthos_varid(2), 'description', len_trim(longname), trim(longname)) 
  if (status .ne. nf_noerr) call handle_err(status)
  status = nf_put_att_text(ncid, benthos_varid(2), 'units', 7, 'mmol/m3')
  if (status .ne. nf_noerr) call handle_err(status)

  longname='Benthos Silicate'
  status = nf_put_att_text(ncid, benthos_varid(3), 'description', len_trim(longname), trim(longname)) 
  if (status .ne. nf_noerr) call handle_err(status)
  status = nf_put_att_text(ncid, benthos_varid(3), 'units', 7, 'mmol/m3')
  if (status .ne. nf_noerr) call handle_err(status)

  longname='Benthos Calcite'
  status = nf_put_att_text(ncid, benthos_varid(4), 'description', len_trim(longname), trim(longname)) 
  if (status .ne. nf_noerr) call handle_err(status)
  status = nf_put_att_text(ncid, benthos_varid(4), 'units', 7, 'mmol/m3')
  if (status .ne. nf_noerr) call handle_err(status)
  
  ! CO2 and Fe
  longname='dFe in surface water from dust'
  status = nf_put_att_text(ncid, aFeIn_varid, 'description', len_trim(longname), trim(longname)) 
  if (status .ne. nf_noerr) call handle_err(status)
  status = nf_put_att_text(ncid, aFeIn_varid, 'units', 9, 'umol/m2/s')
  if (status .ne. nf_noerr) call handle_err(status)
  
  longname='Partial pressure of CO2 in surface water'
  status = nf_put_att_text(ncid, pCO2surf_varid, 'description', len_trim(longname), trim(longname)) 
  if (status .ne. nf_noerr) call handle_err(status)
  status = nf_put_att_text(ncid, pCO2surf_varid, 'units', 4, 'uatm')
  if (status .ne. nf_noerr) call handle_err(status)

  longname='CO2-flux into the surface water: Input to water = positive flux'
  status = nf_put_att_text(ncid, CO2flux_varid, 'description', len_trim(longname), trim(longname)) 
  if (status .ne. nf_noerr) call handle_err(status)
  status = nf_put_att_text(ncid, CO2flux_varid, 'units', 10, 'mmolC/m2/d')
  if (status .ne. nf_noerr) call handle_err(status)
  
  longname='Conc. of H-plus ions in the surface water'
  status = nf_put_att_text(ncid, Hplus_varid, 'description', len_trim(longname), trim(longname)) 
  if (status .ne. nf_noerr) call handle_err(status)
  status = nf_put_att_text(ncid, Hplus_varid, 'units', 6, 'mol/kg')
  if (status .ne. nf_noerr) call handle_err(status)

  ! Diagnostics
  longname='Bio diagnostics: Net pp. phytoplankton'
  status = nf_put_att_text(ncid, diags1_varid, 'description', len_trim(longname), trim(longname)) 
  if (status .ne. nf_noerr) call handle_err(status)
  status = nf_put_att_text(ncid, diags1_varid, 'units', 12, 'mmolC/(m3*d)')
  if (status .ne. nf_noerr) call handle_err(status)
  
  longname='Bio diagnostics: Net pp. diatom'
  status = nf_put_att_text(ncid, diags2_varid, 'description', len_trim(longname), trim(longname)) 
  if (status .ne. nf_noerr) call handle_err(status)
  status = nf_put_att_text(ncid, diags2_varid, 'units', 12, 'mmolC/(m3*d)')
  if (status .ne. nf_noerr) call handle_err(status)
  
  longname='Bio diagnostics: Gross pp. phytoplankton'
  status = nf_put_att_text(ncid, diags3_varid, 'description', len_trim(longname), trim(longname)) 
  if (status .ne. nf_noerr) call handle_err(status)
  status = nf_put_att_text(ncid, diags3_varid, 'units', 12, 'mmolC/(m3*d)')
  if (status .ne. nf_noerr) call handle_err(status)
  
  longname='Bio diagnostics: Gross pp. diatom'
  status = nf_put_att_text(ncid, diags4_varid, 'description', len_trim(longname), trim(longname)) 
  if (status .ne. nf_noerr) call handle_err(status)
  status = nf_put_att_text(ncid, diags4_varid, 'units', 12, 'mmolC/(m3*d)')
  if (status .ne. nf_noerr) call handle_err(status)
  
  longname='Bio diagnostics: Net N-assimilation phytoplankton'
  status = nf_put_att_text(ncid, diags5_varid, 'description', len_trim(longname), trim(longname)) 
  if (status .ne. nf_noerr) call handle_err(status)
  status = nf_put_att_text(ncid, diags5_varid, 'units', 12, 'mmolC/(m3*d)')
  if (status .ne. nf_noerr) call handle_err(status)
  
  longname='Bio diagnostics: Net N-assimilation diatom'
  status = nf_put_att_text(ncid, diags6_varid, 'description', len_trim(longname), trim(longname)) 
  if (status .ne. nf_noerr) call handle_err(status)
  status = nf_put_att_text(ncid, diags6_varid, 'units', 12, 'mmolC/(m3*d)')
  if (status .ne. nf_noerr) call handle_err(status)
  
  longname='Bio diagnostics: Gross N-assimilation phytoplankton'
  status = nf_put_att_text(ncid, diags7_varid, 'description', len_trim(longname), trim(longname)) 
  if (status .ne. nf_noerr) call handle_err(status)
  status = nf_put_att_text(ncid, diags7_varid, 'units', 12, 'mmolC/(m3*d)')
  if (status .ne. nf_noerr) call handle_err(status)
  
  longname='Bio diagnostics: Gross N-assimilation diatom'
  status = nf_put_att_text(ncid, diags8_varid, 'description', len_trim(longname), trim(longname)) 
  if (status .ne. nf_noerr) call handle_err(status)
  status = nf_put_att_text(ncid, diags8_varid, 'units', 12, 'mmolC/(m3*d)')
  if (status .ne. nf_noerr) call handle_err(status)
  
  status = nf_enddef(ncid)
  if (status .ne. nf_noerr) call handle_err(status)

  status=nf_close(ncid)
  if (status .ne. nf_noerr) call handle_err(status)
  
!--  Initializing mean output files  -------------------------------------------------------------------

  filename=trim(ResultPath)//runid//'.'//cyearnew//'.bio.mean.nc'
  
  ! create a file
  status = nf_create(filename, nf_clobber, ncid)
  if (status.ne.nf_noerr) call handle_err(status)
  
  ! Define the dimensions
  status = nf_def_dim(ncid, 'nodes_2d', nod2d, dimid_2d)
  if (status .ne. nf_noerr) call handle_err(status)
  status = nf_def_dim(ncid, 'nodes_3d', nod3d, dimid_3d)
  if (status .ne. nf_noerr) call handle_err(status)
  status = nf_def_dim(ncid, 'T', NF_UNLIMITED, dimid_rec)
  if (status .ne. nf_noerr) call handle_err(status)

  ! Define the time and iteration variables
  status = nf_def_var(ncid, 'time', NF_DOUBLE, 1, dimid_rec, time_varid)
  if (status .ne. nf_noerr) call handle_err(status)
  status = nf_def_var(ncid, 'iter', NF_INT, 1, dimid_rec, iter_varid)
  if (status .ne. nf_noerr) call handle_err(status)

  ! Define the netCDF variables for 2D fields.
  ! In Fortran, the unlimited dimension must come
  ! last on the list of dimids.
  dimids(1) = dimid_2d
  dimids(2) = dimid_rec
  
  ! Mean benthos
  status = nf_def_var(ncid, 'BenN', NF_DOUBLE, 2, dimids, benthos_varid(1))
  if (status .ne. nf_noerr) call handle_err(status)
  status = nf_def_var(ncid, 'BenC', NF_DOUBLE, 2, dimids, benthos_varid(2))
  if (status .ne. nf_noerr) call handle_err(status)
  status = nf_def_var(ncid, 'BenSi', NF_DOUBLE, 2, dimids, benthos_varid(3))
  if (status .ne. nf_noerr) call handle_err(status)
  status = nf_def_var(ncid, 'BenCalc', NF_DOUBLE, 2, dimids, benthos_varid(4))
  if (status .ne. nf_noerr) call handle_err(status)
  
  ! Mean CO2 and Fe
  status = nf_def_var(ncid, 'AtmFeInput_mean', NF_DOUBLE, 2, dimids, aFeIn_varid)
  if (status .ne. nf_noerr) call handle_err(status)
  status = nf_def_var(ncid, 'pCO2surf_mean', NF_DOUBLE, 2, dimids, pCO2surf_varid)
  if (status .ne. nf_noerr) call handle_err(status)
  status = nf_def_var(ncid, 'CO2flux_mean', NF_DOUBLE, 2, dimids, CO2flux_varid)
  if (status .ne. nf_noerr) call handle_err(status)
  status = nf_def_var(ncid, 'Hplus_mean', NF_DOUBLE, 2, dimids, Hplus_varid)
  if (status .ne. nf_noerr) call handle_err(status)
  
  ! Mean diagnostics
  status = nf_def_var(ncid, 'diags_bio1_mean', NF_DOUBLE, 2, dimids, diags1_varid)
  if (status .ne. nf_noerr) call handle_err(status)
  
  status = nf_def_var(ncid, 'diags_bio2_mean', NF_DOUBLE, 2, dimids, diags2_varid)
  if (status .ne. nf_noerr) call handle_err(status)
  
  status = nf_def_var(ncid, 'diags_bio3_mean', NF_DOUBLE, 2, dimids, diags3_varid)
  if (status .ne. nf_noerr) call handle_err(status)
  
  status = nf_def_var(ncid, 'diags_bio4_mean', NF_DOUBLE, 2, dimids, diags4_varid)
  if (status .ne. nf_noerr) call handle_err(status)
  
  status = nf_def_var(ncid, 'diags_bio5_mean', NF_DOUBLE, 2, dimids, diags5_varid)
  if (status .ne. nf_noerr) call handle_err(status)
  
  status = nf_def_var(ncid, 'diags_bio6_mean', NF_DOUBLE, 2, dimids, diags6_varid)
  if (status .ne. nf_noerr) call handle_err(status)
  
  status = nf_def_var(ncid, 'diags_bio7_mean', NF_DOUBLE, 2, dimids, diags7_varid)
  if (status .ne. nf_noerr) call handle_err(status)
  
  status = nf_def_var(ncid, 'diags_bio8_mean', NF_DOUBLE, 2, dimids, diags8_varid)
  if (status .ne. nf_noerr) call handle_err(status)
  
  ! Assign long_name and units attributes to variables.
  longname='Mean Benthos Nitrogen'
  status = nf_put_att_text(ncid, benthos_varid(1), 'description', len_trim(longname), trim(longname)) 
  if (status .ne. nf_noerr) call handle_err(status)
  status = nf_put_att_text(ncid, benthos_varid(1), 'units', 7, 'mmol/m3')
  if (status .ne. nf_noerr) call handle_err(status)

  longname='Mean Benthos Carbon'
  status = nf_put_att_text(ncid, benthos_varid(2), 'description', len_trim(longname), trim(longname)) 
  if (status .ne. nf_noerr) call handle_err(status)
  status = nf_put_att_text(ncid, benthos_varid(2), 'units', 7, 'mmol/m3')
  if (status .ne. nf_noerr) call handle_err(status)
  
  longname='Mean Benthos Silicate'
  status = nf_put_att_text(ncid, benthos_varid(3), 'description', len_trim(longname), trim(longname)) 
  if (status .ne. nf_noerr) call handle_err(status)
  status = nf_put_att_text(ncid, benthos_varid(3), 'units', 7, 'mmol/m3')
  if (status .ne. nf_noerr) call handle_err(status)

  longname='Mean Benthos Calcite'
  status = nf_put_att_text(ncid, benthos_varid(4), 'description', len_trim(longname), trim(longname)) 
  if (status .ne. nf_noerr) call handle_err(status)
  status = nf_put_att_text(ncid, benthos_varid(4), 'units', 7, 'mmol/m3')
  if (status .ne. nf_noerr) call handle_err(status)
  
  ! CO2 and Fe
  longname='Monthly mean of dFe in surface water from dust'
  status = nf_put_att_text(ncid, aFeIn_varid, 'description', len_trim(longname), trim(longname)) 
  if (status .ne. nf_noerr) call handle_err(status)
  status = nf_put_att_text(ncid, aFeIn_varid, 'units', 9, 'umol/m2/s')
  if (status .ne. nf_noerr) call handle_err(status)
  
  longname='Monthly mean of partial pressure of CO2 in surface water'
  status = nf_put_att_text(ncid, pCO2surf_varid, 'description', len_trim(longname), trim(longname)) 
  if (status .ne. nf_noerr) call handle_err(status)
  status = nf_put_att_text(ncid, pCO2surf_varid, 'units', 4, 'uatm')
  if (status .ne. nf_noerr) call handle_err(status)

  longname='Monthly mean of CO2-flux into the surface water: Input to water = positive flux'
  status = nf_put_att_text(ncid, CO2flux_varid, 'description', len_trim(longname), trim(longname)) 
  if (status .ne. nf_noerr) call handle_err(status)
  status = nf_put_att_text(ncid, CO2flux_varid, 'units', 10, 'mmolC/m2/d')
  if (status .ne. nf_noerr) call handle_err(status)
  
  longname='Monthly mean of H-plus ions in the surface water'
  status = nf_put_att_text(ncid, Hplus_varid, 'description', len_trim(longname), trim(longname)) 
  if (status .ne. nf_noerr) call handle_err(status)
  status = nf_put_att_text(ncid, Hplus_varid, 'units', 6, 'mol/kg')
  if (status .ne. nf_noerr) call handle_err(status)
  
  ! Diagnostics
  longname='Bio diagnostics: Mean net pp. phytoplankton'
  status = nf_put_att_text(ncid, diags1_varid, 'description', len_trim(longname), trim(longname)) 
  if (status .ne. nf_noerr) call handle_err(status)
  status = nf_put_att_text(ncid, diags1_varid, 'units', 12, 'mmolC/(m3*d)')
  if (status .ne. nf_noerr) call handle_err(status)
  
  longname='Bio diagnostics: Mean net pp. diatom'
  status = nf_put_att_text(ncid, diags2_varid, 'description', len_trim(longname), trim(longname)) 
  if (status .ne. nf_noerr) call handle_err(status)
  status = nf_put_att_text(ncid, diags2_varid, 'units', 12, 'mmolC/(m3*d)')
  if (status .ne. nf_noerr) call handle_err(status)
  
  longname='Bio diagnostics: Mean gross pp. phytoplankton'
  status = nf_put_att_text(ncid, diags3_varid, 'description', len_trim(longname), trim(longname)) 
  if (status .ne. nf_noerr) call handle_err(status)
  status = nf_put_att_text(ncid, diags3_varid, 'units', 12, 'mmolC/(m3*d)')
  if (status .ne. nf_noerr) call handle_err(status)
  
  longname='Bio diagnostics: Mean gross pp. diatom'
  status = nf_put_att_text(ncid, diags4_varid, 'description', len_trim(longname), trim(longname)) 
  if (status .ne. nf_noerr) call handle_err(status)
  status = nf_put_att_text(ncid, diags4_varid, 'units', 12, 'mmolC/(m3*d)')
  if (status .ne. nf_noerr) call handle_err(status)
  
  longname='Bio diagnostics: Mean net N-assimilation phytoplankton'
  status = nf_put_att_text(ncid, diags5_varid, 'description', len_trim(longname), trim(longname)) 
  if (status .ne. nf_noerr) call handle_err(status)
  status = nf_put_att_text(ncid, diags5_varid, 'units', 12, 'mmolC/(m3*d)')
  if (status .ne. nf_noerr) call handle_err(status)
  
  longname='Bio diagnostics: Mean net N-assimilation diatom'
  status = nf_put_att_text(ncid, diags6_varid, 'description', len_trim(longname), trim(longname)) 
  if (status .ne. nf_noerr) call handle_err(status)
  status = nf_put_att_text(ncid, diags6_varid, 'units', 12, 'mmolC/(m3*d)')
  if (status .ne. nf_noerr) call handle_err(status)
  
  longname='Bio diagnostics: Mean gross N-assimilation phytoplankton'
  status = nf_put_att_text(ncid, diags7_varid, 'description', len_trim(longname), trim(longname)) 
  if (status .ne. nf_noerr) call handle_err(status)
  status = nf_put_att_text(ncid, diags7_varid, 'units', 12, 'mmolC/(m3*d)')
  if (status .ne. nf_noerr) call handle_err(status)
  
  longname='Bio diagnostics: Mean gross N-assimilation diatom'
  status = nf_put_att_text(ncid, diags8_varid, 'description', len_trim(longname), trim(longname)) 
  if (status .ne. nf_noerr) call handle_err(status)
  status = nf_put_att_text(ncid, diags8_varid, 'units', 12, 'mmolC/(m3*d)')
  if (status .ne. nf_noerr) call handle_err(status)
  
  status = nf_enddef(ncid)
  if (status .ne. nf_noerr) call handle_err(status)

  status=nf_close(ncid)
  if (status .ne. nf_noerr) call handle_err(status)

end subroutine REcoM_init_output

!=======================================================================================================
!
!          Subroutine that controls writing of output (diagnostics) from REcoM
!
!=======================================================================================================

subroutine REcoM_control_output
  use REcoM_GloVar
  implicit none
  
  call REcoM_write_snapshots
  
!--- Saving mean ---------------------------------------------------------------------------------------

  call REcoMcalcmean
  call REcoM_write_means
  
!--- Restoring mean arrays to zero ---------------------------------------------------------------------  
  
  diags2D_mean     = 0.d0
  BenthosMean      = 0.d0
  
  GloPCO2surf_mean = 0.d0
  GloCO2flux_mean  = 0.d0
  GloHplus_mean    = 0.d0
  AtmFeInput_mean  = 0.d0
  
end subroutine REcoM_control_output

!=======================================================================================================
!
!          Subroutine that writes output (benthos, CO2, Fe and diagnostics) from REcoM
!
!=======================================================================================================
subroutine REcoM_write_snapshots
  use o_param
  use o_array
  use o_mesh
  use i_array
  use g_clock
  use g_PARFE
  use REcoM_GloVar
  use REcoM_LocVar
  use REcoM_params
  implicit none
  
#include "netcdf.inc"

  integer                   :: time_varid, iter_varid, tottra_varid, totvol_varid, benthos_varid(4)
  integer                   :: diags1_varid, diags2_varid, diags3_varid, diags4_varid
  integer                   :: diags5_varid, diags6_varid, diags7_varid, diags8_varid 
  integer                   :: aCO2_varid, aFeIn_varid, pCO2surf_varid, CO2flux_varid, Hplus_varid
  integer                   :: status, ncid, j
  integer                   :: start(2), count(2), n3
  real(kind=8)              :: sec_in_year
  character(100)            :: filename
  real(kind=8), allocatable :: aux2(:)
  
  allocate(aux2(nod2D))

  if (mype==0) then
    sec_in_year=dt*istep
    
    ! open files
     filename=trim(ResultPath)//runid//'.'//cyearnew//'.bio.nc'
     status = nf_open(filename, nf_write, ncid)
     if (status .ne. nf_noerr) call handle_err(status)
     
     ! inquire variable id
     status=nf_inq_varid(ncid, 'time', time_varid)
     if (status .ne. nf_noerr) call handle_err(status)
     status=nf_inq_varid(ncid, 'iter', iter_varid)
     if (status .ne. nf_noerr) call handle_err(status)
     
     status=nf_inq_varid(ncid, 'AtmCO2pp', aCO2_varid)
     if (status .ne. nf_noerr) call handle_err(status)
     
     if (Test_conservation) then
     status=nf_inq_varid(ncid, 'iter', tottra_varid)
     if (status .ne. nf_noerr) call handle_err(status)
     status=nf_inq_varid(ncid, 'iter', totvol_varid)
     if (status .ne. nf_noerr) call handle_err(status)
     end if ! Test_conservation
     
     status=nf_inq_varid(ncid, 'BenN', benthos_varid(1))
     if (status .ne. nf_noerr) call handle_err(status)
     status=nf_inq_varid(ncid, 'BenC', benthos_varid(2))
     if (status .ne. nf_noerr) call handle_err(status)
     status=nf_inq_varid(ncid, 'BenSi', benthos_varid(3))
     if (status .ne. nf_noerr) call handle_err(status)
     status=nf_inq_varid(ncid, 'BenCalc', benthos_varid(4))
     if (status .ne. nf_noerr) call handle_err(status)
     
     status=nf_inq_varid(ncid, 'AtmFeInput', aFeIn_varid)
     if (status .ne. nf_noerr) call handle_err(status)
     status=nf_inq_varid(ncid, 'pCO2surf', pCO2surf_varid)
     if (status .ne. nf_noerr) call handle_err(status)
     status=nf_inq_varid(ncid, 'CO2flux', CO2flux_varid)
     if (status .ne. nf_noerr) call handle_err(status)
     status=nf_inq_varid(ncid, 'Hplus', Hplus_varid)
     if (status .ne. nf_noerr) call handle_err(status)
     
     status=nf_inq_varid(ncid, 'diags_bio1', diags1_varid)
     if (status .ne. nf_noerr) call handle_err(status)
     status=nf_inq_varid(ncid, 'diags_bio2', diags2_varid)
     if (status .ne. nf_noerr) call handle_err(status)
     status=nf_inq_varid(ncid, 'diags_bio3', diags3_varid)
     if (status .ne. nf_noerr) call handle_err(status)
     status=nf_inq_varid(ncid, 'diags_bio4', diags4_varid)
     if (status .ne. nf_noerr) call handle_err(status)
     status=nf_inq_varid(ncid, 'diags_bio5', diags5_varid)
     if (status .ne. nf_noerr) call handle_err(status)
     status=nf_inq_varid(ncid, 'diags_bio6', diags6_varid)
     if (status .ne. nf_noerr) call handle_err(status)
     status=nf_inq_varid(ncid, 'diags_bio7', diags7_varid)
     if (status .ne. nf_noerr) call handle_err(status)
     status=nf_inq_varid(ncid, 'diags_bio8', diags8_varid)
     if (status .ne. nf_noerr) call handle_err(status)

! write variables

     ! time and iteration
     status=nf_put_vara_double(ncid, time_varid, save_count, 1, sec_in_year)
     if (status .ne. nf_noerr) call handle_err(status)
     status=nf_put_vara_int(ncid, iter_varid, save_count, 1, istep)
     if (status .ne. nf_noerr) call handle_err(status)
     
     status=nf_put_vara_int(ncid, aCO2_varid, save_count, 1, LocAtmCO2)
     if (status .ne. nf_noerr) call handle_err(status)
     write(*,*),'AtmCO2pp=',LocAtmCO2
     
     ! Conservation
     if (Test_conservation) then
     status=nf_put_vara_double(ncid, tottra_varid, save_count, 1, GloTra)
     if (status .ne. nf_noerr) call handle_err(status)
     status=nf_put_vara_double(ncid, totvol_varid, save_count, 1, GloVol)
     if (status .ne. nf_noerr) call handle_err(status)
     end if ! Test_conservation
  end if    !! mype==0 
  
! 2d fields
  ! Benthos
  call broadcast2D(Benthos(1,:),aux2)
  if(mype==0) then            
    start=(/1,save_count/)
    count=(/nod2d, 1/)
    status=nf_put_vara_double(ncid, benthos_varid(1), start, count, aux2) 
    if (status .ne. nf_noerr) call handle_err(status)
  end if
    
  call broadcast2D(Benthos(2,:),aux2)
  if(mype==0) then            
    start=(/1,save_count/)
    count=(/nod2d, 1/)
    status=nf_put_vara_double(ncid, benthos_varid(2), start, count, aux2) 
    if (status .ne. nf_noerr) call handle_err(status)
  end if
    
  call broadcast2D(Benthos(3,:),aux2)
  if(mype==0) then            
    start=(/1,save_count/)
    count=(/nod2d, 1/)
    status=nf_put_vara_double(ncid, benthos_varid(3), start, count, aux2) 
    if (status .ne. nf_noerr) call handle_err(status)
  end if
  
  call broadcast2D(Benthos(4,:),aux2)
  if(mype==0) then            
    start=(/1,save_count/)
    count=(/nod2d, 1/)
    status=nf_put_vara_double(ncid, benthos_varid(4), start, count, aux2) 
    if (status .ne. nf_noerr) call handle_err(status)
  end if
  
  ! CO2 and Fe
  call broadcast2D(AtmFeInput,aux2)
  if(mype==0) then            
    start=(/1,save_count/)
    count=(/nod2d, 1/)
    status=nf_put_vara_double(ncid, aFeIn_varid, start, count, aux2) 
    if (status .ne. nf_noerr) call handle_err(status)
  end if
  
  call broadcast2D(GloPCO2surf,aux2)
  if(mype==0) then            
    start=(/1,save_count/)
    count=(/nod2d, 1/)
    status=nf_put_vara_double(ncid, pCO2surf_varid, start, count, aux2) 
    if (status .ne. nf_noerr) call handle_err(status)
  end if
  
  call broadcast2D(GloCO2flux,aux2)
  if(mype==0) then            
    start=(/1,save_count/)
    count=(/nod2d, 1/)
    status=nf_put_vara_double(ncid, CO2flux_varid, start, count, aux2) 
    if (status .ne. nf_noerr) call handle_err(status)
  end if
  
  call broadcast2D(GloHplus,aux2)
  if(mype==0) then            
    start=(/1,save_count/)
    count=(/nod2d, 1/)
    status=nf_put_vara_double(ncid, Hplus_varid, start, count, aux2) 
    if (status .ne. nf_noerr) call handle_err(status)
  end if
  
  ! Diagnostics
  call broadcast2D(diags2D(1,:),aux2)  
  if(mype==0) then            
     start=(/1,save_count/)
     count=(/nod2d, 1/)
     status=nf_put_vara_double(ncid, diags1_varid, start, count, aux2) 
     if (status .ne. nf_noerr) call handle_err(status)
  end if
  
  call broadcast2D(diags2D(2,:),aux2)  
  if(mype==0) then            
     start=(/1,save_count/)
     count=(/nod2d, 1/)
     status=nf_put_vara_double(ncid, diags2_varid, start, count, aux2) 
     if (status .ne. nf_noerr) call handle_err(status)
  end if
  
  call broadcast2D(diags2D(3,:),aux2)  
  if(mype==0) then            
     start=(/1,save_count/)
     count=(/nod2d, 1/)
     status=nf_put_vara_double(ncid, diags3_varid, start, count, aux2) 
     if (status .ne. nf_noerr) call handle_err(status)
  end if
  
  call broadcast2D(diags2D(4,:),aux2)  
  if(mype==0) then            
     start=(/1,save_count/)
     count=(/nod2d, 1/)
     status=nf_put_vara_double(ncid, diags4_varid, start, count, aux2) 
     if (status .ne. nf_noerr) call handle_err(status)
  end if
  
  call broadcast2D(diags2D(5,:),aux2)  
  if(mype==0) then            
     start=(/1,save_count/)
     count=(/nod2d, 1/)
     status=nf_put_vara_double(ncid, diags5_varid, start, count, aux2) 
     if (status .ne. nf_noerr) call handle_err(status)
  end if
  
  call broadcast2D(diags2D(6,:),aux2)  
  if(mype==0) then            
     start=(/1,save_count/)
     count=(/nod2d, 1/)
     status=nf_put_vara_double(ncid, diags6_varid, start, count, aux2) 
     if (status .ne. nf_noerr) call handle_err(status)
  end if
  
  call broadcast2D(diags2D(7,:),aux2)  
  if(mype==0) then            
     start=(/1,save_count/)
     count=(/nod2d, 1/)
     status=nf_put_vara_double(ncid, diags7_varid, start, count, aux2) 
     if (status .ne. nf_noerr) call handle_err(status)
  end if
  
  call broadcast2D(diags2D(8,:),aux2)  
  if(mype==0) then            
     start=(/1,save_count/)
     count=(/nod2d, 1/)
     status=nf_put_vara_double(ncid, diags8_varid, start, count, aux2) 
     if (status .ne. nf_noerr) call handle_err(status)
  end if
  
  deallocate(aux2)
  
  if(mype==0) then
    status=nf_close(ncid)
    if (status .ne. nf_noerr) call handle_err(status)
  endif
         
end subroutine REcoM_write_snapshots

!======================================================================================
!
!     Subroutine that saves mean values for REcoM (Benthos, CO2, Fe and diagnostics)
!
!======================================================================================

Subroutine REcoM_write_means
  use REcoM_GloVar
  use g_parfe
  use g_clock
  use o_param
  use o_mesh
  implicit none
  
#include "netcdf.inc"
  
  integer                   :: status, ncid
  integer                   :: time_varid, iter_varid, benthos_varid(4)
  integer                   :: diags1_varid, diags2_varid, diags3_varid, diags4_varid
  integer                   :: diags5_varid, diags6_varid, diags7_varid, diags8_varid
  integer                   :: aCO2_varid, aFeIn_varid, pCO2surf_varid, CO2flux_varid, Hplus_varid
  integer                   :: start(2), count(2)
  real(kind=8)              :: sec_in_year
  character(100)            :: filename
  real(kind=8), allocatable :: aux2(:)
  
  allocate(aux2(nod2D))
  
  sec_in_year=dt*istep
  
  if (mype==0) then 

     ! open files
     filename=trim(ResultPath)//runid//'.'//cyearnew//'.bio.mean.nc'
     status = nf_open(filename, nf_write, ncid)
     if (status .ne. nf_noerr) call handle_err(status)
     
     ! inquire variable id
     status=nf_inq_varid(ncid, 'time', time_varid)
     if (status .ne. nf_noerr) call handle_err(status)
     status=nf_inq_varid(ncid, 'iter', iter_varid)
     if (status .ne. nf_noerr) call handle_err(status)
     
     status=nf_inq_varid(ncid, 'BenN', benthos_varid(1))
     if (status .ne. nf_noerr) call handle_err(status)
     status=nf_inq_varid(ncid, 'BenC', benthos_varid(2))
     if (status .ne. nf_noerr) call handle_err(status)
     status=nf_inq_varid(ncid, 'BenSi', benthos_varid(3))
     if (status .ne. nf_noerr) call handle_err(status)
     status=nf_inq_varid(ncid, 'BenCalc', benthos_varid(4))
     if (status .ne. nf_noerr) call handle_err(status)
     
     status=nf_inq_varid(ncid, 'AtmFeInput_mean', aFeIn_varid)
     if (status .ne. nf_noerr) call handle_err(status)
     status=nf_inq_varid(ncid, 'pCO2surf_mean', pCO2surf_varid)
     if (status .ne. nf_noerr) call handle_err(status)
     status=nf_inq_varid(ncid, 'CO2flux_mean', CO2flux_varid)
     if (status .ne. nf_noerr) call handle_err(status)
     status=nf_inq_varid(ncid, 'Hplus_mean', Hplus_varid)
     if (status .ne. nf_noerr) call handle_err(status)
     
     status=nf_inq_varid(ncid, 'diags_bio1_mean', diags1_varid)
     if (status .ne. nf_noerr) call handle_err(status)
     status=nf_inq_varid(ncid, 'diags_bio2_mean', diags2_varid)
     if (status .ne. nf_noerr) call handle_err(status)
	 status=nf_inq_varid(ncid, 'diags_bio3_mean', diags3_varid)
     if (status .ne. nf_noerr) call handle_err(status)
     status=nf_inq_varid(ncid, 'diags_bio4_mean', diags4_varid)
     if (status .ne. nf_noerr) call handle_err(status)
     status=nf_inq_varid(ncid, 'diags_bio5_mean', diags5_varid)
     if (status .ne. nf_noerr) call handle_err(status)
     status=nf_inq_varid(ncid, 'diags_bio6_mean', diags6_varid)
     if (status .ne. nf_noerr) call handle_err(status)
     status=nf_inq_varid(ncid, 'diags_bio7_mean', diags7_varid)
     if (status .ne. nf_noerr) call handle_err(status)
     status=nf_inq_varid(ncid, 'diags_bio8_mean', diags8_varid)
     if (status .ne. nf_noerr) call handle_err(status)
     
     ! write variables

     ! time and iteration
     status=nf_put_vara_double(ncid, time_varid, save_count, 1, sec_in_year)
     if (status .ne. nf_noerr) call handle_err(status)
     status=nf_put_vara_int(ncid, iter_varid, save_count, 1, istep)
     if (status .ne. nf_noerr) call handle_err(status)

     ! 2d fields
     start=(/1,save_count/)
     count=(/nod2d, 1/)
  end if    !! mype=0
  
  ! Benthos
  call broadcast2D(BenthosMean(1,:), aux2)       
  if(mype==0) then
    status=nf_put_vara_real(ncid, benthos_varid(1), start, count, real(aux2,4)) 
    if (status .ne. nf_noerr) call handle_err(status)   
  end if
  call broadcast2D(BenthosMean(2,:), aux2)       
  if(mype==0) then
    status=nf_put_vara_real(ncid, benthos_varid(2), start, count, real(aux2,4)) 
    if (status .ne. nf_noerr) call handle_err(status)   
  end if
  call broadcast2D(BenthosMean(3,:), aux2)       
  if(mype==0) then
    status=nf_put_vara_real(ncid, benthos_varid(3), start, count, real(aux2,4)) 
    if (status .ne. nf_noerr) call handle_err(status)   
  end if
  call broadcast2D(BenthosMean(4,:), aux2)       
  if(mype==0) then
    status=nf_put_vara_real(ncid, benthos_varid(4), start, count, real(aux2,4)) 
    if (status .ne. nf_noerr) call handle_err(status)   
  end if
  
  ! CO2 and Fe
  call broadcast2D(AtmFeInput_mean,aux2)
  if(mype==0) then            
    start=(/1,save_count/)
    count=(/nod2d, 1/)
    status=nf_put_vara_double(ncid, aFeIn_varid, start, count, aux2) 
    if (status .ne. nf_noerr) call handle_err(status)
  end if
  
  call broadcast2D(GloPCO2surf_mean,aux2)
  if(mype==0) then            
    start=(/1,save_count/)
    count=(/nod2d, 1/)
    status=nf_put_vara_double(ncid, pCO2surf_varid, start, count, aux2) 
    if (status .ne. nf_noerr) call handle_err(status)
  end if
  
  call broadcast2D(GloCO2flux_mean,aux2)
  if(mype==0) then            
    start=(/1,save_count/)
    count=(/nod2d, 1/)
    status=nf_put_vara_double(ncid, CO2flux_varid, start, count, aux2) 
    if (status .ne. nf_noerr) call handle_err(status)
  end if
  
  call broadcast2D(GloHplus_mean,aux2)
  if(mype==0) then            
    start=(/1,save_count/)
    count=(/nod2d, 1/)
    status=nf_put_vara_double(ncid, Hplus_varid, start, count, aux2) 
    if (status .ne. nf_noerr) call handle_err(status)
  end if
  
  ! Diagnostics
  call broadcast2D(diags2D_mean(1,:), aux2)       
  if(mype==0) then
    status=nf_put_vara_real(ncid, diags1_varid, start, count, real(aux2,4)) 
    if (status .ne. nf_noerr) call handle_err(status)   
  end if
  call broadcast2D(diags2D_mean(2,:), aux2)       
  if(mype==0) then
    status=nf_put_vara_real(ncid, diags2_varid, start, count, real(aux2,4)) 
    if (status .ne. nf_noerr) call handle_err(status)   
  end if
  call broadcast2D(diags2D_mean(3,:), aux2)       
  if(mype==0) then
    status=nf_put_vara_real(ncid, diags3_varid, start, count, real(aux2,4)) 
    if (status .ne. nf_noerr) call handle_err(status)   
  end if
  call broadcast2D(diags2D_mean(4,:), aux2)       
  if(mype==0) then
    status=nf_put_vara_real(ncid, diags4_varid, start, count, real(aux2,4)) 
    if (status .ne. nf_noerr) call handle_err(status)   
  end if
  call broadcast2D(diags2D_mean(5,:), aux2)       
  if(mype==0) then
    status=nf_put_vara_real(ncid, diags5_varid, start, count, real(aux2,4)) 
    if (status .ne. nf_noerr) call handle_err(status)   
  end if
  call broadcast2D(diags2D_mean(6,:), aux2)       
  if(mype==0) then
    status=nf_put_vara_real(ncid, diags6_varid, start, count, real(aux2,4)) 
    if (status .ne. nf_noerr) call handle_err(status)   
  end if
  call broadcast2D(diags2D_mean(7,:), aux2)       
  if(mype==0) then
    status=nf_put_vara_real(ncid, diags7_varid, start, count, real(aux2,4)) 
    if (status .ne. nf_noerr) call handle_err(status)   
  end if
  call broadcast2D(diags2D_mean(8,:), aux2)       
  if(mype==0) then
    status=nf_put_vara_real(ncid, diags8_varid, start, count, real(aux2,4)) 
    if (status .ne. nf_noerr) call handle_err(status)   
  end if
  
  deallocate(aux2)
  
  if(mype==0) then
    status=nf_close(ncid)
    if (status .ne. nf_noerr) call handle_err(status)
  endif
  
  return 
end subroutine REcoM_write_means

!===================================================================================
!
!   Subroutine for calculating mean arrays for diagnostics
!
!===================================================================================

subroutine REcoMadd2mean
  use REcoM_GloVar
  use g_meanarrays
  use g_parfe
  implicit none
  
  Integer      :: j, row
  
!---Adding to mean-------------------------------------------------------------------
  
  do row = 1,myDim_nod2d
 
    BenthosMean(1,row) = BenthosMean(1,row) + Benthos(1,row)
    BenthosMean(2,row) = BenthosMean(2,row) + Benthos(2,row)
    BenthosMean(3,row) = BenthosMean(3,row) + Benthos(3,row)
    BenthosMean(4,row) = BenthosMean(4,row) + Benthos(4,row)
    
    AtmFeInput_mean(row)  = AtmFeInput_mean(row)  + AtmFeInput(row)
    GloPCO2surf_mean(row) = GloPCO2surf_mean(row) + GloPCO2surf(row) 
    GloCO2flux_mean(row)  = GloCO2flux_mean(row)  + GloCO2flux(row)
    GloHplus_mean(row)    = GloHplus_mean(row)    + GloHplus(row)
    
    diags2D_mean(1,row) = diags2D_mean(1,row) + diags2D(1,row)
    diags2D_mean(2,row) = diags2D_mean(2,row) + diags2D(2,row)
    diags2D_mean(3,row) = diags2D_mean(3,row) + diags2D(3,row)
    diags2D_mean(4,row) = diags2D_mean(4,row) + diags2D(4,row)
    diags2D_mean(5,row) = diags2D_mean(5,row) + diags2D(5,row)
    diags2D_mean(6,row) = diags2D_mean(6,row) + diags2D(6,row)
    diags2D_mean(7,row) = diags2D_mean(7,row) + diags2D(7,row)
    diags2D_mean(8,row) = diags2D_mean(8,row) + diags2D(8,row)
    
  end do

  return
end subroutine REcoMadd2mean

!====================================================================================

subroutine REcoMcalcmean
  use REcoM_GloVar
  use g_meanarrays
  implicit none
  
  Real(kind=8) :: cnt
  
  meancounter = meancounter + 1 ! Updating the mean counter from FESOM
  cnt=float(meancounter)
  
  !---Calculating mean---------------------------------------------------------------

  diags2D_mean          = diags2D_mean     / cnt
  BenthosMean           = BenthosMean      / cnt
  AtmFeInput_mean       = AtmFeInput_mean  / cnt
  GloPCO2surf_mean      = GloPCO2surf_mean / cnt
  GloCO2flux_mean       = GloCO2flux_mean  / cnt
  GloHplus_mean         = GloHplus_mean    / cnt
  
  !--- Restoring the mean counter from FESOM ------------------------------------------

  meancounter = meancounter - 1 
  
  return
end subroutine REcoMcalcmean

