!===============================================================================
! REcoM2 is controlled by these subroutine:
!
!	- subroutine call_REcoM
!	- subroutine REcoM_forcing
!
!===============================================================================
subroutine call_REcoM
!
! After completing advection of tracers, FESOM calls this subroutine to check if
! REcoM2 is turned on.
!
! If yes, this routine will step through all surface nodes and send one water-
! column at a time to the biogeochemical modules. 
! This includes tracer concentrations, benthos concentrations, Hplus value, 
! local wind speed, aeolian input of iron and atm. CO2-concentration.
!
!===============================================================================
  use o_mesh
  use o_elements
  use o_param
  use o_array
  use g_forcing_param
  use g_forcing_arrays
  use g_clock
  use g_parfe
  use g_rotate_grid
  use i_array
  use REcoM_params
  use REcoM_declarations
  use REcoM_LocVar
  use REcoM_GloVar
  implicit none
 
  integer         :: i, row, nn, inds(max_num_layers)
  real(kind=8)    :: z(max_num_layers), C(max_num_layers,bgc_num)
  real(kind=8)    :: Temp(max_num_layers,1),Sali
  real(kind=8)    :: lat, lon, SW, Hplus
  
  if (.not. use_REcoM) return
  
  call REcoM_init_output
  
  do row = 1,myDim_nod2d                                                  ! Steps through all surface nodes
     SW                = shortwave(row)
     SW                = SW * (1 - a_ice(row))
     if (rotated_grid) then
        call r2g(lon, lat, coord_nod2d(1, row), coord_nod2d(2, row))
     else
        lon            = coord_nod2d(1, row)
        lat            = coord_nod2d(2, row)
     end if
     nn                = num_layers_below_nod2D(row)+1
     inds(1:nn)        = nod3D_below_nod2D(1:nn,row)                      ! Nodes are placed in vertical 
                                                                          ! columns below the surface nodes
     Temp(1:nn,1)      = tracer(inds(1:nn),1)
     Sali              = tracer(inds(1),2)
     C(1:nn,1:bgc_num) = tracer(inds(1:nn), 3:bgc_num+2)                  ! C is set equal to the concentration of tracer 3 in the current column
     z(1:nn)           = coord_nod3D(3, inds(1:nn))                       ! Depth of the nodes in the column 
                                                                          ! are calculated
     LocBenthos(1:4)   = Benthos(1:4,row)
     Hplus             = GloHplus(row)                                    ! Local conc of [H+]-ions from last time time step. Stored in LocVar, used as first guess for H+ conc.in subroutine CO2flux 
     ULoc              = sqrt(u_wind(row)**2+v_wind(row)**2)              ! Wind from FESOM is temporarily stored in module LocVar
     
     call Atm_input 
     FeDust            = GloFeDust(row) * (1 - a_ice(row))                ! a_ice(row): Ice concentration in the local node
     LocAtmCO2         = AtmCO2(month)
         
     call REcoM_Forcing(lat,z(1:nn),nn,C(1:nn,1:bgc_num),SW,dt,Temp,Sali) ! REcoM is called and changes concentrations of tracer
     
     tracer(inds(1:nn),3:num_tracer) = C(1:nn,1:bgc_num)                  ! New concentration of tracer is given as new value to tracer 3
     
     ! Local variables that have been changed during the time-step are stored so they can be saved
     Benthos(1:4,row)  = LocBenthos(1:4)                                  ! Updating Benthos values
     Diags2D(:,row)    = LocDiags2D(:)                                    ! Updating diagnostics
     GloPCO2surf(row)  = pco2surf
     GloCO2flux(row)   = dflux
     GloHplus(row)     = hplus
     AtmFeInput(row)   = FeDust
  end do
  
  call REcoMadd2mean
  
  do i=3, num_tracer  
     call com_3D(tracer(:, i))  
  end do
  
end subroutine call_REcoM
!===============================================================================
! REcoM_Forcing
!
! For each watercolumn this routine calls routines to calculate the CO2-flux 
! atmosphere and ocean and changes in tracer concentrations due to biology.
! It also adds the aeolian input of iron to the surface layer.
!
!===============================================================================
subroutine REcoM_Forcing(Latr,zNodes,Nn,state,SurfSW,dt_s,Temp)

  use REcoM_declarations
  use REcoM_params
  use REcoM_constants
  use REcoM_LocVar
  use g_parfe
  Implicit none

! From FESOM
  Real(kind=8)                       :: Latr
  Integer                            :: Nn		             ! Total number of nodes
  Real(kind=8),dimension(Nn)	     :: zNodes		         ! Depth of nodes
  Real(kind=8),dimension(Nn,bgc_num) :: state                ! Biological tracers	
  Real(kind=8)                       :: SurfSW               ! [W/m2] ShortWave radiation at surface
  Real(kind=8)			             :: dt_s                 ! Size of time steps [s]
  Real(kind=8),dimension(Nn)         :: Temp                 ! [degrees C] Ocean temperature
! Subroutine Cobeta
  Real(kind=8)                       :: cosAI
! Subroutine Depth
  Real(kind=8),dimension(Nn+1)       :: zF                   ! [m] Depth of fluxes
  Real(kind=8),dimension(Nn)         :: dzF                  ! [m] Thickness of control volumes  
  Real(kind=8),dimension(Nn)         :: recipdzF             ! [1/m]
  Real(kind=8),dimension(Nn+1,3)     :: SinkVel              ! [m/day]
! Subroutine CO2Flux
  Real(kind=8)                       :: REcoM_DIC            ! [mmol/m3] Conc of DIC in the surface water, used to calculate CO2 flux
  Real(kind=8)                       :: REcoM_Alk            ! [mmol/m3] Conc of Alk in the surface water, used to calculate CO2 flux
  Real(kind=8)                       :: Sali                 ! Salinity of current surface layer
! Subroutine REcoM_sms
  Real(kind=8),dimension(Nn,bgc_num) :: sms                  ! Matrix that entail changes in tracer concentrations
!Diagnostics
  Integer                            :: idiags               ! Diagnostics

  call Cobeta(cosAI,Latr)

  call Depth_calculations(zNodes,Nn,zF,dzF,recipdzF,SinkVel)
  
  allocate(Diags3D(Nn,8))
  Diags3D(:,:) = 0.d0
!-------------------------------------------------------------------------------
! Atmospheric input of CO2
  REcoM_DIC = max(tiny,state(one,idic))
  REcoM_Alk = max(tiny,state(one,ialk))
  call CO2Flux(REcoM_DIC,REcoM_Alk,Temp(one),Sali)
  dflux     = dflux * (1.d0 - Loc_ice_conc)
  
!-------------------------------------------------------------------------------
! Atmospheric iron deposition
  state(one,ife) = state(one,ife)  & 
     	+ FeDust * recipdzF(one) * dt_s 
     	    
!-------------------------------------------------------------------------------
! Source-Minus-Sinks
  call REcoM_sms(cosAI,Nn,state,dzF,recipdzF,SurfSW,dt_s,sms,Temp,SinkVel,zF)
  state = max(tiny,state + sms)
  
!-------------------------------------------------------------------------------
! Diagnostics
  if (Diagnostics) then
	do idiags = one,8
	  LocDiags2D(idiags) = sum(diags3D(:,idiags) * dzF)
	end do
  end if
  deallocate(Diags3D) 


end subroutine REcoM_Forcing
