function [diff_par,kdp] = logdifpar(par,sinle,RH,albedo,DecDOY)
%diffuse radiation model
%inputs 
%par:- Incoming Photosynthetically Active Radiation (PAR)
%albedo:- Ratio of reflected shortwave to incoming  shortwave
%RH:- Relative Humidity of air in percentage (0-100%)
%sinle:- Sine of the solar elevation angle.
%DecDOY:- Decimal day of year
%This can be easily estimated using the site location and local time
%A more accurate estimation can be obtained using the sun_position.m function available at
%http://www.mathworks.com/matlabcentral/fileexchange/4605-sunposition-m
%This is an implementation of the algorithm presented in
%Reda, I., Andreas, A. (2003) Solar position algorithm for solar radiation application. 
%National Renewable Energy Laboratory (NREL) Technical report NREL/TP-560-34302, Revised January 2008
%outputs
%diff_par:- Total diffused PAR in umolm-2s-1
%kdp:- Diffused PAR fraction
%usage [diff_par,kdp] = logdifpar(par,sinle,RH,albedo,DecDOY)

%checking for inputs and outputs
error(nargchk(5, 5, nargin))
error(nargoutchk(2, 2, nargout))
%initilize outputs
diff_par = NaN(size(par,1), size(par,2));
kdp      = NaN(size(par,1), size(par,2));
z        = NaN(size(par,1), size(par,2));
Sp = 2776.4*(1+0.033.*cos((360.*DecDOY)./365)).*sinle; % extra terrestrial PAR
ktp = par./Sp; % PAR clearness Index
ktp(ktp>1)=NaN;
RH = RH*0.01;
%fitted coefficients
ind1 = find(ktp<=0.78);
if isempty(ind1)
error('No clearness index values less than or equal to 0.78 found');
else
a = 2.0196  ;
b = -5.6485 ;
c = 1.3469  ;
d = 0.7309  ;
e = 0.3045  ;
z(ind1) = a + b.*ktp(ind1) + c.*RH(ind1)+ d.*albedo(ind1) +e.*sinle(ind1);
kdp(ind1) = 1./(1+exp(-z(ind1)));
diff_par(ind1) = kdp(ind1).*par(ind1);
end

ind2 = find(ktp>0.78);
if isempty(ind2)
    error('No clearness index values greater than 0.78 found')
else
a = 1.2438 ;
b = -2.3335 ;
c = 0.7046  ;
d = 0.4107  ;
e = -1.9484 ;
z(ind2) = a + b.*ktp(ind2) + c.*RH(ind2) + d.*albedo(ind2) +e.*sinle(ind2);
kdp(ind2) = 1./(1+exp(-z(ind2)));
diff_par(ind2) = kdp(ind2).*par(ind2);
end


