MODULE trcbio_medusa
   !!======================================================================
   !!                         ***  MODULE trcbio  ***
   !! TOP :   MEDUSA
   !!======================================================================
   !! History :
   !!  -   !  1999-07  (M. Levy)              original code
   !!  -   !  2000-12  (E. Kestenare)         assign parameters to name individual tracers
   !!  -   !  2001-03  (M. Levy)              LNO3 + dia2d 
   !! 2.0  !  2007-12  (C. Deltel, G. Madec)  F90
   !!  -   !  2008-08  (K. Popova)            adaptation for MEDUSA
   !!  -   !  2008-11  (A. Yool)              continuing adaptation for MEDUSA
   !!  -   !  2010-03  (A. Yool)              updated for branch inclusion
   !!----------------------------------------------------------------------
#if defined key_medusa
   !!----------------------------------------------------------------------
   !!                                        MEDUSA bio-model
   !!----------------------------------------------------------------------
   !!   trc_bio_medusa        :  
   !!----------------------------------------------------------------------
      USE oce_trc
      USE trc
      USE sms_medusa
      USE lbclnk
      USE prtctl_trc      ! Print control for debugging
      USE trcsed_medusa
#if defined key_iomput
      USE iom
#endif

      IMPLICIT NONE
      PRIVATE
      
      PUBLIC   trc_bio_medusa    ! called in ???

   !!* Substitution
#  include "domzgr_substitute.h90"
   !!----------------------------------------------------------------------
   !! NEMO/TOP 2.0 , LOCEAN-IPSL (2007) 
   !! $Id: trcbio.F90 1146 2008-06-25 11:42:56Z rblod $ 
   !! Software governed by the CeCILL licence (modipsl/doc/NEMO_CeCILL.txt)
   !!----------------------------------------------------------------------

CONTAINS

   SUBROUTINE trc_bio_medusa( kt )
      !!---------------------------------------------------------------------
      !!                     ***  ROUTINE trc_bio  ***
      !!
      !! ** Purpose :   compute the now trend due to biogeochemical processes
      !!              and add it to the general trend of passive tracers equations
      !!
      !! ** Method  :   each now biological flux is calculated in function of now
      !!              concentrations of tracers.
      !!              depending on the tracer, these fluxes are sources or sinks.
      !!              the total of the sources and sinks for each tracer
      !!              is added to the general trend.
      !!        
      !!                      tra = tra + zf...tra - zftra...
      !!                                     |         |
      !!                                     |         |
      !!                                  source      sink
      !!        
      !!              IF 'key_trc_diabio' defined , the biogeochemical trends
      !!              for passive tracers are saved for futher diagnostics.
      !!---------------------------------------------------------------------
      !!
      !!----------------------------------------------------------------------            
      !! Variable conventions
      !!----------------------------------------------------------------------
      !!
      !! names: z*** - state variable 
      !!        f*** - function (or temporary variable used in part of a function)
      !!        x*** - parameter
      !!        b*** - right-hand part (sources and sinks)
      !!        i*** - integer variable (usually used in yes/no flags)
      !!
      !! time (integer timestep)
      INTEGER, INTENT( in ) ::    kt
      !!
      !! spatial array indices
      INTEGER  ::    ji,jj,jk,jn
      !!
      !! AXY (27/07/10): add in indices for depth horizons (for sinking flux
      !!                 and seafloor iron inputs)
      INTEGER  ::    i0100, i0200, i0500, i1000, i1100
      !!
      !! model state variables
      REAL(wp) ::    zchn,zchd,zphn,zphd,zpds,zzmi
      REAL(wp) ::    zzme,zdet,zdtc,zdin,zsil,zfer
      REAL(wp) ::    zage
      !!
      !! integrated source and sink terms
      REAL(wp) ::    b0
      REAL(wp) ::    bchn,bchd,bphn,bphd,bpds,bzmi
      REAL(wp) ::    bzme,bdet,bdin,bsil,bfer,bage
      !! AXY (26/11/08): implicit detrital carbon change
      !! REAL(wp) ::    bzme,bdet,bdtc,bdin,bsil,bfer
      !!
      !! primary production and chl related quantities      
      REAL(wp) ::    fthetan,faln,fchn1,fchn,fjln,fprn,frn
      REAL(wp) ::    fthetad,fald,fchd1,fchd,fjld,fprd,frd
      !! AXY (16/07/09): add in Eppley curve functionality
      REAL(wp) ::    loc_T,fun_T,xvpnT,xvpdT
      INTEGER  ::    ieppley
      !! AXY (01/03/10): add in mixed layer PP diagnostics
      REAL(wp) ::    fprn_ml,fprd_ml
      !!
      !! nutrient limiting factors
      REAL(wp) ::    fnln,ffln            !! N and Fe
      REAL(wp) ::    fnld,ffld,fsld,fsld2 !! N, Fe and Si
      !!
      !! silicon cycle
      REAL(wp) ::    fsin,fnsi,fsin1,fnsi1,fnsi2,fprds,fsdiss
      !!
      !! iron cycle; includes parameters for Parekh et al. (2005) iron scheme
      REAL(wp) ::    ffetop,ffebot,ffescav
      REAL(wp) ::    xLgF, xFeT, xFeF, xFeL, xFree  !! state variables for iron-ligand system
      REAL(wp) ::    xb_coef_tmp, xb2M4ac           !! iron-ligand parameters
      REAL(wp) ::    xmaxFeF,fdeltaFe               !! max Fe' parameters
      !!
      !! microzooplankton grazing
      REAL(wp) ::    fmi1,fmi,fgmipn,fgmid,fgmidc
      REAL(wp) ::    finmi,ficmi,fstarmi,fmith,fmigrow,fmiexcr,fmiresp
      !!
      !! mesozooplankton grazing
      REAL(wp) ::    fme1,fme,fgmepn,fgmepd,fgmepds,fgmezmi,fgmed,fgmedc
      REAL(wp) ::    finme,ficme,fstarme,fmeth,fmegrow,fmeexcr,fmeresp
      !!
      !! mortality/Remineralisation (defunct parameter "fz" removed)
      REAL(wp) ::    fdpn,fdpd,fdpds,fdzmi,fdzme,fdd
      REAL(wp) ::    fdpn2,fdpd2,fdpds2,fdzmi2,fdzme2
      REAL(wp) ::    fslown, fslownflux
      REAL(wp) ::    fregen,fregensi
      REAL(wp), DIMENSION(jpi,jpj) ::    fregenfast,fregenfastsi
      !! AXY (26/11/08): implicit detrital carbon change
      !! REAL(wp) ::    fdpn,fdpd,fdpds,fdzmi,fdzme,fdd,fddc
      !!
      !! particle flux
      REAL(WP) ::    fthk,fdep,fdep1,fdep2,flat,fcaco3
      REAL(WP) ::    ftempn,ftempsi,ftempfe,ftempc,ftempca
      REAL(wp) ::    freminn,freminsi,freminfe,freminc,freminca
      REAL(wp), DIMENSION(jpi,jpj) ::    ffastn,ffastsi,ffastfe,ffastc,ffastca
      REAL(wp) ::    fleftn,fleftsi,fleftfe,fleftc,fleftca
      REAL(wp) ::    fheren,fheresi,fherefe,fherec,fhereca
      REAL(wp) ::    fprotf
      REAL(wp) ::    fsedn,fsedsi,fsedfe,fsedc,fsedca
      REAL(wp), DIMENSION(jpi,jpj) ::    fccd
      !!
      !! conservation law
      REAL(wp) ::    fnit0,fsil0,ffer0 
      !! 
      !! temporary variables
      REAL(wp) ::    fq0,fq1,fq2,fq3,fq4,fq5,fq6,fq7,fq8,fq9
      !!
      !! water column nutrient and flux integrals
      REAL(wp) ::    ftot_n,ftot_si,ftot_fe
      REAL(wp), DIMENSION(jpi,jpj) ::    fflx_n,fflx_si,fflx_fe
      REAL(wp), DIMENSION(jpi,jpj) ::    fifd_n,fifd_si,fifd_fe
      REAL(wp), DIMENSION(jpi,jpj) ::    fofd_n,fofd_si,fofd_fe
      !!
      !! zooplankton grazing integrals
      REAL(wp), DIMENSION(jpi,jpj) ::    fzmi_i,fzmi_o,fzme_i,fzme_o
      !!
      !! limitation term temporary variables
      REAL(wp), DIMENSION(jpi,jpj) ::    ftot_pn,ftot_pd
      !! use ballast scheme (1) or simple exponential scheme (0; a conservation test)
      INTEGER  ::    iball
      INTEGER  ::    ibio_switch
      !!
      !! diagnose fluxes (should only be used in 1D runs)
      INTEGER  ::    idf, idfval

#if defined key_trc_diaadd
      !! local array to store grid cell thickness (for 2D diagnostics)
      REAL(wp), DIMENSION(jpi,jpj,jpk) ::   ze3t
#endif

      !!---------------------------------------------------------------------

      IF( kt == nit000 ) THEN
         IF(lwp) WRITE(numout,*)
         IF(lwp) WRITE(numout,*) ' trc_bio: MEDUSA bio-model'
         IF(lwp) WRITE(numout,*) ' ~~~~~~~'
      ENDIF

      !! not sure what this is for; it's not used anywhere; commenting out
      !! fbodn(:,:) = 0.e0   

#if defined key_trc_diaadd
      !! setup up arrays for 2D diagnostics if these are included
      !!
      !! create array to convert volumetric fluxes to per metre squared per day
      ze3t(:,:,:) = 0.e0
      DO jk = 1,jpk
         !! AXY (04/03/09): fluxes are already per day; no need to correct for seconds
         !! ze3t(:,:,jk) = fse3t(:,:,jk) * 86400.
         ze3t(:,:,jk) = fse3t(:,:,jk)
      END DO 
      !!
      !! blank 2D diagnostic array
      trc2d(:,:,:) = 0.e0
      !!
      !! blank 3D diagnostic array
      trc3d(:,:,:,:) = 0.e0
#endif

      !!----------------------------------------------------------------------
      !! set up aeolian iron parameters
      !! kept local for now to avoid messing the code up; should be moved
      !! to become standard parameters ASAP; incoming iron field appears
      !! to be in g Fe / m2 / month, while surface box flux needs to be in
      !! mmol Fe / m3 / d
      !!----------------------------------------------------------------------
      !!
      !! iron atomic mass (g / mol)
      !! xfe_mass = 55.845
      !!
      !! iron solubility fraction (PISCES favours 0.014; suspiciously not round!)
      !! xfe_sol  = 0.014

      !!----------------------------------------------------------------------
      !! fast detritus ballast scheme (0 = no; 1 = yes)
      !! alternative to ballast scheme is same scheme but with no ballast
      !! protection (not dissimilar to Martin et al., 1987)
      !!----------------------------------------------------------------------
      !!
      iball = 1

      !!----------------------------------------------------------------------
      !! full flux diagnostics (0 = no; 1 = yes); appear in ocean.output
      !! these should *only* be used in 1D since they give comprehensive
      !! output for ecological functions in the model; primarily used in
      !! debugging
      !!----------------------------------------------------------------------
      !!
      idf    = 0
      !!
      !! timer mechanism
      if (kt/120*120.eq.kt) then
         idfval = 1
      else
         idfval = 0
      endif

      !!----------------------------------------------------------------------
      !! blank fast-sinking detritus 2D fields
      !!----------------------------------------------------------------------
      !!
      ffastn(:,:)  = 0.0        !! organic nitrogen
      ffastsi(:,:) = 0.0        !! biogenic silicon
      ffastfe(:,:) = 0.0        !! organic iron
      ffastc(:,:)  = 0.0        !! organic carbon
      ffastca(:,:) = 0.0        !! biogenic calcium carbonate
      !!
      fregenfast(:,:)   = 0.0   !! integrated  N regeneration (fast detritus)
      fregenfastsi(:,:) = 0.0   !! integrated Si regeneration (fast detritus)
      !!
      fccd(:,:)    = 0.0        !! last depth level before CCD

      !!----------------------------------------------------------------------
      !! blank nutrient/flux inventories
      !!----------------------------------------------------------------------
      !!
      fflx_n(:,:)  = 0.0        !! nitrogen flux total
      fflx_si(:,:) = 0.0        !! silicon  flux total
      fflx_fe(:,:) = 0.0        !! iron     flux total
      fifd_n(:,:)  = 0.0        !! nitrogen fast detritus production
      fifd_si(:,:) = 0.0        !! silicon  fast detritus production
      fifd_fe(:,:) = 0.0        !! iron     fast detritus production
      fofd_n(:,:)  = 0.0        !! nitrogen fast detritus remineralisation
      fofd_si(:,:) = 0.0        !! silicon  fast detritus remineralisation
      fofd_fe(:,:) = 0.0        !! iron     fast detritus remineralisation
      ftot_pn(:,:) = 0.0        !! integrated non-diatom phytoplankton
      ftot_pd(:,:) = 0.0        !! integrated diatom     phytoplankton
      !!
      fzmi_i(:,:)  = 0.0        !! material grazed by microzooplankton
      fzmi_o(:,:)  = 0.0        !! ... sum of fate of this material
      fzme_i(:,:)  = 0.0        !! material grazed by  mesozooplankton
      fzme_o(:,:)  = 0.0        !! ... sum of fate of this material

      ! DO jn = 1,jptra
      !    if (lwp) write (numout,*) 'NAN-CHECK', jn, minval(trn(:,:,:,jn)), maxval(trn(:,:,:,jn))
      ! ENDDO	


      !!----------------------------------------------------------------------
      !! Determine jk position of key depths for later calculations 
      !!----------------------------------------------------------------------
      !!
      DO jk = 1,jpk
         !! level thickness
         fthk  = fse3t(ji,jj,jk)
         !! level depth (top of level)
         fdep  = fsdepw(ji,jj,jk)
         !! level depth (bottom of level)
         fdep1 = fdep + fthk
         !! 
         if (fdep.lt.100.0.AND.fdep1.gt.100.0) then        !  100 m
            i0100 = jk
         elseif (fdep.lt.200.0.AND.fdep1.gt.200.0) then    !  200 m
            i0200 = jk
         elseif (fdep.lt.500.0.AND.fdep1.gt.500.0) then    !  500 m
            i0500 = jk
         elseif (fdep.lt.1000.0.AND.fdep1.gt.1000.0) then  ! 1000 m
            i1000 = jk
         elseif (fdep1.lt.1100.0) then                     ! 1100 m (for sedimentary iron)
            i1100 = jk
         endif
      enddo

      !!----------------------------------------------------------------------
      !! MEDUSA has unified equation through the water column
      !! (Diff. from LOBSTER which has two sets: bio- and non-bio layers) 
      !! Statement below in LOBSTER is different: DO jk = 1, jpkbm1          
      !!----------------------------------------------------------------------
      !!
      !! NOTE: the ordering of the loops below differs from that of some other
      !! models; looping over the vertical dimension is the outermost loop and
      !! this complicates some calculations (e.g. storage of vertical fluxes
      !! that can otherwise be done via a singular variable require 2D fields
      !! here); however, these issues are relatively easily resolved, but the
      !! loops CANNOT be reordered without potentially causing code efficiency
      !! problems (e.g. array indexing means that reordering the loops would
      !! require skipping between widely-spaced memory location; potentially
      !! outside those immediately cached)
      !!
      !! OPEN vertical loop
      DO jk = 1,jpk
         !! OPEN horizontal loops
         DO jj = 2,jpjm1
            DO ji = 2,jpim1
               !!======================================================================
               !! SETUP LOCAL GRID CELL
               !!======================================================================
               !!
               !!---------------------------------------------------------------------
               !! Some notes on grid vertical structure
               !! - fsdepw(ji,jj,jk) is the depth of the upper surface of level jk
               !! - fsde3w(ji,jj,jk) is *approximately* the midpoint of level jk
               !! - fse3t(ji,jj,jk)  is the thickness of level jk
               !!---------------------------------------------------------------------
               !!
               !! AXY (11/12/08): set up level thickness
               fthk  = fse3t(ji,jj,jk)
	       !! AXY (25/02/10): set up level depth (top of level)
	       fdep  = fsdepw(ji,jj,jk)
               !! AXY (01/03/10): set up level depth (bottom of level)
               fdep1 = fdep + fthk
               !!
               !! set up model tracers
               !! negative values of state variables are not allowed to
               !! contribute to the calculated fluxes
               zchn = max(0.,trn(ji,jj,jk,jpchn)) !! non-diatom chlorophyll
               zchd = max(0.,trn(ji,jj,jk,jpchd)) !! diatom chlorophyll
               zphn = max(0.,trn(ji,jj,jk,jpphn)) !! non-diatoms
               zphd = max(0.,trn(ji,jj,jk,jpphd)) !! diatoms
               zpds = max(0.,trn(ji,jj,jk,jppds)) !! diatom silicon
               !! AXY (28/01/10): probably need to take account of chl/biomass connection
	       if (zchn.eq.0.) zphn = 0.
	       if (zchd.eq.0.) zphd = 0.
	       if (zphn.eq.0.) zchn = 0.
	       if (zphd.eq.0.) zchd = 0.
               zzmi = max(0.,trn(ji,jj,jk,jpzmi)) !! microzooplankton
               zzme = max(0.,trn(ji,jj,jk,jpzme)) !! mesozooplankton
               zdet = max(0.,trn(ji,jj,jk,jpdet)) !! detrital nitrogen
               zdtc = zdet * xthetad              !! implicit detrital carbon
               !! AXY (26/11/08): implicit detrital carbon change
               !! zdtc = max(0.,trn(ji,jj,jk,jpdtc)) !! detrital carbon
               zdin = max(0.,trn(ji,jj,jk,jpdin)) !! dissolved inorganic nitrogen
               zsil = max(0.,trn(ji,jj,jk,jpsil)) !! dissolved silicic acid
               zfer = max(0.,trn(ji,jj,jk,jpfer)) !! dissolved "iron"

               if (idf.eq.1) then
               !! AXY (15/01/10)
               if (trn(ji,jj,jk,jpdin).lt.0.) then
                  IF (lwp) write (numout,*) '------------------------------'
                  IF (lwp) write (numout,*) 'NEGATIVE DIN ERROR =', trn(ji,jj,jk,jpdin)
                  IF (lwp) write (numout,*) 'NEGATIVE DIN ERROR @', ji, jj, jk, kt
                  !! idf    = 1
                  !! idfval = 1
               endif
               if (trn(ji,jj,jk,jpsil).lt.0.) then
                  IF (lwp) write (numout,*) '------------------------------'
                  IF (lwp) write (numout,*) 'NEGATIVE SIL ERROR =', trn(ji,jj,jk,jpsil)
                  IF (lwp) write (numout,*) 'NEGATIVE SIL ERROR @', ji, jj, jk, kt
                  !! idf    = 1
                  !! idfval = 1
               endif
               endif

               !! report state variable values
               if (idf.eq.1.AND.idfval.eq.1) then
                  IF (lwp) write (numout,*) '------------------------------'
                  IF (lwp) write (numout,*) 'fthk(',jk,') = ', fthk
                  IF (lwp) write (numout,*) 'zphn(',jk,') = ', zphn
                  IF (lwp) write (numout,*) 'zphd(',jk,') = ', zphd
                  IF (lwp) write (numout,*) 'zzmi(',jk,') = ', zzmi
                  IF (lwp) write (numout,*) 'zzme(',jk,') = ', zzme
                  IF (lwp) write (numout,*) 'zdet(',jk,') = ', zdet
                  !! IF (lwp) write (numout,*) 'zdin(',jk,') = ', zdin
                  !! IF (lwp) write (numout,*) 'zsil(',jk,') = ', zsil
                  IF (lwp) write (numout,*) 'zpds(',jk,') = ', zpds
                  IF (lwp) write (numout,*) 'zfer(',jk,') = ', zfer
               endif

               if (idf.eq.1.AND.idfval.eq.1.AND.jk.eq.1) then
                  IF (lwp) write (numout,*) '------------------------------'
                  IF (lwp) write (numout,*) 'dustmo(1) = ', dustmo(ji,jj,1)
                  IF (lwp) write (numout,*) 'dustmo(2) = ', dustmo(ji,jj,2)
                  IF (lwp) write (numout,*) 'dust      = ', dust(ji,jj)
               endif

               !! sum tracers for inventory checks
               ftot_n  = fthk * ( zphn + zphd + zzmi + zzme + zdet + zdin )
               ftot_si = fthk * ( zpds + zsil )
               ftot_fe = fthk * ( xrfn * ( zphn + zphd + zzmi + zzme + zdet ) + zfer )

               !!======================================================================
               !! LOCAL GRID CELL CALCULATIONS
               !!======================================================================
               !!
               !!----------------------------------------------------------------------
               !! Chlorophyll calculations
               !!----------------------------------------------------------------------
               !!
               !! non-diatoms
               fthetan = max(tiny(zchn), (zchn * xxi) / (zphn + tiny(zphn)))
               faln    = xaln * fthetan
               !!
               !! diatoms
               fthetad = max(tiny(zchd), (zchd * xxi) / (zphd + tiny(zphd)))
               fald    = xald * fthetad

               !! report biological calculations
               if (idf.eq.1.AND.idfval.eq.1) then
                  IF (lwp) write (numout,*) '------------------------------'
                  IF (lwp) write (numout,*) 'faln(',jk,') = ', faln
                  IF (lwp) write (numout,*) 'fald(',jk,') = ', fald
               endif

               !!----------------------------------------------------------------------
               !! Phytoplankton light limitation
               !!----------------------------------------------------------------------
               !!
               !! It is assumed xpar is the depth-averaged (vertical layer) PAR 
               !! Light limitation (check self-shading) in W/m2
               !!
               !! Note that there is no temperature dependence in phytoplankton
               !! growth rate or any other function. 
               !! In calculation of Chl/Phy ratio tiny(phyto) is introduced to avoid
               !! NaNs in case of Phy==0.  
               !!
               !! fthetad and fthetan are Chl:C ratio (gChl/gC) in diat and non-diat: 
               !! for 1:1 Chl:P ratio (mgChl/mmolN) theta=0.012
               !!
	       !! AXY (16/07/09)
	       !! temperature for new Eppley style phytoplankton growth
	       loc_T   = tn(ji,jj,jk)
	       fun_T   = 1.066**(1.0 * loc_T)
               if (jphy.eq.1) then
                  xvpnT = xvpn * fun_T
                  xvpdT = xvpd * fun_T
               else
                  xvpnT = xvpn
                  xvpdT = xvpd
               endif
	       !!
               !! non-diatoms
               fchn1   = (xvpnT * xvpnT) + (faln * faln * xpar(ji,jj,jk) * xpar(ji,jj,jk))
               fchn    = xvpnT / sqrt(fchn1)
               fjln    = fchn * faln * xpar(ji,jj,jk) !! non-diatom J term
               !!
               !! diatoms
               fchd1   = (xvpdT * xvpdT) + (fald * fald * xpar(ji,jj,jk) * xpar(ji,jj,jk))
               fchd    = xvpdT / sqrt(fchd1)
               fjld    = fchd * fald * xpar(ji,jj,jk) !! diatom J term
      
               !! report phytoplankton light limitation
               if (idf.eq.1.AND.idfval.eq.1) then
                  IF (lwp) write (numout,*) '------------------------------'
                  IF (lwp) write (numout,*) 'fchn(',jk,') = ', fchn
                  IF (lwp) write (numout,*) 'fchd(',jk,') = ', fchd
                  IF (lwp) write (numout,*) 'fjln(',jk,') = ', fjln
                  IF (lwp) write (numout,*) 'fjld(',jk,') = ', fjld
               endif

               !!----------------------------------------------------------------------
               !! Phytoplankton nutrient limitation
               !!----------------------------------------------------------------------
               !!
               !! non-diatoms (N, Fe)
               fnln = zdin / (zdin + xnln) !! non-diatom Qn term
               ffln = zfer / (zfer + xfln) !! non-diatom Qf term
               !!
               !! diatoms (N, Si, Fe)
               fnld = zdin / (zdin + xnld) !! diatom Qn term
               fsld = zsil / (zsil + xsld) !! diatom Qs term
               ffld = zfer / (zfer + xfld) !! diatom Qf term

               !! report phytoplankton nutrient limitation
               if (idf.eq.1.AND.idfval.eq.1) then
                  IF (lwp) write (numout,*) '------------------------------'
                  IF (lwp) write (numout,*) 'fnln(',jk,') = ', fnln
                  IF (lwp) write (numout,*) 'fnld(',jk,') = ', fnld
                  IF (lwp) write (numout,*) 'ffln(',jk,') = ', ffln
                  IF (lwp) write (numout,*) 'ffld(',jk,') = ', ffld
                  IF (lwp) write (numout,*) 'fsld(',jk,') = ', fsld
               endif

               !!----------------------------------------------------------------------
               !! Primary production (non-diatoms)
               !! (note: still needs multiplying by phytoplankton concentration)
               !!----------------------------------------------------------------------
               !!
               fprn = fjln * fnln * ffln

               !!----------------------------------------------------------------------
               !! Primary production (diatoms)
               !! (note: still needs multiplying by phytoplankton concentration)
               !!
               !! production here is split between nitrogen production and that of
               !! silicon; depending upon the "intracellular" ratio of Si:N, model
               !! diatoms will uptake nitrogen/silicon differentially; this borrows
               !! from the diatom model of Mongin et al. (2006)
               !!----------------------------------------------------------------------
               !!
               !! "intracellular" elemental ratios
               fsin  = zpds / (zphd + tiny(zphd))
               fnsi  = zphd / (zpds + tiny(zpds))
	       !! AXY (23/02/10): these next variables derive from Mongin et al. (2003)
	       fsin1 = 3.0 * xsin0     !! = 0.6
               fnsi1 = 1.0 / fsin1     !! = 1.667
               fnsi2 = 1.0 / xsin0     !! = 5.0
               !!
               !! conditionalities based on ratios
               !! nitrogen (and iron and carbon)
               if (fsin.le.xsin0) then
                  fprd  = 0.0
                  fsld2 = 0.0
               elseif (fsin.lt.fsin1) then
                  fprd  = xuif * ((fsin - xsin0) / fsin) * (fjld * fnld * ffld)
                  fsld2 = xuif * ((fsin - xsin0) / fsin)
               elseif (fsin.ge.fsin1) then
                  fprd  = (fjld * fnld * ffld)
                  fsld2 = 1.0
               endif
               !!
               !! silicon
               if (fsin.lt.fnsi1) then
                  fprds = (fjld * fsld)
               elseif (fsin.lt.fnsi2) then
                  fprds = xuif * ((fnsi - xnsi0) / fnsi) * (fjld * fsld)
               else
                  fprds = 0.0
               endif         

               !! report phytoplankton growth (including diatom silicon submodel)
               if (idf.eq.1.AND.idfval.eq.1) then
                  IF (lwp) write (numout,*) '------------------------------'
                  IF (lwp) write (numout,*) 'fsin(',jk,')   = ', fsin
                  IF (lwp) write (numout,*) 'fnsi(',jk,')   = ', fnsi
                  IF (lwp) write (numout,*) 'fsld2(',jk,')  = ', fsld2
                  IF (lwp) write (numout,*) 'fprn(',jk,')   = ', fprn
                  IF (lwp) write (numout,*) 'fprd(',jk,')   = ', fprd
                  IF (lwp) write (numout,*) 'fprds(',jk,')  = ', fprds
               endif

               !!----------------------------------------------------------------------
               !! Mixed layer primary production
               !! this block calculates the amount of primary production that occurs
               !! within the upper mixed layer; this allows the separate diagnosis
               !! of "sub-surface" primary production; it does assume that short-
               !! term variability in mixed layer depth doesn't mess with things
               !! though
               !!----------------------------------------------------------------------
	       !!
               if (fdep1.le.hmld(ji,jj)) then
                  !! this level is entirely in the mixed layer
                  fq0 = 1.0
               elseif (fdep.ge.hmld(ji,jj)) then
                  !! this level is entirely below the mixed layer
                  fq0 = 0.0
               else
                  !! this level straddles the mixed layer
                  fq0 = (hmld(ji,jj) - fdep) / fthk
               endif
               !!
               fprn_ml = (fprn * zphn * fthk * fq0)
               fprd_ml = (fprd * zphd * fthk * fq0)

               !!----------------------------------------------------------------------
               !! More chlorophyll calculations
               !!----------------------------------------------------------------------
               !!
               !! frn = (xthetam / fthetan) * (fprn / (fthetan * xpar(ji,jj,jk)))
               !! frd = (xthetam / fthetad) * (fprd / (fthetad * xpar(ji,jj,jk)))
               frn = (xthetam * fchn * fnln * ffln       ) / (fthetan + tiny(fthetan))
               !! AXY (12/05/09): there's potentially a problem here; fsld, silicic acid 
               !!   limitation, is used in the following line to regulate chlorophyll 
               !!   growth in a manner that is inconsistent with its use in the regulation 
               !!   of biomass growth; the Mongin term term used in growth is more complex
               !!   than the simple multiplicative function used below
               !! frd = (xthetam * fchd * fnld * ffld * fsld) / (fthetad + tiny(fthetad))
               !! AXY (12/05/09): this replacement line uses the new variable, fsld2, to
               !!   regulate chlorophyll growth
               frd = (xthetamd * fchd * fnld * ffld * fsld2) / (fthetad + tiny(fthetad))

               !! report chlorophyll calculations
               if (idf.eq.1.AND.idfval.eq.1) then
                  IF (lwp) write (numout,*) '------------------------------'
                  IF (lwp) write (numout,*) 'fthetan(',jk,') = ', fthetan
                  IF (lwp) write (numout,*) 'fthetad(',jk,') = ', fthetad
                  IF (lwp) write (numout,*) 'frn(',jk,')     = ', frn
                  IF (lwp) write (numout,*) 'frd(',jk,')     = ', frd
               endif

               !!----------------------------------------------------------------------
               !! Zooplankton Grazing 
               !! this code supplements the base grazing model with one that
               !! considers the C:N ratio of grazed food and balances this against
               !! the requirements of zooplankton growth; this model is derived 
               !! from that of Anderson & Pondaven (2003)
               !!
               !! the current version of the code assumes a fixed C:N ratio for
               !! detritus (in contrast to Anderson & Pondaven, 2003), though the
               !! full equations are retained for future extension
               !!----------------------------------------------------------------------
               !!
               !!----------------------------------------------------------------------
               !! Microzooplankton first
               !!----------------------------------------------------------------------
               !!
               fmi1    = (xkmi * xkmi) + (xpmipn * zphn * zphn) + (xpmid * zdet * zdet)
               fmi     = xgmi * zzmi / fmi1
               fgmipn  = fmi * xpmipn * zphn * zphn   !! grazing on non-diatoms
               fgmid   = fmi * xpmid  * zdet * zdet   !! grazing on detrital nitrogen
               !! AXY (26/11/08): implicit detrital carbon change
               !! fgmidc  = (zdtc / (zdet + tiny(zdet))) * fgmid  !! grazing on detrital carbon
               fgmidc  = xthetad * fgmid              !! grazing on detrital carbon
               !!
               !! which translates to these incoming N and C fluxes
               finmi   = (1.0 - xphi) * (fgmipn + fgmid)
               ficmi   = (1.0 - xphi) * ((xthetapn * fgmipn) + fgmidc)
               !!
               !! the ideal food C:N ratio for microzooplankton
               !! xbetan = 0.77; xthetaz = 5.625; xbetac = 0.64; xkc = 0.80
               fstarmi = (xbetan * xthetazmi) / (xbetac * xkc)
               !!
               !! process these to determine proportioning of grazed N and C
               !! (since there is no explicit consideration of respiration,
               !! only growth and excretion are calculated here)
               fmith   = (ficmi / (finmi + tiny(finmi)))
               if (fmith.ge.fstarmi) then
                  fmigrow = xbetan * finmi
                  fmiexcr = 0.0
                  !! the following line is commented out since respiration not needed until DIC implemented
		  !! fmiresp = (xbetac * ficmi) - (xthetazmi * fmigrow)
               else
                  fmigrow = (xbetac * xkc * ficmi) / xthetazmi
                  fmiexcr = ficmi * ((xbetan / (fmith + tiny(fmith))) - ((xbetac * xkc) / xthetazmi))
                  !! the following line is commented out since respiration not needed until DIC implemented
                  !! fmiresp = (xbetac * ficmi) - (xthetazmi * fmigrow)
               endif

               !! report microzooplankton grazing
               if (idf.eq.1.AND.idfval.eq.1) then
                  IF (lwp) write (numout,*) '------------------------------'
                  IF (lwp) write (numout,*) 'fmi1(',jk,')    = ', fmi1
                  IF (lwp) write (numout,*) 'fmi(',jk,')     = ', fmi
                  IF (lwp) write (numout,*) 'fgmipn(',jk,')  = ', fgmipn
                  IF (lwp) write (numout,*) 'fgmid(',jk,')   = ', fgmid
                  IF (lwp) write (numout,*) 'fgmidc(',jk,')  = ', fgmidc
                  IF (lwp) write (numout,*) 'finmi(',jk,')   = ', finmi
                  IF (lwp) write (numout,*) 'ficmi(',jk,')   = ', ficmi
                  IF (lwp) write (numout,*) 'fstarmi(',jk,') = ', fstarmi
                  IF (lwp) write (numout,*) 'fmith(',jk,')   = ', fmith
                  IF (lwp) write (numout,*) 'fmigrow(',jk,') = ', fmigrow
                  IF (lwp) write (numout,*) 'fmiexcr(',jk,') = ', fmiexcr
               endif

               !!----------------------------------------------------------------------
               !! Mesozooplankton second
               !!----------------------------------------------------------------------
               !!
               fme1    = (xkme * xkme) + (xpmepn * zphn * zphn) + (xpmepd * zphd * zphd) + & 
                         (xpmezmi * zzmi * zzmi) + (xpmed * zdet * zdet)
               fme     = xgme * zzme / fme1
               fgmepn  = fme * xpmepn  * zphn * zphn  !! grazing on non-diatoms
               fgmepd  = fme * xpmepd  * zphd * zphd  !! grazing on diatoms
               fgmepds = fsin * fgmepd                !! grazing on diatom silicon
               fgmezmi = fme * xpmezmi * zzmi * zzmi  !! grazing on microzooplankton
               fgmed   = fme * xpmed   * zdet * zdet  !! grazing on detrital nitrogen
               !! AXY (26/11/08): implicit detrital carbon change
               !! fgmedc  = (zdtc / (zdet + tiny(zdet))) * fgmed  !! grazing on detrital carbon
               fgmedc  = xthetad * fgmed              !! grazing on detrital carbon
               !!
               !! which translates to these incoming N and C fluxes
               finme   = (1.0 - xphi) * (fgmepn + fgmepd + fgmezmi + fgmed)
               ficme   = (1.0 - xphi) * ((xthetapn * fgmepn) + (xthetapd * fgmepd) + &
                        (xthetazmi * fgmezmi) + fgmedc)
               !!
               !! the ideal food C:N ratio for mesozooplankton
               !! xbetan = 0.77; xthetaz = 5.625; xbetac = 0.64; xkc = 0.80
               fstarme = (xbetan * xthetazme) / (xbetac * xkc)
               !!
               !! process these to determine proportioning of grazed N and C
               !! (since there is no explicit consideration of respiration,
               !! only growth and excretion are calculated here)
               fmeth   = (ficme / (finme + tiny(finme)))
               if (fmeth.ge.fstarme) then
                  fmegrow = xbetan * finme
                  fmeexcr = 0.0
                  !! the following line is commented out since respiration not needed until DIC implemented
		  !! fmeresp = (xbetac * ficme) - (xthetazme * fmegrow)
               else
                  fmegrow = (xbetac * xkc * ficme) / xthetazme
                  fmeexcr = ficme * ((xbetan / (fmeth + tiny(fmeth))) - ((xbetac * xkc) / xthetazme))
                  !! the following line is commented out since respiration not needed until DIC implemented
		  !! fmeresp = (xbetac * ficme) - (xthetazme * fmegrow)
               endif

               !! report mesozooplankton grazing
               if (idf.eq.1.AND.idfval.eq.1) then
                  IF (lwp) write (numout,*) '------------------------------'
                  IF (lwp) write (numout,*) 'fme1(',jk,')    = ', fme1
                  IF (lwp) write (numout,*) 'fme(',jk,')     = ', fme
                  IF (lwp) write (numout,*) 'fgmepn(',jk,')  = ', fgmepn
                  IF (lwp) write (numout,*) 'fgmepd(',jk,')  = ', fgmepd
                  IF (lwp) write (numout,*) 'fgmepds(',jk,') = ', fgmepds
                  IF (lwp) write (numout,*) 'fgmezmi(',jk,') = ', fgmezmi
                  IF (lwp) write (numout,*) 'fgmed(',jk,')   = ', fgmed
                  IF (lwp) write (numout,*) 'fgmedc(',jk,')  = ', fgmedc
                  IF (lwp) write (numout,*) 'finme(',jk,')   = ', finme
                  IF (lwp) write (numout,*) 'ficme(',jk,')   = ', ficme
                  IF (lwp) write (numout,*) 'fstarme(',jk,') = ', fstarme
                  IF (lwp) write (numout,*) 'fmeth(',jk,')   = ', fmeth
                  IF (lwp) write (numout,*) 'fmegrow(',jk,') = ', fmegrow
                  IF (lwp) write (numout,*) 'fmeexcr(',jk,') = ', fmeexcr
               endif

               fzmi_i(ji,jj)  = fzmi_i(ji,jj)  + fthk * (  &
                  fgmipn + fgmid )
               fzmi_o(ji,jj)  = fzmi_o(ji,jj)  + fthk * (  &
                  fmigrow + (xphi * (fgmipn + fgmid)) + fmiexcr + ((1.0 - xbetan) * finmi) )
               fzme_i(ji,jj)  = fzme_i(ji,jj)  + fthk * (  &
                  fgmepn + fgmepd + fgmezmi + fgmed )
               fzme_o(ji,jj)  = fzme_o(ji,jj)  + fthk * (  &
                  fmegrow + (xphi * (fgmepn + fgmepd + fgmezmi + fgmed)) + fmeexcr + ((1.0 - xbetan) * finme) )

               !!----------------------------------------------------------------------
               !! Plankton metabolic losses
               !! Linear loss processes assumed to be metabolic in origin
               !!----------------------------------------------------------------------
               !!
	       fdpn2  = xmetapn  * zphn
	       fdpd2  = xmetapd  * zphd
	       fdpds2 = xmetapd  * zpds
	       fdzmi2 = xmetazmi * zzmi
	       fdzme2 = xmetazme * zzme

               !!----------------------------------------------------------------------
               !! Plankton mortality losses
	       !! EKP (26/02/09): phytoplankton hyperbolic mortality term introduced 
               !! to improve performance in gyres
               !!----------------------------------------------------------------------
               !!
               !! non-diatom phytoplankton
	       if (jmpn.eq.1) fdpn = xmpn * zphn               !! linear
	       if (jmpn.eq.2) fdpn = xmpn * zphn * zphn        !! quadratic
	       if (jmpn.eq.3) fdpn = xmpn * zphn * &           !! hyperbolic
                  (zphn / (xkphn + zphn))
	       if (jmpn.eq.4) fdpn = xmpn * zphn * &           !! sigmoid
                  ((zphn * zphn) / (xkphn + (zphn * zphn)))
               !!
               !! diatom phytoplankton
	       if (jmpd.eq.1) fdpd = xmpd * zphd               !! linear
	       if (jmpd.eq.2) fdpd = xmpd * zphd * zphd        !! quadratic
	       if (jmpd.eq.3) fdpd = xmpd * zphd * &           !! hyperbolic
                  (zphd / (xkphd + zphd))
	       if (jmpd.eq.4) fdpd = xmpd * zphd * &           !! sigmoid
                  ((zphd * zphd) / (xkphd + (zphd * zphd)))
               fdpds = fdpd * fsin
               !!
               !! microzooplankton
	       if (jmzmi.eq.1) fdzmi = xmzmi * zzmi            !! linear
	       if (jmzmi.eq.2) fdzmi = xmzmi * zzmi * zzmi     !! quadratic
	       if (jmzmi.eq.3) fdzmi = xmzmi * zzmi * &        !! hyperbolic
                  (zzmi / (xkzmi + zzmi))
	       if (jmzmi.eq.4) fdzmi = xmzmi * zzmi * &        !! sigmoid
                  ((zzmi * zzmi) / (xkzmi + (zzmi * zzmi)))
               !!
               !! mesozooplankton
	       if (jmzme.eq.1) fdzme = xmzme * zzme            !! linear
	       if (jmzme.eq.2) fdzme = xmzme * zzme * zzme     !! quadratic
	       if (jmzme.eq.3) fdzme = xmzme * zzme * &        !! hyperbolic
                  (zzme / (xkzme + zzme))
	       if (jmzme.eq.4) fdzme = xmzme * zzme * &        !! sigmoid
                  ((zzme * zzme) / (xkzme + (zzme * zzme)))

               !!----------------------------------------------------------------------
               !! Detritus remineralisation
               !! Constant or temperature-dependent
               !!----------------------------------------------------------------------
               !!
               if (jmd.eq.1) then
                  !! temperature-dependent
                  fdd  = xmd  * fun_T * zdet
                  !! fddc = xmdc * fun_T * zdtc
               else
                  !! temperature-independent
                  fdd  = xmd  * zdet
                  !! fddc = xmdc * zdtc
               endif

               !! AXY (22/07/09): accelerate detrital remineralisation in the bottom box
               if (jk.eq.(mbathy(ji,jj)-1)) then
                  fdd  = 1.0  * zdet
                  !! fddc = 1.0  * zdtc
               endif

               !! report plankton mortality and remineralisation
               if (idf.eq.1.AND.idfval.eq.1) then
                  IF (lwp) write (numout,*) '------------------------------'
                  IF (lwp) write (numout,*) 'fdpn2(',jk,') = ', fdpn2
                  IF (lwp) write (numout,*) 'fdpd2(',jk,') = ', fdpd2
                  IF (lwp) write (numout,*) 'fdpds2(',jk,')= ', fdpds2
                  IF (lwp) write (numout,*) 'fdzmi2(',jk,')= ', fdzmi2
                  IF (lwp) write (numout,*) 'fdzme2(',jk,')= ', fdzme2
                  IF (lwp) write (numout,*) 'fdpn(',jk,')  = ', fdpn
                  IF (lwp) write (numout,*) 'fdpd(',jk,')  = ', fdpd
                  IF (lwp) write (numout,*) 'fdpds(',jk,') = ', fdpds
                  IF (lwp) write (numout,*) 'fdzmi(',jk,') = ', fdzmi
                  IF (lwp) write (numout,*) 'fdzme(',jk,') = ', fdzme
                  IF (lwp) write (numout,*) 'fdd(',jk,')   = ', fdd
                  !! IF (lwp) write (numout,*) 'fddc(',jk,')  = ', fddc
               endif

               !!----------------------------------------------------------------------
               !! Iron chemistry and fractionation
               !! following the Parekh et al. (2004) scheme adopted by the Met.
               !! Office, Medusa models total iron but considers "free" and
               !! ligand-bound forms for the purposes of scavenging (only "free"
               !! iron can be scavenged
               !!----------------------------------------------------------------------
               !!
               !! total iron concentration (mmol Fe / m3 -> umol Fe / m3)
               xFeT        = zfer * 1.e3
               !!
               !! calculate fractionation (based on Diat-HadOCC; in turn based on Parekh et al., 2004)
               xb_coef_tmp = xk_FeL * (xLgT - xFeT) - 1.0
               xb2M4ac     = max(((xb_coef_tmp * xb_coef_tmp) + (4.0 * xk_FeL * xLgT)), 0.0)
               !!
               !! "free" ligand concentration
               xLgF        = 0.5 * (xb_coef_tmp + (xb2M4ac**0.5)) / xk_FeL
               !!
               !! ligand-bound iron concentration
               xFeL        = xLgT - xLgF
               !!
               !! "free" iron concentration (and convert to mmol Fe / m3)
               xFeF        = (xFeT - xFeL) * 1.e-3
               xFree       = xFeF / (zfer + tiny(zfer))
               !!
               !! scavenging of iron
               !!----------------------------------------------------------------------
	       !! Dutkiewicz et al. (2005)
	       !! This scheme includes a single scavenging term based solely on a
               !! fixed rate and the availablility of "free" iron
	       !!----------------------------------------------------------------------
               !!
               ffescav     = xk_sc_Fe * xFeF                     ! = mmol/m3/d
	       !!
	       !!----------------------------------------------------------------------
               !!
               !! Mick's code contains a further (optional) implicit "scavenging" of 
               !! iron that sets an upper bound on "free" iron concentration, and 
               !! essentially caps the concentration of total iron as xFeL + "free" 
               !! iron; since the former is constrained by a fixed total ligand 
               !! concentration (= 1.0 umol/m3), and the latter isn't allowed above 
               !! this upper bound, total iron is constrained to a maximum of ...
               !!
               !!    xFeL + min(xFeF, 0.3 umol/m3) = 1.0 + 0.3 = 1.3 umol / m3
               !! 
               !! In Mick's code, the actual value of total iron is reset to this
               !! sum (i.e. TFe = FeL + Fe'; but Fe' <= 0.3 umol/m3); this isn't
               !! our favoured approach to tracer updating here (not least because
               !! of the leapfrog), so here the amount scavenged is augmented by an
               !! additional amount that serves to drag total iron back towards that
               !! expected from this limitation on iron concentration ...
               !!
               xmaxFeF     = min((xFeF * 1.e3), 0.3)             ! = umol/m3
               !!
               !! Here, the difference between current total Fe and (FeL + Fe') is
               !! calculated and added to the scavenging flux already calculated
               !! above ...
               !!
               fdeltaFe    = (xFeT - (xFeL + xmaxFeF)) * 1.e-3   ! = mmol/m3
               !!
               !! This assumes that the "excess" iron is dissipated with a time-
               !! scale of 1 day; seems reasonable to me ... (famous last words)
               !!
               ffescav     = ffescav + fdeltaFe                  ! = mmol/m3/d

               !!----------------------------------------------------------------------
               !! Other iron cycle processes
               !!----------------------------------------------------------------------
               !!
               !! aeolian iron deposition
               if (jk.eq.1) then
                  !! dust   is in g Fe / m2 / month
                  !! ffetop is in mmol / m3 / day
                  ffetop  = (((dust(ji,jj) * 1.e3 * xfe_sol) / xfe_mass) / fthk) / 30.
               else
                  ffetop  = 0.0
               endif

               !! report miscellaneous calculations
               if (idf.eq.1.AND.idfval.eq.1) then
                  IF (lwp) write (numout,*) '------------------------------'
                  IF (lwp) write (numout,*) 'xfe_sol  = ', xfe_sol
                  IF (lwp) write (numout,*) 'xfe_mass = ', xfe_mass
                  IF (lwp) write (numout,*) 'ffetop(',jk,')  = ', ffetop
                  IF (lwp) write (numout,*) 'xFree(',jk,')   = ', xFree
                  IF (lwp) write (numout,*) 'ffescav(',jk,') = ', ffescav
               endif
               
               !!----------------------------------------------------------------------
               !! Miscellaneous
               !!----------------------------------------------------------------------
               !!
               !! diatom frustule dissolution
               fsdiss  = xsdiss * zpds

               !! report miscellaneous calculations
               if (idf.eq.1.AND.idfval.eq.1) then
                  IF (lwp) write (numout,*) '------------------------------'
                  IF (lwp) write (numout,*) 'fsdiss(',jk,')  = ', fsdiss
               endif

               !!----------------------------------------------------------------------
	       !! Slow detritus creation
               !!----------------------------------------------------------------------
	       !! this variable integrates the creation of slow sinking detritus
	       !! to allow the split between fast and slow detritus to be 
	       !! diagnosed
	       fslown  = fdpn + fdzmi + ((1.0 - xfdfrac1) * fdpd) + &
               ((1.0 - xfdfrac2) * fdzme) + ((1.0 - xbetan) * (finmi + finme))
               !!
               !! this variable records the slow detrital sinking flux at this
               !! particular depth; it is used in the output of this flux at
               !! standard depths in the diagnostic outputs; needs to be
               !! adjusted from per second to per day because of parameter vsed
               fslownflux = zdet * vsed * 86400.

               !!----------------------------------------------------------------------
               !! Nutrient regeneration
               !! this variable integrates total nitrogen regeneration down the
               !! watercolumn; its value is stored and output as a 2D diagnostic;
               !! the corresponding dissolution flux of silicon (from sources
               !! other than fast detritus) is also integrated
               !!----------------------------------------------------------------------
               !!
	       fregen   = (( (xphi * (fgmipn + fgmid)) +           &
               (xphi * (fgmepn + fgmepd + fgmezmi + fgmed)) +      & 
               fmiexcr + fmeexcr + fdd +                           &
	       fdpn2 + fdpd2 + fdzmi2 + fdzme2) * fthk)
               !!
               fregensi = (( fsdiss + ((1.0 - xfdfrac1) * fdpds) + &
               ((1.0 - xfdfrac2) * fgmepds) +                      &
	       fdpds2) * fthk)

               !!----------------------------------------------------------------------
               !! Fast-sinking detritus terms
               !! "local" variables declared so that conservation can be checked;
               !! the calculated terms are added to the fast-sinking flux later on
               !! only after the flux entering this level has experienced some
               !! remineralisation
               !! note: these fluxes need to be scaled by the level thickness
               !!----------------------------------------------------------------------
               !!
               !! b0 is present for debugging purposes; using b0 = 0 sets the tendency
               !! terms of all biological equations to 0.
               b0 = 1.
               !!
               !! nitrogen:   diatom and mesozooplankton mortality
               ftempn         = b0 * ((xfdfrac1 * fdpd)  + (xfdfrac2 * fdzme))
               !!
               !! silicon:    diatom mortality and grazed diatoms
               ftempsi        = b0 * ((xfdfrac1 * fdpds) + (xfdfrac2 * fgmepds))
               !!
               !! iron:       diatom and mesozooplankton mortality
               ftempfe        = b0 * (((xfdfrac1 * fdpd) + (xfdfrac2 * fdzme)) * xrfn)
               !!
               !! carbon:     diatom and mesozooplankton mortality
               ftempc         = b0 * ((xfdfrac1 * xthetapd * fdpd) + & 
                                (xfdfrac2 * xthetazme * fdzme))
               !!
               !! CaCO3:      latitudinally-based fraction of total primary production
               !!               absolute latitude of current grid cell
               flat           = abs(gphit(ji,jj))
               !!               0.10 at equator; 0.02 at pole
               fcaco3         = xcaco3a + ((xcaco3b - xcaco3a) * ((90.0 - flat) / 90.0))
	       !! AXY (09/03/09): convert CaCO3 production from function of 
               !! primary production into a function of fast-sinking material; 
               !! technically, this is what Dunne et al. (2007) do anyway; they 
               !! convert total primary production estimated from surface 
               !! chlorophyll to an export flux for which they apply conversion 
               !! factors to estimate the various elemental fractions (Si, Ca)
               ftempca        = ftempc * fcaco3

               !! integrate total fast detritus production
               if (idf.eq.1) then
                  fifd_n(ji,jj)  = fifd_n(ji,jj)  + (ftempn  * fthk)
                  fifd_si(ji,jj) = fifd_si(ji,jj) + (ftempsi * fthk)
                  fifd_fe(ji,jj) = fifd_fe(ji,jj) + (ftempfe * fthk)
               endif

               !! report quantities of fast-sinking detritus for each component
               if (idf.eq.1.AND.idfval.eq.1) then
                  IF (lwp) write (numout,*) '------------------------------'
                  IF (lwp) write (numout,*) 'fdpd(',jk,') = ', fdpd
                  IF (lwp) write (numout,*) 'fdzme(',jk,') = ', fdzme
                  IF (lwp) write (numout,*) 'ftempn(',jk,')  = ', ftempn
                  IF (lwp) write (numout,*) 'ftempsi(',jk,') = ', ftempsi
                  IF (lwp) write (numout,*) 'ftempfe(',jk,') = ', ftempfe
                  IF (lwp) write (numout,*) 'ftempc(',jk,')  = ', ftempc
                  IF (lwp) write (numout,*) 'ftempca(',jk,') = ', ftempca
		  IF (lwp) write (numout,*) 'flat(',jk,') = ', flat
		  IF (lwp) write (numout,*) 'fcaco3(',jk,') = ', fcaco3
               endif

               !!----------------------------------------------------------------------
               !! Fast-sinking detritus fluxes, pt. 1: REMINERALISATION
               !! aside from explicitly modelled, slow-sinking detritus, the
               !! model includes an implicit representation of detrital
               !! particles that sink too quickly to be modelled with
               !! explicit state variables; this sinking flux is instead
               !! instantaneously remineralised down the water column using
               !! the version of Armstrong et al. (2002)'s ballast model
               !! used by Dunne et al. (2007); the version of this model
               !! here considers silicon and calcium carbonate ballast
               !! minerals; this section of the code redistributes the fast
               !! sinking material generated locally down the water column;
               !! this differs from Dunne et al. (2007) in that fast sinking
               !! material is distributed at *every* level below that it is
               !! generated, rather than at every level below some fixed
               !! depth; this scheme is also different in that sinking material 
               !! generated in one level is aggregated with that generated by
               !! shallower levels; this should make the ballast model more
               !! self-consistent (famous last words)
               !!----------------------------------------------------------------------
               !!
               if (jk.eq.1) then
                  !! this is the SURFACE OCEAN BOX (no remineralisation)
                  !!
                  freminc  = 0.0
                  freminn  = 0.0
                  freminfe = 0.0
                  freminsi = 0.0
                  freminca = 0.0
               elseif (jk.lt.(mbathy(ji,jj))) then
                  !! this is an OCEAN BOX (remineralise some material)
                  !!
                  !! === organic carbon ===
                  fq0      = ffastc(ji,jj)                        !! how much organic C enters this box        (mol)
                  if (iball.eq.1) then
                     fq1      = (fq0 * xmassc)                    !! how much it weighs                        (mass)
                     fq2      = (ffastca(ji,jj) * xmassca)        !! how much CaCO3 enters this box            (mass)
                     fq3      = (ffastsi(ji,jj) * xmasssi)        !! how much  opal enters this box            (mass)
                     fq4      = (fq2 * xprotca) + (fq3 * xprotsi) !! total protected organic C                 (mass)
                     !! this next term is calculated for C but used for N and Fe as well
                     !! it needs to be protected in case ALL C is protected
                     if (fq4.lt.fq1) then
                        fprotf   = (fq4 / fq1)                    !! protected fraction of total organic C     (non-dim)
                     else
                        fprotf   = 1.0                            !! all organic C is protected                (non-dim)
                     endif
                     fq5      = (1.0 - fprotf)                    !! unprotected fraction of total organic C   (non-dim)
                     fq6      = (fq0 * fq5)                       !! how much organic C is unprotected         (mol)
                     fq7      = (fq6 * exp(-(fthk / xfastc)))     !! how much unprotected C leaves this box    (mol)
                     fq8      = (fq7 + (fq0 * fprotf))            !! how much total C leaves this box          (mol)
                     freminc  = (fq0 - fq8) / fthk                !! C remineralisation in this box            (mol)
                     ffastc(ji,jj) = fq8                          
                     !! report in/out/remin fluxes of carbon for this level
                        if (idf.eq.1.AND.idfval.eq.1) then
                        IF (lwp) write (numout,*) '------------------------------'
                        IF (lwp) write (numout,*) 'totalC(',jk,')  = ', fq1
                        IF (lwp) write (numout,*) 'prtctC(',jk,')  = ', fq4
                        IF (lwp) write (numout,*) 'fprotf(',jk,')  = ', fprotf
                        IF (lwp) write (numout,*) '------------------------------'
                        IF (lwp) write (numout,*) 'IN   C(',jk,')  = ', fq0
                        IF (lwp) write (numout,*) 'LOST C(',jk,')  = ', freminc * fthk
                        IF (lwp) write (numout,*) 'OUT  C(',jk,')  = ', fq8
                        IF (lwp) write (numout,*) 'NEW  C(',jk,')  = ', ftempc * fthk
                     endif
                  else
                     fq1      = fq0 * exp(-(fthk / xfastc))       !! how much organic C leaves this box        (mol)
                     freminc  = (fq0 - fq1) / fthk                !! C remineralisation in this box            (mol)
                     ffastc(ji,jj)  = fq1
                  endif
                  !!
                  !! === organic nitrogen ===
                  fq0      = ffastn(ji,jj)                        !! how much organic N enters this box        (mol)
                  if (iball.eq.1) then
                     fq5      = (1.0 - fprotf)                    !! unprotected fraction of total organic N   (non-dim)
                     fq6      = (fq0 * fq5)                       !! how much organic N is unprotected         (mol)
                     fq7      = (fq6 * exp(-(fthk / xfastc)))     !! how much unprotected N leaves this box    (mol)
                     fq8      = (fq7 + (fq0 * fprotf))            !! how much total N leaves this box          (mol)
                     freminn  = (fq0 - fq8) / fthk                !! N remineralisation in this box            (mol)
                     ffastn(ji,jj)  = fq8
                     !! report in/out/remin fluxes of carbon for this level
                     if (idf.eq.1.AND.idfval.eq.1) then
                        IF (lwp) write (numout,*) '------------------------------'
                        IF (lwp) write (numout,*) 'totalN(',jk,')  = ', fq1
                        IF (lwp) write (numout,*) 'prtctN(',jk,')  = ', fq4
                        IF (lwp) write (numout,*) 'fprotf(',jk,')  = ', fprotf
                        IF (lwp) write (numout,*) '------------------------------'
                        if (freminn < 0.0) then
                           IF (lwp) write (numout,*) '** FREMIN ERROR **'
                        endif
                        IF (lwp) write (numout,*) 'IN   N(',jk,')  = ', fq0
                        IF (lwp) write (numout,*) 'LOST N(',jk,')  = ', freminn * fthk
                        IF (lwp) write (numout,*) 'OUT  N(',jk,')  = ', fq8
                        IF (lwp) write (numout,*) 'NEW  N(',jk,')  = ', ftempn * fthk
                     endif
                  else
                     fq1      = fq0 * exp(-(fthk / xfastc))       !! how much organic N leaves this box        (mol)
                     freminn  = (fq0 - fq1) / fthk                !! N remineralisation in this box            (mol)
                     ffastn(ji,jj)  = fq1
                  endif
                  !!
                  !! === organic iron ===
                  fq0      = ffastfe(ji,jj)                       !! how much organic Fe enters this box       (mol)
                  if (iball.eq.1) then
                     fq5      = (1.0 - fprotf)                    !! unprotected fraction of total organic Fe  (non-dim)
                     fq6      = (fq0 * fq5)                       !! how much organic Fe is unprotected        (mol)
                     fq7      = (fq6 * exp(-(fthk / xfastc)))     !! how much unprotected Fe leaves this box   (mol)
                     fq8      = (fq7 + (fq0 * fprotf))            !! how much total Fe leaves this box         (mol)            
                     freminfe = (fq0 - fq8) / fthk                !! Fe remineralisation in this box           (mol)
                     ffastfe(ji,jj) = fq8
                  else
                     fq1      = fq0 * exp(-(fthk / xfastc))       !! how much total Fe leaves this box         (mol)      
                     freminfe = (fq0 - fq1) / fthk                !! Fe remineralisation in this box           (mol)
                     ffastfe(ji,jj) = fq1
                  endif
                  !!
                  !! === biogenic silicon ===
                  fq0      = ffastsi(ji,jj)                       !! how much  opal centers this box           (mol) 
                  fq1      = fq0 * exp(-(fthk / xfastsi))         !! how much  opal leaves this box            (mol)
                  freminsi = (fq0 - fq1) / fthk                   !! Si remineralisation in this box           (mol)
                  ffastsi(ji,jj) = fq1
                  !!
                  !! === biogenic calcium carbonate ===
                  fq0      = ffastca(ji,jj)                       !! how much CaCO3 enters this box            (mol)
                  if (fdep.le.ocal_ccd(ji,jj)) then
                     !! whole grid cell above CCD
                     fq1      = fq0                               !! above lysocline, no Ca dissolves          (mol)
                     freminca = 0.0                               !! above lysocline, no Ca dissolves          (mol)
                     fccd(ji,jj) = real(jk)                       !! which is the last level above the CCD?    (#)
                  elseif (fdep.ge.ocal_ccd(ji,jj)) then
                     !! whole grid cell below CCD
                     fq1      = fq0 * exp(-(fthk / xfastca))      !! how much CaCO3 leaves this box            (mol)
                     freminca = (fq0 - fq1) / fthk                !! Ca remineralisation in this box           (mol)
                  else
                     !! partial grid cell below CCD
                     fq2      = fdep1 - ocal_ccd(ji,jj)           !! amount of grid cell below CCD             (m)
                     fq1      = fq0 * exp(-(fq2 / xfastca))       !! how much CaCO3 leaves this box            (mol)
                     freminca = (fq0 - fq1) / fthk                !! Ca remineralisation in this box           (mol)
                  endif
                  ffastca(ji,jj) = fq1 
               else
                  !! this is BELOW THE LAST OCEAN BOX (do nothing)
                  freminc  = 0.0
                  freminn  = 0.0
                  freminfe = 0.0
                  freminsi = 0.0
                  freminca = 0.0              
               endif

               !!----------------------------------------------------------------------
               !! Fast-sinking detritus fluxes, pt. 2: UPDATE FAST FLUXES
               !! here locally calculated additions to the fast-sinking flux are added
               !! to the total fast-sinking flux; this is done here such that material
               !! produced in a particular layer is only remineralised below this
               !! layer
               !!----------------------------------------------------------------------
               !!
               !! add sinking material generated in this layer to running totals
               !!
               !! === organic carbon ===                          (diatom and mesozooplankton mortality)
               ffastc(ji,jj)  = ffastc(ji,jj)  + (ftempc  * fthk)
               !!
               !! === organic nitrogen ===                        (diatom and mesozooplankton mortality)
               ffastn(ji,jj)  = ffastn(ji,jj)  + (ftempn  * fthk)
               !!
               !! === organic iron ===                            (diatom and mesozooplankton mortality)
               ffastfe(ji,jj) = ffastfe(ji,jj) + (ftempfe * fthk)
               !!
               !! === biogenic silicon ===                        (diatom mortality and grazed diatoms)
               ffastsi(ji,jj) = ffastsi(ji,jj) + (ftempsi * fthk)
               !!
               !! === biogenic calcium carbonate ===              (latitudinally-based fraction of total primary production)
               ffastca(ji,jj) = ffastca(ji,jj) + (ftempca * fthk)

               !!----------------------------------------------------------------------
               !! Fast-sinking detritus fluxes, pt. 3: SEAFLOOR
               !! remineralise all remaining fast-sinking detritus to dissolved
               !! nutrients; the sedimentation fluxes calculated here allow the
               !! separation of what's remineralised sinking through the final
               !! ocean box from that which is added to the final box by the
               !! remineralisation of material that reaches the seafloor (i.e.
               !! the model assumes that *all* material that hits the seafloor
               !! is remineralised and that none is permanently buried; hey,
               !! this is a giant GCM model that can't be run for long enough
               !! to deal with burial fluxes!)
               !!----------------------------------------------------------------------
               !! 
               if (jk.eq.(mbathy(ji,jj)-1)) then
                  !! this is the BOTTOM OCEAN BOX (remineralise everything)
                  !! 
                  !! === organic carbon ===
                  freminc  = freminc + (ffastc(ji,jj) / fthk)     !! C remineralisation in this box            (mol)
                  fsedc    = ffastc(ji,jj)                        !! record seafloor C
                  ffastc(ji,jj)  = 0.0
                  !!
                  !! === organic nitrogen ===
                  freminn  = freminn + (ffastn(ji,jj) / fthk)     !! N remineralisation in this box            (mol)
                  fsedn    = ffastn(ji,jj)                        !! record seafloor N
                  ffastn(ji,jj)  = 0.0
                  !!
                  !! === organic iron ===
                  freminfe = freminfe + (ffastfe(ji,jj) / fthk)   !! Fe remineralisation in this box           (mol)
                  fsedfe   = ffastfe(ji,jj)                       !! record seafloor Fe
                  ffastfe(ji,jj) = 0.0
                  !!
                  !! === biogenic silicon ===
                  freminsi = freminsi + (ffastsi(ji,jj) / fthk)   !! Si remineralisation in this box           (mol) 
                  fsedsi   = ffastsi(ji,jj)                       !! record seafloor Si
                  ffastsi(ji,jj) = 0.0
                  !!
                  !! === biogenic calcium carbonate ===
                  freminca = freminca + (ffastca(ji,jj) / fthk)   !! Ca remineralisation in this box           (mol) 
                  fsedca   = ffastca(ji,jj)                       !! record seafloor Ca
                  ffastca(ji,jj) = 0.0
               endif

               if (idf.eq.1) then
                  !!----------------------------------------------------------------------
                  !! Integrate total fast detritus remineralisation
                  !!----------------------------------------------------------------------
                  !!
                  fofd_n(ji,jj)  = fofd_n(ji,jj)  + (freminn  * fthk)
                  fofd_si(ji,jj) = fofd_si(ji,jj) + (freminsi * fthk)
                  fofd_fe(ji,jj) = fofd_fe(ji,jj) + (freminfe * fthk)
               endif

               !!----------------------------------------------------------------------
               !! Sort out remineralisation tally of fast-sinking detritus
               !!----------------------------------------------------------------------
               !!
	       !! update fast-sinking regeneration arrays
	       fregenfast(ji,jj)   = fregenfast(ji,jj)   + freminn  * fthk
	       fregenfastsi(ji,jj) = fregenfastsi(ji,jj) + freminsi * fthk

               !!======================================================================
               !! LOCAL GRID CELL TRENDS
               !!======================================================================
               !!
               !!----------------------------------------------------------------------
               !! Determination of trends
               !!----------------------------------------------------------------------
               !!
               !! chlorophyll
               bchn = b0 * ( &
                 + ((frn * fprn * zphn) - fgmipn - fgmepn - fdpn - fdpn2) * (fthetan / xxi) )
               bchd = b0 * ( &
                 + ((frd * fprd * zphd) - fgmepd - fdpd - fdpd2) * (fthetad / xxi) )
               !!
               !! phytoplankton
               bphn = b0 * ( &
                 + (fprn * zphn) - fgmipn - fgmepn - fdpn - fdpn2 )
               bphd = b0 * ( &
                 + (fprd * zphd) - fgmepd - fdpd - fdpd2 )
               bpds = b0 * ( &
                 + (fprds * zpds) - fgmepds - fdpds - fsdiss - fdpds2 )	 
               !!
               !! zooplankton
               bzmi = b0 * ( &
                 + fmigrow - fgmezmi - fdzmi - fdzmi2 )
               bzme = b0 * ( &
                 + fmegrow - fdzme - fdzme2 )
               !!
               !! detritus
               bdet = b0 * ( &
                 + fdpn + fdzmi + ((1.0 - xfdfrac1) * fdpd) + &
                 ((1.0 - xfdfrac2) * fdzme) + &
                 ((1.0 - xbetan) * (finmi + finme)) - fgmid - fgmed - fdd )
               !! AXY (26/11/08): implicit detrital carbon change
               !! bdtc = b0 * ( &
               !! + (xthetapn * fdpn) + (xthetazmi * fdzmi) + &
               !! ((1 - xfdfrac) * ((xthetapd * fdpd) + (xthetazme * fdzme))) + &
               !! ((1 - xbetac) * (ficmi + ficme)) - fgmidc - fgmedc - fddc )
               !!
               !! dissolved inorganic nitrogen nutrient
               bdin = b0 * ( &
                 - (fprn * zphn) - (fprd * zphd) + (xphi * (fgmipn + fgmid)) + &
                 (xphi * (fgmepn + fgmepd + fgmezmi + fgmed)) + & 
                 fmiexcr + fmeexcr + fdd + freminn + &
		 fdpn2 + fdpd2 + fdzmi2 + fdzme2 )
               !!
               !! dissolved silicic acid nutrient
               bsil = b0 * ( &
                 - (fprds * zpds) + fsdiss + & 
                 ((1.0 - xfdfrac1) * fdpds) + &
                 ((1.0 - xfdfrac2) * fgmepds) + &
                 freminsi + fdpds2 )
               !!
               !! dissolved "iron" nutrient
               bfer = b0 * ( &
                 + (xrfn * bdin) + ffetop + ffebot - ffescav )

	       !! report state variable fluxes (not including fast-sinking detritus)
               if (idf.eq.1.AND.idfval.eq.1) then
                  IF (lwp) write (numout,*) '------------------------------'
                  IF (lwp) write (numout,*) 'bchn(',jk,')  = ', bchn
                  IF (lwp) write (numout,*) 'bchd(',jk,')  = ', bchd
                  IF (lwp) write (numout,*) 'bphn(',jk,')  = ', bphn
                  IF (lwp) write (numout,*) 'bphd(',jk,')  = ', bphd
                  IF (lwp) write (numout,*) 'bpds(',jk,')  = ', bpds
                  IF (lwp) write (numout,*) 'bzmi(',jk,')  = ', bzmi
                  IF (lwp) write (numout,*) 'bzme(',jk,')  = ', bzme
                  IF (lwp) write (numout,*) 'bdet(',jk,')  = ', bdet
                  IF (lwp) write (numout,*) 'bdin(',jk,')  = ', bdin
                  IF (lwp) write (numout,*) 'bsil(',jk,')  = ', bsil
                  IF (lwp) write (numout,*) 'bfer(',jk,')  = ', bfer
               endif

               !!----------------------------------------------------------------------
               !! Integrate calculated fluxes for mass balance
               !!----------------------------------------------------------------------
               !!
               !! === nitrogen ===
               fflx_n(ji,jj)  = fflx_n(ji,jj)  + &
                  fthk * ( bphn + bphd + bzmi + bzme + bdet + bdin )
               !! === silicon ===
               fflx_si(ji,jj) = fflx_si(ji,jj) + &
                  fthk * ( bpds + bsil )
               !! === iron ===
               fflx_fe(ji,jj) = fflx_fe(ji,jj) + &
                  fthk * ( ( xrfn * ( bphn + bphd + bzmi + bzme + bdet) ) + bfer )

               !!----------------------------------------------------------------------
               !! Apply calculated tracer fluxes
               !!----------------------------------------------------------------------
               !!
               !! units: [unit of tracer] per second (fluxes are calculated above per day)
               !!
	       ibio_switch = 1
	       if (ibio_switch.eq.1) then
               tra(ji,jj,jk,jpchn) = tra(ji,jj,jk,jpchn) + (bchn / 86400.)
               tra(ji,jj,jk,jpchd) = tra(ji,jj,jk,jpchd) + (bchd / 86400.)
               tra(ji,jj,jk,jpphn) = tra(ji,jj,jk,jpphn) + (bphn / 86400.)
               tra(ji,jj,jk,jpphd) = tra(ji,jj,jk,jpphd) + (bphd / 86400.)
               tra(ji,jj,jk,jppds) = tra(ji,jj,jk,jppds) + (bpds / 86400.)
               tra(ji,jj,jk,jpzmi) = tra(ji,jj,jk,jpzmi) + (bzmi / 86400.)
               tra(ji,jj,jk,jpzme) = tra(ji,jj,jk,jpzme) + (bzme / 86400.)
               tra(ji,jj,jk,jpdet) = tra(ji,jj,jk,jpdet) + (bdet / 86400.)
               !! AXY (26/11/08): implicit detrital carbon change
               !! tra(ji,jj,jk,jpdtc) = tra(ji,jj,jk,jpdtc) + (bdtc / 86400.)
               tra(ji,jj,jk,jpdin) = tra(ji,jj,jk,jpdin) + (bdin / 86400.)
               tra(ji,jj,jk,jpsil) = tra(ji,jj,jk,jpsil) + (bsil / 86400.)
               tra(ji,jj,jk,jpfer) = tra(ji,jj,jk,jpfer) + (bfer / 86400.)
	       endif               

               !!----------------------------------------------------------------------
               !! Check model conservation
               !! these terms merely sum up the tendency terms of the relevant
               !! state variables, which should sum to zero; the iron cycle is
               !! complicated by fluxes that add (aeolian deposition and seafloor
               !! remineralisation) and remove (scavenging) dissolved iron from
               !! the model (i.e. the sum of iron fluxes is unlikely to be zero)
               !!----------------------------------------------------------------------
               !!
               fnit0 = bphn + bphd + bzme + bzmi + bdet + bdin  ! + ftempn
               fsil0 = bpds + bsil                              ! + ftempsi
               ffer0 = (xrfn * fnit0) + bfer
               !!
               if (kt/240*240.eq.kt) then
                  if (ji.eq.2.and.jj.eq.2.and.jk.eq.1) then
                     IF (lwp) write (*,*) '*******!MEDUSA Conservation!*******',kt
                     IF (lwp) write (*,*) fnit0,fsil0,ffer0
                  endif
               endif     

#if defined key_trc_diabio
               !!======================================================================
               !! LOCAL GRID CELL DIAGNOSTICS
               !!======================================================================
               !!
               !!----------------------------------------------------------------------
               !! Full diagnostics key_trc_diabio:
               !! LOBSTER and PISCES support full diagnistics option key_trc_diabio    
               !! which gives an option of FULL output of biological sourses and sinks.
               !! I cannot see any reason for doing this. If needed, it can be done
               !! as shown below.
               !!----------------------------------------------------------------------
               !!
               IF(lwp) WRITE(numout,*) ' MEDUSA does not support key_trc_diabio'
               !!               trbio(ji,jj,jk, 1) = fprn
#endif

#if defined key_trc_diaadd
               !!----------------------------------------------------------------------
               !! Prepare 2D diagnostics
               !!----------------------------------------------------------------------
               !!
               !! if ((kt / 240*240).eq.kt) then
               !!    IF (lwp) write (*,*) '*******!MEDUSA DIAADD!*******',kt
               !! endif     
               trc2d(ji,jj,1)  = trc2d(ji,jj,1)  + ftot_n                   !! nitrogen inventory
               trc2d(ji,jj,2)  = trc2d(ji,jj,2)  + ftot_si                  !! silicon  inventory
               trc2d(ji,jj,3)  = trc2d(ji,jj,3)  + ftot_fe                  !! iron     inventory
               trc2d(ji,jj,4)  = trc2d(ji,jj,4)  + (fprn  * zphn * fthk)    !! non-diatom production
               trc2d(ji,jj,5)  = trc2d(ji,jj,5)  + (fdpn         * fthk)    !! non-diatom non-grazing losses
               trc2d(ji,jj,6)  = trc2d(ji,jj,6)  + (fprd  * zphd * fthk)    !! diatom production
               trc2d(ji,jj,7)  = trc2d(ji,jj,7)  + (fdpd         * fthk)    !! diatom non-grazing losses
               !! diagnostic field  8 is (ostensibly) supplied by trcsed.F            
               trc2d(ji,jj,9)  = trc2d(ji,jj,9)  + (fprds * zpds * fthk)    !! diatom silicon production
               trc2d(ji,jj,10) = trc2d(ji,jj,10) + (fsdiss  * fthk)         !! diatom silicon dissolution
               trc2d(ji,jj,11) = trc2d(ji,jj,11) + (fgmipn  * fthk)         !! microzoo grazing on non-diatoms
               trc2d(ji,jj,12) = trc2d(ji,jj,12) + (fgmid   * fthk)         !! microzoo grazing on detrital nitrogen
               trc2d(ji,jj,13) = trc2d(ji,jj,13) + (fdzmi   * fthk)         !! microzoo non-grazing losses
               trc2d(ji,jj,14) = trc2d(ji,jj,14) + (fgmepn  * fthk)         !! mesozoo  grazing on non-diatoms
               trc2d(ji,jj,15) = trc2d(ji,jj,15) + (fgmepd  * fthk)         !! mesozoo  grazing on diatoms
               trc2d(ji,jj,16) = trc2d(ji,jj,16) + (fgmezmi * fthk)         !! mesozoo  grazing on microzoo
               trc2d(ji,jj,17) = trc2d(ji,jj,17) + (fgmed   * fthk)         !! mesozoo  grazing on detrital nitrogen
               trc2d(ji,jj,18) = trc2d(ji,jj,18) + (fdzme   * fthk)         !! mesozoo  non-grazing losses
               !! diagnostic field 19 is (ostensibly) supplied by trcexp.F
               trc2d(ji,jj,20) = trc2d(ji,jj,20) + (fslown  * fthk)         !! slow sinking detritus N production
               trc2d(ji,jj,21) = trc2d(ji,jj,21) + (fdd     * fthk)         !! detrital remineralisation
               trc2d(ji,jj,22) = trc2d(ji,jj,22) + (ffetop  * fthk)         !! aeolian  iron addition
               trc2d(ji,jj,23) = trc2d(ji,jj,23) + (ffebot  * fthk)         !! seafloor iron addition
               trc2d(ji,jj,24) = trc2d(ji,jj,24) + (ffescav * fthk)         !! "free" iron scavenging
	       trc2d(ji,jj,25) = trc2d(ji,jj,25) + (fjln  * zphn * fthk)    !! non-diatom J  limitation term 
	       trc2d(ji,jj,26) = trc2d(ji,jj,26) + (fnln  * zphn * fthk)    !! non-diatom N  limitation term 
	       trc2d(ji,jj,27) = trc2d(ji,jj,27) + (ffln  * zphn * fthk)    !! non-diatom Fe limitation term 
	       trc2d(ji,jj,28) = trc2d(ji,jj,28) + (fjld  * zphd * fthk)    !! diatom     J  limitation term 
	       trc2d(ji,jj,29) = trc2d(ji,jj,29) + (fnld  * zphd * fthk)    !! diatom     N  limitation term 
	       trc2d(ji,jj,30) = trc2d(ji,jj,30) + (ffld  * zphd * fthk)    !! diatom     Fe limitation term 
	       trc2d(ji,jj,31) = trc2d(ji,jj,31) + (fsld2 * zphd * fthk)    !! diatom     Si limitation term 
	       trc2d(ji,jj,32) = trc2d(ji,jj,32) + (fsld  * zphd * fthk)    !! diatom     Si uptake limitation term
               if (jk.eq.i0100) trc2d(ji,jj,33) = fslownflux                !! slow detritus flux at  100 m
               if (jk.eq.i0200) trc2d(ji,jj,34) = fslownflux                !! slow detritus flux at  200 m
               if (jk.eq.i0500) trc2d(ji,jj,35) = fslownflux                !! slow detritus flux at  500 m
               if (jk.eq.i1000) trc2d(ji,jj,36) = fslownflux                !! slow detritus flux at 1000 m
               trc2d(ji,jj,37) = trc2d(ji,jj,37) + fregen                   !! non-fast N  full column regeneration
               trc2d(ji,jj,38) = trc2d(ji,jj,38) + fregensi                 !! non-fast Si full column regeneration
               if (jk.eq.i0100) trc2d(ji,jj,39) = trc2d(ji,jj,37)           !! non-fast N  regeneration to  100 m
               if (jk.eq.i0200) trc2d(ji,jj,40) = trc2d(ji,jj,37)           !! non-fast N  regeneration to  200 m
               if (jk.eq.i0500) trc2d(ji,jj,41) = trc2d(ji,jj,37)           !! non-fast N  regeneration to  500 m
               if (jk.eq.i1000) trc2d(ji,jj,42) = trc2d(ji,jj,37)           !! non-fast N  regeneration to 1000 m
               trc2d(ji,jj,43) = trc2d(ji,jj,43) + (ftempn  * fthk)         !! fast sinking detritus N production
               trc2d(ji,jj,44) = trc2d(ji,jj,44) + (ftempsi * fthk)         !! fast sinking detritus Si production
               trc2d(ji,jj,45) = trc2d(ji,jj,45) + (ftempfe * fthk)         !! fast sinking detritus Fe production
               trc2d(ji,jj,46) = trc2d(ji,jj,46) + (ftempc  * fthk)         !! fast sinking detritus C production
               trc2d(ji,jj,47) = trc2d(ji,jj,47) + (ftempca * fthk)         !! fast sinking detritus CaCO3 production
               if (jk.eq.i0100) trc2d(ji,jj,48) = ffastn(ji,jj)             !! fast detritus N  flux at  100 m
               if (jk.eq.i0200) trc2d(ji,jj,49) = ffastn(ji,jj)             !! fast detritus N  flux at  200 m
               if (jk.eq.i0500) trc2d(ji,jj,50) = ffastn(ji,jj)             !! fast detritus N  flux at  500 m
               if (jk.eq.i1000) trc2d(ji,jj,51) = ffastn(ji,jj)             !! fast detritus N  flux at 1000 m
               if (jk.eq.i0100) trc2d(ji,jj,52) = fregenfast(ji,jj)         !! N  regeneration to  100 m
               if (jk.eq.i0200) trc2d(ji,jj,53) = fregenfast(ji,jj)         !! N  regeneration to  200 m
               if (jk.eq.i0500) trc2d(ji,jj,54) = fregenfast(ji,jj)         !! N  regeneration to  500 m
               if (jk.eq.i1000) trc2d(ji,jj,55) = fregenfast(ji,jj)         !! N  regeneration to 1000 m
               if (jk.eq.i0100) trc2d(ji,jj,56) = ffastsi(ji,jj)            !! fast detritus Si flux at  100 m
               if (jk.eq.i0200) trc2d(ji,jj,57) = ffastsi(ji,jj)            !! fast detritus Si flux at  200 m
               if (jk.eq.i0500) trc2d(ji,jj,58) = ffastsi(ji,jj)            !! fast detritus Si flux at  500 m
               if (jk.eq.i1000) trc2d(ji,jj,59) = ffastsi(ji,jj)            !! fast detritus Si flux at 1000 m
               if (jk.eq.i0100) trc2d(ji,jj,60) = fregenfastsi(ji,jj)       !! Si regeneration to  100 m
               if (jk.eq.i0200) trc2d(ji,jj,61) = fregenfastsi(ji,jj)       !! Si regeneration to  200 m
               if (jk.eq.i0500) trc2d(ji,jj,62) = fregenfastsi(ji,jj)       !! Si regeneration to  500 m
               if (jk.eq.i1000) trc2d(ji,jj,63) = fregenfastsi(ji,jj)       !! Si regeneration to 1000 m
               trc2d(ji,jj,64) = trc2d(ji,jj,64) + (freminn  * fthk)        !! sum of fast-sinking N  fluxes
               trc2d(ji,jj,65) = trc2d(ji,jj,65) + (freminsi * fthk)        !! sum of fast-sinking Si fluxes
               trc2d(ji,jj,66) = trc2d(ji,jj,66) + (freminfe * fthk)        !! sum of fast-sinking Fe fluxes
               trc2d(ji,jj,67) = trc2d(ji,jj,67) + (freminc  * fthk)        !! sum of fast-sinking C  fluxes
               trc2d(ji,jj,68) = trc2d(ji,jj,68) + (freminca * fthk)        !! sum of fast-sinking Ca fluxes
               if (jk.eq.(mbathy(ji,jj)-1)) then
                  trc2d(ji,jj,69) = fsedn                                   !! N  sedimentation flux                                  
                  trc2d(ji,jj,70) = fsedsi                                  !! Si sedimentation flux
                  trc2d(ji,jj,71) = fsedfe                                  !! Fe sedimentation flux
                  trc2d(ji,jj,72) = fsedc                                   !! C  sedimentation flux
                  trc2d(ji,jj,73) = fsedca                                  !! Ca sedimentation flux
               endif
	       if (jk.eq.1)  trc2d(ji,jj,74) = qsr(ji,jj)
	       if (jk.eq.1)  trc2d(ji,jj,75) = xpar(ji,jj,jk)
               !! diagnostic fields 76 to 80 calculated below
               trc2d(ji,jj,81) = trc2d(ji,jj,81) + fprn_ml                  !! mixed layer non-diatom production
               trc2d(ji,jj,82) = trc2d(ji,jj,82) + fprd_ml                  !! mixed layer     diatom production
               trc2d(ji,jj,83) = ocal_ccd(ji,jj)                            !! calcite CCD depth
               trc2d(ji,jj,84) = fccd(ji,jj)                                !! last model level above calcite CCD depth
               if (jk.eq.1)     trc2d(ji,jj,85) = xFree                     !! surface "free" iron
               if (jk.eq.i0200) trc2d(ji,jj,86) = xFree                     !! "free" iron at  100 m
               if (jk.eq.i0200) trc2d(ji,jj,87) = xFree                     !! "free" iron at  200 m
               if (jk.eq.i0500) trc2d(ji,jj,88) = xFree                     !! "free" iron at  500 m
               if (jk.eq.i1000) trc2d(ji,jj,89) = xFree                     !! "free" iron at 1000 m
               !! 
               ftot_pn(ji,jj)  = ftot_pn(ji,jj)  + (zphn * fthk)            !! vertical integral non-diatom phytoplankton
               ftot_pd(ji,jj)  = ftot_pd(ji,jj)  + (zphd * fthk)            !! vertical integral diatom phytoplankton

               !! other possible future diagnostics include:
               !!   - integrated tracer values (esp. biological)
               !!   - mixed layer tracer values
               !!   - sub-surface chlorophyll maxima (plus depth)
               !!   - different mixed layer depth criteria (T, sigma, var. sigma)

               !!----------------------------------------------------------------------
               !! Prepare 3D diagnostics
               !!----------------------------------------------------------------------
               !!
               trc3d(ji,jj,jk,1)  = (fprn * zphn)        !! non-diatom production  
               trc3d(ji,jj,jk,2)  = (fprd * zphd)        !! diatom production
	       !! trc3d(ji,jj,jk,3)  = xORGscav
	       !! trc3d(ji,jj,jk,4)  = xINORGscav
	       trc3d(ji,jj,jk,3)  = fregen + freminn     !! N  nutrient regeneration
	       trc3d(ji,jj,jk,4)  = fregensi + freminsi  !! Si nutrient regeneration
               !! trc3d(ji,jj,jk,1)  = bdin
               !! trc3d(ji,jj,jk,2)  = bsil
               !! trc3d(ji,jj,jk,3)  = ffastn(ji,jj)
               !! trc3d(ji,jj,jk,4)  = freminn * fthk
               !! trc3d(ji,jj,jk,5)  = ffastsi(ji,jj)
               !! trc3d(ji,jj,jk,6)  = freminsi * fthk
               !! trc3d(ji,jj,jk,7)  = ffastca(ji,jj)
               !! trc3d(ji,jj,jk,8)  = ffetop + ffebot - ffescav
               !! trc3d(ji,jj,jk,7)  = ftempn * fthk
               !! trc3d(ji,jj,jk,8)  = ftempsi * fthk
#endif

            !! CLOSE horizontal loops
            END DO
         END DO
      !! CLOSE vertical loop
      END DO

#if defined key_trc_diaadd
      !!----------------------------------------------------------------------
      !! Output several accumulated diagnostics
      !!   - biomass-average phytoplankton limitation terms
      !!   - integrated tendency terms
      !!----------------------------------------------------------------------
      !! 
      DO jj = 2,jpjm1
         DO ji = 2,jpim1
            !! non-diatom phytoplankton limitations
            trc2d(ji,jj,25) = trc2d(ji,jj,25) / ftot_pn(ji,jj)
            trc2d(ji,jj,26) = trc2d(ji,jj,26) / ftot_pn(ji,jj)
            trc2d(ji,jj,27) = trc2d(ji,jj,27) / ftot_pn(ji,jj)
            !! diatom phytoplankton limitations
            trc2d(ji,jj,28) = trc2d(ji,jj,28) / ftot_pd(ji,jj)
            trc2d(ji,jj,29) = trc2d(ji,jj,29) / ftot_pd(ji,jj)
            trc2d(ji,jj,30) = trc2d(ji,jj,30) / ftot_pd(ji,jj)
            trc2d(ji,jj,31) = trc2d(ji,jj,31) / ftot_pd(ji,jj)
            trc2d(ji,jj,32) = trc2d(ji,jj,32) / ftot_pd(ji,jj)
            !! tendency terms
            trc2d(ji,jj,76) = fflx_n(ji,jj)
            trc2d(ji,jj,77) = fflx_si(ji,jj)
            trc2d(ji,jj,78) = fflx_fe(ji,jj)
            !! integrated phytoplankton biomass
            trc2d(ji,jj,79) = ftot_pn(ji,jj)
            trc2d(ji,jj,80) = ftot_pd(ji,jj)
         END DO
      END DO

      !!----------------------------------------------------------------------
      !! Don't know what this does; belongs to someone else ...
      !!----------------------------------------------------------------------
      !! 
      !! Lateral boundary conditions on trc2d
      DO jn=1,jp_medusa_2d
          CALL lbc_lnk(trc2d(:,:,jn),'T',1. )
      END DO 

      !! Lateral boundary conditions on trc3d
      DO jn=1,jp_medusa_3d
          CALL lbc_lnk(trc3d(:,:,1,jn),'T',1. )
      END DO 
#endif

#if defined key_iomput
      !!----------------------------------------------------------------------
      !! Add in XML diagnostics stuff
      !!----------------------------------------------------------------------
      !!
      !! ** 2D diagnostics
      zw2d = trc2d(:,:, 1)
      CALL iom_put("INVTN"    ,zw2d)
      !!
      zw2d = trc2d(:,:, 2)
      CALL iom_put("INVTSI"   ,zw2d)
      !!
      zw2d = trc2d(:,:, 3)
      CALL iom_put("INVTFE"   ,zw2d)
      !!
      zw2d = trc2d(:,:, 4)
      CALL iom_put("PRN"      ,zw2d)
      !!
      zw2d = trc2d(:,:, 5)
      CALL iom_put("MPN"      ,zw2d)
      !!
      zw2d = trc2d(:,:, 6)
      CALL iom_put("PRD"      ,zw2d)
      !!
      zw2d = trc2d(:,:, 7)
      CALL iom_put("MPD"      ,zw2d)
      !!
      !! zw2d = trc2d(:,:, 8)
      !! CALL iom_put("DSED"     ,zw2d)
      !!
      zw2d = trc2d(:,:, 9)
      CALL iom_put("OPAL"     ,zw2d)
      !!
      zw2d = trc2d(:,:,10)
      CALL iom_put("OPALDISS" ,zw2d)
      !!
      zw2d = trc2d(:,:,11)
      CALL iom_put("GMIPN"    ,zw2d)
      !!
      zw2d = trc2d(:,:,12)
      CALL iom_put("GMID"     ,zw2d)
      !!
      zw2d = trc2d(:,:,13)
      CALL iom_put("MZMI"     ,zw2d)
      !!
      zw2d = trc2d(:,:,14)
      CALL iom_put("GMEPN"    ,zw2d)
      !!
      zw2d = trc2d(:,:,15)
      CALL iom_put("GMEPD"    ,zw2d)
      !!
      zw2d = trc2d(:,:,16)
      CALL iom_put("GMEZMI"   ,zw2d)
      !!
      zw2d = trc2d(:,:,17)
      CALL iom_put("GMED"     ,zw2d)
      !!
      zw2d = trc2d(:,:,18)
      CALL iom_put("MZME"     ,zw2d)
      !!
      !! zw2d = trc2d(:,:,19)
      !! CALL iom_put("DEXP"     ,zw2d)
      !!
      zw2d = trc2d(:,:,20)
      CALL iom_put("DETN"     ,zw2d)
      !!
      zw2d = trc2d(:,:,21)
      CALL iom_put("MDET"     ,zw2d)
      !!
      zw2d = trc2d(:,:,22)
      CALL iom_put("AEOLIAN"  ,zw2d)
      !!
      zw2d = trc2d(:,:,23)
      CALL iom_put("BENTHIC"  ,zw2d)
      !!
      zw2d = trc2d(:,:,24)
      CALL iom_put("SCAVENGE" ,zw2d)
      !!
      zw2d = trc2d(:,:,25)
      CALL iom_put("PN_JLIM"  ,zw2d)
      !!
      zw2d = trc2d(:,:,26)
      CALL iom_put("PN_NLIM"  ,zw2d)
      !!
      zw2d = trc2d(:,:,27)
      CALL iom_put("PN_FELIM" ,zw2d)
      !!
      zw2d = trc2d(:,:,28)
      CALL iom_put("PD_JLIM"  ,zw2d)
      !!
      zw2d = trc2d(:,:,29)
      CALL iom_put("PD_NLIM"  ,zw2d)
      !!
      zw2d = trc2d(:,:,30)
      CALL iom_put("PD_FELIM" ,zw2d)
      !!
      zw2d = trc2d(:,:,31)
      CALL iom_put("PD_SILIM" ,zw2d)
      !!
      zw2d = trc2d(:,:,32)
      CALL iom_put("PDSILIM2" ,zw2d)
      !!
      zw2d = trc2d(:,:,33)
      CALL iom_put("SDT__100" ,zw2d)
      !!
      zw2d = trc2d(:,:,34)
      CALL iom_put("SDT__200" ,zw2d)
      !!
      zw2d = trc2d(:,:,35)
      CALL iom_put("SDT__500" ,zw2d)
      !!
      zw2d = trc2d(:,:,36)
      CALL iom_put("SDT_1000" ,zw2d)
      !!
      zw2d = trc2d(:,:,37)
      CALL iom_put("TOTREG_N" ,zw2d)
      !!
      zw2d = trc2d(:,:,38)
      CALL iom_put("TOTRG_SI" ,zw2d)
      !!
      zw2d = trc2d(:,:,39)
      CALL iom_put("REG__100" ,zw2d)
      !!
      zw2d = trc2d(:,:,40)
      CALL iom_put("REG__200" ,zw2d)
      !!
      zw2d = trc2d(:,:,41)
      CALL iom_put("REG__500" ,zw2d)
      !!
      zw2d = trc2d(:,:,42)
      CALL iom_put("REG_1000" ,zw2d)
      !!
      zw2d = trc2d(:,:,43)
      CALL iom_put("FASTN"    ,zw2d)
      !!
      zw2d = trc2d(:,:,44)
      CALL iom_put("FASTSI"   ,zw2d)
      !!
      zw2d = trc2d(:,:,45)
      CALL iom_put("FASTFE"   ,zw2d)
      !!
      zw2d = trc2d(:,:,46)
      CALL iom_put("FASTC"    ,zw2d)
      !!
      zw2d = trc2d(:,:,47)
      CALL iom_put("FASTCA"   ,zw2d)
      !!
      zw2d = trc2d(:,:,48)
      CALL iom_put("FDT__100" ,zw2d)
      !!
      zw2d = trc2d(:,:,49)
      CALL iom_put("FDT__200" ,zw2d)
      !!
      zw2d = trc2d(:,:,50)
      CALL iom_put("FDT__500" ,zw2d)
      !!
      zw2d = trc2d(:,:,51)
      CALL iom_put("FDT_1000" ,zw2d)
      !!
      zw2d = trc2d(:,:,52)
      CALL iom_put("RG__100F" ,zw2d)
      !!
      zw2d = trc2d(:,:,53)
      CALL iom_put("RG__200F" ,zw2d)
      !!
      zw2d = trc2d(:,:,54)
      CALL iom_put("RG__500F" ,zw2d)
      !!
      zw2d = trc2d(:,:,55)
      CALL iom_put("RG_1000F" ,zw2d)
      !!
      zw2d = trc2d(:,:,56)
      CALL iom_put("FDS__100" ,zw2d)
      !!
      zw2d = trc2d(:,:,57)
      CALL iom_put("FDS__200" ,zw2d)
      !!
      zw2d = trc2d(:,:,58)
      CALL iom_put("FDS__500" ,zw2d)
      !!
      zw2d = trc2d(:,:,59)
      CALL iom_put("FDS_1000" ,zw2d)
      !!
      zw2d = trc2d(:,:,60)
      CALL iom_put("RGS__100" ,zw2d)
      !!
      zw2d = trc2d(:,:,61)
      CALL iom_put("RGS__200" ,zw2d)
      !!
      zw2d = trc2d(:,:,62)
      CALL iom_put("RGS__500" ,zw2d)
      !!
      zw2d = trc2d(:,:,63)
      CALL iom_put("RGS_1000" ,zw2d)
      !!
      zw2d = trc2d(:,:,64)
      CALL iom_put("REMINN"   ,zw2d)
      !!
      zw2d = trc2d(:,:,65)
      CALL iom_put("REMINSI"  ,zw2d)
      !!
      zw2d = trc2d(:,:,66)
      CALL iom_put("REMINFE"  ,zw2d)
      !!
      zw2d = trc2d(:,:,67)
      CALL iom_put("REMINC"   ,zw2d)
      !!
      zw2d = trc2d(:,:,68)
      CALL iom_put("REMINCA"  ,zw2d)
      !!
      zw2d = trc2d(:,:,69)
      CALL iom_put("SEAFLRN"  ,zw2d)
      !!
      zw2d = trc2d(:,:,70)
      CALL iom_put("SEAFLRSI" ,zw2d)
      !!
      zw2d = trc2d(:,:,71)
      CALL iom_put("SEAFLRFE" ,zw2d)
      !!
      zw2d = trc2d(:,:,72)
      CALL iom_put("SEAFLRC"  ,zw2d)
      !!
      zw2d = trc2d(:,:,73)
      CALL iom_put("SEAFLRCA" ,zw2d)
      !!
      zw2d = trc2d(:,:,74)
      CALL iom_put("MED_QSR"  ,zw2d)
      !!
      zw2d = trc2d(:,:,75)
      CALL iom_put("MED_XPAR" ,zw2d)
      !!
      zw2d = trc2d(:,:,76)
      CALL iom_put("INTFLX_N" ,zw2d)
      !!
      zw2d = trc2d(:,:,77)
      CALL iom_put("INTFX_SI" ,zw2d)
      !!
      zw2d = trc2d(:,:,78)
      CALL iom_put("INTFX_FE" ,zw2d)
      !!
      zw2d = trc2d(:,:,79)
      CALL iom_put("INT_PN"   ,zw2d)
      !!
      zw2d = trc2d(:,:,80)
      CALL iom_put("INT_PD"   ,zw2d)
      !!
      zw2d = trc2d(:,:,81)
      CALL iom_put("ML_PRN"   ,zw2d)
      !!
      zw2d = trc2d(:,:,82)
      CALL iom_put("ML_PRD"   ,zw2d)
      !!
      zw2d = trc2d(:,:,83)
      CALL iom_put("OCAL_CCD" ,zw2d)
      !!
      zw2d = trc2d(:,:,84)
      CALL iom_put("OCAL_LVL" ,zw2d)
      !!
      zw2d = trc2d(:,:,85)
      CALL iom_put("BLANK_1"  ,zw2d)
      !!
      zw2d = trc2d(:,:,86)
      CALL iom_put("BLANK_2"  ,zw2d)
      !!
      zw2d = trc2d(:,:,87)
      CALL iom_put("BLANK_3"  ,zw2d)
      !!
      zw2d = trc2d(:,:,88)
      CALL iom_put("BLANK_4"  ,zw2d)
      !!
      zw2d = trc2d(:,:,89)
      CALL iom_put("BLANK_5"  ,zw2d)
      !!
      zw2d = trc2d(:,:,90)
      CALL iom_put("BLANK_6"  ,zw2d)
      !!
      !! ** 3D diagnostics
      zw3d = trc3d(:,:,:, 1)
      CALL iom_put("PRN3"     ,zw3d)
      !!
      zw3d = trc3d(:,:,:, 2)
      CALL iom_put("PRD3"     ,zw3d)
      !!
      zw3d = trc3d(:,:,:, 3)
      CALL iom_put("REMIN3N"  ,zw3d)
      !!
      zw3d = trc3d(:,:,:, 4)
      CALL iom_put("REMIN3SI" ,zw3d)
#endif

#if defined key_trc_diabio
      !! Lateral boundary conditions on trcbio
      DO jn=1,jp_medusa_trd
          CALL lbc_lnk(trbio(:,:,1,jn),'T',1. )
      END DO 
#endif

   END SUBROUTINE trc_bio_medusa

#  else
   !!======================================================================
   !!  Dummy module :                                   No MEDUSA bio-model
   !!======================================================================
CONTAINS
   SUBROUTINE trc_bio_medusa( kt )                   ! Empty routine
      INTEGER, INTENT( in ) ::   kt
      WRITE(*,*) 'trc_bio_medusa: You should not have seen this print! error?', kt
   END SUBROUTINE trc_bio_medusa
#endif 

   !!======================================================================
END MODULE  trcbio_medusa
