/* gnuplot.hpp
 * 
 * Copyright (C) 2010 Sylwester Arabas
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or (at
 * your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 */

#ifndef GNUPLOT_HPP
#  define GNUPLOT_HPP

#  include <cstdio>
#  include "ModelPseudoAdiabaticBulk.hpp"
#  include "SolverCVODESAdamsFunctional.hpp"
#  include "TolerancesUniform.hpp"

#  define self OutputGnuplot
class self : public Output
{

  protected: Model *model;
  protected: InitpTq *initptq;
  protected: SpectraMemLayout *ml;
  protected: Updraft *updraft;  

  private: FILE* tmpf;

  private: vector<long> offs;                    // \ __ TODO: some pre-allocation?
  protected: vector< quantity<si::time> > times; // /

  // TODO: move to GnuplotStuve?
  protected: size_t step, step_last, rec_step;

  protected: quantity<si::temperature> T_last;
  protected: quantity<si::pressure> p_last;
  protected: quantity<si::dimensionless> q_last;

  public: self(Model *model_, InitpTq *initptq_, SpectraMemLayout *ml_, Updraft *updraft_,
    size_t rec_step_
  ) 
  { 
    model = model_; 
    initptq = initptq_;
    ml = ml_;
    updraft = updraft_;

    tmpf = tmpfile();

    cout << scientific;
    cout.precision(12);

    rec_step = rec_step_;
    step = -1;
    step_last = -rec_step;
  }

  public: ~self()
  {
    fclose(tmpf);
  }

  public: void head() {}
  public: void foot(quantity<si::time>, N_Vector, N_Vector) {}

  protected: void moist_adiabat()
  {
    Model *model = new ModelPseudoAdiabaticBulk();
    OutputGnuplot *output = new OutputGnuplot(model, initptq, NULL, updraft, 1);
    ModelParams *params = model->getParams(updraft, NULL, NULL, NULL, NULL, NULL, NULL, NULL);
    TolerancesUniform *tol = new TolerancesUniform(1e-6, 1e-9);
    Solver *solver = new SolverCVODESAdamsFunctional(model, output, params, initptq, 
      NULL, NULL, NULL, NULL, tol, times.back());
    solver->run();
    output->dataSounding();
    delete solver;
    delete params;
    delete output;
    delete model;
  }
 
  protected: void lib() 
  {
    cout << "qs2rs(qs) = qs / (1 - qs)" << endl;
    cout << "rgb(r,g,b) = int(r)*65536 + int(g)*256 + int(b)" << endl;
    cout << "epsilon = " << realtype(constants::epsilon) << endl;
    cout << "R_d = " << realtype(constants::R_d / si::joules * si::kelvins * si::kilograms) << endl;
    cout << "R_v = " << realtype(constants::R_v / si::joules * si::kelvins * si::kilograms) << endl;
    cout << "R(q_v) = q_v * R_v + (1 - q_v) * R_d" << endl;
    cout << "c_p_d = " << realtype(constants::c_p_d / si::joules * si::kelvins * si::kilograms) << endl;
    cout << "p_v(p, q_v) = p * q_v / (epsilon + q_v * (1 - epsilon))" << endl;
    cout << "q_l(q_v) = " << realtype(initptq->getSpecificHumidity()) << " - q_v" << endl;
    cout << "p_s(T) = " << realtype(constants::p_0 / si::pascals) << " * exp( "
      << realtype((
           (constants::l_v_0 + (constants::c_p_w - constants::c_p_v) * constants::T_0) / constants::R_v
         ) / si::kelvins) << " * ( " << realtype(1. * si::kelvins / constants::T_0) << " - 1. / T) "
      << "-" << realtype((constants::c_p_w - constants::c_p_v) / constants::R_v) 
      << " * log(T / " << realtype(constants::T_0 / si::kelvins) << "))" << endl;
    cout << "rho_w_0 = " << realtype(constants::rho_w_0 / si::kilogrammes * si::cubic_metres) << endl;
  }

#  define read(data, type) \
  if (1 != fread(&data, sizeof(type), 1, tmpf)) \
  { \
    cerr << "failed to read data from the temporary file" << endl; \
    throw exception(); \
  }

  protected: void dataSounding(bool moments = false, 
    quantity<si::length> cloud_min = 0. * si::metres, 
    quantity<si::length> cloud_max = 0. * si::metres,
    bool endline = true
  )
  {
    size_t n = model->getStateVectorIndexRadii();
    realtype tmp;

    for (size_t r = 0; r < offs.size(); r++)
    {
      fseek(tmpf, offs[r], SEEK_SET);
      read(tmp, realtype);
      cout << tmp;
      for (size_t i = 0; i < n; ++i) 
      {
        read(tmp, realtype)
        cout << "\t" << tmp;
      }
      if (moments)
      {
        assert(cloud_max > cloud_min);
        quantity<specific_concentration> 
          m0 = 0. / si::kilogrammes;
        quantity<length_over_mass> 
          m1 = 0. * pow<1>(si::metres) / si::kilogrammes;
        quantity<area_over_mass> 
          m2 = 0. * pow<2>(si::metres) / si::kilogrammes;
        quantity<volume_over_mass> 
          m3 = 0. * pow<3>(si::metres) / si::kilogrammes;
        size_t n_spec;
        read(n_spec, size_t);
        for (size_t s = 0; s < n_spec; ++s)
        {
          size_t n_bins;
          read(n_bins, size_t);
          for (size_t b = 0; b < n_bins; ++b)
          {
            realtype tmp;
            quantity<si::length> r_l, r_r;
            quantity<specific_concentration> N;

            read(tmp, realtype); // r_d_l 
            read(tmp, realtype); // To_l 
            read(tmp, realtype); r_l = tmp * si::metres;
            read(tmp, realtype); N = tmp / si::kilogrammes;
            read(tmp, realtype); // r_d_r
            read(tmp, realtype); // To_r
            read(tmp, realtype); r_r = tmp * si::metres;

            if (r_l > cloud_min && r_r < cloud_max)
            {
              // TODO: move to in-bin spectrum?
              quantity<specific_concentration_density> 
                rho = N / (r_r - r_l);
              quantity<si::length> 
                r_l_adj = max(r_l, cloud_min),
                r_r_adj = min(r_r, cloud_max);
              m0 += rho * 1./1. * (pow<1>(r_r_adj) - pow<1>(r_l_adj));
              m1 += rho * 1./2. * (pow<2>(r_r_adj) - pow<2>(r_l_adj));
              m2 += rho * 1./3. * (pow<3>(r_r_adj) - pow<3>(r_l_adj));
              m3 += rho * 1./4. * (pow<4>(r_r_adj) - pow<4>(r_l_adj));
            }
          }
        }
        cout << "\t" << realtype(m0 * si::kilogrammes);
        cout << "\t" << realtype(m1 * si::kilogrammes / pow<1>(si::metres));
        cout << "\t" << realtype(m2 * si::kilogrammes / pow<2>(si::metres));
        cout << "\t" << realtype(m3 * si::kilogrammes / pow<3>(si::metres));
      }
      if (endline) cout << endl;
    }
  }
  
  protected: size_t n_bins(size_t rec, size_t spec)
  {
    fseek(tmpf, offs[rec] + 4 * sizeof(realtype), SEEK_SET);
    size_t n_spec;
    read(n_spec, size_t);
    for (size_t s = 0; s < n_spec; ++s)
    {
      size_t n_bins;
      read(n_bins, size_t);
      if (spec != s) 
      {
        fseek(tmpf, (n_bins * 7) * sizeof(realtype), SEEK_CUR); // TODO: 7 -> block_size
        continue;
      }
      return n_bins;
    }
  }

  protected: void dataSpectrum(size_t rec, size_t spec, bool only_r_l = false, bool emulate_boxes = false)
  {
    realtype t, p, T, q;
    fseek(tmpf, offs[rec], SEEK_SET);
    read(t, realtype);
    read(p, realtype);
    read(T, realtype)
    read(q, realtype)
    size_t n_spec;
    read(n_spec, size_t);
    for (size_t s = 0; s < n_spec; ++s)
    {
      size_t n_bins;
      read(n_bins, size_t);
      if (spec != s) 
      {
        fseek(tmpf, (n_bins * 7) * sizeof(realtype), SEEK_CUR); // TODO: 5 -> block_size
        continue;
      }
      for (size_t b = 0; b < n_bins; ++b)
      {
        realtype r_l, r_r, N, r_d_l, r_d_r, To_l, To_r;
        read(r_d_l, realtype);
        read(To_l, realtype);
        read(r_l, realtype);
        read(N, realtype);
        read(r_d_r, realtype);
        read(To_r, realtype);
        read(r_r, realtype);
        if (!emulate_boxes)
        {
          cout << t << "\t" << r_l;
          if (!only_r_l) cout << "\t" << N << "\t" << r_r << "\t" << r_d_l << "\t" << r_d_r;
          else cout << "\t" << To_l - T;
        }
        else 
        {
          cout << t << "\t" << r_l << "\t" << N / (r_r - r_l) << endl;
          cout << t << "\t" << r_r << "\t" << N / (r_r - r_l);
        }
        cout << endl;
      }
    }
  }

#  undef read

  protected: size_t numRecords()
  {
    return offs.size();
  }

#  define write(data, type) \
  { \
    type tmp; \
    tmp = data;\
    if (1 != fwrite(&tmp, sizeof(type), 1, tmpf)) \
    { \
      cerr << "failed to write data into the temporary file" << endl; \
      throw exception(); \
    } \
  }

  protected: void record(quantity<si::time> t, N_Vector y, N_Vector p)
  {
    tmpfile_write(t, y, p);
  }

  protected: void tmpfile_write(quantity<si::time> t, N_Vector y, N_Vector p)
  {
    // time
    cerr << msgprefix << "recording @ t = " << t << endl;
    offs.push_back(ftell(tmpf));
    times.push_back(t);
    write(t / si::seconds, realtype);

    // thermodynamic state TODO: the order... 
    write(NV_Ith_S(y, model->getStateVectorIndexPressure()), realtype);
    write(NV_Ith_S(y, model->getStateVectorIndexTemperature()), realtype);
    write(NV_Ith_S(y, model->getStateVectorIndexSpecificHumidity()), realtype);

    size_t n = model->getStateVectorIndexRadii(); // TODO...
    if (!model->isBulk())
    {
      write(ml->n_specs(), size_t);
      for (int s = 0; s < ml->n_specs(); ++s) 
      {
        write(ml->n_bins(s), size_t);
        for (int b = 0; ml->bin_r(s, b) != -1; b = ml->bin_r(s, b))
        {
          write(NV_Ith_S(p, ml->r_d_l_ix(s, b)), realtype);
          write(NV_Ith_S(y, n + ml->To_l_ix(s, b)), realtype);
          write(NV_Ith_S(y, n + ml->r_l_ix(s, b)), realtype);
          write(NV_Ith_S(p, ml->N_ix(s, b)), realtype); 
          write(NV_Ith_S(p, ml->r_d_r_ix(s, b)), realtype);
          write(NV_Ith_S(y, n + ml->To_r_ix(s, b)), realtype);
          write(NV_Ith_S(y, n + ml->r_r_ix(s, b)), realtype);
        }
      }
    }

    T_last = NV_Ith_S(y, model->getStateVectorIndexTemperature()) * si::kelvins;
    p_last = NV_Ith_S(y, model->getStateVectorIndexPressure()) * si::pascals;
    q_last = NV_Ith_S(y, model->getStateVectorIndexSpecificHumidity());
  }
#  undef write

};
#  undef self

#endif
